#!/bin/bash

# log-viewer - Aggregate and filter logs across services
# Usage: log-viewer <command> [options]

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Helper functions
print_success() { echo -e "${GREEN}✓${NC} $1"; }
print_error() { echo -e "${RED}✗${NC} $1"; }
print_info() { echo -e "${BLUE}ℹ${NC} $1"; }
print_warning() { echo -e "${YELLOW}⚠${NC} $1"; }

show_usage() {
    cat << EOF
${BLUE}log-viewer${NC} - Aggregate and filter logs across services

${YELLOW}USAGE:${NC}
    log-viewer <command> [options]

${YELLOW}COMMANDS:${NC}
    tail [service]         Show live logs (default: all services)
    search <pattern>       Search logs for a pattern
    errors [service]       Show only error logs
    warnings [service]     Show only warning logs
    aggregate [minutes]    Aggregate logs from last N minutes (default: 30)
    service <name>         Show logs for specific service
    clear                  Clear old log files
    list                   List available services
    help                   Show this help message

${YELLOW}OPTIONS:${NC}
    -f, --follow          Follow log output (live tail)
    -n, --lines N         Show last N lines (default: 100)
    --since TIME          Show logs since timestamp (e.g., "1h", "30m")
    --level LEVEL         Filter by log level (error, warn, info, debug)
    --json                Output in JSON format

${YELLOW}EXAMPLES:${NC}
    # Tail all services
    log-viewer tail

    # Show last 50 lines from a specific service
    log-viewer service supabase-db -n 50

    # Search for errors in the last hour
    log-viewer search "error" --since 1h

    # Follow error logs
    log-viewer errors --follow

    # Show logs from specific Docker service
    log-viewer tail supabase-kong

${YELLOW}DOCKER SERVICES:${NC}
    • supabase-db         - PostgreSQL database
    • supabase-studio     - Supabase Studio UI
    • supabase-kong       - API Gateway
    • supabase-auth       - Auth server
    • supabase-rest       - PostgREST API
    • supabase-realtime   - Realtime subscriptions
    • supabase-storage    - File storage
EOF
}

# Get Docker compose services
get_services() {
    if [ -f "docker-compose.yml" ]; then
        docker compose ps --services 2>/dev/null || docker-compose ps --services 2>/dev/null || echo ""
    else
        # Look for docker-compose in common locations
        if [ -f "$HOME/dev-environment/docker-compose.yml" ]; then
            cd "$HOME/dev-environment"
            docker compose ps --services 2>/dev/null || docker-compose ps --services 2>/dev/null || echo ""
        else
            echo ""
        fi
    fi
}

# List available services
list_services() {
    print_info "Available Docker services:"
    echo ""
    
    services=$(get_services)
    if [ -z "$services" ]; then
        print_warning "No Docker services found. Make sure docker-compose.yml exists."
        return
    fi
    
    echo "$services" | while read -r service; do
        status=$(docker ps --filter "name=$service" --format "{{.Status}}" 2>/dev/null | head -1)
        if [ -n "$status" ]; then
            echo -e "  ${GREEN}●${NC} $service - ${CYAN}$status${NC}"
        else
            echo -e "  ${RED}○${NC} $service - stopped"
        fi
    done
}

# Tail logs from service(s)
tail_logs() {
    local service=$1
    local follow=${2:-false}
    local lines=${3:-100}
    
    if [ "$service" = "all" ] || [ -z "$service" ]; then
        print_info "Tailing logs from all services..."
        if [ "$follow" = true ]; then
            docker compose logs -f --tail="$lines" 2>/dev/null || docker-compose logs -f --tail="$lines"
        else
            docker compose logs --tail="$lines" 2>/dev/null || docker-compose logs --tail="$lines"
        fi
    else
        print_info "Tailing logs from service: $service"
        if [ "$follow" = true ]; then
            docker compose logs -f --tail="$lines" "$service" 2>/dev/null || docker-compose logs -f --tail="$lines" "$service"
        else
            docker compose logs --tail="$lines" "$service" 2>/dev/null || docker-compose logs --tail="$lines" "$service"
        fi
    fi
}

# Search logs for pattern
search_logs() {
    local pattern=$1
    local since=${2:-"24h"}
    
    print_info "Searching logs for: $pattern (since: $since)"
    
    docker compose logs --since="$since" 2>/dev/null | grep -i --color=always "$pattern" || \
    docker-compose logs --since="$since" 2>/dev/null | grep -i --color=always "$pattern" || \
    print_warning "No matches found for: $pattern"
}

# Show error logs
show_errors() {
    local service=$1
    local follow=${2:-false}
    
    print_info "Showing error logs..."
    
    if [ -z "$service" ]; then
        if [ "$follow" = true ]; then
            docker compose logs -f 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)" || \
            docker-compose logs -f 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)"
        else
            docker compose logs --tail=200 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)" || \
            docker-compose logs --tail=200 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)"
        fi
    else
        if [ "$follow" = true ]; then
            docker compose logs -f "$service" 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)" || \
            docker-compose logs -f "$service" 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)"
        else
            docker compose logs --tail=200 "$service" 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)" || \
            docker-compose logs --tail=200 "$service" 2>/dev/null | grep -i --color=always -E "(error|fail|exception|fatal)"
        fi
    fi
}

# Show warning logs
show_warnings() {
    local service=$1
    local follow=${2:-false}
    
    print_info "Showing warning logs..."
    
    if [ -z "$service" ]; then
        if [ "$follow" = true ]; then
            docker compose logs -f 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)" || \
            docker-compose logs -f 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)"
        else
            docker compose logs --tail=200 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)" || \
            docker-compose logs --tail=200 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)"
        fi
    else
        if [ "$follow" = true ]; then
            docker compose logs -f "$service" 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)" || \
            docker-compose logs -f "$service" 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)"
        else
            docker compose logs --tail=200 "$service" 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)" || \
            docker-compose logs --tail=200 "$service" 2>/dev/null | grep -i --color=always -E "(warn|warning|caution)"
        fi
    fi
}

# Aggregate logs
aggregate_logs() {
    local minutes=${1:-30}
    
    print_info "Aggregating logs from last $minutes minutes..."
    echo ""
    
    services=$(get_services)
    if [ -z "$services" ]; then
        print_error "No services found"
        return 1
    fi
    
    echo "$services" | while read -r service; do
        echo -e "${CYAN}━━━ $service ━━━${NC}"
        
        error_count=$(docker compose logs --since="${minutes}m" "$service" 2>/dev/null | grep -ic "error" || echo "0")
        warn_count=$(docker compose logs --since="${minutes}m" "$service" 2>/dev/null | grep -ic "warn" || echo "0")
        
        echo -e "  Errors:   ${RED}$error_count${NC}"
        echo -e "  Warnings: ${YELLOW}$warn_count${NC}"
        echo ""
    done
}

# Clear old logs
clear_logs() {
    print_warning "This will remove all Docker container logs. Continue? (y/N)"
    read -r confirm
    
    if [ "$confirm" = "y" ] || [ "$confirm" = "Y" ]; then
        print_info "Clearing Docker logs..."
        docker ps -aq | xargs -r docker inspect --format='{{.LogPath}}' | xargs -r sudo truncate -s 0 2>/dev/null || \
        print_warning "Some logs couldn't be cleared (may need sudo)"
        print_success "Logs cleared"
    else
        print_info "Cancelled"
    fi
}

# Parse command line arguments
FOLLOW=false
LINES=100
SINCE=""
SERVICE=""

command="${1:-help}"
shift || true

while [ $# -gt 0 ]; do
    case $1 in
        -f|--follow)
            FOLLOW=true
            shift
            ;;
        -n|--lines)
            LINES=$2
            shift 2
            ;;
        --since)
            SINCE=$2
            shift 2
            ;;
        *)
            SERVICE=$1
            shift
            ;;
    esac
done

# Main script logic
case "$command" in
    tail)
        tail_logs "${SERVICE:-all}" "$FOLLOW" "$LINES"
        ;;
    search)
        [ -z "$SERVICE" ] && { print_error "Search pattern required"; show_usage; exit 1; }
        search_logs "$SERVICE" "${SINCE:-24h}"
        ;;
    errors)
        show_errors "$SERVICE" "$FOLLOW"
        ;;
    warnings)
        show_warnings "$SERVICE" "$FOLLOW"
        ;;
    aggregate)
        aggregate_logs "${SERVICE:-30}"
        ;;
    service)
        [ -z "$SERVICE" ] && { print_error "Service name required"; show_usage; exit 1; }
        tail_logs "$SERVICE" "$FOLLOW" "$LINES"
        ;;
    clear)
        clear_logs
        ;;
    list)
        list_services
        ;;
    help|--help|-h)
        show_usage
        ;;
    *)
        print_error "Unknown command: $command"
        show_usage
        exit 1
        ;;
esac
