# lzc-build.yml 构建配置规范

## 概述

`lzc-build.yml` 是懒猫微服应用的构建配置文件，定义了如何构建应用包（.lpk 文件）以及开发环境的配置。

本文档详细介绍其结构和各字段的含义。

## 快速导航

- [基础配置](#基础配置) - buildscript、manifest、contentdir 等必填字段
- [开发环境配置](#开发环境配置-devshell) - devshell 开发依赖和路由
- [高级配置](#高级配置) - compose override 特性

---

## 基础配置

### 必填字段

```yaml
buildscript: sh build.sh        # 构建脚本
manifest: ./lzc-manifest.yml    # manifest 文件路径
contentdir: ./dist              # 打包内容目录
pkgout: ./                      # 输出路径
icon: ./lzc-icon.png            # 应用图标
```

### 字段说明

| 字段名 | 类型 | 是否必填 | 描述 |
|--------|------|----------|------|
| `buildscript` | string | ✅ | 构建脚本路径或 shell 命令 |
| `manifest` | string | ✅ | lzc-manifest.yml 文件路径 |
| `contentdir` | string | ✅ | 打包内容目录（会被打包到 lpk 中） |
| `pkgout` | string | ✅ | lpk 包的输出路径 |
| `icon` | string | ⚠️ | 应用图标路径（仅允许 PNG 格式） |
| `devshell` | DevshellConfig | ❌ | 开发环境配置 |
| `compose_override` | ComposeOverrideConfig | ❌ | 高级 compose 配置（v1.3.0+） |

**注意**:
- 如果不指定 `icon`，构建时会发出警告
- 商店应用必须提供图标

---

## 构建脚本 (buildscript)

### 基本用法

#### 方式 1: 脚本文件路径

```yaml
buildscript: sh build.sh
```

**build.sh 示例**:
```bash
#!/bin/bash
set -e

# 安装依赖
cd ui
npm install

# 构建生产版本
npm run build

# 返回根目录
cd ..
```

#### 方式 2: 直接写命令

```yaml
buildscript: npm run build && cp -r dist/* ./output/
```

### 常见构建脚本示例

#### 前端应用 (Vue/React)

```bash
#!/bin/bash
set -e

cd ui
npm install
npm run build
cd ..
```

#### 前后端分离应用

```bash
#!/bin/bash
set -e

# 构建前端
cd frontend
npm install
npm run build
cd ..

# 构建后端
cd backend
npm install
npm run build
cd ..

# 准备打包目录
mkdir -p dist/frontend
mkdir -p dist/backend

cp -r frontend/dist/* dist/frontend/
cp -r backend/dist/* dist/backend/

# 创建启动脚本
cat > dist/backend/start.sh <<'EOF'
#!/bin/sh
cd /lzcapp/pkg/content/backend
node index.js
EOF
chmod +x dist/backend/start.sh
```

#### Go 应用

```bash
#!/bin/bash
set -e

# 构建前端
cd ui
npm install
npm run build
cd ..

# 构建 Go 后端（静态编译）
cd backend
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o server main.go
cd ..

# 准备打包目录
mkdir -p dist/ui
mkdir -p dist/backend

cp -r ui/dist/* dist/ui/
cp backend/server dist/backend/
chmod +x dist/backend/server
```

#### Python 应用

```bash
#!/bin/bash
set -e

# 构建前端
cd ui
npm install
npm run build
cd ..

# 准备打包目录
mkdir -p dist/ui
mkdir -p dist/backend

cp -r ui/dist/* dist/ui/
cp -r backend/* dist/backend/

# 创建启动脚本
cat > dist/backend/run.sh <<'EOF'
#!/bin/sh
cd /lzcapp/pkg/content/backend

# 安装 Python 依赖（首次运行）
if [ ! -f "/lzcapp/var/.deps_installed" ]; then
    pip install -r requirements.txt --break-system-packages \
        --index-url https://pypi.tuna.tsinghua.edu.cn/simple
    touch /lzcapp/var/.deps_installed
fi

# 启动应用
python main.py
EOF
chmod +x dist/backend/run.sh
```

---

## 打包内容目录 (contentdir)

### 路径说明

`contentdir` 指定的目录内容会被打包到 lpk 中，在运行时映射到容器的 `/lzcapp/pkg/content/` 目录（只读）。

### 常见目录结构

#### 纯静态应用

```yaml
contentdir: ./dist
```

构建后:
```
dist/
├── index.html
├── assets/
│   ├── main.js
│   └── style.css
└── favicon.ico
```

运行时访问: `/lzcapp/pkg/content/index.html`

#### 前后端分离应用

```yaml
contentdir: ./output
```

构建后:
```
output/
├── frontend/          # 静态文件
│   └── dist/
├── backend/           # 后端代码
│   ├── server.js
│   └── start.sh
└── README.md
```

运行时:
- 前端: `/lzcapp/pkg/content/frontend/dist/`
- 后端: `/lzcapp/pkg/content/backend/start.sh`

---

## 开发环境配置 (devshell)

### 概述

`devshell` 配置定义了开发环境的依赖和路由，在执行 `lzc-cli project devshell` 时生效。

### 字段说明

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `routes` | []string | 开发路由规则 |
| `dependencies` | []string | 开发依赖包（自动安装） |
| `setupscript` | string | 手动安装脚本 |
| `image` | string | 自定义基础镜像 |
| `pull_policy` | string | 镜像拉取策略（`build` 时从 Dockerfile 构建） |
| `build` | string | Dockerfile 文件路径 |

**注意**: 如果 `dependencies` 和 `build` 同时存在，优先使用 `dependencies`。

### 基本示例

```yaml
devshell:
  # 开发路由
  routes:
    - /=http://127.0.0.1:5173      # 前端开发服务器
    - /api/=http://127.0.0.1:3000  # 后端开发服务器

  # 依赖包
  dependencies:
    - nodejs
    - npm
    - python3
    - py3-pip

  # 环境设置
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
```

### 路由配置

开发路由将请求转发到本地开发服务器。

#### 前端开发（Vite）

```yaml
devshell:
  routes:
    - /=http://127.0.0.1:5173  # Vite 默认端口
  dependencies:
    - nodejs
    - npm
```

开发流程:
```bash
lzc-cli project devshell
# 进入容器后
npm install
npm run dev
```

#### 前后端开发

```yaml
devshell:
  routes:
    - /=http://127.0.0.1:5173      # 前端
    - /api/=http://127.0.0.1:8080  # 后端
  dependencies:
    - nodejs
    - npm
    - python3
    - py3-pip
```

开发流程（需要两个终端）:
```bash
# 终端 1: 前端
lzc-cli project devshell
cd ui
npm install && npm run dev

# 终端 2: 后端
lzc-cli project devshell
cd backend
pip install -r requirements.txt --break-system-packages
python main.py
```

### 依赖安装

#### 使用 dependencies 自动安装

```yaml
devshell:
  dependencies:
    - nodejs      # Node.js 运行时
    - npm         # npm 包管理器
    - python3     # Python 3
    - py3-pip     # pip 包管理器
    - git         # Git 版本控制
    - curl        # HTTP 工具
    - go          # Go 语言
```

**常用依赖包**:
- 前端: `nodejs`, `npm`, `yarn`
- Python: `python3`, `py3-pip`
- Go: `go`
- 工具: `git`, `curl`, `wget`, `vim`

#### 使用 setupscript 手动配置

```yaml
devshell:
  dependencies:
    - nodejs
    - npm
  setupscript: |
    # 配置 npm 镜像源
    export npm_config_registry=https://registry.npmmirror.com

    # 配置 pip 镜像源
    export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple

    # 配置 Go 代理
    export GOPROXY=https://goproxy.cn

    # 安装全局工具
    npm install -g pnpm
```

### 自定义镜像

#### 方式 1: 使用现成镜像

```yaml
devshell:
  image: registry.lazycat.cloud/golang:1.21-alpine
  routes:
    - /=http://127.0.0.1:8080
```

#### 方式 2: 从 Dockerfile 构建

**lzc-build.yml**:
```yaml
devshell:
  image: ${package}-devshell:${version}
  pull_policy: build
  build: ./Dockerfile.dev
  routes:
    - /=http://127.0.0.1:3000
```

**Dockerfile.dev**:
```dockerfile
FROM node:18-alpine

# 安装额外依赖
RUN apk add --no-cache git python3 make g++

# 配置镜像源
RUN npm config set registry https://registry.npmmirror.com

WORKDIR /workspace
```

### 完整示例

#### Vue 3 + TypeScript

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./ui/dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173
  dependencies:
    - nodejs
    - npm
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    cd ui
    npm install
```

#### Python Flask + Vue

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173      # 前端
    - /api/=http://127.0.0.1:8080  # 后端
  dependencies:
    - nodejs
    - npm
    - python3
    - py3-pip
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
```

#### Go + React

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  image: registry.lazycat.cloud/golang:1.21-alpine
  routes:
    - /=http://127.0.0.1:5173
    - /api/=http://127.0.0.1:8080
  dependencies:
    - nodejs
    - npm
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    export GOPROXY=https://goproxy.cn
```

---

## 高级配置

### Compose Override (v1.3.0+)

`compose_override` 用于配置 lpk 规范目前无法覆盖的运行权限需求。

**需求**:
- lzc-cli >= 1.2.61
- lzc-os >= 1.3.0

### 配置示例

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

# Compose Override 配置
compose_override:
  services:
    # 指定服务名称
    some_container:
      # 删除特定 capability
      cap_drop:
        - SETCAP
        - MKNOD

      # 挂载宿主机文件（只读）
      volumes:
        - /data/playground:/lzcapp/run/playground:ro
```

### 支持的配置

基本上支持所有 docker-compose 的配置项，常用的有：

- `cap_drop` / `cap_add` - 容器权限控制
- `volumes` - 额外的卷挂载
- `devices` - 设备挂载
- `security_opt` - 安全选项
- `sysctls` - 内核参数

详见 [Compose Override 文档](https://developer.lazycat.cloud/advanced-compose-override.html)。

---

## 模板变量

在 lzc-build.yml 中可以使用 manifest 中定义的变量：

```yaml
# lzc-manifest.yml
package: com.example.myapp
version: 1.0.0

# lzc-build.yml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./${package}-v${version}.lpk  # 使用变量

devshell:
  image: ${package}-devshell:${version}
```

---

## 完整示例

### 纯前端应用

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./ui/dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173
  dependencies:
    - nodejs
    - npm
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
```

### 全栈应用

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173
    - /api/=http://127.0.0.1:3000
  dependencies:
    - nodejs
    - npm
    - python3
    - py3-pip
    - postgresql-client  # 用于调试数据库
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
```

### 使用自定义 Dockerfile

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  image: ${package}-devshell:${version}
  pull_policy: build
  build: ./Dockerfile.dev
  routes:
    - /=http://127.0.0.1:3000
  setupscript: |
    npm install
```

---

## 最佳实践

### 1. 构建脚本

**使用脚本文件而非内联命令**:
```yaml
# ✅ 推荐
buildscript: sh build.sh

# ❌ 不推荐（复杂命令难以维护）
buildscript: cd ui && npm install && npm run build && cd .. && mkdir dist && cp -r ui/dist/* dist/
```

### 2. 依赖安装

**使用国内镜像源**:
```yaml
setupscript: |
  export npm_config_registry=https://registry.npmmirror.com
  export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
  export GOPROXY=https://goproxy.cn
```

### 3. 开发路由

**前后端分离时使用不同端口**:
```yaml
routes:
  - /=http://127.0.0.1:5173      # 前端
  - /api/=http://127.0.0.1:3000  # 后端
```

### 4. 图标要求

- 格式: PNG
- 推荐尺寸: 512x512 或 1024x1024
- 背景: 透明或纯色
- 文件大小: < 1MB

### 5. contentdir 目录

**只包含必要文件**:
```bash
# build.sh
npm run build
# 只复制构建产物，不包含源码
cp -r dist/* output/
```

---

## 调试技巧

### 1. 查看构建日志

```bash
lzc-cli project build -o app.lpk
# 查看详细输出
```

### 2. 验证打包内容

```bash
# 解压 lpk 文件
mkdir temp
cd temp
tar -xzf ../app.lpk
# 查看 content 目录
ls -la content/
```

### 3. 测试开发环境

```bash
lzc-cli project devshell
# 验证依赖是否安装
which node
which python3
npm --version
```

---

## 常见问题

### 1. 构建失败

**问题**: `buildscript` 执行失败

**排查**:
1. 检查脚本是否有执行权限: `chmod +x build.sh`
2. 检查脚本语法: `bash -n build.sh`
3. 手动执行脚本查看错误: `sh build.sh`

### 2. 依赖安装失败

**问题**: npm/pip 安装超时

**解决方案**:
```yaml
setupscript: |
  # 使用国内镜像源
  export npm_config_registry=https://registry.npmmirror.com
  export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
```

### 3. 图标警告

**问题**: 构建时提示缺少图标

**解决方案**:
1. 准备 PNG 格式图标
2. 在 lzc-build.yml 中指定:
```yaml
icon: ./lzc-icon.png
```

### 4. devshell 路由不工作

**问题**: 开发环境无法访问

**排查**:
1. 确认开发服务器是否启动
2. 检查端口是否匹配:
```yaml
routes:
  - /=http://127.0.0.1:5173  # Vite 默认 5173
```
3. 查看容器日志

---

## 相关文档

- [Manifest 完整规范](./manifest-spec.md)
- [应用配置示例](./examples.md)
- [路由配置详解](./route-patterns.md)
- [开发环境依赖安装](https://developer.lazycat.cloud/devshell-install-and-use.html)
- [Compose Override 文档](https://developer.lazycat.cloud/advanced-compose-override.html)
