# Docker 应用迁移完全指南

本文档详细介绍如何将现有的 Docker 和 Docker Compose 应用迁移到懒猫微服平台。

## 目录

- [迁移概述](#迁移概述)
- [Docker 命令转换](#docker-命令转换)
- [Docker Compose 转换](#docker-compose-转换)
- [参数映射对照表](#参数映射对照表)
- [存储路径转换规则](#存储路径转换规则)
- [环境变量处理](#环境变量处理)
- [端口映射策略](#端口映射策略)
- [完整迁移案例](#完整迁移案例)
- [常见问题](#常见问题)

---

## 迁移概述

### 迁移流程

```
1. 找到 Docker 启动方式
   ├─ Docker 命令
   └─ docker-compose.yml

2. 分析关键参数
   ├─ 镜像名称和版本
   ├─ 环境变量
   ├─ 端口映射
   └─ 存储卷

3. 转换为懒猫配置
   ├─ lzc-manifest.yml
   └─ lzc-build.yml

4. 测试和调试
   └─ 本地安装验证
```

### 核心概念对应关系

| Docker 概念 | 懒猫微服概念 | 配置位置 |
|-------------|--------------|----------|
| Container | Service | `services.{name}` |
| Volume | Binds | `services.{name}.binds` |
| Environment | Environment | `services.{name}.environment` |
| Port | Routes/Ingress | `application.routes` 或 `application.ingress` |
| Network | 自动管理 | 无需配置（自动创建网络） |

---

## Docker 命令转换

### 步骤 1: 找到 Docker 命令

以 GitLab 为例，官方文档提供的安装命令：

```bash
sudo docker run --detach \
  --hostname gitlab.example.com \
  --env GITLAB_OMNIBUS_CONFIG="external_url 'http://gitlab.example.com'" \
  --publish 443:443 --publish 80:80 --publish 22:22 \
  --name gitlab \
  --restart always \
  --volume $GITLAB_HOME/config:/etc/gitlab \
  --volume $GITLAB_HOME/logs:/var/log/gitlab \
  --volume $GITLAB_HOME/data:/var/opt/gitlab \
  --shm-size 256m \
  gitlab/gitlab-ee:17.2.8-ee.0
```

### 步骤 2: 提取关键参数

| Docker 参数 | 值 | 用途 |
|-------------|-----|------|
| 镜像 | `gitlab/gitlab-ee:17.2.8-ee.0` | services.{name}.image |
| --env | `GITLAB_OMNIBUS_CONFIG=...` | services.{name}.environment |
| --publish | `443:443`, `80:80`, `22:22` | application.routes / ingress |
| --volume | `$GITLAB_HOME/config:/etc/gitlab` | services.{name}.binds |
| --shm-size | `256m` | services.{name}.shm_size |

### 步骤 3: 转换为 lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
name: GitLab
package: cloud.lazycat.app.gitlab
version: 17.2.8-patch1

application:
  subdomain: gitlab

  # HTTP 端口（80） → routes
  routes:
    - /=http://gitlab:80

  # SSH 端口（22） → ingress
  ingress:
    - protocol: tcp
      port: 22
      service: gitlab
      description: GitLab SSH 服务

services:
  gitlab:
    # 镜像
    image: gitlab/gitlab-ee:17.2.8

    # 环境变量
    environment:
      - GITLAB_OMNIBUS_CONFIG=external_url 'http://gitlab.${LAZYCAT_BOX_DOMAIN}'; gitlab_rails['lfs_enabled'] = true;

    # 存储卷（路径必须 /lzcapp 开头）
    binds:
      - /lzcapp/var/config:/etc/gitlab
      - /lzcapp/var/logs:/var/log/gitlab
      - /lzcapp/var/data:/var/opt/gitlab

    # 共享内存
    shm_size: 256M
```

### 步骤 4: 创建 lzc-build.yml

```yaml
pkgout: ./
icon: ./icon.png
manifest: ./lzc-manifest.yml
```

### 步骤 5: 准备图标并安装

```bash
# 下载图标（PNG 格式）
wget -O icon.png https://about.gitlab.com/images/press/logo/png/gitlab-icon-rgb.png

# 构建应用包
lzc-cli project build -o gitlab.lpk

# 安装到微服
lzc-cli app install gitlab.lpk
```

---

## Docker Compose 转换

### Docker Compose 文件示例

还是以 GitLab 为例，官方的 docker-compose.yml：

```yaml
version: '3.6'
services:
  gitlab:
    image: gitlab/gitlab-ee:17.2.8-ee.0
    container_name: gitlab
    restart: always
    hostname: 'gitlab.example.com'
    environment:
      GITLAB_OMNIBUS_CONFIG: |
        external_url 'https://gitlab.example.com'
    ports:
      - '80:80'
      - '443:443'
      - '22:22'
    volumes:
      - '$GITLAB_HOME/config:/etc/gitlab'
      - '$GITLAB_HOME/logs:/var/log/gitlab'
      - '$GITLAB_HOME/data:/var/opt/gitlab'
    shm_size: '256m'
```

### 转换步骤

#### 1. 识别层次结构

docker-compose.yml 的 `services` 结构与 lzc-manifest.yml 非常相似：

```
docker-compose.yml           lzc-manifest.yml
├── services                 ├── services
│   ├── gitlab              │   ├── gitlab
│   │   ├── image           │   │   ├── image
│   │   ├── environment     │   │   ├── environment
│   │   ├── ports           │   │   └── ...
│   │   └── volumes         │   └── ...
```

#### 2. 字段映射

| docker-compose 字段 | lzc-manifest 字段 | 说明 |
|---------------------|-------------------|------|
| `image` | `services.{name}.image` | 直接复制 |
| `environment` | `services.{name}.environment` | 转为列表格式 |
| `ports` | `application.routes` 或 `ingress` | HTTP → routes，TCP/UDP → ingress |
| `volumes` | `services.{name}.binds` | 路径必须 /lzcapp 开头 |
| `depends_on` | `services.{name}.depends_on` | 直接复制 |
| `mem_limit` | `services.{name}.mem_limit` | 直接复制 |
| `cpus` | `services.{name}.cpus` | 直接复制 |

#### 3. 转换后的 lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.gitlab
version: 17.2.8
name: GitLab
description: 企业级 DevOps 平台

application:
  subdomain: gitlab

  routes:
    - /=http://gitlab:80

  ingress:
    - protocol: tcp
      port: 22
      service: gitlab
      description: SSH 服务

services:
  gitlab:
    image: gitlab/gitlab-ee:17.2.8
    environment:
      # docker-compose 的多行环境变量转为单行
      - GITLAB_OMNIBUS_CONFIG=external_url 'http://gitlab.${LAZYCAT_BOX_DOMAIN}';
    binds:
      # 将 volumes 转为 binds，路径改为 /lzcapp 开头
      - /lzcapp/var/config:/etc/gitlab
      - /lzcapp/var/logs:/var/log/gitlab
      - /lzcapp/var/data:/var/opt/gitlab
    shm_size: 256M
```

---

## 参数映射对照表

### 容器配置

| Docker 参数 | docker-compose 字段 | lzc-manifest 字段 | 示例 |
|-------------|---------------------|-------------------|------|
| `--image` | `image` | `services.{name}.image` | `nginx:alpine` |
| `--name` | `container_name` | services 的 key | `myapp` |
| `--env` | `environment` | `services.{name}.environment` | `NODE_ENV=production` |
| `--entrypoint` | `entrypoint` | `services.{name}.entrypoint` | `/bin/sh` |
| `--command` | `command` | `services.{name}.command` | `nginx -g 'daemon off;'` |
| `--workdir` | `working_dir` | `services.{name}.workdir` | `/app` |
| `--user` | `user` | `services.{name}.user` | `1000:1000` |

### 资源限制

| Docker 参数 | docker-compose 字段 | lzc-manifest 字段 | 示例 |
|-------------|---------------------|-------------------|------|
| `--memory` | `mem_limit` | `services.{name}.mem_limit` | `1G` 或 `1073741824` |
| `--cpus` | `cpus` | `services.{name}.cpus` | `1.5` |
| `--cpu-shares` | `cpu_shares` | `services.{name}.cpu_shares` | `512` |
| `--shm-size` | `shm_size` | `services.{name}.shm_size` | `256M` |

### 网络和端口

| Docker 参数 | docker-compose 字段 | lzc-manifest 字段 | 说明 |
|-------------|---------------------|-------------------|------|
| `--publish 80:80` | `ports: ["80:80"]` | `application.routes` | HTTP 服务 → routes |
| `--publish 22:22` | `ports: ["22:22"]` | `application.ingress` | TCP/UDP → ingress |
| `--network host` | `network_mode: host` | `services.{name}.network_mode` | 值为 `host` 或留空 |

### 存储

| Docker 参数 | docker-compose 字段 | lzc-manifest 字段 | 说明 |
|-------------|---------------------|-------------------|------|
| `--volume /host:/container` | `volumes: ["/host:/container"]` | `services.{name}.binds` | 路径必须 /lzcapp 开头 |
| `--tmpfs /tmp` | `tmpfs: ["/tmp"]` | `services.{name}.tmpfs` | 临时文件系统 |

### 健康检查

| Docker 参数 | docker-compose 字段 | lzc-manifest 字段 |
|-------------|---------------------|-------------------|
| `--health-cmd` | `healthcheck.test` | `services.{name}.healthcheck.test` |
| `--health-interval` | `healthcheck.interval` | `services.{name}.healthcheck.interval` |
| `--health-timeout` | `healthcheck.timeout` | `services.{name}.healthcheck.timeout` |
| `--health-retries` | `healthcheck.retries` | `services.{name}.healthcheck.retries` |
| `--health-start-period` | `healthcheck.start_period` | `services.{name}.healthcheck.start_period` |

---

## 存储路径转换规则

### 核心原则

容器的 rootfs 在重启后会丢失，只有以下路径的数据会永久保留：
- `/lzcapp/var` - 永久存储
- `/lzcapp/cache` - 缓存存储

### 转换规则

```
Docker Volume                    →  懒猫微服 Binds
$GITLAB_HOME/config:/etc/gitlab  →  /lzcapp/var/config:/etc/gitlab
$GITLAB_HOME/data:/var/lib/mysql →  /lzcapp/var/mysql:/var/lib/mysql
/tmp/cache:/app/cache            →  /lzcapp/cache/tmp:/app/cache
```

### 路径选择指南

| 数据类型 | 推荐路径 | 示例 |
|----------|----------|------|
| 数据库文件 | `/lzcapp/var` | `/lzcapp/var/mysql:/var/lib/mysql` |
| 应用配置 | `/lzcapp/var` | `/lzcapp/var/config:/etc/myapp` |
| 用户上传文件 | `/lzcapp/var` | `/lzcapp/var/uploads:/app/uploads` |
| 重要日志 | `/lzcapp/var` | `/lzcapp/var/logs:/var/log/app` |
| 缓存数据 | `/lzcapp/cache` | `/lzcapp/cache/tmp:/app/cache` |
| 临时文件 | `/lzcapp/cache` | `/lzcapp/cache/temp:/tmp` |
| 可重建的数据 | `/lzcapp/cache` | `/lzcapp/cache/thumbnails:/app/thumbs` |

### 特殊路径（只读）

| 路径 | 说明 | 使用示例 |
|------|------|----------|
| `/lzcapp/pkg/content/` | lpk 包内容（只读） | 静态文件、可执行文件 |
| `/lzcapp/run/manifest.yml` | 渲染后的 manifest | 调试时查看最终配置 |
| `/lzcapp/run/mnt/home` | 文稿目录（需开启） | `ext_config.enable_document_access` |
| `/lzcapp/run/mnt/media` | 媒体目录（需开启） | `ext_config.enable_media_access` |

### 完整示例

```yaml
services:
  wordpress:
    image: wordpress:6.0
    binds:
      # 永久存储：WordPress 文件和插件
      - /lzcapp/var/wordpress:/var/www/html

      # 缓存存储：临时文件
      - /lzcapp/cache/tmp:/tmp

  mysql:
    image: mysql:8
    binds:
      # 永久存储：数据库数据
      - /lzcapp/var/mysql:/var/lib/mysql

  redis:
    image: redis:7
    binds:
      # 缓存存储：Redis 数据（可接受丢失）
      - /lzcapp/cache/redis:/data
```

---

## 环境变量处理

### 格式转换

#### Docker 命令格式
```bash
--env NODE_ENV=production \
--env DB_HOST=mysql \
--env DB_PASSWORD=secret123
```

#### docker-compose.yml 格式
```yaml
environment:
  NODE_ENV: production
  DB_HOST: mysql
  DB_PASSWORD: secret123
```

#### lzc-manifest.yml 格式
```yaml
environment:
  - NODE_ENV=production
  - DB_HOST=mysql
  - DB_PASSWORD=secret123
```

### 系统变量引用

懒猫微服提供的系统变量：

```yaml
environment:
  # 微服域名
  - APP_URL=https://${SUBDOMAIN}.${LAZYCAT_BOX_DOMAIN}

  # Package ID
  - PACKAGE_ID=${PACKAGE_ID}

  # 应用版本
  - APP_VERSION=${VERSION}
```

### 稳定密码生成

**不要硬编码密码**:
```yaml
environment:
  - DB_PASSWORD=mysecret123  # ❌ 不安全
```

**使用 stable_secret**:
```yaml
environment:
  - DB_PASSWORD={{ stable_secret "db_password" }}  # ✅ 安全
  - REDIS_PASSWORD={{ stable_secret "redis_password" | substr 0 16 }}  # 限制长度
```

### 多行环境变量

docker-compose 的多行格式需要转为单行：

**docker-compose.yml**:
```yaml
environment:
  GITLAB_OMNIBUS_CONFIG: |
    external_url 'https://gitlab.example.com'
    gitlab_rails['lfs_enabled'] = true
```

**lzc-manifest.yml**:
```yaml
environment:
  - GITLAB_OMNIBUS_CONFIG=external_url 'https://gitlab.example.com'; gitlab_rails['lfs_enabled'] = true;
```

---

## 端口映射策略

### HTTP/HTTPS 服务（ports 80/443）

使用 `application.routes` 配置。

#### 单容器 HTTP 服务

```yaml
# docker-compose.yml
ports:
  - "80:80"

# lzc-manifest.yml
application:
  routes:
    - /=http://service-name:80
```

#### 多容器 HTTP 服务

```yaml
# docker-compose.yml
services:
  frontend:
    ports: ["80:80"]
  backend:
    ports: ["3000:3000"]

# lzc-manifest.yml
application:
  routes:
    - /=http://frontend:80
    - /api/=http://backend:3000
```

### TCP/UDP 服务（非 HTTP）

使用 `application.ingress` 配置。

```yaml
# docker-compose.yml
ports:
  - "22:22"   # SSH
  - "3306:3306"  # MySQL

# lzc-manifest.yml
application:
  ingress:
    - protocol: tcp
      port: 22
      service: gitlab
      description: SSH 服务

    - protocol: tcp
      port: 3306
      service: mysql
      description: MySQL 数据库
```

### 端口范围

```yaml
application:
  ingress:
    - protocol: udp
      port: 27015
      service: game-server
      publish_port: "27015-27020"  # 允许的端口范围
```

---

## 完整迁移案例

### 案例 1: WordPress + MySQL

#### 原 docker-compose.yml

```yaml
version: '3'
services:
  wordpress:
    image: wordpress:6.0
    ports:
      - "80:80"
    environment:
      WORDPRESS_DB_HOST: mysql:3306
      WORDPRESS_DB_USER: wordpress
      WORDPRESS_DB_PASSWORD: secret123
      WORDPRESS_DB_NAME: wordpress
    volumes:
      - wordpress_data:/var/www/html
    depends_on:
      - mysql

  mysql:
    image: mysql:8
    environment:
      MYSQL_ROOT_PASSWORD: rootsecret
      MYSQL_DATABASE: wordpress
      MYSQL_USER: wordpress
      MYSQL_PASSWORD: secret123
    volumes:
      - mysql_data:/var/lib/mysql

volumes:
  wordpress_data:
  mysql_data:
```

#### 转换后的 lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.wordpress
version: 6.0.0
name: WordPress
description: 强大的内容管理系统

application:
  subdomain: blog

  # HTTP 路由
  routes:
    - /=http://wordpress:80

services:
  wordpress:
    image: registry.lazycat.cloud/wordpress:6.0
    depends_on:
      - mysql
    environment:
      - WORDPRESS_DB_HOST=mysql:3306
      - WORDPRESS_DB_USER=wordpress
      # 使用稳定密码
      - WORDPRESS_DB_PASSWORD={{ stable_secret "wp_db_password" }}
      - WORDPRESS_DB_NAME=wordpress
    binds:
      - /lzcapp/var/wordpress:/var/www/html
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost"]
      interval: 30s
      timeout: 10s

  mysql:
    image: registry.lazycat.cloud/mysql:8
    environment:
      - MYSQL_ROOT_PASSWORD={{ stable_secret "mysql_root_password" }}
      - MYSQL_DATABASE=wordpress
      - MYSQL_USER=wordpress
      - MYSQL_PASSWORD={{ stable_secret "wp_db_password" }}
    binds:
      - /lzcapp/var/mysql:/var/lib/mysql
    mem_limit: 1G
    shm_size: 256M
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      interval: 10s
```

---

### 案例 2: Nextcloud

#### 原 docker 命令

```bash
docker run -d \
  --name nextcloud \
  -p 8080:80 \
  -v nextcloud:/var/www/html \
  -e MYSQL_HOST=mysql \
  -e MYSQL_DATABASE=nextcloud \
  -e MYSQL_USER=nextcloud \
  -e MYSQL_PASSWORD=secret \
  nextcloud:latest
```

#### 转换后的 lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.nextcloud
version: 1.0.0
name: Nextcloud
description: 私有云存储

application:
  subdomain: cloud
  routes:
    - /=http://nextcloud:80

services:
  nextcloud:
    image: registry.lazycat.cloud/nextcloud:latest
    environment:
      - MYSQL_HOST=mysql
      - MYSQL_DATABASE=nextcloud
      - MYSQL_USER=nextcloud
      - MYSQL_PASSWORD={{ stable_secret "nextcloud_db_password" }}
    binds:
      - /lzcapp/var/nextcloud:/var/www/html
    depends_on:
      - mysql

  mysql:
    image: registry.lazycat.cloud/mysql:8
    environment:
      - MYSQL_ROOT_PASSWORD={{ stable_secret "mysql_root_password" }}
      - MYSQL_DATABASE=nextcloud
      - MYSQL_USER=nextcloud
      - MYSQL_PASSWORD={{ stable_secret "nextcloud_db_password" }}
    binds:
      - /lzcapp/var/mysql:/var/lib/mysql
```

---

### 案例 3: Portainer

#### 原 docker 命令

```bash
docker run -d \
  -p 9000:9000 \
  --name portainer \
  --restart always \
  -v /var/run/docker.sock:/var/run/docker.sock \
  -v portainer_data:/data \
  portainer/portainer-ce:latest
```

#### 转换后的 lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.portainer
version: 1.0.0
name: Portainer
description: Docker 管理界面

application:
  subdomain: portainer
  routes:
    - /=http://portainer:9000

services:
  portainer:
    image: registry.lazycat.cloud/portainer/portainer-ce:latest
    binds:
      # Docker socket（特殊需求，需要管理容器）
      - /var/run/docker.sock:/var/run/docker.sock
      # 数据存储
      - /lzcapp/var/portainer:/data
    healthcheck:
      test: ["CMD", "wget", "--spider", "-q", "http://localhost:9000"]
      interval: 30s
```

---

## 常见问题

### 1. 镜像拉取失败

**问题**: 国内网络无法拉取 Docker Hub 镜像

**解决方案**:

使用 `lzc-cli appstore copy-image` 推送到官方仓库：

```bash
# 上传镜像
lzc-cli appstore copy-image wordpress:6.0

# 输出: registry.lazycat.cloud/yourname/wordpress:hash

# 更新 manifest
image: registry.lazycat.cloud/yourname/wordpress:hash
```

或使用国内镜像（如有）。

### 2. 路径权限问题

**问题**: 容器内无法写入 `/lzcapp/var` 目录

**原因**: 容器使用非 root 用户运行

**解决方案**:

方案 1: 使用 setup_script 修改权限
```yaml
services:
  myapp:
    setup_script: |
      chown -R 1000:1000 /lzcapp/var/myapp
```

方案 2: 指定运行用户
```yaml
services:
  myapp:
    user: "0:0"  # root 用户
```

### 3. 服务依赖启动顺序

**问题**: 应用在数据库未就绪时启动失败

**解决方案**:

使用 `depends_on` + 健康检查：

```yaml
services:
  app:
    depends_on:
      - mysql  # 等待 mysql 健康

  mysql:
    healthcheck:
      test: ["CMD", "mysqladmin", "ping"]
      interval: 5s
      retries: 10
```

### 4. 网络连接问题

**问题**: 容器之间无法通信

**解决方案**:

使用服务名称作为 hostname：

```yaml
environment:
  # ✅ 正确
  - DB_HOST=mysql

  # ❌ 错误
  - DB_HOST=127.0.0.1
  - DB_HOST=localhost
```

### 5. 环境变量未生效

**问题**: 容器内读取不到环境变量

**检查清单**:

1. **格式正确**:
```yaml
# ✅ 正确
environment:
  - KEY=VALUE

# ❌ 错误（YAML 对象格式）
environment:
  KEY: VALUE
```

2. **转义特殊字符**:
```yaml
environment:
  - PASSWORD=my\$ecret  # 转义 $
```

3. **验证**:
```bash
lzc-cli docker exec -it <container> sh
echo $KEY
```

### 6. 数据库初始化失败

**问题**: MySQL/PostgreSQL 首次启动失败

**原因**: 启动时间过长，健康检查超时

**解决方案**:

增加启动等待时间：

```yaml
healthcheck:
  test: ["CMD", "mysqladmin", "ping"]
  interval: 10s
  timeout: 5s
  retries: 10
  start_period: 60s  # 给数据库足够的初始化时间
```

---

## 社区工具

### docker2lzc 转换工具

社区提供的自动转换工具：

```bash
# 安装
npm install -g docker2lzc

# 使用
docker2lzc convert docker-compose.yml
```

项目地址: https://www.npmjs.com/package/docker2lzc

---

## 官方移植仓库

官方维护的移植示例：

https://gitee.com/lazycatcloud/appdb

包含常见应用的完整配置，是学习的最佳资源。

---

## 进阶阅读

- [使用 Github Actions 实现 Docker 应用的自动更新和发布](https://lazycat.cloud/playground/guideline/572)
- [懒猫微服如何使用 OpenID Connect (OIDC)](https://lazycat.cloud/playground/guideline/663)

---

## 相关文档

- [Manifest 完整规范](./manifest-spec.md)
- [路由配置详解](./route-patterns.md)
- [应用配置示例](./examples.md)
- [商店发布流程](./store-publishing.md)
