# 懒猫微服应用配置示例集

本文档提供各种类型应用的完整配置示例，涵盖常见框架和使用场景。

## 目录

- [纯静态前端应用](#纯静态前端应用)
- [Node.js 后端应用](#nodejs-后端应用)
- [Python Web 应用](#python-web-应用)
- [Go 应用](#go-应用)
- [前后端分离应用](#前后端分离应用)
- [带数据库的多服务应用](#带数据库的多服务应用)
- [VitePress 文档站点](#vitepress-文档站点)

---

## 纯静态前端应用

适用于使用 Vite、Vue、React、Angular 等框架构建的纯前端应用。

### 项目结构

```
my-static-app/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
└── ui/                  # 前端源码
    ├── src/
    ├── dist/            # 构建输出目录
    ├── package.json
    └── vite.config.js
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.pptist
version: 1.0.0
name: PPTist
description: 一个基于 Vue3.x + TypeScript 的在线演示文稿（幻灯片）应用

application:
  # 应用访问子域名
  subdomain: pptist

  # 静态文件路由
  routes:
    - /=file:///lzcapp/pkg/content/

  # 文件处理器（可选，用于关联文件类型）
  file_handler:
    mime:
      - x-lzc-extension/pptist  # 自定义文件扩展名
    actions:
      open: /?file=%u             # 打开文件的 URL 路径
```

### lzc-build.yml

```yaml
# 构建脚本路径
buildscript: sh build.sh

# manifest 文件路径
manifest: ./lzc-manifest.yml

# 打包内容目录（构建后的静态文件）
contentdir: ./ui/dist

# 输出路径
pkgout: ./

# 应用图标
icon: ./lzc-icon.png

# 开发环境配置
devshell:
  # 开发路由（转发到 Vite 开发服务器）
  routes:
    - /=http://127.0.0.1:5173

  # 开发依赖
  dependencies:
    - nodejs
    - npm

  # 环境设置脚本
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
```

### build.sh

```bash
#!/bin/bash
set -e

# 进入前端目录
cd ui

# 安装依赖
npm install

# 构建生产版本
npm run build

# 返回项目根目录
cd ..
```

### 部署命令

```bash
# 1. 构建应用包
lzc-cli project build -o release.lpk

# 2. 安装到微服
lzc-cli app install release.lpk
```

---

## Node.js 后端应用

适用于 Express、Koa、NestJS 等 Node.js 框架的后端应用。

### 项目结构

```
my-node-app/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
└── backend/
    ├── src/
    ├── dist/            # TypeScript 编译输出
    ├── package.json
    └── tsconfig.json
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: com.example.node-api
version: 1.0.0
name: Node.js API Server
description: RESTful API 服务

application:
  subdomain: api

  # 使用 exec 路由启动 Node.js 服务
  routes:
    - /=exec://3000,/lzcapp/pkg/content/start.sh

  # 环境变量
  environment:
    - NODE_ENV=production
    - PORT=3000

  # 健康检查
  health_check:
    test_url: http://127.0.0.1:3000/health
    timeout: 5s
    start_period: 30s
```

### lzc-build.yml

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./backend/dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:3000
  dependencies:
    - nodejs
    - npm
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
```

### build.sh

```bash
#!/bin/bash
set -e

cd backend

# 安装依赖
npm install

# 编译 TypeScript
npm run build

# 复制启动脚本
cat > dist/start.sh <<'EOF'
#!/bin/sh
cd /lzcapp/pkg/content
node index.js
EOF
chmod +x dist/start.sh

cd ..
```

---

## Python Web 应用

适用于 Flask、FastAPI、Django 等 Python Web 框架。

### 项目结构

```
todolist-py/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
├── ui/                  # 前端
│   └── dist/
└── backend/             # 后端
    ├── main.py
    ├── requirements.txt
    └── run.sh
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.todolist-py
version: 1.0.0
name: TodoList (Python)
description: 基于 Python + Vue 的待办清单应用

application:
  # 使用 Alpine 镜像（默认已包含 Python）
  image: registry.lazycat.cloud/lzc/alpine:3.21
  subdomain: todolist

  routes:
    # 前端静态文件
    - /=file:///lzcapp/pkg/content/ui/

    # 后端 API（使用 exec 启动 Python 服务）
    - /api/=exec://8080,/lzcapp/pkg/content/backend/run.sh

  # 工作目录
  workdir: /lzcapp/pkg/content

  # 健康检查
  health_check:
    test_url: http://127.0.0.1:8080/api/health
    timeout: 5s
    start_period: 60s
```

### lzc-build.yml

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173      # 前端开发服务器
    - /api/=http://127.0.0.1:8080  # 后端 API
  dependencies:
    - nodejs
    - npm
    - python3
    - py3-pip
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
    export PIP_INDEX_URL=https://pypi.tuna.tsinghua.edu.cn/simple
```

### build.sh

```bash
#!/bin/bash
set -e

# 构建前端
cd ui
npm install
npm run build
cd ..

# 准备打包目录
mkdir -p dist/ui
mkdir -p dist/backend

# 复制前端构建产物
cp -r ui/dist/* dist/ui/

# 复制后端代码
cp -r backend/* dist/backend/

# 创建后端启动脚本
cat > dist/backend/run.sh <<'EOF'
#!/bin/sh
cd /lzcapp/pkg/content/backend

# 安装 Python 依赖（首次运行）
if [ ! -f "/lzcapp/var/.deps_installed" ]; then
    pip install -r requirements.txt --break-system-packages \
        --index-url https://pypi.tuna.tsinghua.edu.cn/simple
    touch /lzcapp/var/.deps_installed
fi

# 启动应用
python main.py
EOF
chmod +x dist/backend/run.sh
```

### backend/main.py

```python
from flask import Flask, jsonify
from flask_cors import CORS

app = Flask(__name__)
CORS(app)

@app.route('/api/health')
def health():
    return jsonify({"status": "ok"})

@app.route('/api/todos')
def get_todos():
    # 从 /lzcapp/var 读取持久化数据
    return jsonify([])

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8080)
```

---

## Go 应用

适用于使用 Go 语言开发的 Web 应用。

### 项目结构

```
todolist-go/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
├── ui/
│   └── dist/
└── backend/
    ├── main.go
    └── go.mod
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.todolist-go
version: 1.0.0
name: TodoList (Go)
description: 基于 Go + Vue 的待办清单应用

application:
  subdomain: todolist-go

  routes:
    # 前端
    - /=file:///lzcapp/pkg/content/ui/

    # 后端（直接运行编译后的二进制文件）
    - /api/=exec://8080,/lzcapp/pkg/content/backend

  health_check:
    test_url: http://127.0.0.1:8080/api/health
    start_period: 10s
```

### lzc-build.yml

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  # 使用 Go 开发环境镜像
  image: registry.lazycat.cloud/golang:1.21-alpine
  routes:
    - /=http://127.0.0.1:5173
    - /api/=http://127.0.0.1:8080
  dependencies:
    - nodejs
    - npm
```

### build.sh

```bash
#!/bin/bash
set -e

# 构建前端
cd ui
npm install
npm run build
cd ..

# 构建 Go 后端（静态编译）
cd backend
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o backend main.go
cd ..

# 准备打包目录
mkdir -p dist/ui
cp -r ui/dist/* dist/ui/
cp backend/backend dist/backend
chmod +x dist/backend
```

### backend/main.go

```go
package main

import (
    "log"
    "net/http"
    "github.com/gin-gonic/gin"
)

func main() {
    r := gin.Default()

    // 健康检查
    r.GET("/api/health", func(c *gin.Context) {
        c.JSON(200, gin.H{"status": "ok"})
    })

    // 启动服务
    log.Fatal(r.Run(":8080"))
}
```

---

## 前后端分离应用

完整的全栈应用示例，包含前端、后端和数据库。

### 项目结构

```
fullstack-app/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
├── frontend/
│   └── dist/
└── backend/
    └── dist/
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: com.example.fullstack
version: 1.0.0
name: Full Stack Application
description: 完整的前后端分离应用

application:
  subdomain: app

  # 使用 upstreams 实现更精细的路由控制
  upstreams:
    # 静态前端
    - location: /
      backend: file:///lzcapp/pkg/content/frontend/

    # 后端 API
    - location: /api
      backend: http://127.0.0.1:3000/
      backend_launch_command: node /lzcapp/pkg/content/backend/server.js
      disable_trim_location: false

  # 环境变量
  environment:
    - NODE_ENV=production
    - DATABASE_URL=postgresql://user:pass@postgres:5432/mydb

  # 健康检查
  health_check:
    test_url: http://127.0.0.1:3000/api/health
    timeout: 5s
    start_period: 60s

# 数据库服务
services:
  postgres:
    image: registry.lazycat.cloud/postgres:15-alpine

    # 环境变量（使用稳定密码）
    environment:
      - POSTGRES_USER=appuser
      - POSTGRES_PASSWORD={{ stable_secret "db_password" }}
      - POSTGRES_DB=mydb

    # 持久化存储
    binds:
      - /lzcapp/var/postgres:/var/lib/postgresql/data

    # 资源限制
    mem_limit: 512M
    shm_size: 128M

    # 健康检查
    healthcheck:
      test: ["CMD", "pg_isready", "-U", "appuser"]
      interval: 10s
      timeout: 5s
      retries: 3
      start_period: 30s
```

### lzc-build.yml

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173
    - /api/=http://127.0.0.1:3000
  dependencies:
    - nodejs
    - npm
    - postgresql-client  # 用于数据库调试
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
```

---

## 带数据库的多服务应用

以 WordPress 为例，展示如何配置包含多个服务的复杂应用。

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.wordpress
version: 6.0.0
name: WordPress
description: 强大的内容管理系统

application:
  subdomain: blog

  # 将所有请求转发到 WordPress 容器
  routes:
    - /=http://wordpress:80

  # 依赖 MySQL 服务
  depends_on:
    - mysql

# 服务定义
services:
  # WordPress 服务
  wordpress:
    image: registry.lazycat.cloud/wordpress:6.0-apache

    # 依赖 MySQL
    depends_on:
      - mysql

    # 环境变量
    environment:
      - WORDPRESS_DB_HOST=mysql:3306
      - WORDPRESS_DB_USER=wordpress
      - WORDPRESS_DB_PASSWORD={{ stable_secret "wp_db_password" }}
      - WORDPRESS_DB_NAME=wordpress

    # 持久化存储
    binds:
      - /lzcapp/var/wordpress:/var/www/html

    # 资源限制
    mem_limit: 1G

    # 健康检查
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost/wp-admin/install.php"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 60s

  # MySQL 数据库
  mysql:
    image: registry.lazycat.cloud/mysql:8.0

    # 环境变量
    environment:
      - MYSQL_ROOT_PASSWORD={{ stable_secret "mysql_root_password" }}
      - MYSQL_DATABASE=wordpress
      - MYSQL_USER=wordpress
      - MYSQL_PASSWORD={{ stable_secret "wp_db_password" }}

    # 持久化存储
    binds:
      - /lzcapp/var/mysql:/var/lib/mysql

    # 资源限制
    mem_limit: 1G
    shm_size: 256M

    # 健康检查
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost", "-u", "root", "-p{{ stable_secret \"mysql_root_password\" }}"]
      interval: 10s
      timeout: 5s
      retries: 5
      start_period: 30s
```

### 更复杂的示例：GitLab

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.gitlab
version: 17.2.8
name: GitLab
description: 企业级 DevOps 平台

application:
  subdomain: gitlab

  # HTTP 路由
  routes:
    - /=http://gitlab:80

  # SSH 服务（TCP 端口暴露）
  ingress:
    - protocol: tcp
      port: 22
      service: gitlab
      description: GitLab SSH 服务
      publish_port: "2222"

services:
  gitlab:
    image: registry.lazycat.cloud/gitlab/gitlab-ee:17.2.8

    # 环境变量
    environment:
      - GITLAB_OMNIBUS_CONFIG=external_url 'http://gitlab.${LAZYCAT_BOX_DOMAIN}'; gitlab_rails['lfs_enabled'] = true;

    # 持久化存储
    binds:
      - /lzcapp/var/config:/etc/gitlab
      - /lzcapp/var/logs:/var/log/gitlab
      - /lzcapp/var/data:/var/opt/gitlab

    # 资源限制（GitLab 需要较多资源）
    mem_limit: 4G
    shm_size: 512M
    cpus: 2

    # 健康检查
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost/-/health"]
      interval: 60s
      timeout: 30s
      retries: 5
      start_period: 300s  # GitLab 启动较慢
```

---

## VitePress 文档站点

基于当前文档项目的真实示例。

### 项目结构

```
lzc-developer-doc/
├── lzc-build.yml
├── lzc-manifest.yml
├── lzc-icon.png
├── build.sh
└── docs/
    ├── .vitepress/
    ├── index.md
    └── ...
```

### lzc-manifest.yml

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.docs.developer
version: 1.0.0
name: 懒猫开发者文档
description: 懒猫微服平台开发者文档

application:
  subdomain: docs

  # 静态文件路由
  routes:
    - /=file:///lzcapp/pkg/content/

  # 不需要健康检查（纯静态站点）
  health_check:
    disable: true
```

### lzc-build.yml

```yaml
buildscript: sh build.sh
manifest: ./lzc-manifest.yml
contentdir: ./docs/.vitepress/dist
pkgout: ./
icon: ./lzc-icon.png

devshell:
  routes:
    - /=http://127.0.0.1:5173
  dependencies:
    - nodejs
    - npm
  setupscript: |
    export npm_config_registry=https://registry.npmmirror.com
```

### build.sh

```bash
#!/bin/bash
set -e

# 安装依赖
npm install

# 构建文档站点
npm run docs:build

# VitePress 会输出到 docs/.vitepress/dist
```

---

## 常见配置模式总结

### 1. 静态文件服务

```yaml
application:
  routes:
    - /=file:///lzcapp/pkg/content/dist
```

### 2. 可执行文件服务

```yaml
application:
  routes:
    - /api/=exec://3000,/lzcapp/pkg/content/backend
```

### 3. HTTP 代理服务

```yaml
application:
  routes:
    - /=http://service-name:8080
```

### 4. 混合路由

```yaml
application:
  routes:
    - /=file:///lzcapp/pkg/content/ui/          # 前端
    - /api/=exec://3000,/lzcapp/pkg/content/api # 后端 API
    - /ws/=http://127.0.0.1:9000                # WebSocket
```

### 5. 环境变量最佳实践

```yaml
application:
  environment:
    # 使用系统变量
    - APP_URL=https://${SUBDOMAIN}.${LAZYCAT_BOX_DOMAIN}

    # 稳定密码
    - SECRET_KEY={{ stable_secret "app_secret" }}

    # 服务引用
    - DB_HOST=postgres.${PACKAGE_ID}.lzcapp
```

### 6. 持久化存储模式

```yaml
services:
  app:
    binds:
      # 永久数据
      - /lzcapp/var/data:/app/data

      # 缓存数据
      - /lzcapp/cache/temp:/app/tmp

      # 日志（建议用 cache）
      - /lzcapp/cache/logs:/app/logs
```

### 7. 多实例适配

```yaml
services:
  app:
    binds:
      # 根据是否多实例选择不同路径
      {{ if .S.IsMultiInstance }}
      - /lzcapp/var/shared:/data
      {{ else }}
      - /lzcapp/run/mnt/home/{{ .S.DeployUID }}/data:/data
      {{ end }}
```

---

## 调试技巧

### 1. 查看最终渲染的 manifest

```yaml
application:
  routes:
    - /debug-manifest=file:///lzcapp/run/manifest.yml
```

### 2. 输出所有模板变量

在 manifest 中添加:

```yaml
xx-debug: {{ . }}
```

### 3. 查看容器日志

```bash
# 查看应用容器日志
lzc-cli docker logs -f cloud-lazycat-app-myapp-app-1

# 查看服务容器日志
lzc-cli docker logs -f cloud-lazycat-app-myapp-postgres-1
```

### 4. 进入容器调试

```bash
# 进入开发环境
lzc-cli project devshell

# 或进入运行中的容器
lzc-cli docker exec -it cloud-lazycat-app-myapp-app-1 sh
```

---

## 下一步

- 了解 [路由配置详解](./route-patterns.md)
- 学习 [Docker 应用迁移](./docker-migration.md)
- 查看 [构建配置规范](./build-spec.md)
- 准备 [应用商店发布](./store-publishing.md)
