# lzc-manifest.yml 完整规范

## 概述

`lzc-manifest.yml` 是懒猫微服应用的核心配置文件，定义了应用的元数据、部署配置、服务容器、路由规则等所有运行时所需的信息。

本文档提供完整的字段规范说明和实用配置示例。

## 快速导航

- [基本信息配置](#基本信息配置) - package、version、name 等必填字段
- [Application 配置](#application-配置) - 应用核心服务配置
- [Services 配置](#services-配置) - Docker 容器服务配置
- [路由配置](#路由配置) - HTTP 路由规则
- [健康检查配置](#健康检查配置) - 容器健康检测
- [本地化配置](#本地化配置) - 多语言支持
- [高级配置](#高级配置) - 实验性特性

---

## 基本信息配置

### 必填字段

```yaml
package: cloud.lazycat.app.myapp  # 应用唯一标识
version: 1.0.0                    # 语义化版本号
name: My Application              # 应用名称
```

| 字段名 | 类型 | 是否必填 | 描述 |
|--------|------|----------|------|
| `package` | string | ✅ | 应用的全球唯一 ID，建议以个人域名开头（如 `com.example.app`） |
| `version` | string | ✅ | 遵循 [语义化版本](https://semver.org/) 规范（X.Y.Z 格式） |
| `name` | string | ✅ | 应用显示名称 |

### 可选字段

| 字段名 | 类型 | 描述 | 示例 |
|--------|------|------|------|
| `description` | string | 应用简短描述 | "一款开源的密码管理器" |
| `usage` | string | 使用须知（首次访问时显示） | "请先设置管理员密码" |
| `license` | string | 许可证类型 | "MIT" |
| `homepage` | string | 应用主页 URL | "https://example.com" |
| `author` | string | 作者名称 | "Lazycat Team" |
| `min_os_version` | string | 最低系统版本要求 | "v1.3.0" |
| `unsupported_platforms` | []string | 不支持的平台列表 | `["ios", "android"]` |

**平台标识符**: `"ios"`, `"android"`, `"windows"`, `"macos"`, `"linux"`, `"tvos"`

### 配置示例

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.bitwarden
version: 1.44.1
name: Bitwarden
description: 一款自由且开源的密码管理服务
license: GPL-3.0
homepage: https://bitwarden.com
author: Bitwarden Inc.
min_os_version: v1.2.0
```

---

## Application 配置

`application` 字段定义应用的核心服务配置（对应名为 `app` 的特殊容器）。

### 基础配置

| 字段名 | 类型 | 默认值 | 描述 |
|--------|------|--------|------|
| `image` | string | `alpine:3.21` | 应用容器镜像（推荐使用系统默认镜像） |
| `subdomain` | string | - | 应用访问域名前缀（如 `myapp` → `myapp.xxx.heiyu.space`） |
| `background_task` | bool | `true` | 是否自动启动且不自动休眠 |
| `workdir` | string | - | 容器工作目录 |
| `environment` | []string | - | 环境变量列表（格式: `KEY=VALUE`） |

### 高级特性

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `multi_instance` | bool | 是否支持多用户实例 |
| `usb_accel` | bool | 挂载 USB 设备到 `/dev/bus/usb` |
| `gpu_accel` | bool | 挂载 GPU 设备到 `/dev/dri` |
| `kvm_accel` | bool | 挂载 KVM 设备到 `/dev/kvm` |
| `depends_on` | []string | 依赖的其他服务（必须健康才能启动） |

### 路由配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `routes` | []string | 简化版路由规则（格式: `PATH=UPSTREAM`） |
| `upstreams` | []UpstreamConfig | 高级路由配置（支持更多选项） |
| `public_path` | []string | 无需鉴权的公开路径列表 |

详见 [路由配置](#路由配置)。

### 网络配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `ingress` | []IngressConfig | TCP/UDP 端口暴露配置 |

#### IngressConfig 字段

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `protocol` | string | 协议类型（`tcp` 或 `udp`） |
| `port` | int | 目标端口号 |
| `service` | string | 服务名称（留空表示 `app` 容器） |
| `description` | string | 服务说明 |
| `publish_port` | string | 允许的入站端口（如 `22` 或 `1000-50000`） |
| `send_port_info` | bool | 发送实际端口信息（小端序 uint16） |
| `yes_i_want_80_443` | bool | 允许转发 80/443 流量（绕过系统鉴权） |

### 文件处理器配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `file_handler` | FileHandlerConfig | 声明支持的文件类型 |

```yaml
application:
  file_handler:
    mime:
      - x-lzc-extension/pptist  # 自定义 MIME 类型
      - application/pdf
    actions:
      open: /?file=%u            # %u 为文件路径
```

### 健康检查配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `health_check` | AppHealthCheckExt | 健康检测配置（扩展版本） |

详见 [健康检查配置](#健康检查配置)。

### 完整示例

```yaml
application:
  # 基础配置
  image: registry.lazycat.cloud/lzc/alpine:3.21
  subdomain: myapp
  background_task: true
  workdir: /lzcapp/pkg/content

  # 环境变量
  environment:
    - NODE_ENV=production
    - PORT=3000

  # 路由配置
  routes:
    - /=file:///lzcapp/pkg/content/dist     # 静态文件
    - /api/=http://127.0.0.1:3000/api/      # API 服务

  # TCP 端口暴露
  ingress:
    - protocol: tcp
      port: 22
      service: app
      description: SSH 服务

  # 硬件加速
  gpu_accel: true

  # 健康检查
  health_check:
    test_url: http://127.0.0.1:3000/health
    timeout: 5s
    start_period: 30s
```

---

## Services 配置

`services` 字段定义额外的 Docker 容器服务（如数据库、缓存等）。

### 容器基础配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `image` | string | Docker 镜像（必填） |
| `environment` | []string | 环境变量（格式: `KEY=VALUE`） |
| `entrypoint` | string | 容器入口点（覆盖镜像默认值） |
| `command` | string | 容器启动命令 |
| `user` | string | 运行用户（UID 或用户名） |
| `workdir` | string | 工作目录 |

### 资源限制

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `cpu_shares` | int64 | CPU 权重份额 |
| `cpus` | float32 | CPU 核心数限制 |
| `mem_limit` | string/int | 内存上限（如 `"512M"` 或 `536870912`） |
| `shm_size` | string/int | `/dev/shm` 大小 |

### 存储配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `binds` | []string | 持久化存储绑定（格式: `SOURCE:TARGET`） |
| `tmpfs` | []string | 临时文件系统挂载 |

**重要规则**:
- 持久化路径必须以 `/lzcapp/var` 或 `/lzcapp/cache` 开头
- `/lzcapp/var` - 永久存储（不会被自动清理）
- `/lzcapp/cache` - 缓存存储（可能被系统清理）

```yaml
binds:
  - /lzcapp/var/mysql:/var/lib/mysql     # 数据库数据
  - /lzcapp/cache/tmp:/tmp               # 临时文件
```

### 网络配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `network_mode` | string | 网络模式（仅支持 `host` 或留空） |
| `netadmin` | bool | 是否需要 NET_ADMIN 权限 |

### 高级配置

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `setup_script` | string | root 权限初始化脚本（与 entrypoint/command 互斥） |
| `runtime` | string | OCI runtime（`runc` 或 `sysbox-runc`） |
| `depends_on` | []string | 依赖的其他服务（不能包含 `app`） |
| `healthcheck` | HealthCheckConfig | 健康检查配置 |

### 完整示例

```yaml
services:
  # PostgreSQL 数据库
  postgres:
    image: registry.lazycat.cloud/postgres:15
    environment:
      - POSTGRES_USER=appuser
      - POSTGRES_PASSWORD={{ stable_secret "db_password" }}
      - POSTGRES_DB=myapp
    binds:
      - /lzcapp/var/postgres:/var/lib/postgresql/data
    mem_limit: 1G
    shm_size: 256M
    healthcheck:
      test: ["CMD", "pg_isready", "-U", "appuser"]
      interval: 10s
      timeout: 5s
      retries: 3

  # Redis 缓存
  redis:
    image: registry.lazycat.cloud/redis:7-alpine
    binds:
      - /lzcapp/cache/redis:/data
    command: redis-server --appendonly yes
    mem_limit: 512M
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 5s

  # Nginx 代理
  nginx:
    image: registry.lazycat.cloud/nginx:alpine
    depends_on:
      - postgres
      - redis
    setup_script: |
      cat <<'EOF' > /etc/nginx/conf.d/default.conf
      server {
        listen 80;
        location / {
          proxy_pass http://app:3000;
        }
      }
      EOF
```

---

## 路由配置

路由配置定义了 HTTP 请求如何被转发到实际的服务。

### 简化版路由 (routes)

格式: `URL_PATH=UPSTREAM`

支持的协议:
- **file://** - 静态文件服务
- **exec://** - 可执行文件服务
- **http(s)://** - HTTP 代理

```yaml
application:
  routes:
    # 静态文件
    - /=file:///lzcapp/pkg/content/dist

    # 可执行文件（端口:路径）
    - /api/=exec://3001,/lzcapp/pkg/content/backend

    # HTTP 服务
    - /proxy/=http://upstream-service:8080
```

### 高级路由 (upstreams)

提供更细粒度的控制。

#### UpstreamConfig 字段

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `location` | string | 匹配的 URL 路径 |
| `domain_prefix` | string | 匹配的域名前缀（用于多域名分流） |
| `backend` | string | 上游地址（支持 http/https/file 协议） |
| `backend_launch_command` | string | 自动启动的命令 |
| `disable_trim_location` | bool | 不自动去除 location 前缀（v1.3.9+） |
| `use_backend_host` | bool | 使用后端的 Host header |
| `disable_backend_ssl_verify` | bool | 跳过 SSL 证书验证 |
| `disable_auto_health_checking` | bool | 禁用自动健康检查 |
| `remove_this_request_headers` | []string | 删除指定的请求头 |
| `trim_url_suffix` | string | 删除 URL 后缀字符 |
| `fix_websocket_header` | bool | 修正 WebSocket 头 |
| `dump_http_headers_when_5xx` | bool | 5xx 错误时输出请求头 |
| `dump_http_headers_when_paths` | []string | 指定路径输出请求头 |

#### 高级示例

```yaml
application:
  subdomain: myapp

  upstreams:
    # 静态文件服务
    - location: /
      backend: file:///lzcapp/pkg/content/dist

    # 可执行文件服务
    - location: /api
      backend: http://127.0.0.1:3001/
      backend_launch_command: /lzcapp/pkg/content/backend -port 3001

    # 外部服务代理
    - location: /search
      backend: https://www.google.com/
      use_backend_host: true
      disable_backend_ssl_verify: false
      remove_this_request_headers:
        - Origin
        - Referer

    # 基于域名前缀的分流
    - location: /
      domain_prefix: admin
      backend: http://admin-panel:8080

    # WebSocket 支持
    - location: /ws
      backend: http://websocket-service:9000
      fix_websocket_header: true
```

### 服务名称解析

在 routes 中引用其他服务的两种方式:

1. **短名称** (推荐): `http://service-name:port`
2. **完整域名**: `http://service-name.package-id.lzcapp:port`

```yaml
package: cloud.lazycat.app.myapp

services:
  backend:
    image: mybackend:latest

application:
  routes:
    # 方式 1: 短名称（推荐）
    - /=http://backend:8080

    # 方式 2: 完整域名
    - /=http://backend.cloud.lazycat.app.myapp.lzcapp:8080
```

---

## 健康检查配置

### AppHealthCheckExt (Application 专用)

适用于 `application.health_check` 字段。

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `test_url` | string | HTTP 健康检查 URL（无需容器内工具） |
| `disable` | bool | 禁用健康检查 |
| `timeout` | string | 单次检测超时时间（如 `"5s"`） |
| `start_period` | string | 启动等待时间（如 `"60s"`） |

```yaml
application:
  health_check:
    test_url: http://127.0.0.1:3000/health
    timeout: 5s
    start_period: 30s
```

### HealthCheckConfig (Services 通用)

适用于 `services.*.healthcheck` 字段。

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `test` | []string | 检测命令（格式: `["CMD", "command", "args"]`） |
| `interval` | string | 检测间隔（如 `"10s"`） |
| `timeout` | string | 单次超时时间 |
| `retries` | int | 失败重试次数（默认 1） |
| `start_period` | string | 启动等待时间 |
| `start_interval` | string | 启动阶段检测间隔 |
| `disable` | bool | 禁用健康检查 |

```yaml
services:
  mysql:
    image: mysql:8
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      interval: 10s
      timeout: 3s
      retries: 3
      start_period: 40s
```

### 常用健康检查命令

```yaml
# MySQL
test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]

# PostgreSQL
test: ["CMD", "pg_isready", "-U", "postgres"]

# Redis
test: ["CMD", "redis-cli", "ping"]

# HTTP 服务
test: ["CMD", "curl", "-f", "http://localhost:8080/health"]

# 端口检查
test: ["CMD", "nc", "-z", "localhost", "3000"]
```

---

## 本地化配置

通过 `locales` 字段支持多语言显示。

### I10nConfigItem 字段

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `name` | string | 应用名称（本地化） |
| `description` | string | 应用描述（本地化） |
| `usage` | string | 使用须知（本地化） |

### Language Key 规范

遵循 [BCP 47 标准](https://en.wikipedia.org/wiki/IETF_language_tag):
- `zh` - 中文
- `zh_CN` - 简体中文
- `en` - 英文
- `ja` - 日文
- `ko` - 韩文

### 完整示例

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.netatalk
version: 0.0.1
name: Apple 时间机器备份
description: Netatalk 服务可用于 Apple 时间机器备份
author: Netatalk

# 多语言配置
locales:
  zh:
    name: "Apple 时间机器备份"
    description: "Netatalk 服务可用于 Apple 时间机器备份"
  zh_CN:
    name: "Apple 时间机器备份"
    description: "Netatalk 服务可用于 Apple 时间机器备份"
  en:
    name: "Time Machine Server"
    description: "Netatalk service can be used for Apple Time Machine backup"
  ja:
    name: "タイムマシンサーバー"
    description: "Netatalk サービスは Apple Time Machine のバックアップに使用できます"

application:
  subdomain: netatalk3
```

---

## 高级配置

### ExtConfig (实验性特性)

需要 lzc-os >= v1.3.0。

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `enable_document_access` | bool | 挂载文稿目录到 `/lzcapp/run/mnt/home` |
| `enable_media_access` | bool | 挂载媒体目录到 `/lzcapp/run/mnt/media` |
| `disable_grpc_web_on_root` | bool | 不劫持应用的 gRPC-Web 流量 |
| `default_prefix_domain` | string | 调整默认打开的域名前缀 |

```yaml
ext_config:
  enable_document_access: true
  enable_media_access: false
  default_prefix_domain: "admin"
```

---

## 完整配置示例

### 纯静态网站

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.static-site
version: 1.0.0
name: My Static Site
description: A simple static website

application:
  subdomain: site
  routes:
    - /=file:///lzcapp/pkg/content/
```

### 前后端分离应用

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.fullstack
version: 1.0.0
name: Full Stack App

application:
  subdomain: app
  environment:
    - NODE_ENV=production

  routes:
    - /=file:///lzcapp/pkg/content/dist
    - /api/=http://127.0.0.1:3000/api/

  upstreams:
    - location: /api
      backend: http://127.0.0.1:3000/
      backend_launch_command: node /lzcapp/pkg/content/server.js

  health_check:
    test_url: http://127.0.0.1:3000/health
    start_period: 30s

services:
  postgres:
    image: registry.lazycat.cloud/postgres:15
    environment:
      - POSTGRES_PASSWORD={{ stable_secret "db_pass" }}
    binds:
      - /lzcapp/var/db:/var/lib/postgresql/data
    healthcheck:
      test: ["CMD", "pg_isready"]
      interval: 10s
```

### 多服务应用

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.wordpress
version: 6.0.0
name: WordPress

application:
  subdomain: blog
  routes:
    - /=http://wordpress:80

services:
  wordpress:
    image: registry.lazycat.cloud/wordpress:6.0-apache
    depends_on:
      - mysql
    environment:
      - WORDPRESS_DB_HOST=mysql:3306
      - WORDPRESS_DB_USER=wordpress
      - WORDPRESS_DB_PASSWORD={{ stable_secret "wp_db_pass" }}
    binds:
      - /lzcapp/var/wordpress:/var/www/html
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost"]
      interval: 30s

  mysql:
    image: registry.lazycat.cloud/mysql:8
    environment:
      - MYSQL_ROOT_PASSWORD={{ stable_secret "mysql_root_pass" }}
      - MYSQL_DATABASE=wordpress
      - MYSQL_USER=wordpress
      - MYSQL_PASSWORD={{ stable_secret "wp_db_pass" }}
    binds:
      - /lzcapp/var/mysql:/var/lib/mysql
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      interval: 10s
```

---

## 常见配置模式

### 1. 环境变量引用

```yaml
application:
  environment:
    - DB_HOST=mysql.${PACKAGE_ID}.lzcapp
    - APP_URL=https://${SUBDOMAIN}.${LAZYCAT_BOX_DOMAIN}
```

### 2. 稳定密码生成

```yaml
services:
  mysql:
    environment:
      - MYSQL_ROOT_PASSWORD={{ stable_secret "root_password" }}
      - MYSQL_PASSWORD={{ stable_secret "user_password" | substr 0 16 }}
```

### 3. 多实例配置

```yaml
services:
  app:
    binds:
      {{ if .S.IsMultiInstance }}
      - /lzcapp/var/data:/data
      {{ else }}
      - /lzcapp/run/mnt/home/{{ .S.DeployUID }}/data:/data
      {{ end }}
```

### 4. 跨域配置

```yaml
upstreams:
  - location: /api/external
    backend: https://api.example.com/
    use_backend_host: true
    remove_this_request_headers:
      - Origin
      - Referer
```

---

## 最佳实践

1. **Package ID 命名**: 使用反向域名格式（如 `com.company.product`）
2. **版本号管理**: 遵循语义化版本规范
3. **镜像选择**: 优先使用官方镜像或 `registry.lazycat.cloud`
4. **存储路径**:
   - 重要数据 → `/lzcapp/var`
   - 缓存数据 → `/lzcapp/cache`
5. **健康检查**: 所有服务都应配置合理的健康检查
6. **密码管理**: 使用 `{{ stable_secret "seed" }}` 生成稳定密码
7. **多语言支持**: 商店应用必须配置 `locales`

---

## 相关文档

- [lzc-build.yml 规范](./build-spec.md)
- [路由配置详解](./route-patterns.md)
- [Docker 迁移指南](./docker-migration.md)
- [部署参数配置](./deploy-params.md)
