# 路由配置深度指南

路由配置是懒猫微服应用的核心部分，决定了 HTTP 请求如何被转发到实际的服务。本文档详细介绍路由的各种配置方式和高级特性。

## 目录

- [路由基础概念](#路由基础概念)
- [简化版路由 (routes)](#简化版路由-routes)
- [高级路由 (upstreams)](#高级路由-upstreams)
- [APP Proxy 高级用法](#app-proxy-高级用法)
- [路由优先级和匹配规则](#路由优先级和匹配规则)
- [常见路由模式](#常见路由模式)
- [故障排查](#故障排查)

---

## 路由基础概念

### 路由格式

路由规则遵循 `URL_PATH=UPSTREAM` 的格式：

- **URL_PATH**: 浏览器访问的实际 URL 路径（不含 hostname）
- **UPSTREAM**: 具体的上游服务地址

### 支持的协议

懒猫微服支持三种上游协议：

1. **file://** - 静态文件服务
2. **exec://** - 可执行文件服务
3. **http(s)://** - HTTP 代理服务

---

## 简化版路由 (routes)

`application.routes` 字段使用简化的路由配置，适合大多数场景。

### 1. 静态文件路由 (file://)

用于加载静态 HTML、CSS、JavaScript 等资源。

#### 基本语法

```yaml
routes:
  - /=file:///lzcapp/pkg/content/
```

#### 完整示例

```yaml
lzc-sdk-version: '0.1'
package: cloud.lazycat.app.pptist
name: PPTist
description: 在线演示文稿应用

application:
  subdomain: pptist
  routes:
    # 静态文件路由
    - /=file:///lzcapp/pkg/content/

  # 文件处理器（关联文件类型）
  file_handler:
    mime:
      - x-lzc-extension/pptist
    actions:
      open: /?file=%u
```

#### 路径说明

- `/lzcapp/pkg/content/` - lpk 包中 contentdir 的内容（只读）
- 文件以只读形式挂载，无法在运行时修改

#### 适用场景

- 纯前端应用（Vue/React/Angular）
- 静态文档站点（VitePress/VuePress）
- 单页应用（SPA）

---

### 2. 可执行文件路由 (exec://)

用于启动并转发到应用内的可执行文件提供的服务。

#### 基本语法

```yaml
routes:
  - /api/=exec://PORT,/path/to/executable
```

**格式**: `exec://端口号,可执行文件路径`

#### 工作原理

1. 系统启动应用时执行指定的可执行文件
2. 假设该文件提供的服务运行在指定端口
3. 将匹配的请求转发到 `http://127.0.0.1:PORT`

**注意**: 系统不会验证服务是否真的由该文件启动（可用于初始化脚本）

#### 示例 1: 单一后端服务

```yaml
package: cloud.lazycat.app.myapp
name: My App

application:
  subdomain: myapp
  routes:
    # 前端静态文件
    - /=file:///lzcapp/pkg/content/dist

    # 后端 API 服务（Node.js）
    - /api/=exec://3001,/lzcapp/pkg/content/backend
```

**backend 脚本示例** (需要可执行权限):
```bash
#!/bin/sh
cd /lzcapp/pkg/content
node server.js
```

#### 示例 2: 多路径共享同一服务

```yaml
application:
  subdomain: file
  routes:
    # 前端
    - /=file:///lzcapp/pkg/content/dist

    # 后端 API（端口 3001）
    - /api/=exec://3001,/lzcapp/pkg/content/backend

    # 文件服务（同一个后端提供）
    - /files/=http://127.0.0.1:3001/files/
```

---

### 3. HTTP 代理路由 (http://)

将请求转发到内网或外网的 HTTP 服务。

#### 基本语法

```yaml
routes:
  - /=http://hostname:port/path
  - /=https://hostname:port/path
```

#### 服务名称解析

在 routes 中引用本应用的其他服务，有两种方式：

##### 方式 1: 短名称（推荐）

```yaml
package: cloud.lazycat.app.bitwarden

application:
  routes:
    - /=http://bitwarden:80

services:
  bitwarden:
    image: bitwarden/nginx:1.44.1
```

**优点**:
- 简洁易读
- 修改 package ID 时无需调整
- lzc-os 1.3.x+ 支持应用隔离

##### 方式 2: 完整域名

```yaml
package: cloud.lazycat.app.bitwarden

application:
  routes:
    - /=http://bitwarden.cloud.lazycat.app.bitwarden.lzcapp:80
```

**格式**: `{service_name}.{package_id}.lzcapp`

**何时使用完整域名**:
1. lzc-os 1.3.x 之前（无应用隔离，避免 service name 冲突）
2. 上游服务会检测 HTTP Host header

#### 示例 1: 内网服务

```yaml
application:
  routes:
    - /=http://wordpress:80

services:
  wordpress:
    image: wordpress:6.0
```

#### 示例 2: 外网服务

```yaml
application:
  subdomain: appstore
  routes:
    # 转发到公网应用商店
    - /=https://appstore.lazycat.cloud
```

**优势**: 前端代码依然可以使用 lzc-sdk，应用逻辑在微服内运行，但页面托管在公有云。

---

## 高级路由 (upstreams)

`application.upstreams` 提供更细粒度的路由控制（v1.3.8+）。

### UpstreamConfig 字段

| 字段名 | 类型 | 描述 |
|--------|------|------|
| `location` | string | 匹配的 URL 路径 |
| `domain_prefix` | string | 匹配的域名前缀 |
| `backend` | string | 上游地址（http/https/file） |
| `backend_launch_command` | string | 自动启动的命令 |
| `disable_trim_location` | bool | 不自动去除 location 前缀（v1.3.9+） |
| `use_backend_host` | bool | 使用后端的 Host header |
| `disable_backend_ssl_verify` | bool | 跳过 SSL 证书验证 |
| `disable_auto_health_checking` | bool | 禁用自动健康检查 |
| `remove_this_request_headers` | []string | 删除指定请求头 |
| `disable_url_raw_path` | bool | 删除 HTTP header 中的 raw URL |
| `trim_url_suffix` | string | 删除 URL 后缀字符 |
| `fix_websocket_header` | bool | 修正 WebSocket 头 |
| `dump_http_headers_when_5xx` | bool | 5xx 错误时输出请求头 |
| `dump_http_headers_when_paths` | []string | 指定路径输出请求头 |

### 完整示例

```yaml
application:
  subdomain: debug

  # 简单路由（与 upstreams 可共存）
  routes:
    - /=http://app1.org.snyh.debug.whoami.lzcapp:80

  upstreams:
    # 1. 外部服务代理（跨域处理）
    - location: /search
      backend: https://baidu.com/
      use_backend_host: true  # 使用 baidu.com 作为 Host
      disable_auto_health_checking: true
      remove_this_request_headers:
        - Origin
        - Referer
      disable_url_raw_path: true

    # 2. 自签 SSL 证书
    - location: /other
      backend: https://app2.snyh.debug.lzcapp:4443
      disable_backend_ssl_verify: true  # 跳过 SSL 验证

    # 3. 基于域名前缀的分流
    - location: /
      domain_prefix: config
      backend: http://config.snyh.debug.lzcapp:80

    # 4. 使用 backend_launch_command 替代 exec 路由
    - location: /api
      backend: http://127.0.0.1:3001/
      backend_launch_command: /lzcapp/pkg/content/my-super-backend -listen :3001
```

### 域名前缀分流

使用 `domain_prefix` 实现同一应用的多域名访问：

```yaml
application:
  subdomain: app
  secondary_domains:
    - admin
    - api

  upstreams:
    # app.xxx.heiyu.space
    - location: /
      backend: file:///lzcapp/pkg/content/frontend/

    # admin-app.xxx.heiyu.space
    - location: /
      domain_prefix: admin
      backend: http://admin-panel:8080

    # api-app.xxx.heiyu.space
    - location: /
      domain_prefix: api
      backend: http://api-server:3000
```

---

## APP Proxy 高级用法

官方提供 APP Proxy 镜像用于实现复杂路由和查看请求日志。

### 镜像信息

```yaml
services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
```

基于 OpenResty（Nginx + Lua），支持两种配置模式。

### 模式 1: 环境变量配置

适用于单一 HTTP 上游服务。

#### 支持的环境变量

| 环境变量 | 作用 | 示例 |
|----------|------|------|
| `UPSTREAM` (必填) | 上游服务地址 | `http://whoami:80` |
| `BASIC_AUTH_HEADER` | 绕过 Basic Auth | `Basic dXNlcjpwYXNzd29yZA==` |
| `REMOVE_REQUEST_HEADERS` | 删除请求头（`;` 分隔） | `Origin;Host;` |

#### 示例 1: 基础代理

```yaml
application:
  routes:
    - /=http://app-proxy:80
  subdomain: app-proxy-test

services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
    environment:
      - UPSTREAM=http://whoami:80

  whoami:
    image: registry.lazycat.cloud/snyh1010/traefik/whoami:c899811bc4a1f63a
```

查看日志:
```bash
lzc-cli docker logs -f cloudlazycatappapp-proxy-test-app-proxy-1
```

#### 示例 2: 绕过 Basic Auth

```yaml
services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
    environment:
      - UPSTREAM=http://whoami:80
      # echo -n "user:password" | base64
      - BASIC_AUTH_HEADER=Basic dXNlcjpwYXNzd29yZA==
```

#### 示例 3: 删除请求头

```yaml
services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
    environment:
      - UPSTREAM=http://whoami:80
      - REMOVE_REQUEST_HEADERS=Origin;Cache-Control;
```

### 模式 2: setup_script 配置

直接覆盖 OpenResty 配置，实现复杂路由。

#### 多域名支持示例

```yaml
application:
  routes:
    - /=http://app-proxy:80
  subdomain: app-proxy-test
  secondary_domains:
    - portainer
    - whoami

services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
    setup_script: |
      cat <<'EOF' > /etc/nginx/conf.d/default.conf
      # 默认域名: app-proxy-test.xxx.heiyu.space
      server {
         server_name  app-proxy-test.*;
         location / {
            root   /usr/local/openresty/nginx/html;
            index  index.html index.htm;
         }
      }

      # portainer-app-proxy-test.xxx.heiyu.space
      server {
         server_name  portainer.*;
         location / {
            proxy_pass http://portainer:9000;
         }
      }

      # whoami-app-proxy-test.xxx.heiyu.space
      server {
         server_name  whoami.*;
         location / {
            proxy_pass http://whoami:80;
         }
      }
      EOF

  portainer:
    image: registry.lazycat.cloud/u8997806945/portainer/portainer-ce:d393c0c7d12aae78

  whoami:
    image: registry.lazycat.cloud/snyh1010/traefik/whoami:c899811bc4a1f63a
```

---

## 路由优先级和匹配规则

### 匹配顺序

路由按以下优先级匹配（从高到低）：

1. **精确路径匹配**
2. **长路径前缀匹配**
3. **短路径前缀匹配**

### 示例

```yaml
routes:
  - /api/v2/users=http://127.0.0.1:8080/  # 优先级 1（最长）
  - /api/v2/=http://127.0.0.1:8080/       # 优先级 2
  - /api/=http://127.0.0.1:3000/          # 优先级 3
  - /=file:///lzcapp/pkg/content/         # 优先级 4（通配符）
```

**请求示例**:
- `/api/v2/users` → `http://127.0.0.1:8080/` ✅
- `/api/v2/posts` → `http://127.0.0.1:8080/` ✅
- `/api/login` → `http://127.0.0.1:3000/` ✅
- `/index.html` → `file:///lzcapp/pkg/content/` ✅

### 最佳实践

```yaml
routes:
  # 精确路径在前
  - /api/v2/=http://new-api:8080/
  - /api/=http://legacy-api:3000/

  # 静态文件
  - /static/=file:///lzcapp/pkg/content/static/

  # 通配符在最后
  - /=file:///lzcapp/pkg/content/
```

---

## 常见路由模式

### 1. 纯前端应用

```yaml
application:
  routes:
    - /=file:///lzcapp/pkg/content/
```

### 2. 前后端分离

```yaml
application:
  routes:
    - /=file:///lzcapp/pkg/content/dist
    - /api/=http://127.0.0.1:3000/api/
```

### 3. 微服务架构

```yaml
application:
  upstreams:
    # 前端
    - location: /
      backend: file:///lzcapp/pkg/content/

    # 用户服务
    - location: /api/users
      backend: http://user-service:8080/

    # 订单服务
    - location: /api/orders
      backend: http://order-service:8081/

    # 支付服务
    - location: /api/payments
      backend: http://payment-service:8082/
```

### 4. WebSocket 支持

```yaml
application:
  upstreams:
    - location: /ws
      backend: http://websocket-server:9000
      fix_websocket_header: true
```

### 5. 外部 API 代理

```yaml
application:
  upstreams:
    - location: /api/external
      backend: https://api.example.com/
      use_backend_host: true
      disable_backend_ssl_verify: false
      remove_this_request_headers:
        - Origin
        - Referer
```

### 6. 混合模式

```yaml
application:
  subdomain: app

  routes:
    # 基础路由
    - /=file:///lzcapp/pkg/content/

  upstreams:
    # 高级 API 路由
    - location: /api/v2
      backend: http://127.0.0.1:8080/
      backend_launch_command: /lzcapp/pkg/content/api-v2
      disable_auto_health_checking: false

    # 外部服务
    - location: /proxy/weather
      backend: https://api.weather.com/
      use_backend_host: true
```

---

## 故障排查

### 问题 1: 404 Not Found

**可能原因**:
- 路由路径配置错误
- 服务未启动
- 端口不正确

**排查步骤**:

1. **检查路由配置**:
```yaml
# 确保路径匹配
routes:
  - /api/=http://127.0.0.1:3000/api/  # ✅ 正确
  - /api/=http://127.0.0.1:3000       # ⚠️ 可能导致路径错误
```

2. **验证服务是否运行**:
```bash
lzc-cli docker ps
lzc-cli docker logs -f <container-name>
```

3. **检查端口监听**:
```bash
lzc-cli docker exec -it <container-name> sh
netstat -tlnp | grep 3000
```

### 问题 2: 502 Bad Gateway

**可能原因**:
- 后端服务未启动
- 健康检查失败
- 端口配置错误

**解决方案**:

1. **检查健康检查**:
```yaml
health_check:
  test_url: http://127.0.0.1:3000/health
  start_period: 60s  # 增加启动时间
```

2. **查看服务日志**:
```bash
lzc-cli docker logs -f <service-container>
```

### 问题 3: CORS 跨域错误

**解决方案 1**: 使用 `remove_this_request_headers`

```yaml
upstreams:
  - location: /api/external
    backend: https://api.example.com/
    remove_this_request_headers:
      - Origin
      - Referer
```

**解决方案 2**: 使用 `use_backend_host`

```yaml
upstreams:
  - location: /api/external
    backend: https://api.example.com/
    use_backend_host: true
```

### 问题 4: WebSocket 连接失败

**解决方案**:

```yaml
upstreams:
  - location: /ws
    backend: http://websocket-server:9000
    fix_websocket_header: true  # 修正 WebSocket 头
```

### 问题 5: 路径前缀被自动去除

**问题**: 转发到后端时，`location` 前缀被自动去掉

**示例**:
- 请求: `/api/users`
- 转发到: `http://backend:3000/users` (去掉了 `/api`)

**解决方案** (v1.3.9+):
```yaml
upstreams:
  - location: /api
    backend: http://backend:3000/
    disable_trim_location: true  # 保留 /api 前缀
```

### 问题 6: 外部服务 SSL 验证失败

**解决方案**:

```yaml
upstreams:
  - location: /internal
    backend: https://internal.company.local:4443
    disable_backend_ssl_verify: true  # 跳过 SSL 验证
```

---

## 调试技巧

### 1. 启用请求日志

使用 APP Proxy:
```yaml
services:
  app-proxy:
    image: registry.lazycat.cloud/app-proxy:v0.1.0
    environment:
      - UPSTREAM=http://backend:3000
```

查看日志:
```bash
lzc-cli docker logs -f <app-proxy-container>
```

### 2. Dump HTTP Headers

```yaml
upstreams:
  - location: /debug
    backend: http://backend:3000
    dump_http_headers_when_5xx: true
    dump_http_headers_when_paths:
      - /debug/test
```

### 3. 临时禁用健康检查

```yaml
upstreams:
  - location: /api
    backend: http://backend:3000
    disable_auto_health_checking: true  # 调试时禁用
```

---

## 相关文档

- [Manifest 完整规范](./manifest-spec.md)
- [应用配置示例](./examples.md)
- [Docker 迁移指南](./docker-migration.md)
