---
name: make-pptx
description: HTML 기반 프레젠테이션을 PPTX 파일로 변환하거나 직접 PPTX 파일을 생성하는 스킬. Pretendard 폰트 필수 적용. PptxGenJS 라이브러리 활용.
allowed-tools: Read, Write, Edit, Bash, Glob
user-invocable: true
---

# PPTX 생성 스킬

HTML 프레젠테이션을 PPTX로 변환하거나, PptxGenJS를 직접 활용하여 PowerPoint 파일을 생성합니다.

## 필수 요구사항

### Pretendard 폰트

**모든 PPTX 파일은 반드시 Pretendard 폰트를 사용해야 합니다.**

시스템에 Pretendard가 없는 경우:

1. https://github.com/orioncactus/pretendard 에서 다운로드
2. 시스템에 폰트 설치

### 의존성

```bash
npm install pptxgenjs sharp
```

## 워크플로우

### 옵션 1: PptxGenJS 직접 사용 (권장)

PptxGenJS를 사용하여 프로그래매틱하게 PPTX를 생성합니다.

#### 기본 구조

```javascript
const PptxGenJS = require('pptxgenjs');

// 프레젠테이션 생성
const pptx = new PptxGenJS();

// 메타데이터 설정
pptx.author = 'Fit Checker';
pptx.title = '적합성 분석 리포트';
pptx.subject = '[회사명] x [지원자명]';

// 레이아웃 설정 (16:9)
pptx.defineLayout({ name: 'CUSTOM', width: 13.33, height: 7.5 });
pptx.layout = 'CUSTOM';
```

#### 슬라이드 1: 타이틀 & 매치율

```javascript
const slide1 = pptx.addSlide();

// 배경색
slide1.background = { color: 'F9FAFB' };

// 타이틀
slide1.addText('[회사명] x [지원자명]', {
  x: 0.5,
  y: 0.5,
  w: '90%',
  h: 1,
  fontSize: 40,
  fontFace: 'Pretendard',
  bold: true,
  color: '111827',
  align: 'center',
});

// 서브타이틀
slide1.addText('적합성 분석 리포트', {
  x: 0.5,
  y: 1.3,
  w: '90%',
  h: 0.5,
  fontSize: 24,
  fontFace: 'Pretendard',
  color: '6B7280',
  align: 'center',
});

// 매치율 원형 (도넛 차트)
slide1.addChart(
  pptx.ChartType.doughnut,
  [
    {
      name: 'Match',
      labels: ['매치', '나머지'],
      values: [78, 22], // 매치율에 따라 조정
    },
  ],
  {
    x: 4.5,
    y: 2,
    w: 4,
    h: 4,
    chartColors: ['10B981', 'E5E7EB'],
    showLegend: false,
    showTitle: false,
    holeSize: 70,
  }
);

// 매치율 텍스트 (차트 중앙)
slide1.addText('78%', {
  x: 5.5,
  y: 3.5,
  w: 2,
  h: 1,
  fontSize: 48,
  fontFace: 'Pretendard',
  bold: true,
  color: '10B981',
  align: 'center',
});
```

#### 슬라이드 2: 잘 맞는 점

```javascript
const slide2 = pptx.addSlide();

slide2.addText('✅ 잘 맞는 점', {
  x: 0.5,
  y: 0.3,
  w: '90%',
  fontSize: 32,
  fontFace: 'Pretendard',
  bold: true,
  color: '111827',
});

// 테이블 데이터
const tableData = [
  [
    { text: '항목', options: { bold: true, fill: '2563EB', color: 'FFFFFF' } },
    {
      text: '회사 요구사항',
      options: { bold: true, fill: '2563EB', color: 'FFFFFF' },
    },
    {
      text: '지원자 보유 역량',
      options: { bold: true, fill: '2563EB', color: 'FFFFFF' },
    },
    {
      text: '적합도',
      options: { bold: true, fill: '2563EB', color: 'FFFFFF' },
    },
  ],
  [
    { text: '기술 스택' },
    { text: 'React, TypeScript' },
    { text: 'Next.js, React, TypeScript' },
    { text: '✅ 완벽 일치', options: { color: '10B981' } },
  ],
  // 추가 행...
];

slide2.addTable(tableData, {
  x: 0.5,
  y: 1,
  w: 12,
  fontFace: 'Pretendard',
  fontSize: 14,
  border: { type: 'solid', pt: 1, color: 'E5E7EB' },
  align: 'left',
  valign: 'middle',
});
```

#### 슬라이드 3: 맞지 않는 점

```javascript
const slide3 = pptx.addSlide();

slide3.addText('⚠️ 맞지 않는 점', {
  x: 0.5,
  y: 0.3,
  w: '90%',
  fontSize: 32,
  fontFace: 'Pretendard',
  bold: true,
  color: '111827',
});

// 카드 스타일 박스
slide3.addShape(pptx.ShapeType.rect, {
  x: 0.5,
  y: 1,
  w: 5.5,
  h: 2.5,
  fill: 'FEF2F2',
  line: { color: 'EF4444', pt: 2, dashType: 'solid' },
});

slide3.addText('❌ 경력 요건', {
  x: 0.7,
  y: 1.2,
  w: 5,
  fontSize: 20,
  fontFace: 'Pretendard',
  bold: true,
  color: 'EF4444',
});

slide3.addText('요구: 3년 이상\n보유: 신입', {
  x: 0.7,
  y: 1.8,
  w: 5,
  fontSize: 16,
  fontFace: 'Pretendard',
  color: '111827',
});
```

#### 슬라이드 4: 보완 포인트

```javascript
const slide4 = pptx.addSlide();

slide4.addText('💡 보완 포인트', {
  x: 0.5,
  y: 0.3,
  w: '90%',
  fontSize: 32,
  fontFace: 'Pretendard',
  bold: true,
  color: '111827',
});

// 우선순위 배지
slide4.addShape(pptx.ShapeType.roundRect, {
  x: 0.5,
  y: 1,
  w: 1.2,
  h: 0.4,
  fill: 'FEE2E2',
});

slide4.addText('높음', {
  x: 0.5,
  y: 1,
  w: 1.2,
  h: 0.4,
  fontSize: 12,
  fontFace: 'Pretendard',
  bold: true,
  color: '991B1B',
  align: 'center',
  valign: 'middle',
});

// 보완 항목 내용
slide4.addText('[보완 항목명]', {
  x: 2,
  y: 1,
  w: 10,
  fontSize: 18,
  fontFace: 'Pretendard',
  bold: true,
  color: '111827',
});

slide4.addText('📌 근거: [채용 공고에서 발췌한 근거]', {
  x: 2,
  y: 1.6,
  w: 10,
  fontSize: 14,
  fontFace: 'Pretendard',
  color: '6B7280',
});
```

#### 파일 저장

```javascript
// 파일로 저장 (Node.js)
pptx
  .writeFile({ fileName: 'output/fit-result.pptx' })
  .then((fileName) => console.log(`Created: ${fileName}`));

// 또는 Buffer로 반환
const buffer = await pptx.write({ outputType: 'nodebuffer' });
```

### 옵션 2: HTML to PPTX 변환

HTML 파일을 PPTX로 변환해야 하는 경우, `html2pptx.md` 참조 문서를 확인하세요.

## 중요 주의사항

### 1. 색상 코드

**PptxGenJS에서 `#` 없이 사용해야 합니다.**

```javascript
// ✅ 올바름
{
  color: '10B981';
}
{
  fill: 'F9FAFB';
}

// ❌ 잘못됨 (파일 손상 가능)
{
  color: '#10B981';
}
{
  fill: '#F9FAFB';
}
```

### 2. 폰트 지정

모든 텍스트 요소에 Pretendard 폰트를 명시적으로 지정:

```javascript
{
  fontFace: 'Pretendard',
  // ...
}
```

### 3. 한글 텍스트

한글 텍스트는 자동으로 처리되지만, 줄바꿈이 필요한 경우 `\n` 사용:

```javascript
slide.addText('첫 번째 줄\n두 번째 줄', { ... });
```

### 4. 이모지 사용

이모지는 일부 환경에서 표시되지 않을 수 있습니다.
중요한 정보는 이모지 외에 텍스트로도 표현하세요.

## 출력 파일

결과물은 `output/` 디렉토리에 저장:

```
output/
├── fit-check-result-YYYY-MM-DD.pptx
└── fit-check-result-YYYY-MM-DD.html  (백업용)
```

## 컬러 팔레트 참조

| 용도           | 색상 코드 | 설명               |
| -------------- | --------- | ------------------ |
| Primary        | `2563EB`  | 주요 강조색        |
| Success        | `10B981`  | 긍정적/완료        |
| Warning        | `F59E0B`  | 주의/선호          |
| Danger         | `EF4444`  | 부정적/필수 미충족 |
| Neutral        | `6B7280`  | 중립/보조 텍스트   |
| Background     | `F9FAFB`  | 배경색             |
| Text Primary   | `111827`  | 주요 텍스트        |
| Text Secondary | `6B7280`  | 보조 텍스트        |
| Border         | `E5E7EB`  | 테두리/구분선      |

## 추가 참조 문서

상세한 기술 정보는 다음 문서를 참조하세요:

- `html2pptx.md` - HTML to PPTX 변환 상세 가이드
- `ooxml.md` - OOXML 직접 조작 기술 참조
