# HTML to PPTX 변환 가이드

HTML 슬라이드를 PowerPoint로 변환하는 상세 가이드입니다.

## 개요

HTML 기반 프레젠테이션을 PPTX로 변환할 때는 특정 규칙을 준수해야 합니다.
이 문서는 변환 시 주의사항과 모범 사례를 설명합니다.

## HTML 슬라이드 작성 규칙

### 레이아웃 크기

프레젠테이션 비율에 맞는 정확한 크기를 사용하세요:

| 비율 | 너비 | 높이 | 용도 |
|------|------|------|------|
| 16:9 | 1280px | 720px | 표준 와이드스크린 (권장) |
| 4:3 | 1024px | 768px | 클래식 비율 |
| 16:10 | 1280px | 800px | 맥북 등 |

```html
<!DOCTYPE html>
<html>
<head>
  <style>
    .slide {
      width: 1280px;
      height: 720px;
      margin: 0;
      padding: 40px;
      box-sizing: border-box;
      font-family: 'Pretendard', sans-serif;
    }
  </style>
</head>
<body>
  <div class="slide">
    <!-- 슬라이드 콘텐츠 -->
  </div>
</body>
</html>
```

### 텍스트 규칙

**중요**: 모든 텍스트는 반드시 다음 태그 안에 있어야 합니다:

- `<p>` - 일반 단락
- `<h1>` ~ `<h6>` - 제목
- `<ul>`, `<ol>` - 목록
- `<li>` - 목록 항목

```html
<!-- ✅ 올바름 -->
<h1>제목입니다</h1>
<p>본문 텍스트입니다.</p>
<ul>
  <li>항목 1</li>
  <li>항목 2</li>
</ul>

<!-- ❌ 잘못됨 - 변환 시 텍스트 누락 -->
<div>이 텍스트는 변환되지 않습니다</div>
<span>이것도 누락됩니다</span>
```

### 인라인 스타일링

지원되는 인라인 스타일:

```html
<p>
  <strong>굵은 텍스트</strong>
  <em>기울임 텍스트</em>
  <u>밑줄 텍스트</u>
  <span style="color: #10B981;">컬러 텍스트</span>
</p>
```

### 폰트 사용

**Pretendard 폰트 필수**

웹 세이프 폰트 대체 지정:

```css
font-family: 'Pretendard', -apple-system, BlinkMacSystemFont,
             'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
```

## 지원되지 않는 기능

### CSS 그라데이션

CSS 그라데이션은 PPTX로 변환되지 않습니다.

```css
/* ❌ 변환 안됨 */
background: linear-gradient(to right, #10B981, #2563EB);
```

**해결책**: 그라데이션이 필요한 경우 PNG 이미지로 먼저 생성:

```javascript
const sharp = require('sharp');

// SVG로 그라데이션 생성 후 PNG로 변환
const svgGradient = `
<svg width="400" height="100">
  <defs>
    <linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="0%">
      <stop offset="0%" style="stop-color:#10B981"/>
      <stop offset="100%" style="stop-color:#2563EB"/>
    </linearGradient>
  </defs>
  <rect width="400" height="100" fill="url(#grad)"/>
</svg>`;

await sharp(Buffer.from(svgGradient))
  .png()
  .toFile('gradient-bg.png');
```

### 복잡한 CSS

변환되지 않는 CSS 속성:

- `box-shadow` (외부 그림자)
- `transform`
- `animation`, `transition`
- `filter`
- `clip-path`
- CSS Grid/Flexbox (레이아웃 계산만 가능)

### 웹 폰트

시스템에 설치되지 않은 웹 폰트는 대체됩니다.
**반드시 Pretendard를 시스템에 설치하세요.**

## 변환 프로세스

### Step 1: HTML 준비

슬라이드별로 HTML 파일 생성:

```
slides/
├── slide1.html  (타이틀)
├── slide2.html  (잘 맞는 점)
├── slide3.html  (맞지 않는 점)
├── slide4.html  (보완 포인트)
└── slide5.html  (요약)
```

### Step 2: 이미지 사전 생성

그라데이션, 아이콘 등 PNG로 미리 생성:

```
assets/
├── gradient-header.png
├── match-circle.png
└── icons/
    ├── check.png
    ├── warning.png
    └── cross.png
```

### Step 3: 변환 실행

PptxGenJS를 사용한 변환 스크립트:

```javascript
const PptxGenJS = require('pptxgenjs');
const fs = require('fs');
const path = require('path');

async function convertHtmlToPptx(slidesDir, outputPath) {
  const pptx = new PptxGenJS();

  // 설정
  pptx.author = 'Fit Checker';
  pptx.title = '적합성 분석 리포트';
  pptx.defineLayout({ name: 'CUSTOM', width: 13.33, height: 7.5 });
  pptx.layout = 'CUSTOM';

  // 각 HTML 파일을 슬라이드로 변환
  const files = fs.readdirSync(slidesDir)
    .filter(f => f.endsWith('.html'))
    .sort();

  for (const file of files) {
    const html = fs.readFileSync(path.join(slidesDir, file), 'utf-8');
    await addSlideFromHtml(pptx, html);
  }

  await pptx.writeFile({ fileName: outputPath });
  console.log(`Created: ${outputPath}`);
}

async function addSlideFromHtml(pptx, html) {
  const slide = pptx.addSlide();

  // HTML 파싱 및 요소 변환 로직
  // (cheerio 또는 jsdom 사용)

  return slide;
}
```

### Step 4: 검증

생성된 PPTX를 PowerPoint에서 열어 확인:

- 텍스트가 모두 표시되는지
- 폰트가 올바르게 적용되는지
- 레이아웃이 의도대로인지
- 이미지가 제대로 삽입되는지

## 테이블 변환

HTML 테이블을 PPTX 테이블로 변환:

```html
<!-- HTML -->
<table>
  <thead>
    <tr>
      <th>항목</th>
      <th>요구사항</th>
      <th>보유 역량</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td>기술 스택</td>
      <td>React</td>
      <td>React, Next.js</td>
    </tr>
  </tbody>
</table>
```

```javascript
// PptxGenJS 변환
const tableData = [
  [
    { text: '항목', options: { bold: true, fill: '2563EB', color: 'FFFFFF' } },
    { text: '요구사항', options: { bold: true, fill: '2563EB', color: 'FFFFFF' } },
    { text: '보유 역량', options: { bold: true, fill: '2563EB', color: 'FFFFFF' } }
  ],
  [
    { text: '기술 스택' },
    { text: 'React' },
    { text: 'React, Next.js' }
  ]
];

slide.addTable(tableData, {
  x: 0.5,
  y: 1.5,
  w: 12,
  fontFace: 'Pretendard',
  fontSize: 14,
  border: { type: 'solid', pt: 1, color: 'E5E7EB' }
});
```

## 모범 사례

1. **단순하게 유지**: 복잡한 CSS보다 기본 스타일 사용
2. **이미지 활용**: 복잡한 시각 요소는 PNG로 미리 생성
3. **테스트 반복**: 변환 후 항상 PowerPoint에서 확인
4. **폰트 명시**: 모든 텍스트 요소에 Pretendard 지정
5. **색상 주의**: `#` 없이 색상 코드 사용 (PptxGenJS)
