# OOXML 기술 참조 문서

PowerPoint 파일(PPTX)의 내부 구조인 Office Open XML(OOXML)에 대한 기술 참조입니다.

## PPTX 파일 구조

PPTX 파일은 ZIP 압축된 XML 파일들의 모음입니다.

### 기본 구조

```
presentation.pptx (ZIP)
├── [Content_Types].xml          # 콘텐츠 타입 정의
├── _rels/
│   └── .rels                    # 루트 관계 파일
├── docProps/
│   ├── app.xml                  # 애플리케이션 속성
│   └── core.xml                 # 핵심 속성 (제목, 작성자 등)
└── ppt/
    ├── presentation.xml         # 메인 프레젠테이션 정의
    ├── presProps.xml            # 프레젠테이션 속성
    ├── tableStyles.xml          # 테이블 스타일
    ├── viewProps.xml            # 뷰 속성
    ├── _rels/
    │   └── presentation.xml.rels  # 프레젠테이션 관계
    ├── slideLayouts/            # 슬라이드 레이아웃
    │   ├── slideLayout1.xml
    │   └── _rels/
    ├── slideMasters/            # 슬라이드 마스터
    │   ├── slideMaster1.xml
    │   └── _rels/
    ├── slides/                  # 실제 슬라이드
    │   ├── slide1.xml
    │   ├── slide2.xml
    │   └── _rels/
    ├── theme/                   # 테마
    │   └── theme1.xml
    └── media/                   # 미디어 파일 (이미지 등)
        ├── image1.png
        └── image2.png
```

## 슬라이드 XML 구조

### 기본 슬라이드 구조

```xml
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<p:sld xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main"
       xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships"
       xmlns:p="http://schemas.openxmlformats.org/presentationml/2006/main">
  <p:cSld>
    <p:spTree>
      <p:nvGrpSpPr>
        <p:cNvPr id="1" name=""/>
        <p:cNvGrpSpPr/>
        <p:nvPr/>
      </p:nvGrpSpPr>
      <p:grpSpPr/>

      <!-- 도형/텍스트 요소들 -->

    </p:spTree>
  </p:cSld>
</p:sld>
```

### 텍스트 요소

```xml
<p:sp>
  <p:nvSpPr>
    <p:cNvPr id="2" name="Title 1"/>
    <p:cNvSpPr/>
    <p:nvPr/>
  </p:nvSpPr>
  <p:spPr>
    <a:xfrm>
      <a:off x="457200" y="274638"/>  <!-- 위치 (EMU 단위) -->
      <a:ext cx="8229600" cy="1143000"/>  <!-- 크기 (EMU 단위) -->
    </a:xfrm>
  </p:spPr>
  <p:txBody>
    <a:bodyPr/>
    <a:lstStyle/>
    <a:p>
      <a:r>
        <a:rPr lang="ko-KR" dirty="0">
          <a:latin typeface="Pretendard"/>
          <a:ea typeface="Pretendard"/>
        </a:rPr>
        <a:t>제목 텍스트</a:t>
      </a:r>
    </a:p>
  </p:txBody>
</p:sp>
```

### 폰트 지정 (Pretendard)

모든 텍스트에 Pretendard 폰트를 적용하려면:

```xml
<a:rPr lang="ko-KR" dirty="0">
  <a:latin typeface="Pretendard" panose="020B0604020202020204"/>
  <a:ea typeface="Pretendard"/>  <!-- 동아시아 문자용 -->
  <a:cs typeface="Pretendard"/>  <!-- 복잡한 스크립트용 -->
</a:rPr>
```

### 색상 지정

```xml
<!-- 단색 채우기 -->
<a:solidFill>
  <a:srgbClr val="10B981"/>  <!-- # 없이 HEX 색상 -->
</a:solidFill>

<!-- 테마 색상 -->
<a:solidFill>
  <a:schemeClr val="accent1"/>
</a:solidFill>
```

## 단위 시스템

OOXML은 EMU(English Metric Units)를 사용합니다:

| 단위 | EMU 값 |
|------|--------|
| 1 인치 | 914400 |
| 1 cm | 360000 |
| 1 pt | 12700 |
| 1 px (96dpi) | 9525 |

### 변환 공식

```javascript
// 인치 → EMU
const emuFromInches = inches => Math.round(inches * 914400);

// 포인트 → EMU
const emuFromPoints = points => Math.round(points * 12700);

// 픽셀 (96dpi) → EMU
const emuFromPixels = pixels => Math.round(pixels * 9525);
```

## 도형 타입

### 사각형

```xml
<p:sp>
  <p:nvSpPr>
    <p:cNvPr id="3" name="Rectangle 1"/>
    <p:cNvSpPr/>
    <p:nvPr/>
  </p:nvSpPr>
  <p:spPr>
    <a:xfrm>
      <a:off x="914400" y="914400"/>
      <a:ext cx="3657600" cy="1828800"/>
    </a:xfrm>
    <a:prstGeom prst="rect">
      <a:avLst/>
    </a:prstGeom>
    <a:solidFill>
      <a:srgbClr val="F9FAFB"/>
    </a:solidFill>
    <a:ln w="12700">  <!-- 선 두께 (EMU) -->
      <a:solidFill>
        <a:srgbClr val="E5E7EB"/>
      </a:solidFill>
    </a:ln>
  </p:spPr>
</p:sp>
```

### 둥근 사각형

```xml
<a:prstGeom prst="roundRect">
  <a:avLst>
    <a:gd name="adj" fmla="val 16667"/>  <!-- 모서리 반경 -->
  </a:avLst>
</a:prstGeom>
```

## 이미지 삽입

### 1. 미디어 파일 추가

이미지를 `ppt/media/` 디렉토리에 추가:

```
ppt/media/image1.png
```

### 2. 관계 파일 업데이트

`ppt/slides/_rels/slide1.xml.rels`:

```xml
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
  <Relationship Id="rId1"
                Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/slideLayout"
                Target="../slideLayouts/slideLayout1.xml"/>
  <Relationship Id="rId2"
                Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/image"
                Target="../media/image1.png"/>
</Relationships>
```

### 3. 슬라이드에 이미지 참조

```xml
<p:pic>
  <p:nvPicPr>
    <p:cNvPr id="4" name="Picture 1"/>
    <p:cNvPicPr/>
    <p:nvPr/>
  </p:nvPicPr>
  <p:blipFill>
    <a:blip r:embed="rId2"/>  <!-- 관계 ID 참조 -->
    <a:stretch>
      <a:fillRect/>
    </a:stretch>
  </p:blipFill>
  <p:spPr>
    <a:xfrm>
      <a:off x="914400" y="914400"/>
      <a:ext cx="3657600" cy="2743200"/>
    </a:xfrm>
    <a:prstGeom prst="rect">
      <a:avLst/>
    </a:prstGeom>
  </p:spPr>
</p:pic>
```

## 콘텐츠 타입

`[Content_Types].xml`에 새 파일 타입 등록:

```xml
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="png" ContentType="image/png"/>
  <Default Extension="jpeg" ContentType="image/jpeg"/>
  <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
  <Default Extension="xml" ContentType="application/xml"/>
  <Override PartName="/ppt/presentation.xml"
            ContentType="application/vnd.openxmlformats-officedocument.presentationml.presentation.main+xml"/>
  <Override PartName="/ppt/slides/slide1.xml"
            ContentType="application/vnd.openxmlformats-officedocument.presentationml.slide+xml"/>
  <!-- 추가 슬라이드... -->
</Types>
```

## 유효성 검사

PPTX 파일 수정 후 검증 체크리스트:

1. **XML 유효성**: 모든 XML 파일이 well-formed인지 확인
2. **관계 파일**: 모든 참조(rId)가 유효한지 확인
3. **콘텐츠 타입**: 새 파일 추가 시 Content_Types.xml 업데이트
4. **미사용 리소스**: 사용하지 않는 미디어 파일 정리
5. **UTF-8 인코딩**: 한글 등 유니코드 문자 인코딩 확인

### 유니코드 이스케이프

특수 문자는 XML 엔티티로 이스케이프:

| 문자 | 이스케이프 |
|------|-----------|
| `<` | `&lt;` |
| `>` | `&gt;` |
| `&` | `&amp;` |
| `"` | `&quot;` 또는 `&#8220;` |
| `'` | `&apos;` |

### 공백 보존

텍스트에서 공백을 보존하려면:

```xml
<a:t xml:space="preserve">  공백이 있는 텍스트  </a:t>
```

## PPTX 언팩/리팩

### 언팩 (압축 해제)

```bash
# macOS/Linux
unzip presentation.pptx -d unpacked/

# 또는 Python
python -c "import zipfile; zipfile.ZipFile('presentation.pptx').extractall('unpacked/')"
```

### 리팩 (재압축)

```bash
# macOS/Linux
cd unpacked/
zip -r ../new_presentation.pptx .

# 또는 Python
import zipfile
import os

def repack_pptx(source_dir, output_path):
    with zipfile.ZipFile(output_path, 'w', zipfile.ZIP_DEFLATED) as zf:
        for root, dirs, files in os.walk(source_dir):
            for file in files:
                file_path = os.path.join(root, file)
                arc_name = os.path.relpath(file_path, source_dir)
                zf.write(file_path, arc_name)
```

## 참조 자료

- [ECMA-376 Office Open XML 표준](https://www.ecma-international.org/publications-and-standards/standards/ecma-376/)
- [PptxGenJS 공식 문서](https://gitbrent.github.io/PptxGenJS/)
- [Open XML SDK 문서](https://docs.microsoft.com/en-us/office/open-xml/open-xml-sdk)
