#!/bin/bash

# enum_detector.sh
# GLOWマスタデータのEnum検知・分析スクリプト

set -e

# パス定義
SCHEMA_DIR="projects/glow-schema/Schema"
PHP_ENUM_DIR="projects/glow-server/api/app/Domain"
CSHARP_ENUM_DIR="projects/glow-client/Assets/GLOW/Scripts/Runtime/Core/Domain/Constants/AutoGenerated"

# 使い方を表示
usage() {
  cat <<EOF
Usage: $0 <command> [options]

Commands:
  list-enums [schema|php|csharp]    - Enum一覧を取得
  find-enum <DataName> <columnName> - カラムに対応するEnum情報を検索
  php-enum <EnumName>               - PHP Enumから値を抽出
  csharp-enum <EnumName>            - C# Enumから値を抽出
  detect <DataName>                 - データ定義内の全Enum列を検出
  compare <EnumName>                - サーバー/クライアント間のEnum値整合性チェック

Examples:
  $0 list-enums schema              # glow-schemaで定義された全Enum名を表示
  $0 find-enum MstUnit roleType     # MstUnit.roleTypeのEnum情報を検索
  $0 php-enum GachaType             # GachaType（PHP）の値を抽出
  $0 csharp-enum ItemType           # ItemType（C#）の値を抽出
  $0 detect OprGacha                # OprGachaの全Enum列を検出
  $0 compare ItemType               # ItemTypeの整合性チェック

EOF
  exit 1
}

# yqがインストールされているか確認
check_yq() {
  if ! command -v yq &> /dev/null; then
    echo "Error: yq is not installed. Please install it first:"
    echo "  brew install yq"
    exit 1
  fi
}

# スキーマディレクトリの存在確認
check_schema_dir() {
  if [ ! -d "$SCHEMA_DIR" ]; then
    echo "Error: Schema directory not found: $SCHEMA_DIR"
    exit 1
  fi
}

# glow-schemaから全Enum名を抽出
list_schema_enums() {
  check_schema_dir
  check_yq

  find "$SCHEMA_DIR" -name "*.yml" -not -path "*/Template/*" -exec yq '.enum[]?.name' {} \; 2>/dev/null | grep -v '^null$' | sort -u
}

# PHP Enum一覧
list_php_enums() {
  find "$PHP_ENUM_DIR" -path "*/Enums/*.php" -exec basename {} .php \; | sort -u
}

# C# Enum一覧
list_csharp_enums() {
  find "$CSHARP_ENUM_DIR" -name "*.cs" -exec basename {} .cs \; | sort -u
}

# Enum一覧を取得
list_enums() {
  local type="$1"

  case "$type" in
    schema)
      list_schema_enums
      ;;
    php)
      list_php_enums
      ;;
    csharp)
      list_csharp_enums
      ;;
    "")
      echo "=== Schema Enums ==="
      list_schema_enums
      echo ""
      echo "=== PHP Enums ==="
      list_php_enums
      echo ""
      echo "=== C# Enums ==="
      list_csharp_enums
      ;;
    *)
      echo "Error: Unknown type: $type"
      echo "Valid types: schema, php, csharp"
      exit 1
      ;;
  esac
}

# PHP Enumファイルを検索
find_php_enum_file() {
  local enum_name="$1"
  find "$PHP_ENUM_DIR" -path "*/Enums/${enum_name}.php" 2>/dev/null | head -1
}

# PHP Enum値を抽出
extract_php_enum() {
  local enum_name="$1"
  local file=$(find_php_enum_file "$enum_name")

  if [ -z "$file" ]; then
    return 1
  fi

  # case CASE_NAME = 'Value'; から Value を抽出
  grep -E "^\s*case\s+\w+\s*=\s*'" "$file" | \
    sed -n "s/^.*= *'\([^']*\)'.*$/\1/p"
}

# C# Enum値を抽出
extract_csharp_enum() {
  local enum_name="$1"
  local file="${CSHARP_ENUM_DIR}/${enum_name}.cs"

  if [ ! -f "$file" ]; then
    return 1
  fi

  # public enum後の値を抽出
  awk '/public enum/,/^}/' "$file" | \
    grep -E '^\s+\w+' | \
    grep -vE '^\s*(public|private|protected|internal|static|readonly|const)' | \
    sed -n 's/^ *\([A-Za-z0-9_]*\).*$/\1/p'
}

# glow-schemaからEnum値を取得
get_schema_enum_values() {
  local enum_name="$1"

  check_schema_dir
  check_yq

  # 全YAMLを検索してEnum値を抽出
  for yml in "$SCHEMA_DIR"/*.yml; do
    yq ".enum[] | select(.name == \"$enum_name\") | .params[].name" "$yml" 2>/dev/null
  done | grep -v '^null$'
}

# PHP Enumを類似検索 + 値マッチングで探す
find_php_enum_fuzzy() {
  local schema_enum="$1"

  # 1. 完全一致を試す
  local exact=$(find_php_enum_file "$schema_enum")
  if [ -n "$exact" ]; then
    basename "$exact" .php
    return 0
  fi

  # 2. 部分一致で候補を探す（末尾のType, Categoryなどを使う）
  local suffix=$(echo "$schema_enum" | grep -oE '[A-Z][a-z]+$')  # 例: Type, Category
  if [ -z "$suffix" ]; then
    return 1
  fi

  local candidates=$(list_php_enums | grep -i "$suffix")
  if [ -z "$candidates" ]; then
    return 1
  fi

  # 3. glow-schemaのEnum値を取得
  local schema_values=$(get_schema_enum_values "$schema_enum" | sort)
  if [ -z "$schema_values" ]; then
    return 1
  fi

  # 4. 各候補の値と比較、一致率が最も高いものを返す
  local best_match=""
  local best_score=0

  while IFS= read -r candidate; do
    local php_values=$(extract_php_enum "$candidate" 2>/dev/null | sort)
    if [ -z "$php_values" ]; then
      continue
    fi

    local common=$(comm -12 <(echo "$schema_values") <(echo "$php_values") | wc -l | tr -d ' ')
    if [ "$common" -gt "$best_score" ]; then
      best_score=$common
      best_match=$candidate
    fi
  done <<< "$candidates"

  # 一致数が半分以上なら採用
  local total=$(echo "$schema_values" | wc -l | tr -d ' ')
  if [ "$best_score" -ge $((total / 2)) ]; then
    echo "$best_match"
  fi
}

# glow-schemaからカラムのEnum型を取得
find_enum_type_for_column() {
  local data_name="$1"
  local column_name="$2"

  check_schema_dir
  check_yq

  # 全YAMLファイルを検索
  for yml_file in "$SCHEMA_DIR"/*.yml; do
    # データ定義を探す
    local enum_type=$(yq ".data[] | select(.name == \"$data_name\") | .params[] | select(.name == \"$column_name\") | .type" "$yml_file" 2>/dev/null | grep -v '^null$')

    if [ -n "$enum_type" ]; then
      # ? を除去（Nullable型の場合）
      echo "$enum_type" | sed 's/?$//'
      return 0
    fi
  done

  return 1
}

# カラムからEnum情報を検索
find_enum_for_column() {
  local data_name="$1"
  local column_name="$2"

  if [ -z "$data_name" ] || [ -z "$column_name" ]; then
    echo "Error: DataName and columnName are required"
    usage
  fi

  echo "Column: $data_name.$column_name"
  echo ""

  # glow-schemaからEnum型を取得
  local enum_type=$(find_enum_type_for_column "$data_name" "$column_name")

  if [ -z "$enum_type" ]; then
    echo "  Enum Type: (not found in glow-schema)"
    return 1
  fi

  echo "  Enum Type (glow-schema): $enum_type"
  echo ""

  # PHP Enum検索（類似検索）
  local php_enum=$(find_php_enum_fuzzy "$enum_type")
  if [ -n "$php_enum" ]; then
    local php_file=$(find_php_enum_file "$php_enum")
    echo "  PHP Enum: $php_enum"
    echo "  PHP File: $php_file"
    echo "  PHP Values:"
    extract_php_enum "$php_enum" | sed 's/^/    /'
    echo ""
  else
    echo "  PHP Enum: (not found)"
    echo ""
  fi

  # C# Enum検索（完全一致）
  if [ -f "${CSHARP_ENUM_DIR}/${enum_type}.cs" ]; then
    echo "  C# Enum: $enum_type"
    echo "  C# File: ${CSHARP_ENUM_DIR}/${enum_type}.cs"
    echo "  C# Values:"
    extract_csharp_enum "$enum_type" | sed 's/^/    /'
  else
    echo "  C# Enum: (not found)"
  fi
}

# データ定義内の全Enum列を検出
detect_enums() {
  local data_name="$1"

  if [ -z "$data_name" ]; then
    echo "Error: DataName is required"
    usage
  fi

  check_schema_dir
  check_yq

  echo "Detecting enum columns in $data_name..."
  echo ""

  # 全YAMLファイルを検索
  local found=false
  for yml_file in "$SCHEMA_DIR"/*.yml; do
    # データ定義を探す
    local data_exists=$(yq ".data[] | select(.name == \"$data_name\")" "$yml_file" 2>/dev/null | grep -v '^null$')

    if [ -n "$data_exists" ]; then
      found=true
      # カラムとEnum型を抽出
      yq ".data[] | select(.name == \"$data_name\") | .params[] | select(.type) | [.name, .type] | @tsv" "$yml_file" 2>/dev/null | \
        while IFS=$'\t' read -r col_name col_type; do
          # Nullable型の場合は ? を除去
          local clean_type=$(echo "$col_type" | sed 's/?$//')

          # Enum型かどうか判定（プリミティブ型でないか）
          if ! echo "$clean_type" | grep -qE '^(string|int|bool|DateTimeOffset|DateTime|TimeSpan|decimal|float|double)$'; then
            # PHP/C#でEnum定義があるか確認
            local php_enum=""
            local csharp_enum=""

            # C#は完全一致
            if [ -f "${CSHARP_ENUM_DIR}/${clean_type}.cs" ]; then
              csharp_enum="$clean_type"
            fi

            # PHPは類似検索
            php_enum=$(find_php_enum_fuzzy "$clean_type")

            if [ -n "$php_enum" ] || [ -n "$csharp_enum" ]; then
              # 出力形式: roleType -> CharacterUnitRoleType(C#), RoleType(PHP)
              local enum_display=""

              # C#とPHPで同名か確認
              if [ -n "$csharp_enum" ] && [ -n "$php_enum" ] && [ "$csharp_enum" = "$php_enum" ]; then
                # 同名の場合
                enum_display="${csharp_enum}(C#/PHP)"
              else
                # 異名の場合
                [ -n "$csharp_enum" ] && enum_display="${csharp_enum}(C#)"
                if [ -n "$php_enum" ]; then
                  [ -n "$enum_display" ] && enum_display="$enum_display, "
                  enum_display="${enum_display}${php_enum}(PHP)"
                fi
              fi

              # Nullable型の場合は表示
              local nullable_mark=""
              [ "$col_type" != "$clean_type" ] && nullable_mark=" (nullable)"

              echo "  $col_name -> $enum_display$nullable_mark"
            fi
          fi
        done
      break
    fi
  done

  if [ "$found" = false ]; then
    echo "Error: Data '$data_name' not found in glow-schema"
    return 1
  fi
}

# Enum値を比較
compare_enums() {
  local enum_name="$1"

  if [ -z "$enum_name" ]; then
    echo "Error: EnumName is required"
    usage
  fi

  local php_values=$(extract_php_enum "$enum_name" 2>/dev/null | sort)
  local csharp_values=$(extract_csharp_enum "$enum_name" 2>/dev/null | sort)

  if [ -z "$php_values" ] && [ -z "$csharp_values" ]; then
    echo "Error: Enum '$enum_name' not found in PHP or C#"
    return 1
  fi

  echo "Comparing $enum_name..."
  echo ""

  # PHP only
  if [ -n "$php_values" ]; then
    local php_only=$(comm -23 <(echo "$php_values") <(echo "$csharp_values") 2>/dev/null)
    echo "PHP only:"
    [ -n "$php_only" ] && echo "$php_only" | sed 's/^/  /' || echo "  (none)"
    echo ""
  fi

  # C# only
  if [ -n "$csharp_values" ]; then
    local csharp_only=$(comm -13 <(echo "$php_values") <(echo "$csharp_values") 2>/dev/null)
    echo "C# only:"
    [ -n "$csharp_only" ] && echo "$csharp_only" | sed 's/^/  /' || echo "  (none)"
    echo ""
  fi

  # Common
  if [ -n "$php_values" ] && [ -n "$csharp_values" ]; then
    local common=$(comm -12 <(echo "$php_values") <(echo "$csharp_values") 2>/dev/null)
    echo "Common:"
    [ -n "$common" ] && echo "$common" | sed 's/^/  /' || echo "  (none)"
  fi
}

# メイン処理
main() {
  # 引数チェック
  if [ $# -eq 0 ]; then
    usage
  fi

  # コマンド実行
  local command="$1"
  shift

  case "$command" in
    list-enums)
      list_enums "$@"
      ;;
    find-enum)
      find_enum_for_column "$@"
      ;;
    php-enum)
      [ -z "$1" ] && usage
      extract_php_enum "$1"
      ;;
    csharp-enum)
      [ -z "$1" ] && usage
      extract_csharp_enum "$1"
      ;;
    detect)
      detect_enums "$@"
      ;;
    compare)
      compare_enums "$@"
      ;;
    help|--help|-h)
      usage
      ;;
    *)
      echo "Error: Unknown command: $command"
      echo
      usage
      ;;
  esac
}

# スクリプト実行
main "$@"
