---
name: moai-workflow-jit-docs/optimization
description: Performance optimization for JIT documentation, indexing, caching, and memory management
---

# JIT Documentation Optimization (v5.0.0)

## Memory-Efficient Caching

### 1. LRU Cache with Size Limits

```typescript
interface CacheEntry<T> {
    value: T;
    size: number;
    timestamp: number;
    accessCount: number;
}

class LRUDocumentCache<T> {
    private cache: Map<string, CacheEntry<T>> = new Map();
    private maxSize: number; // bytes
    private currentSize: number = 0;

    constructor(maxSizeInMB: number = 100) {
        this.maxSize = maxSizeInMB * 1024 * 1024;
    }

    set(key: string, value: T, sizeEstimate: number) {
        // Remove existing entry if present
        if (this.cache.has(key)) {
            const existing = this.cache.get(key)!;
            this.currentSize -= existing.size;
        }

        // Evict LRU items if necessary
        while (this.currentSize + sizeEstimate > this.maxSize && this.cache.size > 0) {
            this.evictLRU();
        }

        // Add new entry
        this.cache.set(key, {
            value,
            size: sizeEstimate,
            timestamp: Date.now(),
            accessCount: 0
        });

        this.currentSize += sizeEstimate;
    }

    get(key: string): T | undefined {
        const entry = this.cache.get(key);
        if (!entry) return undefined;

        // Update access tracking
        entry.accessCount++;
        entry.timestamp = Date.now();

        return entry.value;
    }

    private evictLRU() {
        // Find least recently used item
        let lruKey: string | null = null;
        let lruScore = Infinity;

        for (const [key, entry] of this.cache.entries()) {
            const score = entry.timestamp + (entry.accessCount * 1000);
            if (score < lruScore) {
                lruScore = score;
                lruKey = key;
            }
        }

        if (lruKey) {
            const removed = this.cache.get(lruKey)!;
            this.currentSize -= removed.size;
            this.cache.delete(lruKey);
        }
    }

    getStats() {
        return {
            items: this.cache.size,
            usedBytes: this.currentSize,
            usedMB: (this.currentSize / 1024 / 1024).toFixed(2),
            maxMB: (this.maxSize / 1024 / 1024).toFixed(2),
            hitRate: this.calculateHitRate()
        };
    }
}
```

## Index Optimization

### 1. Incremental Indexing

```typescript
class IncrementalDocumentIndex {
    private index: Map<string, Set<string>> = new Map(); // word -> docIds
    private docHashes: Map<string, string> = new Map();  // docId -> contentHash

    async indexDocuments(docs: DocumentContent[]): Promise<void> {
        for (const doc of docs) {
            const hash = this.calculateHash(doc.content);
            const previousHash = this.docHashes.get(doc.id);

            // Only reindex if content changed
            if (previousHash !== hash) {
                await this.indexDocument(doc);
                this.docHashes.set(doc.id, hash);
            }
        }
    }

    private async indexDocument(doc: DocumentContent) {
        // Extract and tokenize content
        const tokens = this.tokenize(doc.content);

        for (const token of new Set(tokens)) {
            if (!this.index.has(token)) {
                this.index.set(token, new Set());
            }

            this.index.get(token)!.add(doc.id);
        }
    }

    async search(query: string): Promise<string[]> {
        const tokens = this.tokenize(query);

        // Find intersection of document sets
        let results = this.index.get(tokens[0]) || new Set<string>();

        for (let i = 1; i < tokens.length; i++) {
            const tokenDocs = this.index.get(tokens[i]) || new Set();
            results = new Set([...results].filter(id => tokenDocs.has(id)));
        }

        return Array.from(results);
    }

    private tokenize(text: string): string[] {
        return text
            .toLowerCase()
            .match(/\b\w+\b/g) || [];
    }

    private calculateHash(content: string): string {
        return require('crypto')
            .createHash('sha256')
            .update(content)
            .digest('hex');
    }
}
```

## Compression and Storage

### 1. Content Compression

```typescript
import zlib from 'zlib';
import brotli from 'brotli';

class CompressedDocumentStorage {
    async compressDocument(doc: DocumentContent): Promise<CompressedDoc> {
        const json = JSON.stringify(doc);

        // Compress with brotli (better compression)
        const compressed = await this.brotliCompress(json);

        return {
            id: doc.id,
            compressed: compressed.toString('base64'),
            originalSize: Buffer.byteLength(json),
            compressedSize: compressed.length,
            compressionRatio: (compressed.length / Buffer.byteLength(json) * 100).toFixed(1) + '%'
        };
    }

    private async brotliCompress(data: string): Promise<Buffer> {
        return new Promise((resolve, reject) => {
            brotli.compress(Buffer.from(data), (err, output) => {
                if (err) reject(err);
                else resolve(output);
            });
        });
    }

    async decompressDocument(compressed: CompressedDoc): Promise<DocumentContent> {
        const buffer = Buffer.from(compressed.compressed, 'base64');

        return new Promise((resolve, reject) => {
            brotli.decompress(buffer, (err, output) => {
                if (err) reject(err);
                else resolve(JSON.parse(output.toString()));
            });
        });
    }
}
```

## Performance Monitoring

### 1. Load Time Metrics

```typescript
class DocumentLoadMetrics {
    async trackLoadTime(docId: string, stage: 'fetch' | 'parse' | 'enhance' | 'render') {
        const startTime = performance.now();

        try {
            const result = await this.loadDocument(docId);

            const duration = performance.now() - startTime;

            // Log metrics
            logger.info('Document load', {
                doc_id: docId,
                stage,
                duration_ms: duration,
                cache_hit: this.wasCacheHit,
                timestamp: new Date()
            });

            // Alert if slow
            if (duration > 1000) {
                metrics.increment('doc.load.slow', { stage });
                logger.warn('Slow document load', { docId, stage, duration });
            }

            return result;
        } catch (error) {
            metrics.increment('doc.load.error', { stage });
            logger.error('Document load failed', { docId, stage, error });
            throw error;
        }
    }

    // Performance budgets
    validatePerformanceBudget(stage: string, duration: number): boolean {
        const budgets = {
            'fetch': 500,      // 500ms
            'parse': 200,      // 200ms
            'enhance': 1000,   // 1s
            'render': 300      // 300ms
        };

        const budget = budgets[stage as keyof typeof budgets];
        return duration <= budget;
    }
}
```

### 2. Memory Usage Monitoring

```typescript
class MemoryMonitor {
    trackMemoryUsage() {
        const memUsage = process.memoryUsage();

        metrics.gauge('memory.heapUsed', memUsage.heapUsed / 1024 / 1024); // MB
        metrics.gauge('memory.heapTotal', memUsage.heapTotal / 1024 / 1024);
        metrics.gauge('memory.external', memUsage.external / 1024 / 1024);

        // Alert if heap usage > 80%
        const heapUsagePercent = (memUsage.heapUsed / memUsage.heapTotal) * 100;
        if (heapUsagePercent > 80) {
            logger.warn('High memory usage', {
                heapUsagePercent: heapUsagePercent.toFixed(1)
            });

            // Trigger garbage collection
            if (global.gc) {
                global.gc();
            }
        }
    }
}
```

---

**Version**: 5.0.0 | **Last Updated**: 2025-11-22 | **Enterprise Ready**: ✓
