# Mongoose - Getting Started

**Pages:** 33

---

## Queries in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/queries.html

**Contents:**
- Queries in TypeScript
- Using lean() in TypeScript

Mongoose's Query class is a chainable query builder that represents a MongoDB query. When you call find(), findOne(), updateOne(), findOneAndUpdate(), etc. on a model, Mongoose will return a Query instance. Queries have a .then() function that returns a Promise, so you can use them with await.

In TypeScript, the Query class takes the following generic parameters:

The lean() method tells Mongoose to skip hydrating the result documents, making queries faster and more memory efficient. lean() comes with some caveats in TypeScript when working with the query transform() function. In general, we recommend calling lean() before using the transform() function to ensure accurate types.

In general, if you're having trouble with lean() inferring the correct type, you can try moving lean() earlier in the query chain.

**Examples:**

Example 1 (php):
```php
class Query<
  ResultType, // The type of the result of the query, like `DocType[]`
  DocType, // The hydrated document type of the query's associated model
  THelpers = {}, // Query helpers
  RawDocType = unknown, // The "lean" document type of the query's associated model
  QueryOp = 'find', // The operation that will be executed, like 'find', 'findOne', 'updateOne', etc.
  TDocOverrides = Record<string, never> // Methods and virtuals on the hydrated document
>
```

Example 2 (csharp):
```csharp
// Put `lean()` **before** `transform()` in TypeScript because `transform` modifies the query ResultType into a shape
// that `lean()` does not know how to handle.
const result = await ProjectModel
  .find()
  .lean()
  .transform((docs) => new Map(docs.map((doc) => [doc._id.toString(), doc])));

// Do **not** do the following
const result = await ProjectModel
  .find()
  .transform((docs) => new Map(docs.map((doc) => [doc._id.toString(), doc])))
  .lean();
```

---

## TypeScript Support

**URL:** https://mongoosejs.com/docs/5.x/docs/typescript.html

**Contents:**
- TypeScript Support
  - Creating Your First Document
  - Using extends Document
  - Using Custom Bindings
  - Next Up

Mongoose introduced officially supported TypeScript bindings in v5.11.0. Mongoose's index.d.ts file supports a wide variety of syntaxes and strives to be compatible with @types/mongoose where possible. This guide describes Mongoose's recommended approach to working with Mongoose in TypeScript.

To get started with Mongoose in TypeScript, you need to:

You as the developer are responsible for ensuring that your document interface lines up with your Mongoose schema. For example, Mongoose won't report an error if email is required in your Mongoose schema but optional in your document interface.

Alternatively, your document interface can extend Mongoose's Document class. Many Mongoose TypeScript codebases use the below approach.

This approach works, but we recommend your document interface not extend Document. Using extends Document makes it difficult for Mongoose to infer which properties are present on query filters, lean documents, and other cases.

We recommend your document interface contain the properties defined in your schema and line up with what your documents look like in MongoDB. Although you can add instance methods to your document interface, we do not recommend doing so.

If Mongoose's built-in index.d.ts file does not work for you, you can remove it in a postinstall script in your package.json as shown below. However, before you do, please open an issue on Mongoose's GitHub page and describe the issue you're experiencing.

Now that you've seen the basics of how to use Mongoose in TypeScript, let's take a look at statics in TypeScript.

**Examples:**

Example 1 (typescript):
```typescript
import { Schema, model, connect } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface User {
  name: string;
  email: string;
  avatar?: string;
}

// 2. Create a Schema corresponding to the document interface.
const schema = new Schema<User>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// 3. Create a Model.
const UserModel = model<User>('User', schema);

run().catch(err => console.log(err));

async function run(): Promise<void> {
  // 4. Connect to MongoDB
  await connect('mongodb://localhost:27017/test', {
    useNewUrlParser: true,
    useUnifiedTopology: true
  });

  const doc = new UserModel({
    name: 'Bill',
    email: 'bill@initech.com',
    avatar: 'https://i.imgur.com/dM7Thhn.png'
  });
  await doc.save();

  console.log(doc.email); // 'bill@initech.com'
}
```

Example 2 (typescript):
```typescript
import { Document, Schema, model, connect } from 'mongoose';

interface User extends Document {
  name: string;
  email: string;
  avatar?: string;
}
```

Example 3 (json):
```json
{
  "postinstall": "rm ./node_modules/mongoose/index.d.ts"
}
```

---

## Connections

**URL:** https://mongoosejs.com/docs/7.x/docs/connections.html

**Contents:**
- Connections
- Operation Buffering
- Error Handling
- Options
- serverSelectionTimeoutMS
- Callback
- Connection String Options
- Connection Events
- A note about keepAlive
- Replica Set Connections

You can connect to MongoDB with the mongoose.connect() method.

This is the minimum needed to connect the myapp database running locally on the default port (27017). For local MongoDB databases, we recommend using 127.0.0.1 instead of localhost. That is because Node.js 18 and up prefer IPv6 addresses, which means, on many machines, Node.js will resolve localhost to the IPv6 address ::1 and Mongoose will be unable to connect, unless the mongodb instance is running with ipv6 enabled.

You can also specify several more parameters in the uri:

See the mongodb connection string spec for more details.

Mongoose lets you start using your models immediately, without waiting for mongoose to establish a connection to MongoDB.

That's because mongoose buffers model function calls internally. This buffering is convenient, but also a common source of confusion. Mongoose will not throw any errors by default if you use a model without connecting.

To disable buffering, turn off the bufferCommands option on your schema. If you have bufferCommands on and your connection is hanging, try turning bufferCommands off to see if you haven't opened a connection properly. You can also disable bufferCommands globally:

Note that buffering is also responsible for waiting until Mongoose creates collections if you use the autoCreate option. If you disable buffering, you should also disable the autoCreate option and use createCollection() to create capped collections or collections with collations.

There are two classes of errors that can occur with a Mongoose connection.

To handle initial connection errors, you should use .catch() or try/catch with async/await.

To handle errors after initial connection was established, you should listen for error events on the connection. However, you still need to handle initial connection errors as shown above.

Note that Mongoose does not necessarily emit an 'error' event if it loses connectivity to MongoDB. You should listen to the disconnected event to report when Mongoose is disconnected from MongoDB.

The connect method also accepts an options object which will be passed on to the underlying MongoDB driver.

A full list of options can be found on the MongoDB Node.js driver docs for MongoClientOptions. Mongoose passes options to the driver without modification, modulo a few exceptions that are explained below.

Below are some of the options that are important for tuning Mongoose.

The serverSelectionTimeoutMS option is extremely important: it controls how long the MongoDB Node.js driver will attempt to retry any operation before erroring out. This includes initial connection, like await mongoose.connect(), as well as any operations that make requests to MongoDB, like save() or find().

By default, serverSelectionTimeoutMS is 30000 (30 seconds). This means that, for example, if you call mongoose.connect() when your standalone MongoDB server is down, your mongoose.connect() call will only throw an error after 30 seconds.

Similarly, if your standalone MongoDB server goes down after initial connection, any find() or save() calls will error out after 30 seconds, unless your MongoDB server is restarted.

While 30 seconds seems like a long time, serverSelectionTimeoutMS means you're unlikely to see any interruptions during a replica set failover. If you lose your replica set primary, the MongoDB Node driver will ensure that any operations you send during the replica set election will eventually execute, assuming that the replica set election takes less than serverSelectionTimeoutMS.

To get faster feedback on failed connections, you can reduce serverSelectionTimeoutMS to 5000 as follows. We don't recommend reducing serverSelectionTimeoutMS unless you are running a standalone MongoDB server rather than a replica set, or unless you are using a serverless runtime like AWS Lambda.

There is no way to tune serverSelectionTimeoutMS independently for mongoose.connect() vs for queries. If you want to reduce serverSelectionTimeoutMS for queries and other operations, but still retry mongoose.connect() for longer, you are responsible for retrying the connect() calls yourself using a for loop or a tool like p-retry.

The connect() function also accepts a callback parameter and returns a promise.

You can also specify driver options in your connection string as parameters in the query string portion of the URI. This only applies to options passed to the MongoDB driver. You can't set Mongoose-specific options like bufferCommands in the query string.

The disadvantage of putting options in the query string is that query string options are harder to read. The advantage is that you only need a single configuration option, the URI, rather than separate options for socketTimeoutMS, etc. Best practice is to put options that likely differ between development and production, like replicaSet or ssl, in the connection string, and options that should remain constant, like socketTimeoutMS or maxPoolSize, in the options object.

The MongoDB docs have a full list of supported connection string options. Below are some options that are often useful to set in the connection string because they are closely associated with the hostname and authentication information.

Connections inherit from Node.js' EventEmitter class, and emit events when something happens to the connection, like losing connectivity to the MongoDB server. Below is a list of events that a connection may emit.

When you're connecting to a single MongoDB server (a "standalone"), Mongoose will emit 'disconnected' if it gets disconnected from the standalone server, and 'connected' if it successfully connects to the standalone. In a replica set, Mongoose will emit 'disconnected' if it loses connectivity to the replica set primary, and 'connected' if it manages to reconnect to the replica set primary.

Before Mongoose 5.2.0, you needed to enable the keepAlive option to initiate TCP keepalive to prevent "connection closed" errors errors. However, keepAlive has been true by default since Mongoose 5.2.0, and the keepAlive is deprecated as of Mongoose 7.2.0. Please remove keepAlive and keepAliveInitialDelay options from your Mongoose connections.

To connect to a replica set you pass a comma delimited list of hosts to connect to rather than a single host.

To connect to a single node replica set, specify the replicaSet option.

The underlying MongoDB driver uses a process known as server selection to connect to MongoDB and send operations to MongoDB. If the MongoDB driver can't find a server to send an operation to after serverSelectionTimeoutMS, you'll get the below error:

You can configure the timeout using the serverSelectionTimeoutMS option to mongoose.connect():

A MongoTimeoutError has a reason property that explains why server selection timed out. For example, if you're connecting to a standalone server with an incorrect password, reason will contain an "Authentication failed" error.

MongoDB replica sets rely on being able to reliably figure out the domain name for each member.On Linux and OSX, the MongoDB server uses the output of the hostname command to figure out the domain name to report to the replica set. This can cause confusing errors if you're connecting to a remote MongoDB replica set running on a machine that reports its hostname as localhost:

If you're experiencing a similar error, connect to the replica set using the mongo shell and run the rs.conf() command to check the host names of each replica set member. Follow this page's instructions to change a replica set member's host name.

You can also check the reason.servers property of MongooseServerSelectionError to see what the MongoDB Node driver thinks the state of your replica set is. The reason.servers property contains a map of server descriptions.

You can also connect to multiple mongos instances for high availability in a sharded cluster. You do not need to pass any special options to connect to multiple mongos in mongoose 5.x.

So far we've seen how to connect to MongoDB using Mongoose's default connection. Mongoose creates a default connection when you call mongoose.connect(). You can access the default connection using mongoose.connection.

You may need multiple connections to MongoDB for several reasons. One reason is if you have multiple databases or multiple MongoDB clusters. Another reason is to work around slow trains. The mongoose.createConnection() function takes the same arguments as mongoose.connect() and returns a new connection.

This connection object is then used to create and retrieve models. Models are always scoped to a single connection.

The createConnection() function returns a connection instance, not a promise. If you want to use await to make sure Mongoose successfully connects to MongoDB, use the asPromise() function:

If you use multiple connections, you should make sure you export schemas, not models. Exporting a model from a file is called the export model pattern. The export model pattern is limited because you can only use one connection.

If you use the export schema pattern, you still need to create models somewhere. There are two common patterns. First is to export a connection and register the models on the connection in the file:

Another alternative is to register connections with a dependency injector or another inversion of control (IOC) pattern.

Each connection, whether created with mongoose.connect or mongoose.createConnection are all backed by an internal configurable connection pool defaulting to a maximum size of 100. Adjust the pool size using your connection options:

The connection pool size is important because MongoDB currently can only process one operation per socket. So maxPoolSize functions as a cap on the number of concurrent operations.

In the context of Mongoose, a multi-tenant architecture typically means a case where multiple different clients talk to MongoDB through a single Mongoose application. This typically means each client makes queries and executes updates through a single Mongoose application, but has a distinct MongoDB database within the same MongoDB cluster.

We recommend reading this article about multi-tenancy with Mongoose; it has a good description of how we define multi-tenancy and a more detailed overview of our recommended patterns.

There are two patterns we recommend for multi-tenancy in Mongoose:

The following is an example of pattern (1). We recommend pattern (1) for cases where you have a small number of tenants, or if each individual tenant's workload is light (approximately < 1 request per second, all requests take < 10ms of database processing time). Pattern (1) is simpler to implement and simpler to manage in production, because there is only 1 connection pool. But, under high load, you will likely run into issues where some tenants' operations slow down other tenants' operations due to slow trains.

The following is an example of pattern (2). Pattern (2) is more flexible and better for use cases with > 10k tenants and > 1 requests/second. Because each tenant has a separate connection pool, one tenants' slow operations will have minimal impact on other tenants. However, this pattern is harder to implement and manage in production. In particular, MongoDB does have a limit on the number of open connections, and MongoDB Atlas has separate limits on the number of open connections, so you need to make sure the total number of sockets in your connection pools doesn't go over MongoDB's limits.

Now that we've covered connections, let's take a look at models.

**Examples:**

Example 1 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
```

Example 2 (javascript):
```javascript
mongoose.connect('mongodb://username:password@host:port/database?options...');
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Works
await MyModel.findOne();
```

Example 4 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const promise = MyModel.findOne();

setTimeout(function() {
  mongoose.connect('mongodb://127.0.0.1:27017/myapp');
}, 60000);

// Will just hang until mongoose successfully connects
await promise;
```

---

## Statics in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/statics-and-methods.html

**Contents:**
- Statics in TypeScript
- With Generics
- Using loadClass() with TypeScript
  - Basic Usage
  - Typing this Inside Methods
  - Getters / Setters Limitation
  - Full Example Code
  - When Should I Use loadClass()?

To use Mongoose's automatic type inference to define types for your statics and methods, you should define your methods and statics using the methods and statics schema options as follows. Do not use the Schema.prototype.method() and Schema.prototype.static() functions, because Mongoose's automatic type inference system cannot detect methods and statics defined using those functions.

We recommend using Mongoose's automatic type inference where possible, but you can use Schema and Model generics to set up type inference for your statics and methods. Mongoose models do not have an explicit generic parameter for statics. If your model has statics, we recommend creating an interface that extends Mongoose's Model interface as shown below.

You should pass methods as the 3rd generic param to the Schema constructor as follows.

Mongoose supports applying ES6 classes to a schema using schema.loadClass() as an alternative to defining statics and methods in your schema. When using TypeScript, there are a few important typing details to understand.

loadClass() copies static methods, instance methods, and ES getters/setters from the class onto the schema.

Mongoose does not automatically update TypeScript types for class members. To get full type support, you must manually define types using Mongoose's Model and HydratedDocument generics.

You can annotate this in methods to enable full safety, using the Model and HydratedDocument types you defined. Note that this must be done for each method individually; it is not possible to set a this type for the entire class at once.

TypeScript currently does not allow this parameters on getters/setters:

This is a TypeScript limitation. See: TypeScript issue #52923

As a workaround, you can cast this to the document type inside your getter:

loadClass() is useful for defining methods and statics in classes. If you have a strong preference for classes, you can use loadClass(); however, we recommend defining statics and methods in schema options as described in the first section.

The major downside of loadClass() in TypeScript is that it requires manual TypeScript types. If you want better type inference, you can use schema options methods and statics.

**Examples:**

Example 1 (json):
```json
const userSchema = new mongoose.Schema(
  { name: { type: String, required: true } },
  {
    methods: {
      updateName(name: string) {
        this.name = name;
        return this.save();
      }
    },
    statics: {
      createWithName(name: string) {
        return this.create({ name });
      }
    }
  }
);
const UserModel = mongoose.model('User', userSchema);

const doc = new UserModel({ name: 'test' });
// Compiles correctly
doc.updateName('foo');
// Compiles correctly
UserModel.createWithName('bar');
```

Example 2 (typescript):
```typescript
import { Model, Schema, model } from 'mongoose';

interface IUser {
  name: string;
}

interface UserModelType extends Model<IUser> {
  myStaticMethod(): number;
}

const schema = new Schema<IUser, UserModelType>({ name: String });
schema.static('myStaticMethod', function myStaticMethod() {
  return 42;
});

const User = model<IUser, UserModelType>('User', schema);

const answer: number = User.myStaticMethod(); // 42
```

Example 3 (typescript):
```typescript
import { Model, Schema, model } from 'mongoose';

interface IUser {
  name: string;
}

interface UserMethods {
  updateName(name: string): Promise<any>;
}

const schema = new Schema<IUser, Model<IUser>, UserMethods>({ name: String });
schema.method('updateName', function updateName(name) {
  this.name = name;
  return this.save();
});

const User = model('User', schema);
const doc = new User({ name: 'test' });
// Compiles correctly
doc.updateName('foo');
```

Example 4 (php):
```php
class MyClass {
  myMethod() {
    return 42;
  }

  static myStatic() {
    return 42;
  }

  get myVirtual() {
    return 42;
  }
}

const schema = new Schema({ property1: String });
schema.loadClass(MyClass);
```

---

## Connections

**URL:** https://mongoosejs.com/docs/6.x/docs/connections.html

**Contents:**
- Connections
- Operation Buffering
- Error Handling
- Options
- Callback
- Connection String Options
- Connection Events
- A note about keepAlive
- Replica Set Connections
- Server Selection

You can connect to MongoDB with the mongoose.connect() method.

This is the minimum needed to connect the myapp database running locally on the default port (27017). If connecting fails on your machine, try using 127.0.0.1 instead of localhost.

You can also specify several more parameters in the uri:

See the mongodb connection string spec for more details.

Mongoose lets you start using your models immediately, without waiting for mongoose to establish a connection to MongoDB.

That's because mongoose buffers model function calls internally. This buffering is convenient, but also a common source of confusion. Mongoose will not throw any errors by default if you use a model without connecting.

To disable buffering, turn off the bufferCommands option on your schema. If you have bufferCommands on and your connection is hanging, try turning bufferCommands off to see if you haven't opened a connection properly. You can also disable bufferCommands globally:

Note that buffering is also responsible for waiting until Mongoose creates collections if you use the autoCreate option. If you disable buffering, you should also disable the autoCreate option and use createCollection() to create capped collections or collections with collations.

There are two classes of errors that can occur with a Mongoose connection.

To handle initial connection errors, you should use .catch() or try/catch with async/await.

To handle errors after initial connection was established, you should listen for error events on the connection. However, you still need to handle initial connection errors as shown above.

Note that Mongoose does not necessarily emit an 'error' event if it loses connectivity to MongoDB. You should listen to the disconnected event to report when Mongoose is disconnected from MongoDB.

The connect method also accepts an options object which will be passed on to the underlying MongoDB driver.

A full list of options can be found on the MongoDB Node.js driver docs for MongoClientOptions. Mongoose passes options to the driver without modification, modulo a few exceptions that are explained below.

Below are some of the options that are important for tuning Mongoose.

The serverSelectionTimeoutMS option also handles how long mongoose.connect() will retry initial connection before erroring out. mongoose.connect() will retry for 30 seconds by default (default serverSelectionTimeoutMS) before erroring out. To get faster feedback on failed operations, you can reduce serverSelectionTimeoutMS to 5000 as shown below.

See this page for more information about connectTimeoutMS and socketTimeoutMS

The connect() function also accepts a callback parameter and returns a promise.

You can also specify driver options in your connection string as parameters in the query string portion of the URI. This only applies to options passed to the MongoDB driver. You can't set Mongoose-specific options like bufferCommands in the query string.

The disadvantage of putting options in the query string is that query string options are harder to read. The advantage is that you only need a single configuration option, the URI, rather than separate options for socketTimeoutMS, connectTimeoutMS, etc. Best practice is to put options that likely differ between development and production, like replicaSet or ssl, in the connection string, and options that should remain constant, like connectTimeoutMS or maxPoolSize, in the options object.

The MongoDB docs have a full list of supported connection string options. Below are some options that are often useful to set in the connection string because they are closely associated with the hostname and authentication information.

Connections inherit from Node.js' EventEmitter class, and emit events when something happens to the connection, like losing connectivity to the MongoDB server. Below is a list of events that a connection may emit.

When you're connecting to a single MongoDB server (a "standalone"), Mongoose will emit 'disconnected' if it gets disconnected from the standalone server, and 'connected' if it successfully connects to the standalone. In a replica set, Mongoose will emit 'disconnected' if it loses connectivity to the replica set primary, and 'connected' if it manages to reconnect to the replica set primary.

For long running applications, it is often prudent to enable keepAlive with a number of milliseconds. Without it, after some period of time you may start to see "connection closed" errors for what seems like no reason. If so, after reading this, you may decide to enable keepAlive:

keepAliveInitialDelay is the number of milliseconds to wait before initiating keepAlive on the socket. keepAlive is true by default since mongoose 5.2.0.

To connect to a replica set you pass a comma delimited list of hosts to connect to rather than a single host.

To connect to a single node replica set, specify the replicaSet option.

The underlying MongoDB driver uses a process known as server selection to connect to MongoDB and send operations to MongoDB. If the MongoDB driver can't find a server to send an operation to after serverSelectionTimeoutMS, you'll get the below error:

You can configure the timeout using the serverSelectionTimeoutMS option to mongoose.connect():

A MongoTimeoutError has a reason property that explains why server selection timed out. For example, if you're connecting to a standalone server with an incorrect password, reason will contain an "Authentication failed" error.

MongoDB replica sets rely on being able to reliably figure out the domain name for each member. On Linux and OSX, the MongoDB server uses the output of the hostname command to figure out the domain name to report to the replica set. This can cause confusing errors if you're connecting to a remote MongoDB replica set running on a machine that reports its hostname as localhost:

If you're experiencing a similar error, connect to the replica set using the mongo shell and run the rs.conf() command to check the host names of each replica set member. Follow this page's instructions to change a replica set member's host name.

You can also check the reason.servers property of MongooseServerSelectionError to see what the MongoDB Node driver thinks the state of your replica set is. The reason.servers property contains a map of server descriptions.

You can also connect to multiple mongos instances for high availability in a sharded cluster. You do not need to pass any special options to connect to multiple mongos in mongoose 5.x.

So far we've seen how to connect to MongoDB using Mongoose's default connection. Mongoose creates a default connection when you call mongoose.connect(). You can access the default connection using mongoose.connection.

You may need multiple connections to MongoDB for several reasons. One reason is if you have multiple databases or multiple MongoDB clusters. Another reason is to work around slow trains. The mongoose.createConnection() function takes the same arguments as mongoose.connect() and returns a new connection.

This connection object is then used to create and retrieve models. Models are always scoped to a single connection.

If you use multiple connections, you should make sure you export schemas, not models. Exporting a model from a file is called the export model pattern. The export model pattern is limited because you can only use one connection.

If you use the export schema pattern, you still need to create models somewhere. There are two common patterns. First is to export a connection and register the models on the connection in the file:

Another alternative is to register connections with a dependency injector or another inversion of control (IOC) pattern.

Each connection, whether created with mongoose.connect or mongoose.createConnection are all backed by an internal configurable connection pool defaulting to a maximum size of 100. Adjust the pool size using your connection options:

Now that we've covered connections, let's take a look at models.

**Examples:**

Example 1 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
```

Example 2 (javascript):
```javascript
mongoose.connect('mongodb://username:password@host:port/database?options...');
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Works
MyModel.findOne(function(error, result) { /* ... */ });
```

Example 4 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Will just hang until mongoose successfully connects
MyModel.findOne(function(error, result) { /* ... */ });

setTimeout(function() {
  mongoose.connect('mongodb://127.0.0.1:27017/myapp');
}, 60000);
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/guides.html

**Contents:**
- Guides
  - Mongoose Core Concepts
  - Advanced Topics
  - Integrations
  - Migration Guides

Mongoose guides provide detailed tutorials on Mongoose's core concepts and integrating Mongoose with external tools and frameworks.

---

## TypeScript Support

**URL:** https://mongoosejs.com/docs/7.x/docs/typescript.html

**Contents:**
- TypeScript Support
- Creating Your First Document
- ObjectIds and Other Mongoose Types
- Using Custom Bindings
- Next Up

Mongoose introduced officially supported TypeScript bindings in v5.11.0. Mongoose's index.d.ts file supports a wide variety of syntaxes and strives to be compatible with @types/mongoose where possible. This guide describes Mongoose's recommended approach to working with Mongoose in TypeScript.

To get started with Mongoose in TypeScript, you need to:

You as the developer are responsible for ensuring that your document interface lines up with your Mongoose schema. For example, Mongoose won't report an error if email is required in your Mongoose schema but optional in your document interface.

The User() constructor returns an instance of HydratedDocument<IUser>. IUser is a document interface, it represents the raw object structure that IUser objects look like in MongoDB. HydratedDocument<IUser> represents a hydrated Mongoose document, with methods, virtuals, and other Mongoose-specific features.

To define a property of type ObjectId, you should use Types.ObjectId in the TypeScript document interface. You should use 'ObjectId' or Schema.Types.ObjectId in your schema definition.

That's because Schema.Types.ObjectId is a class that inherits from SchemaType, not the class you use to create a new MongoDB ObjectId.

If Mongoose's built-in index.d.ts file does not work for you, you can remove it in a postinstall script in your package.json as shown below. However, before you do, please open an issue on Mongoose's GitHub page and describe the issue you're experiencing.

Now that you've seen the basics of how to use Mongoose in TypeScript, let's take a look at statics in TypeScript.

**Examples:**

Example 1 (typescript):
```typescript
import { Schema, model, connect } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  avatar?: string;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// 3. Create a Model.
const User = model<IUser>('User', userSchema);

run().catch(err => console.log(err));

async function run() {
  // 4. Connect to MongoDB
  await connect('mongodb://127.0.0.1:27017/test');

  const user = new User({
    name: 'Bill',
    email: 'bill@initech.com',
    avatar: 'https://i.imgur.com/dM7Thhn.png'
  });
  await user.save();

  console.log(user.email); // 'bill@initech.com'
}
```

Example 2 (sql):
```sql
import { HydratedDocument } from 'mongoose';

const user: HydratedDocument<IUser> = new User({
  name: 'Bill',
  email: 'bill@initech.com',
  avatar: 'https://i.imgur.com/dM7Thhn.png'
});
```

Example 3 (typescript):
```typescript
import { Schema, Types } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  // Use `Types.ObjectId` in document interface...
  organization: Types.ObjectId;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  // And `Schema.Types.ObjectId` in the schema definition.
  organization: { type: Schema.Types.ObjectId, ref: 'Organization' }
});
```

Example 4 (json):
```json
{
  "postinstall": "rm ./node_modules/mongoose/index.d.ts"
}
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/connections.html

**Contents:**
- Connections
  - Operation Buffering
  - Error Handling
  - Options
  - Callback
  - Connection String Options
  - Connection Events
  - A note about keepAlive
  - Replica Set Connections
  - Server Selection

You can connect to MongoDB with the mongoose.connect() method.

This is the minimum needed to connect the myapp database running locally on the default port (27017). If connecting fails on your machine, try using 127.0.0.1 instead of localhost.

You can also specify several more parameters in the uri:

See the mongodb connection string spec for more detail.

Mongoose lets you start using your models immediately, without waiting for mongoose to establish a connection to MongoDB.

That's because mongoose buffers model function calls internally. This buffering is convenient, but also a common source of confusion. Mongoose will not throw any errors by default if you use a model without connecting.

To disable buffering, turn off the bufferCommands option on your schema. If you have bufferCommands on and your connection is hanging, try turning bufferCommands off to see if you haven't opened a connection properly. You can also disable bufferCommands globally:

Note that buffering is also responsible for waiting until Mongoose creates collections if you use the autoCreate option. If you disable buffering, you should also disable the autoCreate option and use createCollection() to create capped collections or collections with collations.

There are two classes of errors that can occur with a Mongoose connection.

To handle initial connection errors, you should use .catch() or try/catch with async/await.

To handle errors after initial connection was established, you should listen for error events on the connection. However, you still need to handle initial connection errors as shown above.

Note that Mongoose does not necessarily emit an 'error' event if it loses connectivity to MongoDB. You should listen to the disconnected event to report when Mongoose is disconnected from MongoDB.

The connect method also accepts an options object which will be passed on to the underlying MongoDB driver.

A full list of options can be found on the MongoDB Node.js driver docs for connect(). Mongoose passes options to the driver without modification, modulo a few exceptions that are explained below.

Below are some of the options that are important for tuning Mongoose.

The following options are important for tuning Mongoose only if you are running without the useUnifiedTopology option:

The following options are important for tuning Mongoose only if you are running with the useUnifiedTopology option:

The serverSelectionTimeoutMS option also handles how long mongoose.connect() will retry initial connection before erroring out. With useUnifiedTopology, mongoose.connect() will retry for 30 seconds by default (default serverSelectionTimeoutMS) before erroring out. To get faster feedback on failed operations, you can reduce serverSelectionTimeoutMS to 5000 as shown below.

See this page for more information about connectTimeoutMS and socketTimeoutMS

The connect() function also accepts a callback parameter and returns a promise.

You can also specify driver options in your connection string as parameters in the query string portion of the URI. This only applies to options passed to the MongoDB driver. You can't set Mongoose-specific options like bufferCommands in the query string.

The disadvantage of putting options in the query string is that query string options are harder to read. The advantage is that you only need a single configuration option, the URI, rather than separate options for socketTimeoutMS, connectTimeoutMS, etc. Best practice is to put options that likely differ between development and production, like replicaSet or ssl, in the connection string, and options that should remain constant, like connectTimeoutMS or poolSize, in the options object.

The MongoDB docs have a full list of supported connection string options. Below are some options that are often useful to set in the connection string because they are closely associated with the hostname and authentication information.

Connections inherit from Node.js' EventEmitter class, and emit events when something happens to the connection, like losing connectivity to the MongoDB server. Below is a list of events that a connection may emit.

When you're connecting to a single MongoDB server (a "standalone"), Mongoose will emit 'disconnected' if it gets disconnected from the standalone server, and 'connected' if it successfully connects to the standalone. In a replica set with useUnifiedTopology = true, Mongoose will emit 'disconnected' if it loses connectivity to every server in the replica set, and 'connected' if it manages to reconnect to at least one server in the replica set.

For long running applications, it is often prudent to enable keepAlive with a number of milliseconds. Without it, after some period of time you may start to see "connection closed" errors for what seems like no reason. If so, after reading this, you may decide to enable keepAlive:

keepAliveInitialDelay is the number of milliseconds to wait before initiating keepAlive on the socket. keepAlive is true by default since mongoose 5.2.0.

To connect to a replica set you pass a comma delimited list of hosts to connect to rather than a single host.

To connect to a single node replica set, specify the replicaSet option.

If you enable the useUnifiedTopology option, the underlying MongoDB driver will use server selection to connect to MongoDB and send operations to MongoDB. If the MongoDB driver can't find a server to send an operation to after serverSelectionTimeoutMS, you'll get the below error:

You can configure the timeout using the serverSelectionTimeoutMS option to mongoose.connect():

A MongoTimeoutError has a reason property that explains why server selection timed out. For example, if you're connecting to a standalone server with an incorrect password, reason will contain an "Authentication failed" error.

MongoDB replica sets rely on being able to reliably figure out the domain name for each member. On Linux and OSX, the MongoDB server uses the output of the hostname command to figure out the domain name to report to the replica set. This can cause confusing errors if you're connecting to a remote MongoDB replica set running on a machine that reports its hostname as localhost:

If you're experiencing a similar error, connect to the replica set using the mongo shell and run the rs.conf() command to check the host names of each replica set member. Follow this page's instructions to change a replica set member's host name.

You can also connect to multiple mongos instances for high availability in a sharded cluster. You do not need to pass any special options to connect to multiple mongos in mongoose 5.x.

So far we've seen how to connect to MongoDB using Mongoose's default connection. Mongoose creates a default connection when you call mongoose.connect(). You can access the default connection using mongoose.connection.

You may need multiple connections to MongoDB for several reasons. One reason is if you have multiple databases or multiple MongoDB clusters. Another reason is to work around slow trains. The mongoose.createConnection() function takes the same arguments as mongoose.connect() and returns a new connection.

This connection object is then used to create and retrieve models. Models are always scoped to a single connection.

If you use multiple connections, you should make sure you export schemas, not models. Exporting a model from a file is called the export model pattern. The export model pattern is limited because you can only use one connection.

If you use the export schema pattern, you still need to create models somewhere. There are two common patterns. First is to export a connection and register the models on the connection in the file:

Another alternative is to register connections with a dependency injector or another inversion of control (IOC) pattern.

Each connection, whether created with mongoose.connect or mongoose.createConnection are all backed by an internal configurable connection pool defaulting to a maximum size of 5. Adjust the pool size using your connection options:

You may see the following deprecation warning if upgrading from 4.x to 5.x and you didn't use the useMongoClient option in 4.x:

In older version of the MongoDB driver you had to specify distinct options for server connections, replica set connections, and mongos connections:

In mongoose v5.x you can instead declare these options at the top level, without all that extra nesting. Here's the list of all supported options.

Now that we've covered connections, let's take a look at models.

**Examples:**

Example 1 (javascript):
```javascript
mongoose.connect('mongodb://localhost:27017/myapp', {useNewUrlParser: true});
```

Example 2 (javascript):
```javascript
mongoose.connect('mongodb://username:password@host:port/database?options...', {useNewUrlParser: true});
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://localhost:27017/myapp', {useNewUrlParser: true});
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Works
MyModel.findOne(function(error, result) { /* ... */ });
```

Example 4 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Will just hang until mongoose successfully connects
MyModel.findOne(function(error, result) { /* ... */ });

setTimeout(function() {
  mongoose.connect('mongodb://localhost:27017/myapp', {useNewUrlParser: true});
}, 60000);
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/index.html

**Contents:**
- Getting Started
  - Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

We have a pending connection to the test database running on localhost. We now need to get notified if we connect successfully or if a connection error occurs:

Once our connection opens, our callback will be called. For brevity, let's assume that all following code is within this callback.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method. The first argument to the callback will be an error if any occurred.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens to the callback.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
$ npm install mongoose --save
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');
mongoose.connect('mongodb://localhost:27017/test', {useNewUrlParser: true, useUnifiedTopology: true});
```

Example 3 (javascript):
```javascript
const db = mongoose.connection;
db.on('error', console.error.bind(console, 'connection error:'));
db.once('open', function() {
  // we're connected!
});
```

Example 4 (javascript):
```javascript
const kittySchema = new mongoose.Schema({
  name: String
});
```

---

## Guides

**URL:** https://mongoosejs.com/docs/7.x/docs/guides.html

**Contents:**
- Guides
- Mongoose Core Concepts
- Advanced Topics
- Integrations
- Other Guides
- Migration Guides

Mongoose guides provide detailed tutorials on Mongoose's core concepts and integrating Mongoose with external tools and frameworks.

---

## Virtuals in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/virtuals.html

**Contents:**
- Virtuals in TypeScript
- Automatically Inferred Types
- Set virtuals type manually
- Override the Type of this in Your Virtual

Virtuals are computed properties: you can access virtuals on hydrated Mongoose documents, but virtuals are not stored in MongoDB. Mongoose supports auto typed virtuals so you don't need to define additional typescript interface anymore but you are still able to do so.

To make mongoose able to infer virtuals type, You have to define them in schema constructor as following:

Note that Mongoose does not include virtuals in the returned type from InferSchemaType. That is because InferSchemaType returns a value similar to the raw document interface, which represents the structure of the data stored in MongoDB.

However, Mongoose does add the virtuals to the model type.

You shouldn't define virtuals in your TypeScript document interface. Instead, you should define a separate interface for your virtuals, and pass this interface to Model and Schema.

For example, suppose you have a UserDoc interface, and you want to define a fullName virtual. Below is how you can define a separate UserVirtuals interface for fullName.

In case the value of this in your virtual is incorrect for some reason, you can always override it using the generic parameter in the virtual() function.

**Examples:**

Example 1 (json):
```json
import { Schema, Model, model } from 'mongoose';

const schema = new Schema(
  {
    firstName: String,
    lastName: String
  },
  {
    virtuals: {
      fullName: {
        get() {
          return `${this.firstName} ${this.lastName}`;
        }
        // virtual setter and options can be defined here as well.
      }
    }
  }
);
```

Example 2 (typescript):
```typescript
type User = InferSchemaType<typeof schema>;

const user: User = {};
// Property 'fullName' does not exist on type '{ firstName?: string | undefined; ... }'.
user.fullName;
```

Example 3 (typescript):
```typescript
const UserModel = model('User', schema);

const user = new UserModel({ firstName: 'foo' });
// Works
user.fullName;

// Here's how to get the hydrated document type
type UserDocument = ReturnType<(typeof UserModel)['hydrate']>;
```

Example 4 (typescript):
```typescript
import { Schema, Model, model } from 'mongoose';

interface UserDoc {
  firstName: string;
  lastName: string;
}

interface UserVirtuals {
  fullName: string;
}

type UserModelType = Model<UserDoc, {}, {}, UserVirtuals>; // <-- add virtuals here...

const schema = new Schema<UserDoc, UserModelType, {}, {}, UserVirtuals>({ // <-- and here
  firstName: String,
  lastName: String
});

schema.virtual('fullName').get(function() {
  return `${this.firstName} ${this.lastName}`;
});
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/8.x/index.html

---

## 

**URL:** https://mongoosejs.com/docs/5.x/index.html

**Contents:**
- Getting Started
  - Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

We have a pending connection to the test database running on localhost. We now need to get notified if we connect successfully or if a connection error occurs:

Once our connection opens, our callback will be called. For brevity, let's assume that all following code is within this callback.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method. The first argument to the callback will be an error if any occurred.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens to the callback.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
$ npm install mongoose --save
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');
mongoose.connect('mongodb://localhost:27017/test', {useNewUrlParser: true, useUnifiedTopology: true});
```

Example 3 (javascript):
```javascript
const db = mongoose.connection;
db.on('error', console.error.bind(console, 'connection error:'));
db.once('open', function() {
  // we're connected!
});
```

Example 4 (javascript):
```javascript
const kittySchema = new mongoose.Schema({
  name: String
});
```

---

## Guides

**URL:** https://mongoosejs.com/docs/guides.html

**Contents:**
- Guides
- Mongoose Core Concepts
- Advanced Topics
- Integrations
- Other Guides
- Migration Guides

Mongoose guides provide detailed tutorials on Mongoose's core concepts and integrating Mongoose with external tools and frameworks.

---

## TLS/SSL Connections

**URL:** https://mongoosejs.com/docs/tutorials/ssl.html

**Contents:**
- TLS/SSL Connections
- TLS/SSL Validation
- X.509 Authentication
- X.509 Authentication with MongoDB Atlas

Mongoose supports connecting to MongoDB clusters that require TLS/SSL connections. Setting the tls option to true in mongoose.connect() or your connection string is enough to connect to a MongoDB cluster using TLS/SSL:

The tls option defaults to false for connection strings that start with mongodb://. However, the tls option defaults to true for connection strings that start with mongodb+srv://. So if you are using an srv connection string to connect to MongoDB Atlas, TLS/SSL is enabled by default.

If you try to connect to a MongoDB cluster that requires TLS/SSL without enabling the tls/ssl option, mongoose.connect() will error out with the below error:

By default, Mongoose validates the TLS/SSL certificate against a certificate authority to ensure the TLS/SSL certificate is valid. To disable this validation, set the tlsAllowInvalidCertificates (or tlsInsecure) option to true.

In most cases, you should not disable TLS/SSL validation in production. However, tlsAllowInvalidCertificates: true is often helpful for debugging SSL connection issues. If you can connect to MongoDB with tlsAllowInvalidCertificates: true, but not with tlsAllowInvalidCertificates: false, then you can confirm Mongoose can connect to the server and the server is configured to use TLS/SSL correctly, but there's some issue with the certificate.

For example, a common issue is the below error message:

This error is often caused by self-signed MongoDB certificates or other situations where the certificate sent by the MongoDB server is not registered with an established certificate authority. The solution is to set the tlsCAFile option, which essentially sets a list of allowed SSL certificates.

Another common issue is the below error message:

The SSL certificate's common name must line up with the host name in your connection string. If the SSL certificate is for hostname2.mydomain.com, your connection string must connect to hostname2.mydomain.com, not any other hostname or IP address that may be equivalent to hostname2.mydomain.com. For replica sets, this also means that the SSL certificate's common name must line up with the machine's hostname. To disable this validation, set the tlsAllowInvalidHostnames option to true.

If you're using X.509 authentication, you should set the user name in the connection string, not the connect() options.

With MongoDB Atlas, X.509 certificates are not Root CA certificates and will not work with the tlsCAFile parameter as self-signed certificates would. If the tlsCAFile parameter is used an error similar to the following would be raised:

To connect to a MongoDB Atlas cluster using X.509 authentication the correct option to set is tlsCertificateKeyFile. The connection string already specifies the authSource and authMechanism, however they're included below as connect() options for completeness:

Note The connection string options must be URL escaped correctly.

**Examples:**

Example 1 (css):
```css
mongoose.connect('mongodb://127.0.0.1:27017/test', { tls: true });

// Equivalent:
mongoose.connect('mongodb://127.0.0.1:27017/test?tls=true');
```

Example 2 (css):
```css
mongoose.connect('mongodb://127.0.0.1:27017/test', {
  tls: true,
  tlsAllowInvalidCertificates: true,
});
```

Example 3 (python):
```python
await mongoose.connect('mongodb://127.0.0.1:27017/test', {
  tls: true,
  // For example, see https://medium.com/@rajanmaharjan/secure-your-mongodb-connections-ssl-tls-92e2addb3c89
  // for where the `rootCA.pem` file comes from.
  tlsCAFile: `${__dirname}/rootCA.pem`,
});
```

Example 4 (python):
```python
// Do this:
const username = 'myusername';
await mongoose.connect(`mongodb://${encodeURIComponent(username)}@127.0.0.1:27017/test`, {
  tls: true,
  tlsCAFile: `${__dirname}/rootCA.pem`,
  authMechanism: 'MONGODB-X509',
});

// Not this:
await mongoose.connect('mongodb://127.0.0.1:27017/test', {
  tls: true,
  tlsCAFile: `${__dirname}/rootCA.pem`,
  authMechanism: 'MONGODB-X509',
  auth: { username },
});
```

---

## Query Helpers in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/query-helpers.html

**Contents:**
- Query Helpers in TypeScript
- Manually Typed Query Helpers
- Auto Typed Query Helpers
- Using Query Helper Overrides For Different Query Shapes

Query helpers let you define custom helper methods on Mongoose queries. Query helpers make queries more semantic using chaining syntax.

The following is an example of how query helpers work in JavaScript.

In TypeScript, you can define query helpers using a separate query helpers interface. Mongoose's Model takes 3 generic parameters:

The 2nd generic parameter, TQueryHelpers, should be an interface that contains a function signature for each of your query helpers. Below is an example of creating a ProjectModel with a byName query helper.

Mongoose does support auto typed Query Helpers that it are supplied in schema options. Query Helpers functions can be defined as following:

Sometimes you want a query helper to return a different type depending on whether the query is "lean" or not. For example, suppose you want a toMap() query helper that converts the results of a query into a Map keyed by _id. If you call .lean(), you want the map values to be plain objects; otherwise, you want hydrated documents.

To achieve this, you can use TypeScript function overloads on your query helper based on the value of this. Here's an example of how to type a toMap() query helper so that it returns the correct type for both lean and non-lean queries:

With this approach, TypeScript will infer the correct return type for .toMap() depending on whether you use .lean() or not. This ensures type safety and prevents accidental misuse of the query helper on queries that don't return arrays of documents.

**Examples:**

Example 1 (swift):
```swift
ProjectSchema.query.byName = function(name) {
  return this.find({ name: name });
};
const Project = mongoose.model('Project', ProjectSchema);

// Works. Any Project query, whether it be `find()`, `findOne()`,
// `findOneAndUpdate()`, `delete()`, etc. now has a `byName()` helper
Project.find().where('stars').gt(1000).byName('mongoose');
```

Example 2 (typescript):
```typescript
import { HydratedDocument, Model, QueryWithHelpers, Schema, model, connect } from 'mongoose';

interface Project {
  name?: string;
  stars?: number;
}

interface ProjectQueryHelpers {
  byName(name: string): QueryWithHelpers<
    HydratedDocument<Project>[],
    HydratedDocument<Project>,
    ProjectQueryHelpers
  >
}

type ProjectModelType = Model<Project, ProjectQueryHelpers>;

const ProjectSchema = new Schema<
  Project,
  Model<Project, ProjectQueryHelpers>,
  {},
  ProjectQueryHelpers
>({
  name: String,
  stars: Number
});

ProjectSchema.query.byName = function byName(
  this: QueryWithHelpers<any, HydratedDocument<Project>, ProjectQueryHelpers>,
  name: string
) {
  return this.find({ name: name });
};

// 2nd param to `model()` is the Model class to return.
const ProjectModel = model<Project, ProjectModelType>('Project', ProjectSchema);

run().catch(err => console.log(err));

async function run(): Promise<void> {
  await connect('mongodb://127.0.0.1:27017/test');

  // Equivalent to `ProjectModel.find({ stars: { $gt: 1000 }, name: 'mongoose' })`
  await ProjectModel.find().where('stars').gt(1000).byName('mongoose');
}
```

Example 3 (csharp):
```csharp
import { Schema, model } from 'mongoose';

const ProjectSchema = new Schema({
  name: String,
  stars: Number
}, {
  query: {
    byName(name: string) {
      return this.find({ name });
    }
  }
});

const ProjectModel = model('Project', ProjectSchema);

// Equivalent to `ProjectModel.find({ stars: { $gt: 1000 }, name: 'mongoose' })`
await ProjectModel.find().where('stars').gt(1000).byName('mongoose');
```

Example 4 (typescript):
```typescript
import { Model, HydratedDocument, QueryWithHelpers, Schema, model, Types } from 'mongoose';

// Query helper interface with overloads for lean and non-lean queries
export interface ToMapQueryHelpers<RawDocType, HydratedDocType> {
  // For non-lean queries: returns Map<string, HydratedDocType>
  toMap(this: QueryWithHelpers<HydratedDocType[], HydratedDocType>): QueryWithHelpers<Map<string, HydratedDocType>, HydratedDocType>;
  // For lean queries: returns Map<string, RawDocType>
  toMap(this: QueryWithHelpers<RawDocType[], HydratedDocType>): QueryWithHelpers<Map<string, RawDocType>, HydratedDocType>;
}

// Query helpers definition. Will be used in schema options
const query: ToMapQueryHelpers<IUser, UserHydratedDocument> = {
  // Chainable query helper that converts an array of documents to
  // a map of document _id (as a string) to the document
  toMap() {
    return this.transform((docs) => {
      // The `if` statements are type gymnastics to help TypeScript
      // handle the `IUser[] | UserHydratedDocument[]` union. Not necessary
      // for runtime correctness.
      if (docs.length === 0) return new Map();
      if (docs[0] instanceof Document) return new Map(docs.map(doc => [doc._id.toString(), doc]));
      return new Map(docs.map(doc => [doc._id.toString(), doc]));
    });
  }
};

export interface IUser {
  _id: Types.ObjectId;
  name: string;
}

export type UserHydratedDocument = HydratedDocument<IUser>;

export type UserModelType = Model<
  IUser,
  ToMapQueryHelpers<IUser, UserHydratedDocument>
>;

const userSchema = new Schema({ name: String }, { query });
const User = model<IUser, UserModelType>('User', userSchema);

async function run() {
  // Non-lean: Map<string, UserHydratedDocument>
  const hydratedMap = await User.find().toMap();
  // hydratedMap.get('someId') is a hydrated document

  // Lean: Map<string, IUser>
  const leanMap = await User.find().lean().toMap();
  // leanMap.get('someId') is a plain object

  // The following will fail at compile time, as expected, because `toMap()` shouldn't work with single documents or numbers
  // await User.findOne().toMap();
  // await User.countDocuments().toMap();
}
```

---

## Getting Started

**URL:** https://mongoosejs.com/docs/6.x/docs/index.html

**Contents:**
- Getting Started
- Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

For brevity, let's assume that all following code is within the main() function.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method. The first argument to the callback will be an error if any occurred.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens to the callback.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
$ npm install mongoose --save
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');

main().catch(err => console.log(err));

async function main() {
  await mongoose.connect('mongodb://127.0.0.1:27017/test');

  // use `await mongoose.connect('mongodb://user:password@127.0.0.1:27017/test');` if your database has auth enabled
}
```

Example 3 (javascript):
```javascript
const kittySchema = new mongoose.Schema({
  name: String
});
```

Example 4 (javascript):
```javascript
const Kitten = mongoose.model('Kitten', kittySchema);
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/guide.html

**Contents:**
- Schemas
  - Defining your schema
  - Creating a model
  - Ids
  - Instance methods
  - Statics
  - Query Helpers
  - Indexes
  - Virtuals
  - Aliases

If you haven't yet done so, please take a minute to read the quickstart to get an idea of how Mongoose works. If you are migrating from 4.x to 5.x please take a moment to read the migration guide.

Everything in Mongoose starts with a Schema. Each schema maps to a MongoDB collection and defines the shape of the documents within that collection.

If you want to add additional keys later, use the Schema#add method.

Each key in our code blogSchema defines a property in our documents which will be cast to its associated SchemaType. For example, we've defined a property title which will be cast to the String SchemaType and property date which will be cast to a Date SchemaType.

Notice above that if a property only requires a type, it can be specified using a shorthand notation (contrast the title property above with the date property).

Keys may also be assigned nested objects containing further key/type definitions like the meta property above. This will happen whenever a key's value is a POJO that doesn't have a type property.

In these cases, Mongoose only creates actual schema paths for leaves in the tree. (like meta.votes and meta.favs above), and the branches do not have actual paths. A side-effect of this is that meta above cannot have its own validation. If validation is needed up the tree, a path needs to be created up the tree - see the Subdocuments section for more information on how to do this. Also read the Mixed subsection of the SchemaTypes guide for some gotchas.

The permitted SchemaTypes are:

Read more about SchemaTypes here.

Schemas not only define the structure of your document and casting of properties, they also define document instance methods, static Model methods, compound indexes, and document lifecycle hooks called middleware.

To use our schema definition, we need to convert our blogSchema into a Model we can work with. To do so, we pass it into mongoose.model(modelName, schema):

By default, Mongoose adds an _id property to your schemas.

When you create a new document with the automatically added _id property, Mongoose creates a new _id of type ObjectId to your document.

You can also overwrite Mongoose's default _id with your own _id. Just be careful: Mongoose will refuse to save a document that doesn't have an _id, so you're responsible for setting _id if you define your own _id path.

Instances of Models are documents. Documents have many of their own built-in instance methods. We may also define our own custom document instance methods.

Now all of our animal instances have a findSimilarTypes method available to them.

You can also add static functions to your model. There are two equivalent ways to add a static:

Do not declare statics using ES6 arrow functions (=>). Arrow functions explicitly prevent binding this, so the above examples will not work because of the value of this.

You can also add query helper functions, which are like instance methods but for mongoose queries. Query helper methods let you extend mongoose's chainable query builder API.

MongoDB supports secondary indexes. With mongoose, we define these indexes within our Schema at the path level or the schema level. Defining indexes at the schema level is necessary when creating compound indexes.

When your application starts up, Mongoose automatically calls createIndex for each defined index in your schema. Mongoose will call createIndex for each index sequentially, and emit an 'index' event on the model when all the createIndex calls succeeded or when there was an error. While nice for development, it is recommended this behavior be disabled in production since index creation can cause a significant performance impact. Disable the behavior by setting the autoIndex option of your schema to false, or globally on the connection by setting the option autoIndex to false.

Mongoose will emit an index event on the model when indexes are done building or an error occurred.

See also the Model#ensureIndexes method.

Virtuals are document properties that you can get and set but that do not get persisted to MongoDB. The getters are useful for formatting or combining fields, while setters are useful for de-composing a single value into multiple values for storage.

Suppose you want to print out the person's full name. You could do it yourself:

But concatenating the first and last name every time can get cumbersome. And what if you want to do some extra processing on the name, like removing diacritics? A virtual property getter lets you define a fullName property that won't get persisted to MongoDB.

Now, mongoose will call your getter function every time you access the fullName property:

If you use toJSON() or toObject() mongoose will not include virtuals by default. This includes the output of calling JSON.stringify() on a Mongoose document, because JSON.stringify() calls toJSON(). Pass { virtuals: true } to either toObject() or toJSON().

You can also add a custom setter to your virtual that will let you set both first name and last name via the fullName virtual.

Virtual property setters are applied before other validation. So the example above would still work even if the first and last name fields were required.

Only non-virtual properties work as part of queries and for field selection. Since virtuals are not stored in MongoDB, you can't query with them.

You can learn more about virtuals here.

Aliases are a particular type of virtual where the getter and setter seamlessly get and set another property. This is handy for saving network bandwidth, so you can convert a short property name stored in the database into a longer name for code readability.

You can also declare aliases on nested paths. It is easier to use nested schemas and subdocuments, but you can also declare nested path aliases inline as long as you use the full nested path nested.myProp as the alias.

Schemas have a few configurable options which can be passed to the constructor or to the set method:

By default, Mongoose's init() function creates all the indexes defined in your model's schema by calling Model.createIndexes() after you successfully connect to MongoDB. Creating indexes automatically is great for development and test environments. But index builds can also create significant load on your production database. If you want to manage indexes carefully in production, you can set autoIndex to false.

The autoIndex option is set to true by default. You can change this default by setting mongoose.set('autoIndex', false);

Before Mongoose builds indexes, it calls Model.createCollection() to create the underlying collection in MongoDB if autoCreate is set to true. Calling createCollection() sets the collection's default collation based on the collation option and establishes the collection as a capped collection if you set the capped schema option. Like autoIndex, setting autoCreate to true is helpful for development and test environments.

Unfortunately, createCollection() cannot change an existing collection. For example, if you add capped: 1024 to your schema and the existing collection is not capped, createCollection() will throw an error. Generally, autoCreate should be false for production environments.

Unlike autoIndex, autoCreate is false by default. You can change this default by setting mongoose.set('autoCreate', true);

By default, mongoose buffers commands when the connection goes down until the driver manages to reconnect. To disable buffering, set bufferCommands to false.

The schema bufferCommands option overrides the global bufferCommands option.

If bufferCommands is on, this option sets the maximum amount of time Mongoose buffering will wait before throwing an error. If not specified, Mongoose will use 10000 (10 seconds).

Mongoose supports MongoDBs capped collections. To specify the underlying MongoDB collection be capped, set the capped option to the maximum size of the collection in bytes.

The capped option may also be set to an object if you want to pass additional options like max or autoIndexId. In this case you must explicitly pass the size option, which is required.

Mongoose by default produces a collection name by passing the model name to the utils.toCollectionName method. This method pluralizes the name. Set this option if you need a different name for your collection.

When you define a discriminator, Mongoose adds a path to your schema that stores which discriminator a document is an instance of. By default, Mongoose adds an __t path, but you can set discriminatorKey to overwrite this default.

Mongoose assigns each of your schemas an id virtual getter by default which returns the document's _id field cast to a string, or in the case of ObjectIds, its hexString. If you don't want an id getter added to your schema, you may disable it by passing this option at schema construction time.

Mongoose assigns each of your schemas an _id field by default if one is not passed into the Schema constructor. The type assigned is an ObjectId to coincide with MongoDB's default behavior. If you don't want an _id added to your schema at all, you may disable it using this option.

You can only use this option on subdocuments. Mongoose can't save a document without knowing its id, so you will get an error if you try to save a document without an _id.

Mongoose will, by default, "minimize" schemas by removing empty objects.

This behavior can be overridden by setting minimize option to false. It will then store empty objects.

To check whether an object is empty, you can use the $isEmpty() helper:

Allows setting query#read options at the schema level, providing us a way to apply default ReadPreferences to all queries derived from a model.

The alias of each pref is also permitted so instead of having to type out 'secondaryPreferred' and getting the spelling wrong, we can simply pass 'sp'.

The read option also allows us to specify tag sets. These tell the driver from which members of the replica-set it should attempt to read. Read more about tag sets here and here.

NOTE: you may also specify the driver read pref strategy option when connecting:

Allows setting write concern at the schema level.

The shardKey option is used when we have a sharded MongoDB architecture. Each sharded collection is given a shard key which must be present in all insert/update operations. We just need to set this schema option to the same shard key and we’ll be all set.

Note that Mongoose does not send the shardcollection command for you. You must configure your shards yourself.

The strict option, (enabled by default), ensures that values passed to our model constructor that were not specified in our schema do not get saved to the db.

This also affects the use of doc.set() to set a property value.

This value can be overridden at the model instance level by passing a second boolean argument:

The strict option may also be set to "throw" which will cause errors to be produced instead of dropping the bad data.

NOTE: Any key/val set on the instance that does not exist in your schema is always ignored, regardless of schema option.

For backwards compatibility, the strict option does not apply to the filter parameter for queries.

The strict option does apply to updates.

Mongoose has a separate strictQuery option to toggle strict mode for the filter parameter to queries.

Exactly the same as the toObject option but only applies when the document's toJSON method is called.

To see all available toJSON/toObject options, read this.

Documents have a toObject method which converts the mongoose document into a plain JavaScript object. This method accepts a few options. Instead of applying these options on a per-document basis, we may declare the options at the schema level and have them applied to all of the schema's documents by default.

To have all virtuals show up in your console.log output, set the toObject option to { getters: true }:

To see all available toObject options, read this.

By default, if you have an object with key 'type' in your schema, mongoose will interpret it as a type declaration.

However, for applications like geoJSON, the 'type' property is important. If you want to control which key mongoose uses to find type declarations, set the 'typeKey' schema option.

By default, documents are automatically validated before they are saved to the database. This is to prevent saving an invalid document. If you want to handle validation manually, and be able to save objects which don't pass validation, you can set validateBeforeSave to false.

The versionKey is a property set on each document when first created by Mongoose. This keys value contains the internal revision of the document. The versionKey option is a string that represents the path to use for versioning. The default is __v. If this conflicts with your application you can configure as such:

Note that Mongoose's default versioning is not a full optimistic concurrency solution. Mongoose's default versioning only operates on arrays as shown below.

If you need optimistic concurrency support for save(), you can set the optimisticConcurrency option

Document versioning can also be disabled by setting the versionKey to false. DO NOT disable versioning unless you know what you are doing.

Mongoose only updates the version key when you use save(). If you use update(), findOneAndUpdate(), etc. Mongoose will not update the version key. As a workaround, you can use the below middleware.

Optimistic concurrency is a strategy to ensure the document you're updating didn't change between when you loaded it using find() or findOne(), and when you update it using save().

For example, suppose you have a House model that contains a list of photos, and a status that represents whether this house shows up in searches. Suppose that a house that has status 'APPROVED' must have at least two photos. You might implement the logic of approving a house document as shown below:

The markApproved() function looks right in isolation, but there might be a potential issue: what if another function removes the house's photos between the findOne() call and the save() call? For example, the below code will succeed:

If you set the optimisticConcurrency option on the House model's schema, the above script will throw an error.

Sets a default collation for every query and aggregation. Here's a beginner-friendly overview of collations.

skipVersioning allows excluding paths from versioning (i.e., the internal revision will not be incremented even if these paths are updated). DO NOT do this unless you know what you're doing. For subdocuments, include this on the parent document using the fully qualified path.

The timestamps option tells mongoose to assign createdAt and updatedAt fields to your schema. The type assigned is Date.

By default, the names of the fields are createdAt and updatedAt. Customize the field names by setting timestamps.createdAt and timestamps.updatedAt.

By default, Mongoose uses new Date() to get the current time. If you want to overwrite the function Mongoose uses to get the current time, you can set the timestamps.currentTime option. Mongoose will call the timestamps.currentTime function whenever it needs to get the current time.

Write operations like update(), updateOne(), updateMany(), and findOneAndUpdate() only check the top-level schema's strict mode setting.

If you set useNestedStrict to true, mongoose will use the child schema's strict option for casting updates.

By default, Mongoose will automatically select() any populated paths for you, unless you explicitly exclude them.

To opt out of selecting populated fields by default, set selectPopulatedPaths to false in your schema.

For legacy reasons, when there is a validation error in subpath of a single nested schema, Mongoose will record that there was a validation error in the single nested schema path as well. For example:

Set the storeSubdocValidationError to false on the child schema to make Mongoose only reports the parent error.

Schemas have a loadClass() method that you can use to create a Mongoose schema from an ES6 class:

Here's an example of using loadClass() to create a schema from an ES6 class:

Schemas are also pluggable which allows us to package up reusable features into plugins that can be shared with the community or just between your projects.

Here's an alternative introduction to Mongoose schemas.

To get the most out of MongoDB, you need to learn the basics of MongoDB schema design. SQL schema design (third normal form) was designed to minimize storage costs, whereas MongoDB schema design is about making common queries as fast as possible. The 6 Rules of Thumb for MongoDB Schema Design blog series is an excellent resource for learning the basic rules for making your queries fast.

Users looking to master MongoDB schema design in Node.js should look into The Little MongoDB Schema Design Book by Christian Kvalheim, the original author of the MongoDB Node.js driver. This book shows you how to implement performant schemas for a laundry list of use cases, including e-commerce, wikis, and appointment bookings.

Now that we've covered Schemas, let's take a look at SchemaTypes.

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
  const { Schema } = mongoose;

  const blogSchema = new Schema({
    title:  String, // String is shorthand for {type: String}
    author: String,
    body:   String,
    comments: [{ body: String, date: Date }],
    date: { type: Date, default: Date.now },
    hidden: Boolean,
    meta: {
      votes: Number,
      favs:  Number
    }
  });
```

Example 2 (javascript):
```javascript
const Blog = mongoose.model('Blog', blogSchema);
  // ready to go!
```

Example 3 (javascript):
```javascript
const schema = new Schema();

schema.path('_id'); // ObjectId { ... }
```

Example 4 (javascript):
```javascript
const Model = mongoose.model('Test', schema);

const doc = new Model();
doc._id instanceof mongoose.Types.ObjectId; // true
```

---

## Schemas

**URL:** https://mongoosejs.com/docs/guide.html

**Contents:**
- Schemas
- Defining your schema
- Creating a model
- Ids
- Instance methods
- Statics
- Query Helpers
- Indexes
- Virtuals
- Aliases

If you haven't yet done so, please take a minute to read the quickstart to get an idea of how Mongoose works. If you are migrating from 7.x to 8.x please take a moment to read the migration guide.

Everything in Mongoose starts with a Schema. Each schema maps to a MongoDB collection and defines the shape of the documents within that collection.

If you want to add additional keys later, use the Schema#add method.

Each key in our code blogSchema defines a property in our documents which will be cast to its associated SchemaType. For example, we've defined a property title which will be cast to the String SchemaType and property date which will be cast to a Date SchemaType.

Notice above that if a property only requires a type, it can be specified using a shorthand notation (contrast the title property above with the date property).

Keys may also be assigned nested objects containing further key/type definitions like the meta property above. This will happen whenever a key's value is a POJO that doesn't have a type property.

In these cases, Mongoose only creates actual schema paths for leaves in the tree. (like meta.votes and meta.favs above), and the branches do not have actual paths. A side-effect of this is that meta above cannot have its own validation. If validation is needed up the tree, a path needs to be created up the tree - see the Subdocuments section for more information on how to do this. Also read the Mixed subsection of the SchemaTypes guide for some gotchas.

The permitted SchemaTypes are:

Read more about SchemaTypes here.

Schemas not only define the structure of your document and casting of properties, they also define document instance methods, static Model methods, compound indexes, and document lifecycle hooks called middleware.

To use our schema definition, we need to convert our blogSchema into a Model we can work with. To do so, we pass it into mongoose.model(modelName, schema):

By default, Mongoose adds an _id property to your schemas.

When you create a new document with the automatically added _id property, Mongoose creates a new _id of type ObjectId to your document.

You can also overwrite Mongoose's default _id with your own _id. Just be careful: Mongoose will refuse to save a top-level document that doesn't have an _id, so you're responsible for setting _id if you define your own _id path.

Mongoose also adds an _id property to subdocuments. You can disable the _id property on your subdocuments as follows. Mongoose does allow saving subdocuments without an _id property.

Alternatively, you can disable _id using the following syntax:

Instances of Models are documents. Documents have many of their own built-in instance methods. We may also define our own custom document instance methods.

Now all of our animal instances have a findSimilarTypes method available to them.

You can also add static functions to your model. There are three equivalent ways to add a static:

Do not declare statics using ES6 arrow functions (=>). Arrow functions explicitly prevent binding this, so the above examples will not work because of the value of this.

You can also add query helper functions, which are like instance methods but for mongoose queries. Query helper methods let you extend mongoose's chainable query builder API.

MongoDB supports secondary indexes. With mongoose, we define these indexes within our Schema at the path level or the schema level. Defining indexes at the schema level is necessary when creating compound indexes.

See SchemaType#index() for other index options.

When your application starts up, Mongoose automatically calls createIndex for each defined index in your schema. Mongoose will call createIndex for each index sequentially, and emit an 'index' event on the model when all the createIndex calls succeeded or when there was an error. While nice for development, it is recommended this behavior be disabled in production since index creation can cause a significant performance impact. Disable the behavior by setting the autoIndex option of your schema to false, or globally on the connection by setting the option autoIndex to false.

Mongoose will emit an index event on the model when indexes are done building or an error occurred.

See also the Model#ensureIndexes method.

Virtuals are document properties that you can get and set but that do not get persisted to MongoDB. The getters are useful for formatting or combining fields, while setters are useful for de-composing a single value into multiple values for storage.

Suppose you want to print out the person's full name. You could do it yourself:

But concatenating the first and last name every time can get cumbersome. And what if you want to do some extra processing on the name, like removing diacritics? A virtual property getter lets you define a fullName property that won't get persisted to MongoDB.

Now, mongoose will call your getter function every time you access the fullName property:

If you use toJSON() or toObject() Mongoose will not include virtuals by default. Pass { virtuals: true } to toJSON() or toObject() to include virtuals.

The above caveat for toJSON() also includes the output of calling JSON.stringify() on a Mongoose document, because JSON.stringify() calls toJSON(). To include virtuals in JSON.stringify() output, you can either call toObject({ virtuals: true }) on the document before calling JSON.stringify(), or set the toJSON: { virtuals: true } option on your schema.

You can also add a custom setter to your virtual that will let you set both first name and last name via the fullName virtual.

Virtual property setters are applied before other validation. So the example above would still work even if the first and last name fields were required.

Only non-virtual properties work as part of queries and for field selection. Since virtuals are not stored in MongoDB, you can't query with them.

You can learn more about virtuals here.

Aliases are a particular type of virtual where the getter and setter seamlessly get and set another property. This is handy for saving network bandwidth, so you can convert a short property name stored in the database into a longer name for code readability.

You can also declare aliases on nested paths. It is easier to use nested schemas and subdocuments, but you can also declare nested path aliases inline as long as you use the full nested path nested.myProp as the alias.

Schemas have a few configurable options which can be passed to the constructor or to the set method:

By default, Mongoose's init() function creates all the indexes defined in your model's schema by calling Model.createIndexes() after you successfully connect to MongoDB. Creating indexes automatically is great for development and test environments. But index builds can also create significant load on your production database. If you want to manage indexes carefully in production, you can set autoIndex to false.

The autoIndex option is set to true by default. You can change this default by setting mongoose.set('autoIndex', false);

Before Mongoose builds indexes, it calls Model.createCollection() to create the underlying collection in MongoDB by default. Calling createCollection() sets the collection's default collation based on the collation option and establishes the collection as a capped collection if you set the capped schema option.

You can disable this behavior by setting autoCreate to false using mongoose.set('autoCreate', false). Like autoIndex, autoCreate is helpful for development and test environments, but you may want to disable it for production to avoid unnecessary database calls.

Unfortunately, createCollection() cannot change an existing collection. For example, if you add capped: { size: 1024 } to your schema and the existing collection is not capped, createCollection() will not overwrite the existing collection. That is because the MongoDB server does not allow changing a collection's options without dropping the collection first.

By default, mongoose buffers commands when the connection goes down until the driver manages to reconnect. To disable buffering, set bufferCommands to false.

The schema bufferCommands option overrides the global bufferCommands option.

If bufferCommands is on, this option sets the maximum amount of time Mongoose buffering will wait before throwing an error. If not specified, Mongoose will use 10000 (10 seconds).

Mongoose supports MongoDBs capped collections. To specify the underlying MongoDB collection be capped, set the capped option to the maximum size of the collection in bytes.

The capped option may also be set to an object if you want to pass additional options like max. In this case you must explicitly pass the size option, which is required.

Mongoose by default produces a collection name by passing the model name to the utils.toCollectionName method. This method pluralizes the name. Set this option if you need a different name for your collection.

When you define a discriminator, Mongoose adds a path to your schema that stores which discriminator a document is an instance of. By default, Mongoose adds an __t path, but you can set discriminatorKey to overwrite this default.

When excludeIndexes is true, Mongoose will not create indexes from the given subdocument schema. This option only works when the schema is used in a subdocument path or document array path, Mongoose ignores this option if set on the top-level schema for a model. Defaults to false.

Mongoose assigns each of your schemas an id virtual getter by default which returns the document's _id field cast to a string, or in the case of ObjectIds, its hexString. If you don't want an id getter added to your schema, you may disable it by passing this option at schema construction time.

Mongoose assigns each of your schemas an _id field by default if one is not passed into the Schema constructor. The type assigned is an ObjectId to coincide with MongoDB's default behavior. If you don't want an _id added to your schema at all, you may disable it using this option.

You can only use this option on subdocuments. Mongoose can't save a document without knowing its id, so you will get an error if you try to save a document without an _id.

Mongoose will, by default, "minimize" schemas by removing empty objects.

This behavior can be overridden by setting minimize option to false. It will then store empty objects.

To check whether an object is empty, you can use the $isEmpty() helper:

Allows setting query#read options at the schema level, providing us a way to apply default ReadPreferences to all queries derived from a model.

The alias of each pref is also permitted so instead of having to type out 'secondaryPreferred' and getting the spelling wrong, we can simply pass 'sp'.

The read option also allows us to specify tag sets. These tell the driver from which members of the replica-set it should attempt to read. Read more about tag sets in the MongoDB documentation on replica set tag sets and in the MongoDB core documentation on read preference.

NOTE: you may also specify the driver read preference strategy option when connecting:

Allows setting write concern at the schema level.

The shardKey option is used when we have a sharded MongoDB architecture. Each sharded collection is given a shard key which must be present in all insert/update operations. We just need to set this schema option to the same shard key and we’ll be all set.

Note that Mongoose does not send the shardcollection command for you. You must configure your shards yourself.

The strict option, (enabled by default), ensures that values passed to our model constructor that were not specified in our schema do not get saved to the db.

This also affects the use of doc.set() to set a property value.

This value can be overridden at the model instance level by passing a second boolean argument:

The strict option may also be set to "throw" which will cause errors to be produced instead of dropping the bad data.

NOTE: Any key/val set on the instance that does not exist in your schema is always ignored, regardless of schema option.

Mongoose supports a separate strictQuery option to avoid strict mode for query filters. This is because empty query filters cause Mongoose to return all documents in the model, which can cause issues.

The strict option does apply to updates. The strictQuery option is just for query filters.

Mongoose has a separate strictQuery option to toggle strict mode for the filter parameter to queries.

In general, we do not recommend passing user-defined objects as query filters:

In Mongoose 7, strictQuery is false by default. However, you can override this behavior globally:

Exactly the same as the toObject option but only applies when the document's toJSON method is called.

To see all available toJSON/toObject options, read this.

Documents have a toObject method which converts the mongoose document into a plain JavaScript object. This method accepts a few options. Instead of applying these options on a per-document basis, we may declare the options at the schema level and have them applied to all of the schema's documents by default.

To have all virtuals show up in your console.log output, set the toObject option to { getters: true }:

To see all available toObject options, read this.

By default, if you have an object with key 'type' in your schema, mongoose will interpret it as a type declaration.

However, for applications like geoJSON, the 'type' property is important. If you want to control which key mongoose uses to find type declarations, set the 'typeKey' schema option.

By default, documents are automatically validated before they are saved to the database. This is to prevent saving an invalid document. If you want to handle validation manually, and be able to save objects which don't pass validation, you can set validateBeforeSave to false.

The versionKey is a property set on each document when first created by Mongoose. This keys value contains the internal revision of the document. The versionKey option is a string that represents the path to use for versioning. The default is __v. If this conflicts with your application you can configure as such:

Note that Mongoose's default versioning is not a full optimistic concurrency solution. Mongoose's default versioning only operates on arrays as shown below.

If you need optimistic concurrency support for save(), you can set the optimisticConcurrency option.

Document versioning can also be disabled by setting the versionKey to false. DO NOT disable versioning unless you know what you are doing.

Mongoose only updates the version key when you use save(). If you use update(), findOneAndUpdate(), etc. Mongoose will not update the version key. As a workaround, you can use the below middleware.

Optimistic concurrency is a strategy to ensure the document you're updating didn't change between when you loaded it using find() or findOne(), and when you update it using save().

For example, suppose you have a House model that contains a list of photos, and a status that represents whether this house shows up in searches. Suppose that a house that has status 'APPROVED' must have at least two photos. You might implement the logic of approving a house document as shown below:

The markApproved() function looks right in isolation, but there might be a potential issue: what if another function removes the house's photos between the findOne() call and the save() call? For example, the below code will succeed:

If you set the optimisticConcurrency option on the House model's schema, the above script will throw an error.

You can also set optimisticConcurrency to an array of field names to only use optimistic concurrency when one of those fields is modified. Note that setting optimisticConcurrency to an array of field names replaces the default array versioning behavior. For example, if you set optimisticConcurrency: ['status'], Mongoose will only throw a VersionError if status is modified concurrently, and will not throw a VersionError if an array like photos is modified concurrently.

You can also set optimisticConcurrency to an object with an exclude property to exclude certain fields from optimistic concurrency. This enables optimistic concurrency for all fields except the excluded fields, while still replacing default array versioning behavior.

Sets a default collation for every query and aggregation. Here's a beginner-friendly overview of collations.

If you set the timeseries option on a schema, Mongoose will create a timeseries collection for any model that you create from that schema.

skipVersioning allows excluding paths from versioning (i.e., the internal revision will not be incremented even if these paths are updated). DO NOT do this unless you know what you're doing. For subdocuments, include this on the parent document using the fully qualified path.

The timestamps option tells Mongoose to assign createdAt and updatedAt fields to your schema. The type assigned is Date.

By default, the names of the fields are createdAt and updatedAt. Customize the field names by setting timestamps.createdAt and timestamps.updatedAt.

The way timestamps works under the hood is:

By default, Mongoose uses new Date() to get the current time. If you want to overwrite the function Mongoose uses to get the current time, you can set the timestamps.currentTime option. Mongoose will call the timestamps.currentTime function whenever it needs to get the current time.

Mongoose supports defining global plugins, plugins that apply to all schemas.

Sometimes, you may only want to apply a given plugin to some schemas. In that case, you can add pluginTags to a schema:

If you call plugin() with a tags option, Mongoose will only apply that plugin to schemas that have a matching entry in pluginTags.

By default, Mongoose will automatically select() any populated paths for you, unless you explicitly exclude them.

To opt out of selecting populated fields by default, set selectPopulatedPaths to false in your schema.

For legacy reasons, when there is a validation error in subpath of a single nested schema, Mongoose will record that there was a validation error in the single nested schema path as well. For example:

Set the storeSubdocValidationError to false on the child schema to make Mongoose only reports the parent error.

Options like collation and capped affect the options Mongoose passes to MongoDB when creating a new collection. Mongoose schemas support most MongoDB createCollection() options, but not all. You can use the collectionOptions option to set any createCollection() options; Mongoose will use collectionOptions as the default values when calling createCollection() for your schema.

Similar to autoIndex, except for automatically creates any Atlas search indexes defined in your schema. Unlike autoIndex, this option defaults to false.

Read concerns are similar to writeConcern, but for read operations like find() and findOne(). To set a default readConcern, pass the readConcern option to the schema constructor as follows.

Schemas have a loadClass() method that you can use to create a Mongoose schema from an ES6 class:

Here's an example of using loadClass() to create a schema from an ES6 class:

Schemas are also pluggable which allows us to package up reusable features into plugins that can be shared with the community or just between your projects.

Here's an alternative introduction to Mongoose schemas.

To get the most out of MongoDB, you need to learn the basics of MongoDB schema design. SQL schema design (third normal form) was designed to minimize storage costs, whereas MongoDB schema design is about making common queries as fast as possible. The 6 Rules of Thumb for MongoDB Schema Design blog series is an excellent resource for learning the basic rules for making your queries fast.

Users looking to master MongoDB schema design in Node.js should look into The Little MongoDB Schema Design Book by Christian Kvalheim, the original author of the MongoDB Node.js driver. This book shows you how to implement performant schemas for a laundry list of use cases, including e-commerce, wikis, and appointment bookings.

Now that we've covered Schemas, let's take a look at SchemaTypes.

**Examples:**

Example 1 (sql):
```sql
import mongoose from 'mongoose';
const { Schema } = mongoose;

const blogSchema = new Schema({
  title: String, // String is shorthand for {type: String}
  author: String,
  body: String,
  comments: [{ body: String, date: Date }],
  date: { type: Date, default: Date.now },
  hidden: Boolean,
  meta: {
    votes: Number,
    favs: Number
  }
});
```

Example 2 (javascript):
```javascript
const Blog = mongoose.model('Blog', blogSchema);
// ready to go!
```

Example 3 (javascript):
```javascript
const schema = new Schema();

schema.path('_id'); // ObjectId { ... }
```

Example 4 (javascript):
```javascript
const Model = mongoose.model('Test', schema);

const doc = new Model();
doc._id instanceof mongoose.Types.ObjectId; // true
```

---

## Handling Subdocuments in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/subdocuments.html

**Contents:**
- Handling Subdocuments in TypeScript
- Subdocument Arrays

Subdocuments are tricky in TypeScript. By default, Mongoose treats object properties in document interfaces as nested properties rather than subdocuments.

Mongoose provides a mechanism to override types in the hydrated document. Define a separate THydratedDocumentType and pass it as the 5th generic param to mongoose.Model<>. THydratedDocumentType controls what type Mongoose uses for "hydrated documents", that is, what await UserModel.findOne(), UserModel.hydrate(), and new UserModel() return.

You can also override arrays to properly type subdocument arrays using TMethodsAndOverrides:

**Examples:**

Example 1 (typescript):
```typescript
// Setup
import { Schema, Types, model, Model } from 'mongoose';

// Subdocument definition
interface Names {
  _id: Types.ObjectId;
  firstName: string;
}

// Document definition
interface User {
  names: Names;
}

// Models and schemas
type UserModelType = Model<User>;
const userSchema = new Schema<User, UserModelType>({
  names: new Schema<Names>({ firstName: String })
});
const UserModel = model<User, UserModelType>('User', userSchema);

// Create a new document:
const doc = new UserModel({ names: { _id: '0'.repeat(24), firstName: 'foo' } });

// "Property 'ownerDocument' does not exist on type 'Names'."
// Means that `doc.names` is not a subdocument!
doc.names.ownerDocument();
```

Example 2 (typescript):
```typescript
import { HydratedSingleSubdocument } from 'mongoose';

// Define property overrides for hydrated documents
type THydratedUserDocument = {
  names?: HydratedSingleSubdocument<Names>
}
type UserModelType = mongoose.Model<User, {}, {}, {}, THydratedUserDocument>;

const userSchema = new mongoose.Schema<User, UserModelType>({
  names: new mongoose.Schema<Names>({ firstName: String })
});
const UserModel = mongoose.model<User, UserModelType>('User', userSchema);

const doc = new UserModel({ names: { _id: '0'.repeat(24), firstName: 'foo' } });
doc.names!.ownerDocument(); // Works, `names` is a subdocument!
doc.names!.firstName; // 'foo'
```

Example 3 (typescript):
```typescript
// Subdocument definition
interface Names {
  _id: Types.ObjectId;
  firstName: string;
}
// Document definition
interface User {
  names: Names[];
}

// TMethodsAndOverrides
type THydratedUserDocument = {
  names?: Types.DocumentArray<Names>
}
type UserModelType = Model<User, {}, {}, {}, THydratedUserDocument>;

// Create model
const UserModel = model<User, UserModelType>('User', new Schema<User, UserModelType>({
  names: [new Schema<Names>({ firstName: String })]
}));

const doc = new UserModel({});
doc.names[0].ownerDocument(); // Works!
doc.names[0].firstName; // string
```

---

## Populate with TypeScript

**URL:** https://mongoosejs.com/docs/typescript/populate.html

**Contents:**
- Populate with TypeScript
- Using PopulatedDoc

Mongoose's TypeScript bindings add a generic parameter Paths to the populate():

An alternative approach is to define a PopulatedParent interface and use Pick<> to pull the properties you're populating.

Mongoose also exports a PopulatedDoc type that helps you define populated documents in your document interface:

However, we recommend using the .populate<{ child: Child }> syntax from the first section instead of PopulatedDoc. Here's two reasons why:

**Examples:**

Example 1 (typescript):
```typescript
import { Schema, model, Document, Types } from 'mongoose';

// `Parent` represents the object as it is stored in MongoDB
interface Parent {
  child?: Types.ObjectId,
  name?: string
}
const ParentModel = model<Parent>('Parent', new Schema({
  child: { type: Schema.Types.ObjectId, ref: 'Child' },
  name: String
}));

interface Child {
  name: string;
}
const childSchema = new Schema({ name: String });
const ChildModel = model<Child>('Child', childSchema);

// Populate with `Paths` generic `{ child: Child }` to override `child` path
ParentModel.findOne({}).populate<{ child: Child }>('child').orFail().then(doc => {
  // Works
  const t: string = doc.child.name;
});
```

Example 2 (typescript):
```typescript
import { Schema, model, Document, Types } from 'mongoose';

// `Parent` represents the object as it is stored in MongoDB
interface Parent {
  child?: Types.ObjectId,
  name?: string
}
interface Child {
  name: string;
}
interface PopulatedParent {
  child: Child | null;
}
const ParentModel = model<Parent>('Parent', new Schema({
  child: { type: Schema.Types.ObjectId, ref: 'Child' },
  name: String
}));
const childSchema = new Schema({ name: String });
const ChildModel = model<Child>('Child', childSchema);

// Populate with `Paths` generic `{ child: Child }` to override `child` path
ParentModel.findOne({}).populate<Pick<PopulatedParent, 'child'>>('child').orFail().then(doc => {
  // Works
  const t: string = doc.child.name;
});
```

Example 3 (typescript):
```typescript
import { Schema, model, Document, PopulatedDoc } from 'mongoose';

// `child` is either an ObjectId or a populated document
interface Parent {
  child?: PopulatedDoc<Document<ObjectId> & Child>,
  name?: string
}
const ParentModel = model<Parent>('Parent', new Schema({
  child: { type: 'ObjectId', ref: 'Child' },
  name: String
}));

interface Child {
  name?: string;
}
const childSchema = new Schema({ name: String });
const ChildModel = model<Child>('Child', childSchema);

ParentModel.findOne({}).populate('child').orFail().then((doc: Parent) => {
  const child = doc.child;
  if (child == null || child instanceof ObjectId) {
    throw new Error('should be populated');
  } else {
    // Works
    doc.child.name.trim();
  }
});
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/7.x/index.html

---

## Mongoose

**URL:** https://mongoosejs.com/docs/6.x/index.html

---

## Connections

**URL:** https://mongoosejs.com/docs/connections.html

**Contents:**
- Connections
- Operation Buffering
- Error Handling
- Options
- serverSelectionTimeoutMS
- Using Promises and Async/Await
- Connection String Options
- Connection Events
- A note about keepAlive
- Replica Set Connections

You can connect to MongoDB with the mongoose.connect() method.

This is the minimum needed to connect the myapp database running locally on the default port (27017). For local MongoDB databases, we recommend using 127.0.0.1 instead of localhost. That is because Node.js 18 and up prefer IPv6 addresses, which means, on many machines, Node.js will resolve localhost to the IPv6 address ::1 and Mongoose will be unable to connect, unless the mongodb instance is running with ipv6 enabled.

You can also specify several more parameters in the uri:

See the mongodb connection string spec for more details.

Mongoose lets you start using your models immediately, without waiting for mongoose to establish a connection to MongoDB.

That's because mongoose buffers model function calls internally. This buffering is convenient, but also a common source of confusion. Mongoose will not throw any errors by default if you use a model without connecting.

To disable buffering, turn off the bufferCommands option on your schema. If you have bufferCommands on and your connection is hanging, try turning bufferCommands off to see if you haven't opened a connection properly. You can also disable bufferCommands globally:

Note that buffering is also responsible for waiting until Mongoose creates collections if you use the autoCreate option. If you disable buffering, you should also disable the autoCreate option and use createCollection() to create capped collections or collections with collations.

There are two classes of errors that can occur with a Mongoose connection.

To handle initial connection errors, you should use .catch() or try/catch with async/await.

To handle errors after initial connection was established, you should listen for error events on the connection. However, you still need to handle initial connection errors as shown above.

Note that Mongoose does not necessarily emit an 'error' event if it loses connectivity to MongoDB. You should listen to the disconnected event to report when Mongoose is disconnected from MongoDB.

The connect method also accepts an options object which will be passed on to the underlying MongoDB driver.

A full list of options can be found on the MongoDB Node.js driver docs for MongoClientOptions. Mongoose passes options to the driver without modification, modulo a few exceptions that are explained below.

Below are some of the options that are important for tuning Mongoose.

The serverSelectionTimeoutMS option is extremely important: it controls how long the MongoDB Node.js driver will attempt to retry any operation before erroring out. This includes initial connection, like await mongoose.connect(), as well as any operations that make requests to MongoDB, like save() or find().

By default, serverSelectionTimeoutMS is 30000 (30 seconds). This means that, for example, if you call mongoose.connect() when your standalone MongoDB server is down, your mongoose.connect() call will only throw an error after 30 seconds.

Similarly, if your standalone MongoDB server goes down after initial connection, any find() or save() calls will error out after 30 seconds, unless your MongoDB server is restarted.

While 30 seconds seems like a long time, serverSelectionTimeoutMS means you're unlikely to see any interruptions during a replica set failover. If you lose your replica set primary, the MongoDB Node driver will ensure that any operations you send during the replica set election will eventually execute, assuming that the replica set election takes less than serverSelectionTimeoutMS.

To get faster feedback on failed connections, you can reduce serverSelectionTimeoutMS to 5000 as follows. We don't recommend reducing serverSelectionTimeoutMS unless you are running a standalone MongoDB server rather than a replica set, or unless you are using a serverless runtime like AWS Lambda.

There is no way to tune serverSelectionTimeoutMS independently for mongoose.connect() vs for queries. If you want to reduce serverSelectionTimeoutMS for queries and other operations, but still retry mongoose.connect() for longer, you are responsible for retrying the connect() calls yourself using a for loop or a tool like p-retry.

The connect() function returns a promise.

You can also specify driver options in your connection string as parameters in the query string portion of the URI. This only applies to options passed to the MongoDB driver. You can't set Mongoose-specific options like bufferCommands in the query string.

The disadvantage of putting options in the query string is that query string options are harder to read. The advantage is that you only need a single configuration option, the URI, rather than separate options for socketTimeoutMS, etc. Best practice is to put options that likely differ between development and production, like replicaSet or ssl, in the connection string, and options that should remain constant, like socketTimeoutMS or maxPoolSize, in the options object.

The MongoDB docs have a full list of supported connection string options. Below are some options that are often useful to set in the connection string because they are closely associated with the hostname and authentication information.

Connections inherit from Node.js' EventEmitter class, and emit events when something happens to the connection, like losing connectivity to the MongoDB server. Below is a list of events that a connection may emit.

When you're connecting to a single MongoDB server (a "standalone"), Mongoose will emit disconnected if it gets disconnected from the standalone server, and connected if it successfully connects to the standalone. In a replica set, Mongoose will emit disconnected if it loses connectivity to the replica set primary, and connected if it manages to reconnect to the replica set primary.

If you are using mongoose.connect(), you can use the following to listen to the above events:

With mongoose.createConnection(), use the following instead:

Before Mongoose 5.2.0, you needed to enable the keepAlive option to initiate TCP keepalive to prevent "connection closed" errors. However, keepAlive has been true by default since Mongoose 5.2.0, and the keepAlive is deprecated as of Mongoose 7.2.0. Please remove keepAlive and keepAliveInitialDelay options from your Mongoose connections.

To connect to a replica set you pass a comma delimited list of hosts to connect to rather than a single host.

To connect to a single node replica set, specify the replicaSet option.

The underlying MongoDB driver uses a process known as server selection to connect to MongoDB and send operations to MongoDB. If the MongoDB driver can't find a server to send an operation to after serverSelectionTimeoutMS, you'll get the below error:

You can configure the timeout using the serverSelectionTimeoutMS option to mongoose.connect():

A MongoTimeoutError has a reason property that explains why server selection timed out. For example, if you're connecting to a standalone server with an incorrect password, reason will contain an "Authentication failed" error.

MongoDB replica sets rely on being able to reliably figure out the domain name for each member.On Linux and OSX, the MongoDB server uses the output of the hostname command to figure out the domain name to report to the replica set. This can cause confusing errors if you're connecting to a remote MongoDB replica set running on a machine that reports its hostname as localhost:

If you're experiencing a similar error, connect to the replica set using the mongo shell and run the rs.conf() command to check the host names of each replica set member. Follow this page's instructions to change a replica set member's host name.

You can also check the reason.servers property of MongooseServerSelectionError to see what the MongoDB Node driver thinks the state of your replica set is. The reason.servers property contains a map of server descriptions.

You can also connect to multiple mongos instances for high availability in a sharded cluster. You do not need to pass any special options to connect to multiple mongos in mongoose 5.x.

So far we've seen how to connect to MongoDB using Mongoose's default connection. Mongoose creates a default connection when you call mongoose.connect(). You can access the default connection using mongoose.connection.

You may need multiple connections to MongoDB for several reasons. One reason is if you have multiple databases or multiple MongoDB clusters. Another reason is to work around slow trains. The mongoose.createConnection() function takes the same arguments as mongoose.connect() and returns a new connection.

This connection object is then used to create and retrieve models. Models are always scoped to a single connection.

The createConnection() function returns a connection instance, not a promise. If you want to use await to make sure Mongoose successfully connects to MongoDB, use the asPromise() function:

If you use multiple connections, you should make sure you export schemas, not models. Exporting a model from a file is called the export model pattern. The export model pattern is limited because you can only use one connection.

If you use the export schema pattern, you still need to create models somewhere. There are two common patterns. The first is to create a function that instantiates a new connection and registers all models on that connection. With this pattern, you may also register connections with a dependency injector or another inversion of control (IOC) pattern.

Exporting a function that creates a new connection is the most flexible pattern. However, that pattern can make it tricky to get access to your connection from your route handlers or wherever your business logic is. An alternative pattern is to export a connection and register the models on the connection in the file's top-level scope as follows.

You can create separate files for each connection, like connections/web.js and connections/mobile.js if you want to create separate connections for your web API backend and your mobile API backend. Your business logic can then require() or import the connection it needs.

Each connection, whether created with mongoose.connect or mongoose.createConnection are all backed by an internal configurable connection pool defaulting to a maximum size of 100. Adjust the pool size using your connection options:

The connection pool size is important because MongoDB currently can only process one operation per socket. So maxPoolSize functions as a cap on the number of concurrent operations.

In the context of Mongoose, a multi-tenant architecture typically means a case where multiple different clients talk to MongoDB through a single Mongoose application. This typically means each client makes queries and executes updates through a single Mongoose application, but has a distinct MongoDB database within the same MongoDB cluster.

We recommend reading this article about multi-tenancy with Mongoose; it has a good description of how we define multi-tenancy and a more detailed overview of our recommended patterns.

There are two patterns we recommend for multi-tenancy in Mongoose:

The following is an example of pattern (1). We recommend pattern (1) for cases where you have a small number of tenants, or if each individual tenant's workload is light (approximately < 1 request per second, all requests take < 10ms of database processing time). Pattern (1) is simpler to implement and simpler to manage in production, because there is only 1 connection pool. But, under high load, you will likely run into issues where some tenants' operations slow down other tenants' operations due to slow trains.

The following is an example of pattern (2). Pattern (2) is more flexible and better for use cases with > 10k tenants and > 1 requests/second. Because each tenant has a separate connection pool, one tenants' slow operations will have minimal impact on other tenants. However, this pattern is harder to implement and manage in production. In particular, MongoDB does have a limit on the number of open connections, and MongoDB Atlas has separate limits on the number of open connections, so you need to make sure the total number of sockets in your connection pools doesn't go over MongoDB's limits.

Now that we've covered connections, let's take a look at models.

**Examples:**

Example 1 (json):
```json
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
```

Example 2 (python):
```python
mongoose.connect('mongodb://username:password@host:port/database?options...');
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1:27017/myapp');
const MyModel = mongoose.model('Test', new Schema({ name: String }));
// Works
await MyModel.findOne();
```

Example 4 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const promise = MyModel.findOne();

setTimeout(function() {
  mongoose.connect('mongodb://127.0.0.1:27017/myapp');
}, 60000);

// Will just hang until mongoose successfully connects
await promise;
```

---

## TypeScript Support

**URL:** https://mongoosejs.com/docs/6.x/docs/typescript.html

**Contents:**
- TypeScript Support
- Creating Your First Document
- ObjectIds and Other Mongoose Types
- Using extends Document
- Using Custom Bindings
- Next Up

Mongoose introduced officially supported TypeScript bindings in v5.11.0. Mongoose's index.d.ts file supports a wide variety of syntaxes and strives to be compatible with @types/mongoose where possible. This guide describes Mongoose's recommended approach to working with Mongoose in TypeScript.

To get started with Mongoose in TypeScript, you need to:

You as the developer are responsible for ensuring that your document interface lines up with your Mongoose schema. For example, Mongoose won't report an error if email is required in your Mongoose schema but optional in your document interface.

The User() constructor returns an instance of HydratedDocument<IUser>. IUser is a document interface, it represents the raw object structure that IUser objects look like in MongoDB. HydratedDocument<IUser> represents a hydrated Mongoose document, with methods, virtuals, and other Mongoose-specific features.

To define a property of type ObjectId, you should use Types.ObjectId in the TypeScript document interface. You should use 'ObjectId' or Schema.Types.ObjectId in your schema definition.

That's because Schema.Types.ObjectId is a class that inherits from SchemaType, not the class you use to create a new MongoDB ObjectId.

Alternatively, your document interface can extend Mongoose's Document class.

We strongly recommend against using this approach, its support will be dropped in the next major version as it causes major performance issues. Many Mongoose TypeScript codebases use the below approach.

This approach works, but we recommend your document interface not extend Document. Using extends Document makes it difficult for Mongoose to infer which properties are present on query filters, lean documents, and other cases.

We recommend your document interface contain the properties defined in your schema and line up with what your documents look like in MongoDB. Although you can add instance methods to your document interface, we do not recommend doing so.

If Mongoose's built-in index.d.ts file does not work for you, you can remove it in a postinstall script in your package.json as shown below. However, before you do, please open an issue on Mongoose's GitHub page and describe the issue you're experiencing.

Now that you've seen the basics of how to use Mongoose in TypeScript, let's take a look at statics in TypeScript.

**Examples:**

Example 1 (typescript):
```typescript
import { Schema, model, connect } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  avatar?: string;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// 3. Create a Model.
const User = model<IUser>('User', userSchema);

run().catch(err => console.log(err));

async function run() {
  // 4. Connect to MongoDB
  await connect('mongodb://127.0.0.1:27017/test');

  const user = new User({
    name: 'Bill',
    email: 'bill@initech.com',
    avatar: 'https://i.imgur.com/dM7Thhn.png'
  });
  await user.save();

  console.log(user.email); // 'bill@initech.com'
}
```

Example 2 (sql):
```sql
import { HydratedDocument } from 'mongoose';

const user: HydratedDocument<IUser> = new User({
  name: 'Bill',
  email: 'bill@initech.com',
  avatar: 'https://i.imgur.com/dM7Thhn.png'
});
```

Example 3 (typescript):
```typescript
import { Schema, Types } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  // Use `Types.ObjectId` in document interface...
  organization: Types.ObjectId;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  // And `Schema.Types.ObjectId` in the schema definition.
  organization: { type: Schema.Types.ObjectId, ref: 'Organization' }
});
```

Example 4 (typescript):
```typescript
import { Document, Schema, model, connect } from 'mongoose';

interface IUser extends Document {
  name: string;
  email: string;
  avatar?: string;
}
```

---

## Schema

**URL:** https://mongoosejs.com/docs/api/schema.html

**Contents:**
- Schema
  - Schema()
      - Parameters:
      - Inherits:
    - Example:
    - Options:
    - Options for Nested Schemas:
    - Note:
  - Schema.Types
      - Type:

When nesting schemas, (children in the example above), always declare the child schema first before passing it into its parent.

The various built-in Mongoose Schema Types.

Using this exposed access to the Mixed SchemaType, we can use them in our schema.

Creates a new schema with the given definition and options. Equivalent to new Schema(definition, options).

Schema.create() is primarily useful for automatic schema type inference in TypeScript.

The allowed index types

Adds key path / schema type pairs to this schema.

Add an alias for path. This means getting or setting the alias is equivalent to getting or setting the path.

Array of child schemas (from document arrays and single nested subdocs) and their corresponding compiled models. Each element of the array is an object with 2 properties: schema and model.

This property is typically only useful for plugin authors and advanced users. You do not need to interact with this property at all to use mongoose.

Remove all indexes from this schema.

clearIndexes only removes indexes from your schema object. Does not affect the indexes in MongoDB.

Returns a deep copy of the schema

Inherit a Schema by applying a discriminator on an existing Schema.

Iterates the schemas paths similar to Array#forEach.

The callback is passed the pathname and the schemaType instance.

Sets the encryption type of the schema, if a value is provided, otherwise returns the encryption type.

Gets a schema option.

Defines an index (most likely compound) for this schema.

Returns a list of indexes that this schema declares, via schema.index() or by index: true in a path's options. Indexes are expressed as an array [spec, options].

Plugins can use the return value of this function to modify a schema's indexes. For example, the below plugin makes every index unique by default.

Loads an ES6 class into a schema. Maps setters + getters, static methods, and instance methods to schema virtuals, statics, and methods.

Adds an instance method to documents constructed from Models compiled from this schema.

If a hash of name/fn pairs is passed as the only argument, each name/fn pair will be added as methods.

NOTE: Schema.method() adds instance methods to the Schema.methods object. You can also add instance methods directly to the Schema.methods object as seen in the guide

The original object passed to the schema constructor

Returns a new schema that has the paths from the original schema, minus the omitted ones.

This method is analagous to Lodash's omit() function for Mongoose schemas.

Gets/sets schema paths.

Sets a path (if arity 2) Gets a path (if arity 1)

Returns the pathType of path for this schema.

Given a path, returns whether it is a real, virtual, nested, or ad-hoc/undefined path.

The paths defined on this schema. The keys are the top-level paths in this schema, and the values are instances of the SchemaType class.

Returns a new schema that has the picked paths from this schema.

This method is analagous to Lodash's pick() function for Mongoose schemas.

Registers a plugin for this schema.

Defines a post hook for the document

Defines a pre hook for the model.

Adds a method call to the queue.

Removes the given path (or [paths]).

Remove an index by name or index specification.

removeIndex only removes indexes from your schema object. Does not affect the indexes in MongoDB.

Removes the given virtual or virtuals from the schema.

Returns an Array of path strings that are required by this schema.

Add an Atlas search index that Mongoose will create using Model.createSearchIndex(). This function only works when connected to MongoDB Atlas.

Sets a schema option.

Adds static "class" methods to Models compiled from this schema.

If a hash of name/fn pairs is passed as the only argument, each name/fn pair will be added as methods.

If a hash of name/fn pairs is passed as the only argument, each name/fn pair will be added as statics.

Returns a JSON schema representation of this Schema.

By default, returns normal JSON schema representation, which is not typically what you want to use with MongoDB's $jsonSchema collection option. Use the useBsonType: true option to return MongoDB $jsonSchema syntax instead.

In addition to types, jsonSchema() supports the following Mongoose validators:

const schema = new Schema({ name: String }); // { required: ['_id'], properties: { name: { type: ['string', 'null'] }, _id: { type: 'string' } } } schema.toJSONSchema();

// { required: ['_id'], properties: { name: { bsonType: ['string', 'null'] }, _id: { bsonType: 'objectId' } } } schema.toJSONSchema({ useBsonType: true });

Creates a virtual type with the given name.

Returns the virtual type with the given name.

Object containing all virtuals defined on this schema. The objects' keys are the virtual paths and values are instances of VirtualType.

This property is typically only useful for plugin authors and advanced users. You do not need to interact with this property at all to use mongoose.

Reserved document keys.

Keys in this object are names that are warned in schema declarations because they have the potential to break Mongoose/ Mongoose plugins functionality. If you create a schema using new Schema() with one of these property names, Mongoose will log a warning.

NOTE: Use of these terms as method names is permitted, but play at your own risk, as they may be existing mongoose document methods you are stomping on.

schemaMaps are JSON schemas, which use the following structure to represent objects: { field: { bsonType: 'object', properties: { ... } } }

for example, a schema that looks like this { a: { b: int32 } } would be encoded as { a: { bsonType: 'object', properties: { b: < encryption configuration > } } }

This function takes an array of path segments, an output object (that gets mutated) and a value to be associated with the full path, and constructs a valid CSFLE JSON schema path for the object. This works for deeply nested properties as well.

**Examples:**

Example 1 (swift):
```swift
const child = new Schema({ name: String });
const schema = new Schema({ name: String, age: Number, children: [child] });
const Tree = mongoose.model('Tree', schema);

// setting schema options
new Schema({ name: String }, { id: false, autoIndex: false })
```

Example 2 (javascript):
```javascript
const mongoose = require('mongoose');
const ObjectId = mongoose.Schema.Types.ObjectId;
```

Example 3 (css):
```css
const Mixed = mongoose.Schema.Types.Mixed;
new mongoose.Schema({ _user: Mixed })
```

Example 4 (css):
```css
const schema = Schema.create({ name: String }, { toObject: { virtuals: true } });
// Equivalent:
const schema2 = new Schema({ name: String }, { toObject: { virtuals: true } });
```

---

## Getting Started

**URL:** https://mongoosejs.com/docs/index.html

**Contents:**
- Getting Started
- Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

For brevity, let's assume that all following code is within the main() function.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
npm install mongoose
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');

main().catch(err => console.log(err));

async function main() {
  await mongoose.connect('mongodb://127.0.0.1:27017/test');

  // use `await mongoose.connect('mongodb://user:password@127.0.0.1:27017/test');` if your database has auth enabled
}
```

Example 3 (css):
```css
const kittySchema = new mongoose.Schema({
  name: String
});
```

Example 4 (javascript):
```javascript
const Kitten = mongoose.model('Kitten', kittySchema);
```

---

## TypeScript Support

**URL:** https://mongoosejs.com/docs/typescript.html

**Contents:**
- TypeScript Support
- Creating Your First Document
- Using Generics
- Using Custom Bindings
- Next Up

Mongoose introduced officially supported TypeScript bindings in v5.11.0. Mongoose's index.d.ts file supports a wide variety of syntaxes and strives to be compatible with @types/mongoose where possible. This guide describes Mongoose's recommended approach to working with Mongoose in TypeScript.

To get started with Mongoose in TypeScript, you need to:

By default, Mongoose automatically infers the shape of your documents based on your schema definition. However, if you modify your schema after your new Schema() call (like with plugins) then Mongoose's inferred type may be incorrect. For cases where Mongoose's automatic schema type inference is incorrect, you can define a raw document interface that tells Mongoose the type of documents in your database as follows.

You as the developer are responsible for ensuring that your document interface lines up with your Mongoose schema. For example, Mongoose won't report an error if email is required in your Mongoose schema but optional in your document interface.

The User() constructor returns an instance of HydratedDocument<IUser>. IUser is a document interface, it represents the raw object structure that IUser objects look like in MongoDB. HydratedDocument<IUser> represents a hydrated Mongoose document, with methods, virtuals, and other Mongoose-specific features.

To define a property of type ObjectId, you should use Types.ObjectId in the TypeScript document interface. You should use 'ObjectId' or Schema.Types.ObjectId in your schema definition.

That's because Schema.Types.ObjectId is a class that inherits from SchemaType, not the class you use to create a new MongoDB ObjectId.

If Mongoose's built-in index.d.ts file does not work for you, you can remove it in a postinstall script in your package.json as shown below. However, before you do, please open an issue on Mongoose's GitHub page and describe the issue you're experiencing.

Now that you've seen the basics of how to use Mongoose in TypeScript, let's take a look at methods in TypeScript.

**Examples:**

Example 1 (javascript):
```javascript
import { Schema, model, connect } from 'mongoose';

// 1. Create a Schema corresponding to the document interface.
const userSchema = new Schema({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// 2. Create a Model.
const User = model('User', userSchema);

run().catch(err => console.log(err));

async function run() {
  // 3. Connect to MongoDB
  await connect('mongodb://127.0.0.1:27017/test');

  const user = new User({
    name: 'Bill',
    email: 'bill@initech.com',
    avatar: 'https://i.imgur.com/dM7Thhn.png'
  });
  await user.save();

  const email: string = user.email;
  console.log(email); // 'bill@initech.com'
}
```

Example 2 (javascript):
```javascript
import { Schema, model, connect } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  avatar?: string;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// 3. Create a Model.
const User = model<IUser>('User', userSchema);

run().catch(err => console.log(err));

async function run() {
  // 4. Connect to MongoDB
  await connect('mongodb://127.0.0.1:27017/test');

  const user = new User({
    name: 'Bill',
    email: 'bill@initech.com',
    avatar: 'https://i.imgur.com/dM7Thhn.png'
  });
  await user.save();

  console.log(user.email); // 'bill@initech.com'
}
```

Example 3 (sql):
```sql
import { HydratedDocument } from 'mongoose';

const user: HydratedDocument<IUser> = new User({
  name: 'Bill',
  email: 'bill@initech.com',
  avatar: 'https://i.imgur.com/dM7Thhn.png'
});
```

Example 4 (typescript):
```typescript
import { Schema, Types } from 'mongoose';

// 1. Create an interface representing a document in MongoDB.
interface IUser {
  name: string;
  email: string;
  // Use `Types.ObjectId` in document interface...
  organization: Types.ObjectId;
}

// 2. Create a Schema corresponding to the document interface.
const userSchema = new Schema<IUser>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  // And `Schema.Types.ObjectId` in the schema definition.
  organization: { type: Schema.Types.ObjectId, ref: 'Organization' }
});
```

---

## Guides

**URL:** https://mongoosejs.com/docs/6.x/docs/guides.html

**Contents:**
- Guides
- Mongoose Core Concepts
- Advanced Topics
- Integrations
- Other Guides
- Migration Guides

Mongoose guides provide detailed tutorials on Mongoose's core concepts and integrating Mongoose with external tools and frameworks.

---

## Schemas in TypeScript

**URL:** https://mongoosejs.com/docs/typescript/schemas.html

**Contents:**
- Schemas in TypeScript
- Automatic type inference
- Separate document interface definition
- Generic parameters
- Schema vs Interface fields
- Arrays

Mongoose schemas are how you tell Mongoose what your documents look like. Mongoose schemas are separate from TypeScript interfaces, so you need to either define both a raw document interface and a schema; or rely on Mongoose to automatically infer the type from the schema definition.

Mongoose can automatically infer the document type from your schema definition as follows. We recommend relying on automatic type inference when defining schemas and models.

There are a few caveats for using automatic type inference:

If you must define your schema separately, use as const (const schemaDefinition = { ... } as const;) to prevent type widening. TypeScript will automatically widen types like required: false to required: boolean, which will cause Mongoose to assume the field is required. Using as const forces TypeScript to retain these types.

If you need to explicitly get the raw document type (the value returned from doc.toObject(), await Model.findOne().lean(), etc.) from your schema definition, you can use Mongoose's inferRawDocType helper as follows:

If automatic type inference doesn't work for you, you can always fall back to document interface definitions.

If automatic type inference doesn't work for you, you can define a separate raw document interface as follows.

By default, Mongoose does not check if your raw document interface lines up with your schema. For example, the above code won't throw an error if email is optional in the document interface, but required in schema.

The Mongoose Schema class in TypeScript has 9 generic parameters:

The first generic param, DocType, represents the type of documents that Mongoose will store in MongoDB. Mongoose wraps DocType in a Mongoose document for cases like the this parameter to document middleware. For example:

The second generic param, M, is the model used with the schema. Mongoose uses the M type in model middleware defined in the schema.

The third generic param, TInstanceMethods is used to add types for instance methods defined in the schema.

The 4th param, TQueryHelpers, is used to add types for chainable query helpers.

Mongoose checks to make sure that every path in your schema is defined in your document interface.

For example, the below code will fail to compile because email is a path in the schema, but not in the DocType interface.

However, Mongoose does not check for paths that exist in the document interface, but not in the schema. For example, the below code compiles.

This is because Mongoose has numerous features that add paths to your schema that should be included in the DocType interface without you explicitly putting these paths in the Schema() constructor. For example, timestamps and plugins.

When you define an array in a document interface, we recommend using vanilla JavaScript arrays, not Mongoose's Types.Array type or Types.DocumentArray type. Instead, use the THydratedDocumentType generic for models and schemas to define that the hydrated document type has paths of type Types.Array and Types.DocumentArray.

Use HydratedArraySubdocument<RawDocType> for the type of array subdocuments, and HydratedSingleSubdocument<RawDocType> for single subdocuments.

If you are not using schema methods, middleware, or virtuals, you can omit the last 7 generic parameters to Schema() and just define your schema using new mongoose.Schema<IOrder, OrderModelType>(...). The THydratedDocumentType parameter for schemas is primarily for setting the value of this on methods and virtuals.

**Examples:**

Example 1 (sql):
```sql
import { Schema, model } from 'mongoose';
// Schema
const schema = new Schema({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});

// `UserModel` will have `name: string`, etc.
const UserModel = mongoose.model('User', schema);

const doc = new UserModel({ name: 'test', email: 'test' });
doc.name; // string
doc.email; // string
doc.avatar; // string | undefined | null
```

Example 2 (typescript):
```typescript
import { Schema, InferRawDocType, model } from 'mongoose';

const schemaDefinition = {
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
} as const;
const schema = new Schema(schemaDefinition);

const UserModel = model('User', schema);
const doc = new UserModel({ name: 'test', email: 'test' });

type RawUserDocument = InferRawDocType<typeof schemaDefinition>;

useRawDoc(doc.toObject());

function useRawDoc(doc: RawUserDocument) {
  // ...
}
```

Example 3 (typescript):
```typescript
import { Schema } from 'mongoose';

// Raw document interface. Contains the data type as it will be stored
// in MongoDB. So you can ObjectId, Buffer, and other custom primitive data types.
// But no Mongoose document arrays or subdocuments.
interface User {
  name: string;
  email: string;
  avatar?: string;
}

// Schema
const schema = new Schema<User>({
  name: { type: String, required: true },
  email: { type: String, required: true },
  avatar: String
});
```

Example 4 (gdscript):
```gdscript
export class Schema<
  RawDocType = any,
  TModelType = Model<RawDocType, any, any, any>,
  TInstanceMethods = {},
  TQueryHelpers = {},
  TVirtuals = {},
  TStaticMethods = {},
  TSchemaOptions = DefaultSchemaOptions,
  DocType = ...,
  THydratedDocumentType = HydratedDocument<FlatRecord<DocType>, TVirtuals & TInstanceMethods>
>
  extends events.EventEmitter {
  // ...
}
```

---

## Getting Started

**URL:** https://mongoosejs.com/docs/7.x/docs/index.html

**Contents:**
- Getting Started
- Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

For brevity, let's assume that all following code is within the main() function.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method. The first argument to the callback will be an error if any occurred.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens to the callback.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
npm install mongoose --save
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');

main().catch(err => console.log(err));

async function main() {
  await mongoose.connect('mongodb://127.0.0.1:27017/test');

  // use `await mongoose.connect('mongodb://user:password@127.0.0.1:27017/test');` if your database has auth enabled
}
```

Example 3 (javascript):
```javascript
const kittySchema = new mongoose.Schema({
  name: String
});
```

Example 4 (javascript):
```javascript
const Kitten = mongoose.model('Kitten', kittySchema);
```

---

## Schemas

**URL:** https://mongoosejs.com/docs/7.x/docs/guide.html

**Contents:**
- Schemas
- Defining your schema
- Creating a model
- Ids
- Instance methods
- Statics
- Query Helpers
- Indexes
- Virtuals
- Aliases

If you haven't yet done so, please take a minute to read the quickstart to get an idea of how Mongoose works. If you are migrating from 6.x to 7.x please take a moment to read the migration guide.

Everything in Mongoose starts with a Schema. Each schema maps to a MongoDB collection and defines the shape of the documents within that collection.

If you want to add additional keys later, use the Schema#add method.

Each key in our code blogSchema defines a property in our documents which will be cast to its associated SchemaType. For example, we've defined a property title which will be cast to the String SchemaType and property date which will be cast to a Date SchemaType.

Notice above that if a property only requires a type, it can be specified using a shorthand notation (contrast the title property above with the date property).

Keys may also be assigned nested objects containing further key/type definitions like the meta property above. This will happen whenever a key's value is a POJO that doesn't have a type property.

In these cases, Mongoose only creates actual schema paths for leaves in the tree. (like meta.votes and meta.favs above), and the branches do not have actual paths. A side-effect of this is that meta above cannot have its own validation. If validation is needed up the tree, a path needs to be created up the tree - see the Subdocuments section for more information on how to do this. Also read the Mixed subsection of the SchemaTypes guide for some gotchas.

The permitted SchemaTypes are:

Read more about SchemaTypes here.

Schemas not only define the structure of your document and casting of properties, they also define document instance methods, static Model methods, compound indexes, and document lifecycle hooks called middleware.

To use our schema definition, we need to convert our blogSchema into a Model we can work with. To do so, we pass it into mongoose.model(modelName, schema):

By default, Mongoose adds an _id property to your schemas.

When you create a new document with the automatically added _id property, Mongoose creates a new _id of type ObjectId to your document.

You can also overwrite Mongoose's default _id with your own _id. Just be careful: Mongoose will refuse to save a document that doesn't have an _id, so you're responsible for setting _id if you define your own _id path.

Instances of Models are documents. Documents have many of their own built-in instance methods. We may also define our own custom document instance methods.

Now all of our animal instances have a findSimilarTypes method available to them.

You can also add static functions to your model. There are three equivalent ways to add a static:

Do not declare statics using ES6 arrow functions (=>). Arrow functions explicitly prevent binding this, so the above examples will not work because of the value of this.

You can also add query helper functions, which are like instance methods but for mongoose queries. Query helper methods let you extend mongoose's chainable query builder API.

MongoDB supports secondary indexes. With mongoose, we define these indexes within our Schema at the path level or the schema level. Defining indexes at the schema level is necessary when creating compound indexes.

See SchemaType#index() for other index options.

When your application starts up, Mongoose automatically calls createIndex for each defined index in your schema. Mongoose will call createIndex for each index sequentially, and emit an 'index' event on the model when all the createIndex calls succeeded or when there was an error. While nice for development, it is recommended this behavior be disabled in production since index creation can cause a significant performance impact. Disable the behavior by setting the autoIndex option of your schema to false, or globally on the connection by setting the option autoIndex to false.

Mongoose will emit an index event on the model when indexes are done building or an error occurred.

See also the Model#ensureIndexes method.

Virtuals are document properties that you can get and set but that do not get persisted to MongoDB. The getters are useful for formatting or combining fields, while setters are useful for de-composing a single value into multiple values for storage.

Suppose you want to print out the person's full name. You could do it yourself:

But concatenating the first and last name every time can get cumbersome. And what if you want to do some extra processing on the name, like removing diacritics? A virtual property getter lets you define a fullName property that won't get persisted to MongoDB.

Now, mongoose will call your getter function every time you access the fullName property:

If you use toJSON() or toObject() Mongoose will not include virtuals by default. Pass { virtuals: true } to toJSON() or toObject() to include virtuals.

The above caveat for toJSON() also includes the output of calling JSON.stringify() on a Mongoose document, because JSON.stringify() calls toJSON(). To include virtuals in JSON.stringify() output, you can either call toObject({ virtuals: true }) on the document before calling JSON.stringify(), or set the toJSON: { virtuals: true } option on your schema.

You can also add a custom setter to your virtual that will let you set both first name and last name via the fullName virtual.

Virtual property setters are applied before other validation. So the example above would still work even if the first and last name fields were required.

Only non-virtual properties work as part of queries and for field selection. Since virtuals are not stored in MongoDB, you can't query with them.

You can learn more about virtuals here.

Aliases are a particular type of virtual where the getter and setter seamlessly get and set another property. This is handy for saving network bandwidth, so you can convert a short property name stored in the database into a longer name for code readability.

You can also declare aliases on nested paths. It is easier to use nested schemas and subdocuments, but you can also declare nested path aliases inline as long as you use the full nested path nested.myProp as the alias.

Schemas have a few configurable options which can be passed to the constructor or to the set method:

By default, Mongoose's init() function creates all the indexes defined in your model's schema by calling Model.createIndexes() after you successfully connect to MongoDB. Creating indexes automatically is great for development and test environments. But index builds can also create significant load on your production database. If you want to manage indexes carefully in production, you can set autoIndex to false.

The autoIndex option is set to true by default. You can change this default by setting mongoose.set('autoIndex', false);

Before Mongoose builds indexes, it calls Model.createCollection() to create the underlying collection in MongoDB by default. Calling createCollection() sets the collection's default collation based on the collation option and establishes the collection as a capped collection if you set the capped schema option.

You can disable this behavior by setting autoCreate to false using mongoose.set('autoCreate', false). Like autoIndex, autoCreate is helpful for development and test environments, but you may want to disable it for production to avoid unnecessary database calls.

Unfortunately, createCollection() cannot change an existing collection. For example, if you add capped: { size: 1024 } to your schema and the existing collection is not capped, createCollection() will not overwrite the existing collection. That is because the MongoDB server does not allow changing a collection's options without dropping the collection first.

By default, mongoose buffers commands when the connection goes down until the driver manages to reconnect. To disable buffering, set bufferCommands to false.

The schema bufferCommands option overrides the global bufferCommands option.

If bufferCommands is on, this option sets the maximum amount of time Mongoose buffering will wait before throwing an error. If not specified, Mongoose will use 10000 (10 seconds).

Mongoose supports MongoDBs capped collections. To specify the underlying MongoDB collection be capped, set the capped option to the maximum size of the collection in bytes.

The capped option may also be set to an object if you want to pass additional options like max. In this case you must explicitly pass the size option, which is required.

Mongoose by default produces a collection name by passing the model name to the utils.toCollectionName method. This method pluralizes the name. Set this option if you need a different name for your collection.

When you define a discriminator, Mongoose adds a path to your schema that stores which discriminator a document is an instance of. By default, Mongoose adds an __t path, but you can set discriminatorKey to overwrite this default.

When excludeIndexes is true, Mongoose will not create indexes from the given subdocument schema. This option only works when the schema is used in a subdocument path or document array path, Mongoose ignores this option if set on the top-level schema for a model. Defaults to false.

Mongoose assigns each of your schemas an id virtual getter by default which returns the document's _id field cast to a string, or in the case of ObjectIds, its hexString. If you don't want an id getter added to your schema, you may disable it by passing this option at schema construction time.

Mongoose assigns each of your schemas an _id field by default if one is not passed into the Schema constructor. The type assigned is an ObjectId to coincide with MongoDB's default behavior. If you don't want an _id added to your schema at all, you may disable it using this option.

You can only use this option on subdocuments. Mongoose can't save a document without knowing its id, so you will get an error if you try to save a document without an _id.

Mongoose will, by default, "minimize" schemas by removing empty objects.

This behavior can be overridden by setting minimize option to false. It will then store empty objects.

To check whether an object is empty, you can use the $isEmpty() helper:

Allows setting query#read options at the schema level, providing us a way to apply default ReadPreferences to all queries derived from a model.

The alias of each pref is also permitted so instead of having to type out 'secondaryPreferred' and getting the spelling wrong, we can simply pass 'sp'.

The read option also allows us to specify tag sets. These tell the driver from which members of the replica-set it should attempt to read. Read more about tag sets here and here.

NOTE: you may also specify the driver read preference strategy option when connecting:

Allows setting write concern at the schema level.

The shardKey option is used when we have a sharded MongoDB architecture. Each sharded collection is given a shard key which must be present in all insert/update operations. We just need to set this schema option to the same shard key and we’ll be all set.

Note that Mongoose does not send the shardcollection command for you. You must configure your shards yourself.

The strict option, (enabled by default), ensures that values passed to our model constructor that were not specified in our schema do not get saved to the db.

This also affects the use of doc.set() to set a property value.

This value can be overridden at the model instance level by passing a second boolean argument:

The strict option may also be set to "throw" which will cause errors to be produced instead of dropping the bad data.

NOTE: Any key/val set on the instance that does not exist in your schema is always ignored, regardless of schema option.

Mongoose supports a separate strictQuery option to avoid strict mode for query filters. This is because empty query filters cause Mongoose to return all documents in the model, which can cause issues.

The strict option does apply to updates. The strictQuery option is just for query filters.

Mongoose has a separate strictQuery option to toggle strict mode for the filter parameter to queries.

In general, we do not recommend passing user-defined objects as query filters:

In Mongoose 7, strictQuery is false by default. However, you can override this behavior globally:

Exactly the same as the toObject option but only applies when the document's toJSON method is called.

To see all available toJSON/toObject options, read this.

Documents have a toObject method which converts the mongoose document into a plain JavaScript object. This method accepts a few options. Instead of applying these options on a per-document basis, we may declare the options at the schema level and have them applied to all of the schema's documents by default.

To have all virtuals show up in your console.log output, set the toObject option to { getters: true }:

To see all available toObject options, read this.

By default, if you have an object with key 'type' in your schema, mongoose will interpret it as a type declaration.

However, for applications like geoJSON, the 'type' property is important. If you want to control which key mongoose uses to find type declarations, set the 'typeKey' schema option.

By default, documents are automatically validated before they are saved to the database. This is to prevent saving an invalid document. If you want to handle validation manually, and be able to save objects which don't pass validation, you can set validateBeforeSave to false.

The versionKey is a property set on each document when first created by Mongoose. This keys value contains the internal revision of the document. The versionKey option is a string that represents the path to use for versioning. The default is __v. If this conflicts with your application you can configure as such:

Note that Mongoose's default versioning is not a full optimistic concurrency solution. Mongoose's default versioning only operates on arrays as shown below.

If you need optimistic concurrency support for save(), you can set the optimisticConcurrency option

Document versioning can also be disabled by setting the versionKey to false. DO NOT disable versioning unless you know what you are doing.

Mongoose only updates the version key when you use save(). If you use update(), findOneAndUpdate(), etc. Mongoose will not update the version key. As a workaround, you can use the below middleware.

Optimistic concurrency is a strategy to ensure the document you're updating didn't change between when you loaded it using find() or findOne(), and when you update it using save().

For example, suppose you have a House model that contains a list of photos, and a status that represents whether this house shows up in searches. Suppose that a house that has status 'APPROVED' must have at least two photos. You might implement the logic of approving a house document as shown below:

The markApproved() function looks right in isolation, but there might be a potential issue: what if another function removes the house's photos between the findOne() call and the save() call? For example, the below code will succeed:

If you set the optimisticConcurrency option on the House model's schema, the above script will throw an error.

Sets a default collation for every query and aggregation. Here's a beginner-friendly overview of collations.

If you set the timeseries option on a schema, Mongoose will create a timeseries collection for any model that you create from that schema.

skipVersioning allows excluding paths from versioning (i.e., the internal revision will not be incremented even if these paths are updated). DO NOT do this unless you know what you're doing. For subdocuments, include this on the parent document using the fully qualified path.

The timestamps option tells Mongoose to assign createdAt and updatedAt fields to your schema. The type assigned is Date.

By default, the names of the fields are createdAt and updatedAt. Customize the field names by setting timestamps.createdAt and timestamps.updatedAt.

The way timestamps works under the hood is:

By default, Mongoose uses new Date() to get the current time. If you want to overwrite the function Mongoose uses to get the current time, you can set the timestamps.currentTime option. Mongoose will call the timestamps.currentTime function whenever it needs to get the current time.

Mongoose supports defining global plugins, plugins that apply to all schemas.

Sometimes, you may only want to apply a given plugin to some schemas. In that case, you can add pluginTags to a schema:

If you call plugin() with a tags option, Mongoose will only apply that plugin to schemas that have a matching entry in pluginTags.

By default, Mongoose will automatically select() any populated paths for you, unless you explicitly exclude them.

To opt out of selecting populated fields by default, set selectPopulatedPaths to false in your schema.

For legacy reasons, when there is a validation error in subpath of a single nested schema, Mongoose will record that there was a validation error in the single nested schema path as well. For example:

Set the storeSubdocValidationError to false on the child schema to make Mongoose only reports the parent error.

Options like collation and capped affect the options Mongoose passes to MongoDB when creating a new collection. Mongoose schemas support most MongoDB createCollection() options, but not all. You can use the collectionOptions option to set any createCollection() options; Mongoose will use collectionOptions as the default values when calling createCollection() for your schema.

Schemas have a loadClass() method that you can use to create a Mongoose schema from an ES6 class:

Here's an example of using loadClass() to create a schema from an ES6 class:

Schemas are also pluggable which allows us to package up reusable features into plugins that can be shared with the community or just between your projects.

Here's an alternative introduction to Mongoose schemas.

To get the most out of MongoDB, you need to learn the basics of MongoDB schema design. SQL schema design (third normal form) was designed to minimize storage costs, whereas MongoDB schema design is about making common queries as fast as possible. The 6 Rules of Thumb for MongoDB Schema Design blog series is an excellent resource for learning the basic rules for making your queries fast.

Users looking to master MongoDB schema design in Node.js should look into The Little MongoDB Schema Design Book by Christian Kvalheim, the original author of the MongoDB Node.js driver. This book shows you how to implement performant schemas for a laundry list of use cases, including e-commerce, wikis, and appointment bookings.

Now that we've covered Schemas, let's take a look at SchemaTypes.

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
const { Schema } = mongoose;

const blogSchema = new Schema({
  title: String, // String is shorthand for {type: String}
  author: String,
  body: String,
  comments: [{ body: String, date: Date }],
  date: { type: Date, default: Date.now },
  hidden: Boolean,
  meta: {
    votes: Number,
    favs: Number
  }
});
```

Example 2 (javascript):
```javascript
const Blog = mongoose.model('Blog', blogSchema);
// ready to go!
```

Example 3 (javascript):
```javascript
const schema = new Schema();

schema.path('_id'); // ObjectId { ... }
```

Example 4 (javascript):
```javascript
const Model = mongoose.model('Test', schema);

const doc = new Model();
doc._id instanceof mongoose.Types.ObjectId; // true
```

---

## Schemas

**URL:** https://mongoosejs.com/docs/6.x/docs/guide.html

**Contents:**
- Schemas
- Defining your schema
- Creating a model
- Ids
- Instance methods
- Statics
- Query Helpers
- Indexes
- Virtuals
- Aliases

If you haven't yet done so, please take a minute to read the quickstart to get an idea of how Mongoose works. If you are migrating from 5.x to 6.x please take a moment to read the migration guide.

Everything in Mongoose starts with a Schema. Each schema maps to a MongoDB collection and defines the shape of the documents within that collection.

If you want to add additional keys later, use the Schema#add method.

Each key in our code blogSchema defines a property in our documents which will be cast to its associated SchemaType. For example, we've defined a property title which will be cast to the String SchemaType and property date which will be cast to a Date SchemaType.

Notice above that if a property only requires a type, it can be specified using a shorthand notation (contrast the title property above with the date property).

Keys may also be assigned nested objects containing further key/type definitions like the meta property above. This will happen whenever a key's value is a POJO that doesn't have a type property.

In these cases, Mongoose only creates actual schema paths for leaves in the tree. (like meta.votes and meta.favs above), and the branches do not have actual paths. A side-effect of this is that meta above cannot have its own validation. If validation is needed up the tree, a path needs to be created up the tree - see the Subdocuments section for more information on how to do this. Also read the Mixed subsection of the SchemaTypes guide for some gotchas.

The permitted SchemaTypes are:

Read more about SchemaTypes here.

Schemas not only define the structure of your document and casting of properties, they also define document instance methods, static Model methods, compound indexes, and document lifecycle hooks called middleware.

To use our schema definition, we need to convert our blogSchema into a Model we can work with. To do so, we pass it into mongoose.model(modelName, schema):

By default, Mongoose adds an _id property to your schemas.

When you create a new document with the automatically added _id property, Mongoose creates a new _id of type ObjectId to your document.

You can also overwrite Mongoose's default _id with your own _id. Just be careful: Mongoose will refuse to save a document that doesn't have an _id, so you're responsible for setting _id if you define your own _id path.

Instances of Models are documents. Documents have many of their own built-in instance methods. We may also define our own custom document instance methods.

Now all of our animal instances have a findSimilarTypes method available to them.

You can also add static functions to your model. There are three equivalent ways to add a static:

Do not declare statics using ES6 arrow functions (=>). Arrow functions explicitly prevent binding this, so the above examples will not work because of the value of this.

You can also add query helper functions, which are like instance methods but for mongoose queries. Query helper methods let you extend mongoose's chainable query builder API.

MongoDB supports secondary indexes. With mongoose, we define these indexes within our Schema at the path level or the schema level. Defining indexes at the schema level is necessary when creating compound indexes.

See SchemaType#index() for other index options.

When your application starts up, Mongoose automatically calls createIndex for each defined index in your schema. Mongoose will call createIndex for each index sequentially, and emit an 'index' event on the model when all the createIndex calls succeeded or when there was an error. While nice for development, it is recommended this behavior be disabled in production since index creation can cause a significant performance impact. Disable the behavior by setting the autoIndex option of your schema to false, or globally on the connection by setting the option autoIndex to false.

Mongoose will emit an index event on the model when indexes are done building or an error occurred.

See also the Model#ensureIndexes method.

Virtuals are document properties that you can get and set but that do not get persisted to MongoDB. The getters are useful for formatting or combining fields, while setters are useful for de-composing a single value into multiple values for storage.

Suppose you want to print out the person's full name. You could do it yourself:

But concatenating the first and last name every time can get cumbersome. And what if you want to do some extra processing on the name, like removing diacritics? A virtual property getter lets you define a fullName property that won't get persisted to MongoDB.

Now, mongoose will call your getter function every time you access the fullName property:

If you use toJSON() or toObject() mongoose will not include virtuals by default. This includes the output of calling JSON.stringify() on a Mongoose document, because JSON.stringify() calls toJSON(). Pass { virtuals: true } to either toObject() or toJSON().

You can also add a custom setter to your virtual that will let you set both first name and last name via the fullName virtual.

Virtual property setters are applied before other validation. So the example above would still work even if the first and last name fields were required.

Only non-virtual properties work as part of queries and for field selection. Since virtuals are not stored in MongoDB, you can't query with them.

You can learn more about virtuals here.

Aliases are a particular type of virtual where the getter and setter seamlessly get and set another property. This is handy for saving network bandwidth, so you can convert a short property name stored in the database into a longer name for code readability.

You can also declare aliases on nested paths. It is easier to use nested schemas and subdocuments, but you can also declare nested path aliases inline as long as you use the full nested path nested.myProp as the alias.

Schemas have a few configurable options which can be passed to the constructor or to the set method:

By default, Mongoose's init() function creates all the indexes defined in your model's schema by calling Model.createIndexes() after you successfully connect to MongoDB. Creating indexes automatically is great for development and test environments. But index builds can also create significant load on your production database. If you want to manage indexes carefully in production, you can set autoIndex to false.

The autoIndex option is set to true by default. You can change this default by setting mongoose.set('autoIndex', false);

Before Mongoose builds indexes, it calls Model.createCollection() to create the underlying collection in MongoDB by default. Calling createCollection() sets the collection's default collation based on the collation option and establishes the collection as a capped collection if you set the capped schema option.

You can disable this behavior by setting autoCreate to false using mongoose.set('autoCreate', false). Like autoIndex, autoCreate is helpful for development and test environments, but you may want to disable it for production to avoid unnecessary database calls.

Unfortunately, createCollection() cannot change an existing collection. For example, if you add capped: { size: 1024 } to your schema and the existing collection is not capped, createCollection() will not overwrite the existing collection. That is because the MongoDB server does not allow changing a collection's options without dropping the collection first.

By default, mongoose buffers commands when the connection goes down until the driver manages to reconnect. To disable buffering, set bufferCommands to false.

The schema bufferCommands option overrides the global bufferCommands option.

If bufferCommands is on, this option sets the maximum amount of time Mongoose buffering will wait before throwing an error. If not specified, Mongoose will use 10000 (10 seconds).

Mongoose supports MongoDBs capped collections. To specify the underlying MongoDB collection be capped, set the capped option to the maximum size of the collection in bytes.

The capped option may also be set to an object if you want to pass additional options like max. In this case you must explicitly pass the size option, which is required.

Mongoose by default produces a collection name by passing the model name to the utils.toCollectionName method. This method pluralizes the name. Set this option if you need a different name for your collection.

When you define a discriminator, Mongoose adds a path to your schema that stores which discriminator a document is an instance of. By default, Mongoose adds an __t path, but you can set discriminatorKey to overwrite this default.

Mongoose assigns each of your schemas an id virtual getter by default which returns the document's _id field cast to a string, or in the case of ObjectIds, its hexString. If you don't want an id getter added to your schema, you may disable it by passing this option at schema construction time.

Mongoose assigns each of your schemas an _id field by default if one is not passed into the Schema constructor. The type assigned is an ObjectId to coincide with MongoDB's default behavior. If you don't want an _id added to your schema at all, you may disable it using this option.

You can only use this option on subdocuments. Mongoose can't save a document without knowing its id, so you will get an error if you try to save a document without an _id.

Mongoose will, by default, "minimize" schemas by removing empty objects.

This behavior can be overridden by setting minimize option to false. It will then store empty objects.

To check whether an object is empty, you can use the $isEmpty() helper:

Allows setting query#read options at the schema level, providing us a way to apply default ReadPreferences to all queries derived from a model.

The alias of each pref is also permitted so instead of having to type out 'secondaryPreferred' and getting the spelling wrong, we can simply pass 'sp'.

The read option also allows us to specify tag sets. These tell the driver from which members of the replica-set it should attempt to read. Read more about tag sets here and here.

NOTE: you may also specify the driver read preference strategy option when connecting:

Allows setting write concern at the schema level.

The shardKey option is used when we have a sharded MongoDB architecture. Each sharded collection is given a shard key which must be present in all insert/update operations. We just need to set this schema option to the same shard key and we’ll be all set.

Note that Mongoose does not send the shardcollection command for you. You must configure your shards yourself.

The strict option, (enabled by default), ensures that values passed to our model constructor that were not specified in our schema do not get saved to the db.

This also affects the use of doc.set() to set a property value.

This value can be overridden at the model instance level by passing a second boolean argument:

The strict option may also be set to "throw" which will cause errors to be produced instead of dropping the bad data.

NOTE: Any key/val set on the instance that does not exist in your schema is always ignored, regardless of schema option.

Mongoose supports a separate strictQuery option to avoid strict mode for query filters. This is because empty query filters cause Mongoose to return all documents in the model, which can cause issues.

The strict option does apply to updates. The strictQuery option is just for query filters.

Mongoose has a separate strictQuery option to toggle strict mode for the filter parameter to queries.

In general, we do not recommend passing user-defined objects as query filters:

In Mongoose 6, strictQuery is equal to strict by default. However, you can override this behavior globally:

In Mongoose 7, strictQuery default value will be switched back to false. You can prepare for the change by specifying:

Exactly the same as the toObject option but only applies when the document's toJSON method is called.

To see all available toJSON/toObject options, read this.

Documents have a toObject method which converts the mongoose document into a plain JavaScript object. This method accepts a few options. Instead of applying these options on a per-document basis, we may declare the options at the schema level and have them applied to all of the schema's documents by default.

To have all virtuals show up in your console.log output, set the toObject option to { getters: true }:

To see all available toObject options, read this.

By default, if you have an object with key 'type' in your schema, mongoose will interpret it as a type declaration.

However, for applications like geoJSON, the 'type' property is important. If you want to control which key mongoose uses to find type declarations, set the 'typeKey' schema option.

By default, documents are automatically validated before they are saved to the database. This is to prevent saving an invalid document. If you want to handle validation manually, and be able to save objects which don't pass validation, you can set validateBeforeSave to false.

The versionKey is a property set on each document when first created by Mongoose. This keys value contains the internal revision of the document. The versionKey option is a string that represents the path to use for versioning. The default is __v. If this conflicts with your application you can configure as such:

Note that Mongoose's default versioning is not a full optimistic concurrency solution. Mongoose's default versioning only operates on arrays as shown below.

If you need optimistic concurrency support for save(), you can set the optimisticConcurrency option

Document versioning can also be disabled by setting the versionKey to false. DO NOT disable versioning unless you know what you are doing.

Mongoose only updates the version key when you use save(). If you use update(), findOneAndUpdate(), etc. Mongoose will not update the version key. As a workaround, you can use the below middleware.

Optimistic concurrency is a strategy to ensure the document you're updating didn't change between when you loaded it using find() or findOne(), and when you update it using save().

For example, suppose you have a House model that contains a list of photos, and a status that represents whether this house shows up in searches. Suppose that a house that has status 'APPROVED' must have at least two photos. You might implement the logic of approving a house document as shown below:

The markApproved() function looks right in isolation, but there might be a potential issue: what if another function removes the house's photos between the findOne() call and the save() call? For example, the below code will succeed:

If you set the optimisticConcurrency option on the House model's schema, the above script will throw an error.

Sets a default collation for every query and aggregation. Here's a beginner-friendly overview of collations.

If you set the timeseries option on a schema, Mongoose will create a timeseries collection for any model that you create from that schema.

skipVersioning allows excluding paths from versioning (i.e., the internal revision will not be incremented even if these paths are updated). DO NOT do this unless you know what you're doing. For subdocuments, include this on the parent document using the fully qualified path.

The timestamps option tells Mongoose to assign createdAt and updatedAt fields to your schema. The type assigned is Date.

By default, the names of the fields are createdAt and updatedAt. Customize the field names by setting timestamps.createdAt and timestamps.updatedAt.

The way timestamps works under the hood is:

By default, Mongoose uses new Date() to get the current time. If you want to overwrite the function Mongoose uses to get the current time, you can set the timestamps.currentTime option. Mongoose will call the timestamps.currentTime function whenever it needs to get the current time.

Mongoose supports defining global plugins, plugins that apply to all schemas.

Sometimes, you may only want to apply a given plugin to some schemas. In that case, you can add pluginTags to a schema:

If you call plugin() with a tags option, Mongoose will only apply that plugin to schemas that have a matching entry in pluginTags.

By default, Mongoose will automatically select() any populated paths for you, unless you explicitly exclude them.

To opt out of selecting populated fields by default, set selectPopulatedPaths to false in your schema.

For legacy reasons, when there is a validation error in subpath of a single nested schema, Mongoose will record that there was a validation error in the single nested schema path as well. For example:

Set the storeSubdocValidationError to false on the child schema to make Mongoose only reports the parent error.

Schemas have a loadClass() method that you can use to create a Mongoose schema from an ES6 class:

Here's an example of using loadClass() to create a schema from an ES6 class:

Schemas are also pluggable which allows us to package up reusable features into plugins that can be shared with the community or just between your projects.

Here's an alternative introduction to Mongoose schemas.

To get the most out of MongoDB, you need to learn the basics of MongoDB schema design. SQL schema design (third normal form) was designed to minimize storage costs, whereas MongoDB schema design is about making common queries as fast as possible. The 6 Rules of Thumb for MongoDB Schema Design blog series is an excellent resource for learning the basic rules for making your queries fast.

Users looking to master MongoDB schema design in Node.js should look into The Little MongoDB Schema Design Book by Christian Kvalheim, the original author of the MongoDB Node.js driver. This book shows you how to implement performant schemas for a laundry list of use cases, including e-commerce, wikis, and appointment bookings.

Now that we've covered Schemas, let's take a look at SchemaTypes.

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
const { Schema } = mongoose;

const blogSchema = new Schema({
  title: String, // String is shorthand for {type: String}
  author: String,
  body: String,
  comments: [{ body: String, date: Date }],
  date: { type: Date, default: Date.now },
  hidden: Boolean,
  meta: {
    votes: Number,
    favs: Number
  }
});
```

Example 2 (javascript):
```javascript
const Blog = mongoose.model('Blog', blogSchema);
// ready to go!
```

Example 3 (javascript):
```javascript
const schema = new Schema();

schema.path('_id'); // ObjectId { ... }
```

Example 4 (javascript):
```javascript
const Model = mongoose.model('Test', schema);

const doc = new Model();
doc._id instanceof mongoose.Types.ObjectId; // true
```

---
