# Mongoose - Middleware

**Pages:** 11

---

## Middleware

**URL:** https://mongoosejs.com/docs/7.x/docs/middleware.html

**Contents:**
- Middleware
- Types of Middleware
- Pre
  - Use Cases
  - Errors in Pre Hooks
- Post middleware
- Asynchronous Post Hooks
- Define Middleware Before Compiling Models
- Save/Validate Hooks
- Accessing Parameters in Middleware

Middleware (also called pre and post hooks) are functions which are passed control during execution of asynchronous functions. Middleware is specified on the schema level and is useful for writing plugins.

Mongoose has 4 types of middleware: document middleware, model middleware, aggregate middleware, and query middleware.

Document middleware is supported for the following document functions. In Mongoose, a document is an instance of a Model class. In document middleware functions, this refers to the document. To access the model, use this.constructor.

Query middleware is supported for the following Query functions. Query middleware executes when you call exec() or then() on a Query object, or await on a Query object. In query middleware functions, this refers to the query.

Aggregate middleware is for MyModel.aggregate(). Aggregate middleware executes when you call exec() on an aggregate object. In aggregate middleware, this refers to the aggregation object.

Model middleware is supported for the following model functions. Don't confuse model middleware and document middleware: model middleware hooks into static functions on a Model class, document middleware hooks into methods on a Model class. In model middleware functions, this refers to the model.

Here are the possible strings that can be passed to pre()

All middleware types support pre and post hooks. How pre and post hooks work is described in more detail below.

Note: Mongoose registers updateOne middleware on Query.prototype.updateOne() by default. This means that both doc.updateOne() and Model.updateOne() trigger updateOne hooks, but this refers to a query, not a document. To register updateOne middleware as document middleware, use schema.pre('updateOne', { document: true, query: false }).

Note: Like updateOne, Mongoose registers deleteOne middleware on Query.prototype.deleteOne by default. That means that Model.deleteOne() will trigger deleteOne hooks, and this will refer to a query. However, doc.deleteOne() does not fire deleteOne query middleware for legacy reasons. To register deleteOne middleware as document middleware, use schema.pre('deleteOne', { document: true, query: false }).

Note: The create() function fires save() hooks.

Note: Query middlewares are not executed on subdocuments.

Pre middleware functions are executed one after another, when each middleware calls next.

In mongoose 5.x, instead of calling next() manually, you can use a function that returns a promise. In particular, you can use async/await.

If you use next(), the next() call does not stop the rest of the code in your middleware function from executing. Use the early return pattern to prevent the rest of your middleware function from running when you call next().

Middleware are useful for atomizing model logic. Here are some other ideas:

If any pre hook errors out, mongoose will not execute subsequent middleware or the hooked function. Mongoose will instead pass an error to the callback and/or reject the returned promise. There are several ways to report an error in middleware:

Calling next() multiple times is a no-op. If you call next() with an error err1 and then throw an error err2, mongoose will report err1.

post middleware are executed after the hooked method and all of its pre middleware have completed.

If your post hook function takes at least 2 parameters, mongoose will assume the second parameter is a next() function that you will call to trigger the next middleware in the sequence.

Calling pre() or post() after compiling a model does not work in Mongoose in general. For example, the below pre('save') middleware will not fire.

This means that you must add all middleware and plugins before calling mongoose.model(). The below script will print out "Hello from pre save":

As a consequence, be careful about exporting Mongoose models from the same file that you define your schema. If you choose to use this pattern, you must define global plugins before calling require() on your model file.

The save() function triggers validate() hooks, because mongoose has a built-in pre('save') hook that calls validate(). This means that all pre('validate') and post('validate') hooks get called before any pre('save') hooks.

Mongoose provides 2 ways to get information about the function call that triggered the middleware. For query middleware, we recommend using this, which will be a Mongoose Query instance.

For document middleware, like pre('save'), Mongoose passes the 1st parameter to save() as the 2nd argument to your pre('save') callback. You should use the 2nd argument to get access to the save() call's options, because Mongoose documents don't store all the options you can pass to save().

Mongoose has both query and document hooks for deleteOne().

You can pass options to Schema.pre() and Schema.post() to switch whether Mongoose calls your deleteOne() hook for Document.prototype.deleteOne() or Query.prototype.deleteOne(). Note here that you need to set both document and query properties in the passed object:

Mongoose also has both query and document hooks for validate(). Unlike deleteOne and updateOne, validate middleware applies to Document.prototype.validate by default.

Pre and post save() hooks are not executed on update(), findOneAndUpdate(), etc. You can see a more detailed discussion why in this GitHub issue. Mongoose 4.0 introduced distinct hooks for these functions.

Query middleware differs from document middleware in a subtle but important way: in document middleware, this refers to the document being updated. In query middleware, mongoose doesn't necessarily have a reference to the document being updated, so this refers to the query object rather than the document being updated.

For instance, if you wanted to add an updatedAt timestamp to every updateOne() call, you would use the following pre hook.

You cannot access the document being updated in pre('updateOne') or pre('findOneAndUpdate') query middleware. If you need to access the document that will be updated, you need to execute an explicit query for the document.

However, if you define pre('updateOne') document middleware, this will be the document being updated. That's because pre('updateOne') document middleware hooks into Document#updateOne() rather than Query#updateOne().

Middleware execution normally stops the first time a piece of middleware calls next() with an error. However, there is a special kind of post middleware called "error handling middleware" that executes specifically when an error occurs. Error handling middleware is useful for reporting errors and making error messages more readable.

Error handling middleware is defined as middleware that takes one extra parameter: the 'error' that occurred as the first parameter to the function. Error handling middleware can then transform the error however you want.

Error handling middleware also works with query middleware. You can also define a post update() hook that will catch MongoDB duplicate key errors.

Error handling middleware can transform an error, but it can't remove the error. Even if you call next() with no error as shown above, the function call will still error out.

You can also define hooks for the Model.aggregate() function. In aggregation middleware functions, this refers to the Mongoose Aggregate object. For example, suppose you're implementing soft deletes on a Customer model by adding an isDeleted property. To make sure aggregate() calls only look at customers that aren't soft deleted, you can use the below middleware to add a $match stage to the beginning of each aggregation pipeline.

The Aggregate#pipeline() function lets you access the MongoDB aggregation pipeline that Mongoose will send to the MongoDB server. It is useful for adding stages to the beginning of the pipeline from middleware.

Certain Mongoose hooks are synchronous, which means they do not support functions that return promises or receive a next() callback. Currently, only init hooks are synchronous, because the init() function is synchronous. Below is an example of using pre and post init hooks.

To report an error in an init hook, you must throw a synchronous error. Unlike all other middleware, init middleware does not handle promise rejections.

Now that we've covered middleware, let's take a look at Mongoose's approach to faking JOINs with its query population helper.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new mongoose.Schema({
  name: String
});

const mainSchema = new mongoose.Schema({
  child: [childSchema]
});

mainSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on parent document'); // Will be executed
});

childSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on subdocument'); // Will not be executed
});
```

Example 2 (javascript):
```javascript
const schema = new Schema({ /* ... */ });
schema.pre('save', function(next) {
  // do stuff
  next();
});
```

Example 3 (javascript):
```javascript
schema.pre('save', function() {
  return doStuff().
    then(() => doMoreStuff());
});

// Or, in Node.js >= 7.6.0:
schema.pre('save', async function() {
  await doStuff();
  await doMoreStuff();
});
```

Example 4 (javascript):
```javascript
const schema = new Schema({ /* ... */ });
schema.pre('save', function(next) {
  if (foo()) {
    console.log('calling next!');
    // `return next();` will make sure the rest of this function doesn't run
    /* return */ next();
  }
  // Unless you comment out the `return` above, 'after next' will print
  console.log('after next');
});
```

---

## Further Reading

**URL:** https://mongoosejs.com/docs/7.x/docs/further_reading.html

**Contents:**
- Further Reading
- Video Courses
- Books
- Blog Posts

There's a lot of great content out there to learn more about Mongoose. This page has a list of video courses, books, and blog posts curated by Mongoose maintainers. We've gone through every course, book, and article on this page to make sure it is high quality.

This course is a great introduction to building a RESTful API with Express and Mongoose. The instructor is Scott Moss, a serial entrepreneur and former instructor at Hack Reactor, San Francisco's number one coding bootcamp.

Thinkster has created a full stack web development course for just about every backend and frontend framework you can think of. Want to learn how to build a full stack app with Vue and Django, or with Angular and Rails? Thinkster has a course for that. The Node.js tutorial walks you through building a production-ready RESTful API with Express and Mongoose from scratch.

Know someone who wants to get started building Node.js apps but doesn't have a lot of software development experience? This course is more beginner-friendly than API Design and is a great resource for beginners looking to get started.

Looking for a course to take you from zero to Express + Mongoose REST API fast? This is the one. This course is halfway between API Design and Introduction to Mongoose. It focuses more on Express, but also doesn't go into as much detail as API Design.

This course walks you through building a full-stack web application using the VENoM Stack. Try this course if you're a Vue.js expert looking to expand into backend engineering or an experienced MongoDB dev looking to learn about frontend dev.

Stuck maintaining a legacy codebase on Mongoose 3.x? This course will give you a detailed overview of the breaking changes and new features in Mongoose 4.x, so you can upgrade with confidence.

This is the book on Mongoose. Even though it is from 2013, Mongoose for Application Development has stood the test of time. This book does an excellent job summarizing the core ideas that make Mongoose so powerful. Just sub out callbacks for promises.

Christian Kvalheim originally wrote the MongoDB Node.js driver in early 2010, and maintained the driver almost singlehandedly until 2017. As maintainer of the Node.js driver, Christian saw more Node.js+MongoDB code bases than anyone, and his experience is unmatched. This book distills 6 years of hard-learned lessons into concrete examples of how to design MongoDB schemas for massive scale. If you want to learn how to structure MongoDB schemas for an e-commerce store, a category tree, or a blog that can support hundreds of millions of requests per day, this is the book for you.

Coming from an SQL background and having trouble understanding Mongoose and MongoDB? This article explains the core concepts of Mongoose with references to similar concepts in SQL.

Virtually every JavaScript developer has read MDN docs. They're usually one of the top 3 results when you Google "javascript startswith" or any other core JavaScript function. This tutorial provides an overview of Mongoose in MDN's standard style and tone.

---

## Migrating from 8.x to 9.x

**URL:** https://mongoosejs.com/docs/migrating_to_9.html

**Contents:**
- Migrating from 8.x to 9.x
- Pre middleware no longer supports next()
- Schema.prototype.doValidate() now returns a promise
- Update pipelines disallowed by default
- Removed background option for indexes
- mongoose.isValidObjectId() returns false for numbers
- Subdocument deleteOne() hooks execute only when subdocument is deleted
- Hooks for custom methods and statics no longer support callbacks
- Document.prototype.updateOne no longer accepts a callback
- Removed promiseOrCallback

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 8.x to Mongoose 9.x.

If you're still on Mongoose 7.x or earlier, please read the Mongoose 7.x to 8.x migration guide and upgrade to Mongoose 8.x first before upgrading to Mongoose 9.

In Mongoose 9, pre middleware no longer receives a next() parameter. Instead, you should use async functions or promises to handle async pre middleware.

Schema.prototype.doValidate() now returns a promise that rejects with a validation error if one occurred. In Mongoose 8.x, doValidate() took a callback and did not return a promise.

As of MongoDB 4.2, you can pass an array of pipeline stages to updateOne(), updateMany(), and findOneAndUpdate() to modify the document in multiple stages. Mongoose does not cast update pipelines at all, so for Mongoose 9 we've made using update pipelines throw an error by default.

Set updatePipeline: true to enable update pipelines.

You can also set updatePipeline globally to enable update pipelines for all update operations by default.

MongoDB no longer supports the background option for indexes as of MongoDB 4.2. Mongoose 9 will no longer set the background option by default and Mongoose 9 no longer supports setting the background option on Schema.prototype.index().

In Mongoose 8, you could create a new ObjectId from a number, and isValidObjectId() would return true for numbers. In Mongoose 9, isValidObjectId() will return false for numbers and you can no longer create a new ObjectId from a number.

Currently, calling deleteOne() on a subdocument will execute the deleteOne() hooks on the subdocument regardless of whether the subdocument is actually deleted.

Previously, you could use Mongoose middleware with custom methods and statics that took callbacks. In Mongoose 9, this is no longer supported. If you want to use Mongoose middleware with a custom method or static, that custom method or static must be an async function or return a Promise.

Document.prototype.updateOne still supported callbacks in Mongoose 8. In Mongoose 9, the callback parameter was removed.

Mongoose 9 removed the promiseOrCallback helper function.

Mongoose 9 no longer supports isAsync middleware. Middleware functions that use the legacy signature with both next and done callbacks (i.e., function(next, done)) are not supported. We recommend middleware now use promises or async/await.

If you have code that uses isAsync middleware, you must refactor it to use async functions or return a promise instead.

In Mongoose 8, Mongoose queries store an _executionStack property that stores the stack trace of where the query was originally executed for debugging Query was already executed errors. This behavior can cause performance issues with bundlers and source maps. skipOriginalStackTraces was added to work around this behavior. In Mongoose 9, this option is no longer necessary because Mongoose no longer stores the original stack trace.

Mongoose 9 requires Node.js 18 or higher.

Mongoose 9 now returns UUID objects as instances of bson.UUID. In Mongoose 8, UUIDs were Mongoose Buffers that were converted to strings via a getter.

With this change, UUIDs will be represented in hex string format in JSON, even if getters: true is not set.

If you want to convert UUIDs to strings via a getter by default, you can use mongoose.Schema.Types.UUID.get():

In Mongoose 8, certain schema type instances had a caster property which contained either the embedded schema type or embedded subdocument constructor. In Mongoose 9, to make types and internal logic more consistent, we removed the caster property in favor of embeddedSchemaType and Constructor.

In Mongoose 8, there was also an internal $embeddedSchemaType property. That property has been replaced with embeddedSchemaType, which is now part of the public API.

In Mongoose 8, the 3rd parameter to Model() and Document() was either a boolean or options object. If a boolean, Mongoose would interpret the 3rd parameter as the skipId option. In Mongoose 9, the 3rd parameter is always an options object, passing a boolean is no longer supported.

mongoose.Query had a use$geoWithin property that could configure converting $geoWithin to $within to support MongoDB versions before 2.4. That property has been removed in Mongoose 9. $geoWithin is now never converted to $within, because MongoDB no longer supports $within.

The noListener option has been removed from connections and from the useDb() method. In Mongoose 8.x, you could call useDb() with { noListener: true } to prevent the new connection object from listening to state changes on the base connection, which was sometimes useful to reduce memory usage when dynamically creating connections for every request.

In Mongoose 9.x, the noListener option is no longer supported or documented. The second argument to useDb() now only supports { useCache }.

In Mongoose 9, FilterQuery (the first parameter to Model.find(), Model.findOne(), etc.) was renamed to QueryFilter.

In Mongoose 9, the QueryFilter type, which is the type of the first param to Model.find(), Model.findOne(), etc. now enforces stronger types for top-level keys.

This change is backwards breaking if you use generics when creating queries as shown in the following example. If you run into the following issue or any similar issues, you can use as QueryFilter.

In Mongoose 8, create() and insertOne() accepted a generic parameter, which meant TypeScript let you pass any value to the function.

In Mongoose 9, create() and insertOne() no longer accept a generic parameter. Instead, they accept Partial<RawDocType> with some additional query casting applied that allows objects for maps, strings for ObjectIds, and POJOs for subdocuments and document arrays.

If your parameters to create() don't match Partial<RawDocType>, you can use as to cast as follows.

In Mongoose 8 and earlier, id was a property on the Document class that was set to any. This was inconsistent with runtime behavior, where id is a virtual property that returns _id as a string, unless there is already an id property on the schema or the schema has the id option set to false.

Mongoose 9 appends id as a string property to TVirtuals. The Document class no longer has an id property.

**Examples:**

Example 1 (javascript):
```javascript
// Worked in Mongoose 8.x, no longer supported in Mongoose 9!
schema.pre('save', function(next) {
  // Do something async
  next();
});

// Mongoose 9.x example usage
schema.pre('save', async function() {
  // Do something async
});
// or use promises:
schema.pre('save', function() {
  return new Promise((resolve, reject) => {
    // Do something async
    resolve();
  });
});
```

Example 2 (javascript):
```javascript
// Mongoose 8.x function signature
function doValidate(value, cb, scope, options) {}

// Mongoose 8.x example usage
schema.doValidate(value, function(error) {
  if (error) {
    // Handle validation error
  }
}, scope, options);

// Mongoose 9.x function signature
async function doValidate(value, scope, options) {}

// Mongoose 9.x example usage
try {
  await schema.doValidate(value, scope, options);
} catch (error) {
  // Handle validation error
}
```

Example 3 (css):
```css
// Throws in Mongoose 9. Works in Mongoose 8
await Model.updateOne({}, [{ $set: { newProp: 'test2' } }]);
```

Example 4 (css):
```css
// Works in Mongoose 9
await Model.updateOne({}, [{ $set: { newProp: 'test2' } }], { updatePipeline: true });
```

---

## Deprecation Warnings

**URL:** https://mongoosejs.com/docs/deprecations.html

**Contents:**
- Deprecation Warnings

There are no current deprecation warnings for Mongoose 9.x.

---

## Further Reading

**URL:** https://mongoosejs.com/docs/further_reading.html

**Contents:**
- Further Reading
- Video Courses
- Books
- Blog Posts

There's a lot of great content out there to learn more about Mongoose. This page has a list of video courses, books, and blog posts curated by Mongoose maintainers. We've gone through every course, book, and article on this page to make sure it is high quality.

This course is a great introduction to building a RESTful API with Express and Mongoose. The instructor is Scott Moss, a serial entrepreneur and former instructor at Hack Reactor, San Francisco's number one coding bootcamp.

Thinkster has created a full stack web development course for just about every backend and frontend framework you can think of. Want to learn how to build a full stack app with Vue and Django, or with Angular and Rails? Thinkster has a course for that. The Node.js tutorial walks you through building a production-ready RESTful API with Express and Mongoose from scratch.

Know someone who wants to get started building Node.js apps but doesn't have a lot of software development experience? This course is more beginner-friendly than API Design and is a great resource for beginners looking to get started.

Looking for a course to take you from zero to Express + Mongoose REST API fast? This is the one. This course is halfway between API Design and Introduction to Mongoose. It focuses more on Express, but also doesn't go into as much detail as API Design.

This course walks you through building a full-stack web application using the VENoM Stack. Try this course if you're a Vue.js expert looking to expand into backend engineering or an experienced MongoDB dev looking to learn about frontend dev.

Stuck maintaining a legacy codebase on Mongoose 3.x? This course will give you a detailed overview of the breaking changes and new features in Mongoose 4.x, so you can upgrade with confidence.

This is the book on Mongoose. Even though it is from 2013, Mongoose for Application Development has stood the test of time. This book does an excellent job summarizing the core ideas that make Mongoose so powerful. Just sub out callbacks for promises.

Christian Kvalheim originally wrote the MongoDB Node.js driver in early 2010, and maintained the driver almost singlehandedly until 2017. As maintainer of the Node.js driver, Christian saw more Node.js+MongoDB code bases than anyone, and his experience is unmatched. This book distills 6 years of hard-learned lessons into concrete examples of how to design MongoDB schemas for massive scale. If you want to learn how to structure MongoDB schemas for an e-commerce store, a category tree, or a blog that can support hundreds of millions of requests per day, this is the book for you.

Coming from an SQL background and having trouble understanding Mongoose and MongoDB? This article explains the core concepts of Mongoose with references to similar concepts in SQL.

Virtually every JavaScript developer has read MDN docs. They're usually one of the top 3 results when you Google "javascript startswith" or any other core JavaScript function. This tutorial provides an overview of Mongoose in MDN's standard style and tone.

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/further_reading.html

**Contents:**
- Further Reading
- Video Courses
- Books
- Blog Posts

There's a lot of great content out there to learn more about Mongoose. This page has a list of video courses, books, and blog posts curated by Mongoose maintainers. We've gone through every course, book, and article on this page to make sure it is high quality.

This course is a great introduction to building a RESTful API with Express and Mongoose. The instructor is Scott Moss, a serial entrepreneur and former intructor at Hack Reactor, San Francisco's number one coding bootcamp.

Thinkster has created a full stack web development course for just about every backend and frontend framework you can think of. Want to learn how to build a full stack app with Vue and Django, or with Angular and Rails? Thinkster has a course for that. The Node.js tutorial walks you through building a production-ready RESTful API with Express and Mongoose from scratch.

Know someone who wants to get started building Node.js apps but doesn't have a lot of software development experience? This course is more beginner-friendly than API Design and is a great resource for beginners looking to get started.

Looking for a course to take you from zero to Express + Mongoose REST API fast? This is the one. This course is halfway between API Design and Introduction to Mongoose. It focuses more on Express, but also doesn't go into as much detail as API Design.

This course walks you through building a full-stack web application using the VENoM Stack. Try this course if you're a Vue.js expert looking to expand into backend engineering or an experienced MongoDB dev looking to learn about frontend dev.

Stuck maintaining a legacy codebase on Mongoose 3.x? This course will give you a detailed overview of the breaking changes and new features in Mongoose 4.x, so you can upgrade with confidence.

This is the book on Mongoose. Even though it is from 2013, Mongoose for Application Development has stood the test of time. This book does an excellent job summarizing the core ideas that make Mongoose so powerful. Just sub out callbacks for promises.

Christian Kvalheim originally wrote the MongoDB Node.js driver in early 2010, and maintained the driver almost singlehandedly until 2017. As maintainer of the Node.js driver, Christian saw more Node.js+MongoDB code bases than anyone, and his experience is unmatched. This book distills 6 years of hard-learned lessons into concrete examples of how to design MongoDB schemas for massive scale. If you want to learn how to structure MongoDB schemas for an ecommerce store, a category tree, or a blog that can support hundreds of millions of requests per day, this is the book for you.

Coming from an SQL background and having trouble understanding Mongoose and MongoDB? This article explains the core concepts of Mongoose with references to similar concepts in SQL.

Virtually every JavaScript developer has read MDN docs. They're usually one of the top 3 results when you Google "javascript startswith" or any other core JavaScript function. This tutorial provides an overview of Mongoose in MDN's standard style and tone.

---

## Further Reading

**URL:** https://mongoosejs.com/docs/6.x/docs/further_reading.html

**Contents:**
- Further Reading
- Video Courses
- Books
- Blog Posts

There's a lot of great content out there to learn more about Mongoose. This page has a list of video courses, books, and blog posts curated by Mongoose maintainers. We've gone through every course, book, and article on this page to make sure it is high quality.

This course is a great introduction to building a RESTful API with Express and Mongoose. The instructor is Scott Moss, a serial entrepreneur and former instructor at Hack Reactor, San Francisco's number one coding bootcamp.

Thinkster has created a full stack web development course for just about every backend and frontend framework you can think of. Want to learn how to build a full stack app with Vue and Django, or with Angular and Rails? Thinkster has a course for that. The Node.js tutorial walks you through building a production-ready RESTful API with Express and Mongoose from scratch.

Know someone who wants to get started building Node.js apps but doesn't have a lot of software development experience? This course is more beginner-friendly than API Design and is a great resource for beginners looking to get started.

Looking for a course to take you from zero to Express + Mongoose REST API fast? This is the one. This course is halfway between API Design and Introduction to Mongoose. It focuses more on Express, but also doesn't go into as much detail as API Design.

This course walks you through building a full-stack web application using the VENoM Stack. Try this course if you're a Vue.js expert looking to expand into backend engineering or an experienced MongoDB dev looking to learn about frontend dev.

Stuck maintaining a legacy codebase on Mongoose 3.x? This course will give you a detailed overview of the breaking changes and new features in Mongoose 4.x, so you can upgrade with confidence.

This is the book on Mongoose. Even though it is from 2013, Mongoose for Application Development has stood the test of time. This book does an excellent job summarizing the core ideas that make Mongoose so powerful. Just sub out callbacks for promises.

Christian Kvalheim originally wrote the MongoDB Node.js driver in early 2010, and maintained the driver almost singlehandedly until 2017. As maintainer of the Node.js driver, Christian saw more Node.js+MongoDB code bases than anyone, and his experience is unmatched. This book distills 6 years of hard-learned lessons into concrete examples of how to design MongoDB schemas for massive scale. If you want to learn how to structure MongoDB schemas for an e-commerce store, a category tree, or a blog that can support hundreds of millions of requests per day, this is the book for you.

Coming from an SQL background and having trouble understanding Mongoose and MongoDB? This article explains the core concepts of Mongoose with references to similar concepts in SQL.

Virtually every JavaScript developer has read MDN docs. They're usually one of the top 3 results when you Google "javascript startswith" or any other core JavaScript function. This tutorial provides an overview of Mongoose in MDN's standard style and tone.

---

## Middleware

**URL:** https://mongoosejs.com/docs/6.x/docs/middleware.html

**Contents:**
- Middleware
- Types of Middleware
- Pre
  - Use Cases
  - Errors in Pre Hooks
- Post middleware
- Asynchronous Post Hooks
- Define Middleware Before Compiling Models
- Save/Validate Hooks
- Naming Conflicts

Middleware (also called pre and post hooks) are functions which are passed control during execution of asynchronous functions. Middleware is specified on the schema level and is useful for writing plugins.

Mongoose has 4 types of middleware: document middleware, model middleware, aggregate middleware, and query middleware.

Document middleware is supported for the following document functions. In Mongoose, a document is an instance of a Model class. In document middleware functions, this refers to the document. To access the model, use this.constructor.

Query middleware is supported for the following Query functions. Query middleware executes when you call exec() or then() on a Query object, or await on a Query object. In query middleware functions, this refers to the query.

Aggregate middleware is for MyModel.aggregate(). Aggregate middleware executes when you call exec() on an aggregate object. In aggregate middleware, this refers to the aggregation object.

Model middleware is supported for the following model functions. Don't confuse model middleware and document middleware: model middleware hooks into static functions on a Model class, document middleware hooks into methods on a Model class. In model middleware functions, this refers to the model.

Here are the possible strings that can be passed to pre()

All middleware types support pre and post hooks. How pre and post hooks work is described in more detail below.

Note: If you specify schema.pre('remove'), Mongoose will register this middleware for doc.remove() by default. If you want your middleware to run on Query.remove() use schema.pre('remove', { query: true, document: false }, fn).

Note: Unlike schema.pre('remove'), Mongoose registers updateOne and deleteOne middleware on Query#updateOne() and Query#deleteOne() by default. This means that both doc.updateOne() and Model.updateOne() trigger updateOne hooks, but this refers to a query, not a document. To register updateOne or deleteOne middleware as document middleware, use schema.pre('updateOne', { document: true, query: false }).

Note: The create() function fires save() hooks.

Note: Query middlewares are not executed on subdocuments.

Pre middleware functions are executed one after another, when each middleware calls next.

In mongoose 5.x, instead of calling next() manually, you can use a function that returns a promise. In particular, you can use async/await.

If you use next(), the next() call does not stop the rest of the code in your middleware function from executing. Use the early return pattern to prevent the rest of your middleware function from running when you call next().

Middleware are useful for atomizing model logic. Here are some other ideas:

If any pre hook errors out, mongoose will not execute subsequent middleware or the hooked function. Mongoose will instead pass an error to the callback and/or reject the returned promise. There are several ways to report an error in middleware:

Calling next() multiple times is a no-op. If you call next() with an error err1 and then throw an error err2, mongoose will report err1.

post middleware are executed after the hooked method and all of its pre middleware have completed.

If your post hook function takes at least 2 parameters, mongoose will assume the second parameter is a next() function that you will call to trigger the next middleware in the sequence.

Calling pre() or post() after compiling a model does not work in Mongoose in general. For example, the below pre('save') middleware will not fire.

This means that you must add all middleware and plugins before calling mongoose.model(). The below script will print out "Hello from pre save":

As a consequence, be careful about exporting Mongoose models from the same file that you define your schema. If you choose to use this pattern, you must define global plugins before calling require() on your model file.

The save() function triggers validate() hooks, because mongoose has a built-in pre('save') hook that calls validate(). This means that all pre('validate') and post('validate') hooks get called before any pre('save') hooks.

Mongoose has both query and document hooks for remove().

You can pass options to Schema.pre() and Schema.post() to switch whether Mongoose calls your remove() hook for Document.remove() or Model.remove(). Note here that you need to set both document and query properties in the passed object:

Pre and post save() hooks are not executed on update(), findOneAndUpdate(), etc. You can see a more detailed discussion why in this GitHub issue. Mongoose 4.0 introduced distinct hooks for these functions.

Query middleware differs from document middleware in a subtle but important way: in document middleware, this refers to the document being updated. In query middleware, mongoose doesn't necessarily have a reference to the document being updated, so this refers to the query object rather than the document being updated.

For instance, if you wanted to add an updatedAt timestamp to every updateOne() call, you would use the following pre hook.

You cannot access the document being updated in pre('updateOne') or pre('findOneAndUpdate') query middleware. If you need to access the document that will be updated, you need to execute an explicit query for the document.

However, if you define pre('updateOne') document middleware, this will be the document being updated. That's because pre('updateOne') document middleware hooks into Document#updateOne() rather than Query#updateOne().

Middleware execution normally stops the first time a piece of middleware calls next() with an error. However, there is a special kind of post middleware called "error handling middleware" that executes specifically when an error occurs. Error handling middleware is useful for reporting errors and making error messages more readable.

Error handling middleware is defined as middleware that takes one extra parameter: the 'error' that occurred as the first parameter to the function. Error handling middleware can then transform the error however you want.

Error handling middleware also works with query middleware. You can also define a post update() hook that will catch MongoDB duplicate key errors.

Error handling middleware can transform an error, but it can't remove the error. Even if you call next() with no error as shown above, the function call will still error out.

You can also define hooks for the Model.aggregate() function. In aggregation middleware functions, this refers to the Mongoose Aggregate object. For example, suppose you're implementing soft deletes on a Customer model by adding an isDeleted property. To make sure aggregate() calls only look at customers that aren't soft deleted, you can use the below middleware to add a $match stage to the beginning of each aggregation pipeline.

The Aggregate#pipeline() function lets you access the MongoDB aggregation pipeline that Mongoose will send to the MongoDB server. It is useful for adding stages to the beginning of the pipeline from middleware.

Certain Mongoose hooks are synchronous, which means they do not support functions that return promises or receive a next() callback. Currently, only init hooks are synchronous, because the init() function is synchronous. Below is an example of using pre and post init hooks.

To report an error in an init hook, you must throw a synchronous error. Unlike all other middleware, init middleware does not handle promise rejections.

Now that we've covered middleware, let's take a look at Mongoose's approach to faking JOINs with its query population helper.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new mongoose.Schema({
  name: String
});

const mainSchema = new mongoose.Schema({
  child: [childSchema]
});

mainSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on parent document'); // Will be executed
});

childSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on subdocument'); // Will not be executed
});
```

Example 2 (javascript):
```javascript
const schema = new Schema({ /* ... */ });
schema.pre('save', function(next) {
  // do stuff
  next();
});
```

Example 3 (javascript):
```javascript
schema.pre('save', function() {
  return doStuff().
    then(() => doMoreStuff());
});

// Or, in Node.js >= 7.6.0:
schema.pre('save', async function() {
  await doStuff();
  await doMoreStuff();
});
```

Example 4 (javascript):
```javascript
const schema = new Schema({ /* ... */ });
schema.pre('save', function(next) {
  if (foo()) {
    console.log('calling next!');
    // `return next();` will make sure the rest of this function doesn't run
    /* return */ next();
  }
  // Unless you comment out the `return` above, 'after next' will print
  console.log('after next');
});
```

---

## Middleware

**URL:** https://mongoosejs.com/docs/middleware.html

**Contents:**
- Middleware
- Types of Middleware
- Pre
  - Use Cases
  - Errors in Pre Hooks
- Post middleware
- Asynchronous Post Hooks
- Define Middleware Before Compiling Models
- Save/Validate Hooks
- Accessing Parameters in Middleware

Middleware (also called pre and post hooks) are functions which are passed control during execution of asynchronous functions. Middleware is specified on the schema level and is useful for writing plugins.

Mongoose has 4 types of middleware: document middleware, model middleware, aggregate middleware, and query middleware.

Document middleware is supported for the following document functions. In Mongoose, a document is an instance of a Model class. In document middleware functions, this refers to the document. To access the model, use this.constructor.

Query middleware is supported for the following Query functions. Query middleware executes when you call exec() or then() on a Query object, or await on a Query object. In query middleware functions, this refers to the query.

Aggregate middleware is for MyModel.aggregate(). Aggregate middleware executes when you call exec() on an aggregate object. In aggregate middleware, this refers to the aggregation object.

Model middleware is supported for the following model functions. Don't confuse model middleware and document middleware: model middleware hooks into static functions on a Model class, document middleware hooks into methods on a Model class. In model middleware functions, this refers to the model.

Here are the possible strings that can be passed to pre()

All middleware types support pre and post hooks. How pre and post hooks work is described in more detail below.

Note: Mongoose registers updateOne middleware on Query.prototype.updateOne() by default. This means that both doc.updateOne() and Model.updateOne() trigger updateOne hooks, but this refers to a query, not a document. To register updateOne middleware as document middleware, use schema.pre('updateOne', { document: true, query: false }).

Note: Like updateOne, Mongoose registers deleteOne middleware on Query.prototype.deleteOne by default. That means that Model.deleteOne() will trigger deleteOne hooks, and this will refer to a query. However, doc.deleteOne() does not fire deleteOne query middleware for legacy reasons. To register deleteOne middleware as document middleware, use schema.pre('deleteOne', { document: true, query: false }).

Note: The create() function fires save() hooks.

Note: Query middlewares are not executed on subdocuments.

Pre middleware functions are executed one after another.

You can also use a function that returns a promise, including async functions. Mongoose will wait until the promise resolves to move on to the next middleware.

Middleware is useful for atomizing model logic. Here are some other ideas:

If any pre hook errors out, mongoose will not execute subsequent middleware or the hooked function. Mongoose will instead pass an error to the callback and/or reject the returned promise. There are several ways to report an error in middleware:

post middleware are executed after the hooked method and all of its pre middleware have completed.

If your post hook function takes at least 2 parameters, mongoose will assume the second parameter is a next() function that you will call to trigger the next middleware in the sequence.

You can also pass an async function to post(). If you pass an async function that takes at least 2 parameters, you are still responsible for calling next(). However, you can also pass in an async function that takes less than 2 parameters, and Mongoose will wait for the promise to resolve.

Calling pre() or post() after compiling a model does not work in Mongoose in general. For example, the below pre('save') middleware will not fire.

This means that you must add all middleware and plugins before calling mongoose.model(). The below script will print out "Hello from pre save":

As a consequence, be careful about exporting Mongoose models from the same file that you define your schema. If you choose to use this pattern, you must define global plugins before calling require() on your model file.

The save() function triggers validate() hooks, because mongoose has a built-in pre('save') hook that calls validate(). This means that all pre('validate') and post('validate') hooks get called before any pre('save') hooks.

Mongoose provides 2 ways to get information about the function call that triggered the middleware. For query middleware, we recommend using this, which will be a Mongoose Query instance.

Mongoose also passes the 1st parameter to the hooked function, like save(), as the 1st argument to your pre('save') function. You should use the argument to get access to the save() call's options, because Mongoose documents don't store all the options you can pass to save().

Mongoose has both query and document hooks for deleteOne().

You can pass options to Schema.pre() and Schema.post() to switch whether Mongoose calls your deleteOne() hook for Document.prototype.deleteOne() or Query.prototype.deleteOne(). Note here that you need to set both document and query properties in the passed object:

Mongoose also has both query and document hooks for validate(). Unlike deleteOne and updateOne, validate middleware applies to Document.prototype.validate by default.

Pre and post save() hooks are not executed on update(), findOneAndUpdate(), etc. You can see a more detailed discussion why in this GitHub issue. Mongoose 4.0 introduced distinct hooks for these functions.

Query middleware differs from document middleware in a subtle but important way: in document middleware, this refers to the document being updated. In query middleware, mongoose doesn't necessarily have a reference to the document being updated, so this refers to the query object rather than the document being updated.

For instance, if you wanted to add an updatedAt timestamp to every updateOne() call, you would use the following pre hook.

You cannot access the document being updated in pre('updateOne') or pre('findOneAndUpdate') query middleware. If you need to access the document that will be updated, you need to execute an explicit query for the document.

However, if you define pre('updateOne') document middleware, this will be the document being updated. That's because pre('updateOne') document middleware hooks into Document#updateOne() rather than Query#updateOne().

Middleware execution normally stops the first time a piece of middleware throws an error, or returns a promise that rejects. However, there is a special kind of post middleware called "error handling middleware" that executes specifically when an error occurs. Error handling middleware is useful for reporting errors and making error messages more readable.

Error handling middleware is defined as middleware that takes one extra parameter: the 'error' that occurred as the first parameter to the function. Error handling middleware can then transform the error however you want.

Error handling middleware also works with query middleware. You can also define a post update() hook that will catch MongoDB duplicate key errors.

Error handling middleware can transform an error, but it can't remove the error. Even if the error handling middleware succeeds, the function call will still error out.

You can also define hooks for the Model.aggregate() function. In aggregation middleware functions, this refers to the Mongoose Aggregate object. For example, suppose you're implementing soft deletes on a Customer model by adding an isDeleted property. To make sure aggregate() calls only look at customers that aren't soft deleted, you can use the below middleware to add a $match stage to the beginning of each aggregation pipeline.

The Aggregate#pipeline() function lets you access the MongoDB aggregation pipeline that Mongoose will send to the MongoDB server. It is useful for adding stages to the beginning of the pipeline from middleware.

Certain Mongoose hooks are synchronous, which means they do not support functions that return promises. Currently, only init hooks are synchronous, because the init() function is synchronous. Below is an example of using pre and post init hooks.

To report an error in an init hook, you must throw a synchronous error. Unlike all other middleware, init middleware does not handle promise rejections.

Now that we've covered middleware, let's take a look at Mongoose's approach to faking JOINs with its query population helper.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new mongoose.Schema({
  name: String
});

const mainSchema = new mongoose.Schema({
  child: [childSchema]
});

mainSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on parent document'); // Will be executed
});

childSchema.pre('findOneAndUpdate', function() {
  console.log('Middleware on subdocument'); // Will not be executed
});
```

Example 2 (javascript):
```javascript
const schema = new Schema({ /* ... */ });
schema.pre('save', function() {
  // do stuff
});
```

Example 3 (swift):
```swift
schema.pre('save', function() {
  return doStuff().
    then(() => doMoreStuff());
});

// Or, using async functions
schema.pre('save', async function() {
  await doStuff();
  await doMoreStuff();
});

schema.pre('save', function() {
  // Will execute **after** `await doMoreStuff()` is done
});
```

Example 4 (javascript):
```javascript
schema.pre('save', function() {
  const err = new Error('something went wrong');
  throw err;
});

schema.pre('save', function() {
  // You can also return a promise that rejects
  return new Promise((resolve, reject) => {
    reject(new Error('something went wrong'));
  });
});

schema.pre('save', function() {
  // You can also throw a synchronous error
  throw new Error('something went wrong');
});

schema.pre('save', async function() {
  await Promise.resolve();
  // You can also throw an error in an `async` function
  throw new Error('something went wrong');
});

// later...

// Changes will not be persisted to MongoDB because a pre hook errored out
try {
  await myDoc.save();
} catch (err) {
  console.log(err.message); // something went wrong
}
```

---

## AggregationCursor

**URL:** https://mongoosejs.com/docs/api/aggregationcursor.html

**Contents:**
- AggregationCursor
  - AggregationCursor()
      - Parameters:
      - Inherits:
  - AggregationCursor.prototype.addCursorFlag()
      - Parameters:
      - Returns:
  - AggregationCursor.prototype.close()
      - Returns:
      - See:

An AggregationCursor is a concurrency primitive for processing aggregation results one document at a time. It is analogous to QueryCursor.

An AggregationCursor fulfills the Node.js streams3 API, in addition to several other mechanisms for loading documents from MongoDB one at a time.

Creating an AggregationCursor executes the model's pre aggregate hooks, but not the model's post aggregate hooks.

Unless you're an advanced user, do not instantiate this class directly. Use Aggregate#cursor() instead.

Adds a cursor flag. Useful for setting the noCursorTimeout and tailable flags.

Marks this cursor as closed. Will stop streaming and subsequent calls to next() will error.

Execute fn for every document in the cursor. If fn returns a promise, will wait for the promise to resolve before iterating on to the next one. Returns a promise that resolves when done.

Registers a transform function which subsequently maps documents retrieved via the streams interface or .next()

Get the next document from this cursor. Will return null when there are no documents left.

Returns an asyncIterator for use with for/await/of loops You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Node.js 10.x supports async iterators natively without any flags. You can enable async iterators in Node.js 8.x using the --harmony_async_iteration flag.

Note: This function is not set if Symbol.asyncIterator is undefined. If Symbol.asyncIterator is undefined, that means your Node.js version does not support async iterators.

Handles error emitted from pre middleware. In particular, checks for skipWrappedFunction, which allows skipping the actual aggregation and overwriting the function's return value. Because aggregation cursors don't return a value, we need to make sure the user doesn't accidentally set a value in skipWrappedFunction.

**Examples:**

Example 1 (javascript):
```javascript
// Map documents returned by `data` events
Thing.
  find({ name: /^hello/ }).
  cursor().
  map(function (doc) {
   doc.foo = "bar";
   return doc;
  })
  on('data', function(doc) { console.log(doc.foo); });

// Or map documents returned by `.next()`
const cursor = Thing.find({ name: /^hello/ }).
  cursor().
  map(function (doc) {
    doc.foo = "bar";
    return doc;
  });
cursor.next(function(error, doc) {
  console.log(doc.foo);
});
```

Example 2 (javascript):
```javascript
// Async iterator without explicitly calling `cursor()`. Mongoose still
// creates an AggregationCursor instance internally.
const agg = Model.aggregate([{ $match: { age: { $gte: 25 } } }]);
for await (const doc of agg) {
  console.log(doc.name);
}

// You can also use an AggregationCursor instance for async iteration
const cursor = Model.aggregate([{ $match: { age: { $gte: 25 } } }]).cursor();
for await (const doc of cursor) {
  console.log(doc.name);
}
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/middleware.html

**Contents:**
- Middleware
  - Types of Middleware
  - Pre
    - Use Cases
    - Errors in Pre Hooks
  - Post middleware
  - Asynchronous Post Hooks
  - Define Middleware Before Compiling Models
  - Save/Validate Hooks
  - Naming Conflicts

Middleware (also called pre and post hooks) are functions which are passed control during execution of asynchronous functions. Middleware is specified on the schema level and is useful for writing plugins.

Mongoose has 4 types of middleware: document middleware, model middleware, aggregate middleware, and query middleware. Document middleware is supported for the following document functions. In document middleware functions, this refers to the document.

Query middleware is supported for the following Model and Query functions. In query middleware functions, this refers to the query.

Aggregate middleware is for MyModel.aggregate(). Aggregate middleware executes when you call exec() on an aggregate object. In aggregate middleware, this refers to the aggregation object.

Model middleware is supported for the following model functions. In model middleware functions, this refers to the model.

All middleware types support pre and post hooks. How pre and post hooks work is described in more detail below.

Note: If you specify schema.pre('remove'), Mongoose will register this middleware for doc.remove() by default. If you want to your middleware to run on Query.remove() use schema.pre('remove', { query: true, document: false }, fn).

Note: Unlike schema.pre('remove'), Mongoose registers updateOne and deleteOne middleware on Query#updateOne() and Query#deleteOne() by default. This means that both doc.updateOne() and Model.updateOne() trigger updateOne hooks, but this refers to a query, not a document. To register updateOne or deleteOne middleware as document middleware, use schema.pre('updateOne', { document: true, query: false }).

Note: The create() function fires save() hooks.

Pre middleware functions are executed one after another, when each middleware calls next.

In mongoose 5.x, instead of calling next() manually, you can use a function that returns a promise. In particular, you can use async/await.

If you use next(), the next() call does not stop the rest of the code in your middleware function from executing. Use the early return pattern to prevent the rest of your middleware function from running when you call next().

Middleware are useful for atomizing model logic. Here are some other ideas:

If any pre hook errors out, mongoose will not execute subsequent middleware or the hooked function. Mongoose will instead pass an error to the callback and/or reject the returned promise. There are several ways to report an error in middleware:

Calling next() multiple times is a no-op. If you call next() with an error err1 and then throw an error err2, mongoose will report err1.

post middleware are executed after the hooked method and all of its pre middleware have completed.

If your post hook function takes at least 2 parameters, mongoose will assume the second parameter is a next() function that you will call to trigger the next middleware in the sequence.

Calling pre() or post() after compiling a model does not work in Mongoose in general. For example, the below pre('save') middleware will not fire.

This means that you must add all middleware and plugins before calling mongoose.model(). The below script will print out "Hello from pre save":

As a consequence, be careful about exporting Mongoose models from the same file that you define your schema. If you choose to use this pattern, you must define global plugins before calling require() on your model file.

The save() function triggers validate() hooks, because mongoose has a built-in pre('save') hook that calls validate(). This means that all pre('validate') and post('validate') hooks get called before any pre('save') hooks.

Mongoose has both query and document hooks for remove().

You can pass options to Schema.pre() and Schema.post() to switch whether Mongoose calls your remove() hook for Document.remove() or Model.remove(). Note here that you need to set both document and query properties in the passed object:

Pre and post save() hooks are not executed on update(), findOneAndUpdate(), etc. You can see a more detailed discussion why in this GitHub issue. Mongoose 4.0 introduced distinct hooks for these functions.

Query middleware differs from document middleware in a subtle but important way: in document middleware, this refers to the document being updated. In query middleware, mongoose doesn't necessarily have a reference to the document being updated, so this refers to the query object rather than the document being updated.

For instance, if you wanted to add an updatedAt timestamp to every updateOne() call, you would use the following pre hook.

You cannot access the document being updated in pre('updateOne') or pre('findOneAndUpdate') query middleware. If you need to access the document that will be updated, you need to execute an explicit query for the document.

However, if you define pre('updateOne') document middleware, this will be the document being updated. That's because pre('updateOne') document middleware hooks into Document#updateOne() rather than Query#updateOne().

Middleware execution normally stops the first time a piece of middleware calls next() with an error. However, there is a special kind of post middleware called "error handling middleware" that executes specifically when an error occurs. Error handling middleware is useful for reporting errors and making error messages more readable.

Error handling middleware is defined as middleware that takes one extra parameter: the 'error' that occurred as the first parameter to the function. Error handling middleware can then transform the error however you want.

Error handling middleware also works with query middleware. You can also define a post update() hook that will catch MongoDB duplicate key errors.

Error handling middleware can transform an error, but it can't remove the error. Even if you call next() with no error as shown above, the function call will still error out.

You can also define hooks for the Model.aggregate() function. In aggregation middleware functions, this refers to the Mongoose Aggregate object. For example, suppose you're implementing soft deletes on a Customer model by adding an isDeleted property. To make sure aggregate() calls only look at customers that aren't soft deleted, you can use the below middleware to add a $match stage to the beginning of each aggregation pipeline.

The Aggregate#pipeline() function lets you access the MongoDB aggregation pipeline that Mongoose will send to the MongoDB server. It is useful for adding stages to the beginning of the pipeline from middleware.

Certain Mongoose hooks are synchronous, which means they do not support functions that return promises or receive a next() callback. Currently, only init hooks are synchronous, because the init() function is synchronous. Below is an example of using pre and post init hooks.

To report an error in an init hook, you must throw a synchronous error. Unlike all other middleware, init middleware does not handle promise rejections.

Now that we've covered middleware, let's take a look at Mongoose's approach to faking JOINs with its query population helper.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema(..);
schema.pre('save', function(next) {
  // do stuff
  next();
});
```

Example 2 (javascript):
```javascript
schema.pre('save', function() {
  return doStuff().
    then(() => doMoreStuff());
});

// Or, in Node.js >= 7.6.0:
schema.pre('save', async function() {
  await doStuff();
  await doMoreStuff();
});
```

Example 3 (javascript):
```javascript
const schema = new Schema(..);
schema.pre('save', function(next) {
  if (foo()) {
    console.log('calling next!');
    // `return next();` will make sure the rest of this function doesn't run
    /*return*/ next();
  }
  // Unless you comment out the `return` above, 'after next' will print
  console.log('after next');
});
```

Example 4 (javascript):
```javascript
schema.pre('save', function(next) {
  const err = new Error('something went wrong');
  // If you call `next()` with an argument, that argument is assumed to be
  // an error.
  next(err);
});

schema.pre('save', function() {
  // You can also return a promise that rejects
  return new Promise((resolve, reject) => {
    reject(new Error('something went wrong'));
  });
});

schema.pre('save', function() {
  // You can also throw a synchronous error
  throw new Error('something went wrong');
});

schema.pre('save', async function() {
  await Promise.resolve();
  // You can also throw an error in an `async` function
  throw new Error('something went wrong');
});

// later...

// Changes will not be persisted to MongoDB because a pre hook errored out
myDoc.save(function(err) {
  console.log(err.message); // something went wrong
});
```

---
