# Mongoose - Models

**Pages:** 11

---

## Documents

**URL:** https://mongoosejs.com/docs/documents.html

**Contents:**
- Documents
- Documents vs Models
- Retrieving
- Updating Using save()
- Setting Nested Properties
- Updating Using Queries
- Validating
- Overwriting
- Next Up

Mongoose documents represent a one-to-one mapping to documents as stored in MongoDB. Each document is an instance of its Model.

Document and Model are distinct classes in Mongoose. The Model class is a subclass of the Document class. When you use the Model constructor, you create a new document.

In Mongoose, a "document" generally means an instance of a model. You should not have to create an instance of the Document class without going through a model.

When you load documents from MongoDB using model functions like findOne(), you get a Mongoose document back.

Mongoose documents track changes. You can modify a document using vanilla JavaScript assignments and Mongoose will convert it into MongoDB update operators.

The save() method returns a promise. If save() succeeds, the promise resolves to the document that was saved.

If the document with the corresponding _id is not found, Mongoose will report a DocumentNotFoundError:

Mongoose documents have a set() function that you can use to safely set deeply nested properties.

Mongoose documents also have a get() function that lets you safely read deeply nested properties. get() lets you avoid having to explicitly check for nullish values, similar to JavaScript's optional chaining operator ?..

You can use optional chaining ?. and nullish coalescing ?? with Mongoose documents. However, be careful when using nullish coalescing assignments ??= to create nested paths with Mongoose documents.

The save() function is generally the right way to update a document with Mongoose. With save(), you get full validation and middleware.

For cases when save() isn't flexible enough, Mongoose lets you create your own MongoDB updates with casting, middleware, and limited validation.

Note that update(), updateMany(), findOneAndUpdate(), etc. do not execute save() middleware. If you need save middleware and full validation, first query for the document and then save() it.

Documents are casted and validated before they are saved. Mongoose first casts values to the specified type and then validates them. Internally, Mongoose calls the document's validate() method before saving.

Mongoose also supports limited validation on updates using the runValidators option. Mongoose casts parameters to query functions like findOne(), updateOne() by default. However, Mongoose does not run validation on query function parameters by default. You need to set runValidators: true for Mongoose to validate.

Read the validation guide for more details.

There are 2 different ways to overwrite a document (replacing all keys in the document). One way is to use the Document#overwrite() function followed by save().

The other way is to use Model.replaceOne().

Now that we've covered Documents, let's take a look at Subdocuments.

**Examples:**

Example 1 (css):
```css
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const doc = new MyModel();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 2 (javascript):
```javascript
const doc = await MyModel.findOne();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 3 (css):
```css
doc.name = 'foo';

// Mongoose sends an `updateOne({ _id: doc._id }, { $set: { name: 'foo' } })`
// to MongoDB.
await doc.save();
```

Example 4 (javascript):
```javascript
doc.save().then(savedDoc => {
  savedDoc === doc; // true
});
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/models.html

**Contents:**
- Models
  - Compiling your first model
  - Constructing Documents
  - Querying
  - Deleting
  - Updating
  - Change Streams
  - Yet more
  - Next Up

Models are fancy constructors compiled from Schema definitions. An instance of a model is called a document. Models are responsible for creating and reading documents from the underlying MongoDB database.

When you call mongoose.model() on a schema, Mongoose compiles a model for you.

The first argument is the singular name of the collection your model is for. Mongoose automatically looks for the plural, lowercased version of your model name. Thus, for the example above, the model Tank is for the tanks collection in the database.

Note: The .model() function makes a copy of schema. Make sure that you've added everything you want to schema, including hooks, before calling .model()!

An instance of a model is called a document. Creating them and saving to the database is easy.

Note that no tanks will be created/removed until the connection your model uses is open. Every model has an associated connection. When you use mongoose.model(), your model will use the default mongoose connection.

If you create a custom connection, use that connection's model() function instead.

Finding documents is easy with Mongoose, which supports the rich query syntax of MongoDB. Documents can be retreived using each models find, findById, findOne, or where static methods.

See the chapter on queries for more details on how to use the Query api.

Models have static deleteOne() and deleteMany() functions for removing all documents matching the given filter.

Each model has its own update method for modifying documents in the database without returning them to your application. See the API docs for more detail.

If you want to update a single document in the db and return it to your application, use findOneAndUpdate instead.

New in MongoDB 3.6.0 and Mongoose 5.0.0

Change streams provide a way for you to listen to all inserts and updates going through your MongoDB database. Note that change streams do not work unless you're connected to a MongoDB replica set.

The output from the above async function will look like what you see below.

You can read more about change streams in mongoose in this blog post.

The API docs cover many additional methods available like count, mapReduce, aggregate, and more.

Now that we've covered Models, let's take a look at Documents.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new mongoose.Schema({ name: 'string', size: 'string' });
const Tank = mongoose.model('Tank', schema);
```

Example 2 (javascript):
```javascript
const Tank = mongoose.model('Tank', yourSchema);

const small = new Tank({ size: 'small' });
small.save(function (err) {
  if (err) return handleError(err);
  // saved!
});

// or

Tank.create({ size: 'small' }, function (err, small) {
  if (err) return handleError(err);
  // saved!
});

// or, for inserting large batches of documents
Tank.insertMany([{ size: 'small' }], function(err) {

});
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://localhost/gettingstarted', {useNewUrlParser: true});
```

Example 4 (javascript):
```javascript
const connection = mongoose.createConnection('mongodb://localhost:27017/test');
const Tank = connection.model('Tank', yourSchema);
```

---

## Documents

**URL:** https://mongoosejs.com/docs/6.x/docs/documents.html

**Contents:**
- Documents
- Documents vs Models
- Retrieving
- Updating Using save()
- Updating Using Queries
- Validating
- Overwriting
- Next Up

Mongoose documents represent a one-to-one mapping to documents as stored in MongoDB. Each document is an instance of its Model.

Document and Model are distinct classes in Mongoose. The Model class is a subclass of the Document class. When you use the Model constructor, you create a new document.

In Mongoose, a "document" generally means an instance of a model. You should not have to create an instance of the Document class without going through a model.

When you load documents from MongoDB using model functions like findOne(), you get a Mongoose document back.

Mongoose documents track changes. You can modify a document using vanilla JavaScript assignments and Mongoose will convert it into MongoDB update operators.

The save() method returns a promise. If save() succeeds, the promise resolves to the document that was saved.

If the document with the corresponding _id is not found, Mongoose will report a DocumentNotFoundError:

The save() function is generally the right way to update a document with Mongoose. With save(), you get full validation and middleware.

For cases when save() isn't flexible enough, Mongoose lets you create your own MongoDB updates with casting, middleware, and limited validation.

Note that update(), updateMany(), findOneAndUpdate(), etc. do not execute save() middleware. If you need save middleware and full validation, first query for the document and then save() it.

Documents are casted and validated before they are saved. Mongoose first casts values to the specified type and then validates them. Internally, Mongoose calls the document's validate() method before saving.

Mongoose also supports limited validation on updates using the runValidators option. Mongoose casts parameters to query functions like findOne(), updateOne() by default. However, Mongoose does not run validation on query function parameters by default. You need to set runValidators: true for Mongoose to validate.

Read the validation guide for more details.

There are 2 different ways to overwrite a document (replacing all keys in the document). One way is to use the Document#overwrite() function followed by save().

The other way is to use Model.replaceOne().

Now that we've covered Documents, let's take a look at Subdocuments.

**Examples:**

Example 1 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const doc = new MyModel();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 2 (javascript):
```javascript
const doc = await MyModel.findOne();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 3 (javascript):
```javascript
doc.name = 'foo';

// Mongoose sends an `updateOne({ _id: doc._id }, { $set: { name: 'foo' } })`
// to MongoDB.
await doc.save();
```

Example 4 (javascript):
```javascript
doc.save().then(savedDoc => {
  savedDoc === doc; // true
});
```

---

## Migrating from 6.x to 7.x

**URL:** https://mongoosejs.com/docs/7.x/docs/migrating_to_7.html

**Contents:**
- Migrating from 6.x to 7.x
- Version Requirements
- strictQuery
- Removed remove()
- Dropped callback support
- Removed update()
- ObjectId requires new
- id Setter
- Discriminator schemas use base schema options by default
- Removed castForQueryWrapper, updated castForQuery() signature

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 6.x to Mongoose 7.x.

If you're still on Mongoose 5.x, please read the Mongoose 5.x to 6.x migration guide and upgrade to Mongoose 6.x first.

Mongoose now requires Node.js >= 14.0.0 and MongoDB Node Driver >= 5.0.0.

See the MongoDB Node Driver migration guide for detailed info.

strictQuery is now false by default.

The remove() method on documents and models has been removed. Use deleteOne() or deleteMany() instead.

Keep in mind that deleteOne() hooks are treated as query middleware by default. So for middleware, please do the following:

The following functions no longer accept callbacks. They always return promises.

If you are using the above functions with callbacks, we recommend switching to async/await, or promises if async functions don't work for you. If you need help refactoring a legacy codebase, this tool from Mastering JS callbacks to async await using ChatGPT.

Model.update(), Query.prototype.update(), and Document.prototype.update() have been removed. Use updateOne() instead.

In Mongoose 6 and older, you could define a new ObjectId without using the new keyword:

In Mongoose 7, ObjectId is now a JavaScript class, so you need to use the new keyword.

Starting in Mongoose 7.4, Mongoose's built-in id virtual (which stores the document's _id as a string) has a setter which allows modifying the document's _id property via id.

This can cause surprising behavior if you create a new TestModel(obj) where obj contains both an id and an _id, or if you use doc.set()

The id setter was later removed in Mongoose 8 due to compatibility issues.

When you use Model.discriminator(), Mongoose will now use the discriminator base schema's options by default. This means you don't need to explicitly set child schema options to match the base schema's.

Mongoose now always calls SchemaType castForQuery() method with 3 arguments: $conditional, value, and context. If you've implemented a custom schema type that defines its own castForQuery() method, you need to update the method as follows.

Mongoose now copies user defined schema options when adding one schema to another. For example, childSchema below will get baseSchema's id and toJSON options.

This applies both when creating a new schema using an array of schemas, as well as when calling add() as follows.

The internal _bsontype property on ObjectIds is equal to 'ObjectId' in Mongoose 7, as opposed to 'ObjectID' in Mongoose 6.

Please update any places where you use _bsontype to check if an object is an ObjectId. This may also affect libraries that use Mongoose.

MongoDB no longer supports mapReduce, so Mongoose 7 no longer has a Model.mapReduce() function. Use the aggregation framework as a replacement for mapReduce().

Mongoose 7 no longer supports plugging in custom promise libraries. So the following no longer makes Mongoose return Bluebird promises in Mongoose 7.

If you want to use Bluebird for all promises globally, you can do the following:

Before Mongoose 5.2.0, you needed to enable the keepAlive option to initiate TCP keepalive to prevent "connection closed" errors. However, keepAlive has been true by default since Mongoose 5.2.0, and the keepAlive is deprecated as of Mongoose 7.2.0. Please remove keepAlive and keepAliveInitialDelay options from your Mongoose connections.

Mongoose 7 no longer exports a LeanDocument type, and no longer supports passing a document type that extends Document into Model<>.

Mongoose's HydratedDocument type transforms a raw document interface into the type of the hydrated Mongoose document, including virtuals, methods, etc. In Mongoose 7, the generic parameters to HydratedDocument have changed. In Mongoose 6, the generic parameters were:

In Mongoose 7, the new type is as follows.

In Mongoose 7, the first parameter is the raw document interface, the 2nd parameter is any document-specific overrides (usually virtuals and methods), and the 3rd parameter is any query helpers associated with the document's model.

The key difference is that, in Mongoose 6, the 3rd generic param was the document's virtuals. In Mongoose 7, the 3rd generic param is the document's query helpers.

**Examples:**

Example 1 (javascript):
```javascript
const mySchema = new Schema({ field: Number });
const MyModel = mongoose.model('Test', mySchema);

// Mongoose will not strip out `notInSchema: 1` because `strictQuery` is false by default
const docs = await MyModel.find({ notInSchema: 1 });
// Empty array in Mongoose 7. In Mongoose 6, this would contain all documents in MyModel
docs;
```

Example 2 (javascript):
```javascript
const mySchema = new Schema({ field: Number });
const MyModel = mongoose.model('Test', mySchema);

// Change this:
await MyModel.remove(filter);

// To this:
await MyModel.deleteOne(filter);
// Or this, if you want to delete multiple:
await MyModel.deleteMany(filter);

// For documents, change this:
await doc.remove();

// To this:
await doc.deleteOne();
```

Example 3 (javascript):
```javascript
// Replace this:
schema.pre('remove', function() {
  /* ... */
});

// With this:
schema.pre('deleteOne', { document: true, query: false }, function() {
  /* ... */
});
```

Example 4 (javascript):
```javascript
// Before
conn.startSession(function(err, session) {
  // ...
});

// After
const session = await conn.startSession();
// Or:
conn.startSession().then(sesson => { /* ... */ });

// With error handling
try {
  await conn.startSession();
} catch (err) { /* ... */ }
// Or:
const [err, session] = await conn.startSession().then(
  session => ([null, session]),
  err => ([err, null])
);
```

---

## Documents

**URL:** https://mongoosejs.com/docs/7.x/docs/documents.html

**Contents:**
- Documents
- Documents vs Models
- Retrieving
- Updating Using save()
- Updating Using Queries
- Validating
- Overwriting
- Next Up

Mongoose documents represent a one-to-one mapping to documents as stored in MongoDB. Each document is an instance of its Model.

Document and Model are distinct classes in Mongoose. The Model class is a subclass of the Document class. When you use the Model constructor, you create a new document.

In Mongoose, a "document" generally means an instance of a model. You should not have to create an instance of the Document class without going through a model.

When you load documents from MongoDB using model functions like findOne(), you get a Mongoose document back.

Mongoose documents track changes. You can modify a document using vanilla JavaScript assignments and Mongoose will convert it into MongoDB update operators.

The save() method returns a promise. If save() succeeds, the promise resolves to the document that was saved.

If the document with the corresponding _id is not found, Mongoose will report a DocumentNotFoundError:

The save() function is generally the right way to update a document with Mongoose. With save(), you get full validation and middleware.

For cases when save() isn't flexible enough, Mongoose lets you create your own MongoDB updates with casting, middleware, and limited validation.

Note that update(), updateMany(), findOneAndUpdate(), etc. do not execute save() middleware. If you need save middleware and full validation, first query for the document and then save() it.

Documents are casted and validated before they are saved. Mongoose first casts values to the specified type and then validates them. Internally, Mongoose calls the document's validate() method before saving.

Mongoose also supports limited validation on updates using the runValidators option. Mongoose casts parameters to query functions like findOne(), updateOne() by default. However, Mongoose does not run validation on query function parameters by default. You need to set runValidators: true for Mongoose to validate.

Read the validation guide for more details.

There are 2 different ways to overwrite a document (replacing all keys in the document). One way is to use the Document#overwrite() function followed by save().

The other way is to use Model.replaceOne().

Now that we've covered Documents, let's take a look at Subdocuments.

**Examples:**

Example 1 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const doc = new MyModel();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 2 (javascript):
```javascript
const doc = await MyModel.findOne();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 3 (javascript):
```javascript
doc.name = 'foo';

// Mongoose sends an `updateOne({ _id: doc._id }, { $set: { name: 'foo' } })`
// to MongoDB.
await doc.save();
```

Example 4 (javascript):
```javascript
doc.save().then(savedDoc => {
  savedDoc === doc; // true
});
```

---

## Defaults

**URL:** https://mongoosejs.com/docs/defaults.html

**Contents:**
- Defaults
- Declaring Defaults in Your Schema
- Default Functions
- The setDefaultsOnInsert Option
- Default functions and this

Your schemas can define default values for certain paths. If you create a new document without that path set, the default will kick in.

Note: Mongoose only applies a default if the value of the path is strictly undefined.

You can also set the default schema option to a function. Mongoose will execute that function and use the return value as the default.

Mongoose also sets defaults on update() and findOneAndUpdate() when the upsert option is set by adding your schema's defaults to a MongoDB $setOnInsert operator. You can disable this behavior by setting the setDefaultsOnInsert option to false.

You can also set setDefaultsOnInsert to false globally:

Unless it is running on a query with setDefaultsOnInsert, a default function's this refers to the document.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({
  name: String,
  role: { type: String, default: 'guitarist' }
});

const Person = db.model('Person', schema);

const axl = new Person({ name: 'Axl Rose', role: 'singer' });
assert.equal(axl.role, 'singer');

const slash = new Person({ name: 'Slash' });
assert.equal(slash.role, 'guitarist');

const izzy = new Person({ name: 'Izzy', role: undefined });
assert.equal(izzy.role, 'guitarist');

// Defaults do **not** run on `null`, `''`, or value other than `undefined`.
const foo = new Person({ name: 'Bar', role: null });
assert.strictEqual(foo.role, null);

await Person.create(axl, slash);

const docs = await Person.find({ role: 'guitarist' });

assert.equal(docs.length, 1);
assert.equal(docs[0].name, 'Slash');
```

Example 2 (typescript):
```typescript
const schema = new Schema({
  title: String,
  date: {
    type: Date,
    // `Date.now()` returns the current unix timestamp as a number
    default: Date.now
  }
});

const BlogPost = db.model('BlogPost', schema);

const post = new BlogPost({ title: '5 Best Arnold Schwarzenegger Movies' });

// The post has a default Date set to now
assert.ok(post.date.getTime() >= Date.now() - 1000);
assert.ok(post.date.getTime() <= Date.now());
```

Example 3 (javascript):
```javascript
const schema = new Schema({
  title: String,
  genre: { type: String, default: 'Action' }
});

const Movie = db.model('Movie', schema);

const query = {};
const update = { title: 'The Terminator' };
const options = {
  // Return the document after updates are applied
  new: true,
  // Create a document if one isn't found.
  upsert: true
};

let doc = await Movie.findOneAndUpdate(query, update, options).lean();
doc.genre; // 'Action', Mongoose set a default value.

await Movie.deleteMany({});

doc = await Movie.findOneAndUpdate(query, update, { new: true, upsert: true, setDefaultsOnInsert: false }).lean();
doc.genre; // undefined, Mongoose did not set a default value
```

Example 4 (unknown):
```unknown
mongoose.set('setDefaultsOnInsert', false);
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/documents.html

**Contents:**
- Documents
- Documents vs Models
- Retrieving
- Updating Using save()
- Updating Using Queries
- Validating
- Overwriting
  - Next Up

Mongoose documents represent a one-to-one mapping to documents as stored in MongoDB. Each document is an instance of its Model.

Document and Model are distinct classes in Mongoose. The Model class is a subclass of the Document class. When you use the Model constructor, you create a new document.

In Mongoose, a "document" generally means an instance of a model. You should not have to create an instance of the Document class without going through a model.

When you load documents from MongoDB using model functions like findOne(), you get a Mongoose document back.

Mongoose documents track changes. You can modify a document using vanilla JavaScript assignments and Mongoose will convert it into MongoDB update operators.

The save() method returns a promise. If save() succeeds, the promise resolves to the document that was saved.

If the document with the corresponding _id is not found, Mongoose will report a DocumentNotFoundError:

The save() function is generally the right way to update a document with Mongoose. With save(), you get full validation and middleware.

For cases when save() isn't flexible enough, Mongoose lets you create your own MongoDB updates with casting, middleware, and limited validation.

Note that update(), updateMany(), findOneAndUpdate(), etc. do not execute save() middleware. If you need save middleware and full validation, first query for the document and then save() it.

Documents are casted and validated before they are saved. Mongoose first casts values to the specified type and then validates them. Internally, Mongoose calls the document's validate() method before saving.

Mongoose also supports limited validation on updates using the runValidators option. Mongoose casts parameters to query functions like findOne(), updateOne() by default. However, Mongoose does not run validation on query function parameters by default. You need to set runValidators: true for Mongoose to validate.

Read the validation guide for more details.

There are 2 different ways to overwrite a document (replacing all keys in the document). One way is to use the Document#overwrite() function followed by save().

The other way is to use Model.replaceOne().

Now that we've covered Documents, let's take a look at Subdocuments.

**Examples:**

Example 1 (javascript):
```javascript
const MyModel = mongoose.model('Test', new Schema({ name: String }));
const doc = new MyModel();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 2 (javascript):
```javascript
const doc = await MyModel.findOne();

doc instanceof MyModel; // true
doc instanceof mongoose.Model; // true
doc instanceof mongoose.Document; // true
```

Example 3 (javascript):
```javascript
doc.name = 'foo';

// Mongoose sends an `updateOne({ _id: doc._id }, { $set: { name: 'foo' } })`
// to MongoDB.
await doc.save();
```

Example 4 (javascript):
```javascript
doc.save().then(savedDoc => {
  savedDoc === doc; // true
});
```

---

## Migrating from 6.x to 7.x

**URL:** https://mongoosejs.com/docs/migrating_to_7.html

**Contents:**
- Migrating from 6.x to 7.x
- Version Requirements
- strictQuery
- Removed remove()
- Dropped callback support
- Removed update()
- ObjectId requires new
- id Setter
- Discriminator schemas use base schema options by default
- Removed castForQueryWrapper, updated castForQuery() signature

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 6.x to Mongoose 7.x.

If you're still on Mongoose 5.x, please read the Mongoose 5.x to 6.x migration guide and upgrade to Mongoose 6.x first.

Mongoose now requires Node.js >= 14.0.0 and MongoDB Node Driver >= 5.0.0.

See the MongoDB Node Driver migration guide for detailed info.

strictQuery is now false by default.

The remove() method on documents and models has been removed. Use deleteOne() or deleteMany() instead.

Keep in mind that deleteOne() hooks are treated as query middleware by default. So for middleware, please do the following:

The following functions no longer accept callbacks. They always return promises.

If you are using the above functions with callbacks, we recommend switching to async/await, or promises if async functions don't work for you. If you need help refactoring a legacy codebase, this tool from Mastering JS callbacks to async await using ChatGPT.

Model.update(), Query.prototype.update(), and Document.prototype.update() have been removed. Use updateOne() instead.

In Mongoose 6 and older, you could define a new ObjectId without using the new keyword:

In Mongoose 7, ObjectId is now a JavaScript class, so you need to use the new keyword.

Starting in Mongoose 7.4, Mongoose's built-in id virtual (which stores the document's _id as a string) has a setter which allows modifying the document's _id property via id.

This can cause surprising behavior if you create a new TestModel(obj) where obj contains both an id and an _id, or if you use doc.set()

The id setter was later removed in Mongoose 8 due to compatibility issues.

When you use Model.discriminator(), Mongoose will now use the discriminator base schema's options by default. This means you don't need to explicitly set child schema options to match the base schema's.

Mongoose now always calls SchemaType castForQuery() method with 3 arguments: $conditional, value, and context. If you've implemented a custom schema type that defines its own castForQuery() method, you need to update the method as follows.

Mongoose now copies user defined schema options when adding one schema to another. For example, childSchema below will get baseSchema's id and toJSON options.

This applies both when creating a new schema using an array of schemas, as well as when calling add() as follows.

The internal _bsontype property on ObjectIds is equal to 'ObjectId' in Mongoose 7, as opposed to 'ObjectID' in Mongoose 6.

Please update any places where you use _bsontype to check if an object is an ObjectId. This may also affect libraries that use Mongoose.

MongoDB no longer supports mapReduce, so Mongoose 7 no longer has a Model.mapReduce() function. Use the aggregation framework as a replacement for mapReduce().

Mongoose 7 no longer supports plugging in custom promise libraries. So the following no longer makes Mongoose return Bluebird promises in Mongoose 7.

If you want to use Bluebird for all promises globally, you can do the following:

Before Mongoose 5.2.0, you needed to enable the keepAlive option to initiate TCP keepalive to prevent "connection closed" errors. However, keepAlive has been true by default since Mongoose 5.2.0, and the keepAlive is deprecated as of Mongoose 7.2.0. Please remove keepAlive and keepAliveInitialDelay options from your Mongoose connections.

Mongoose 7 no longer exports a LeanDocument type, and no longer supports passing a document type that extends Document into Model<>.

Mongoose's HydratedDocument type transforms a raw document interface into the type of the hydrated Mongoose document, including virtuals, methods, etc. In Mongoose 7, the generic parameters to HydratedDocument have changed. In Mongoose 6, the generic parameters were:

In Mongoose 7, the new type is as follows.

In Mongoose 7, the first parameter is the raw document interface, the 2nd parameter is any document-specific overrides (usually virtuals and methods), and the 3rd parameter is any query helpers associated with the document's model.

The key difference is that, in Mongoose 6, the 3rd generic param was the document's virtuals. In Mongoose 7, the 3rd generic param is the document's query helpers.

**Examples:**

Example 1 (javascript):
```javascript
const mySchema = new Schema({ field: Number });
const MyModel = mongoose.model('Test', mySchema);

// Mongoose will not strip out `notInSchema: 1` because `strictQuery` is false by default
const docs = await MyModel.find({ notInSchema: 1 });
// Empty array in Mongoose 7. In Mongoose 6, this would contain all documents in MyModel
docs;
```

Example 2 (javascript):
```javascript
const mySchema = new Schema({ field: Number });
const MyModel = mongoose.model('Test', mySchema);

// Change this:
await MyModel.remove(filter);

// To this:
await MyModel.deleteOne(filter);
// Or this, if you want to delete multiple:
await MyModel.deleteMany(filter);

// For documents, change this:
await doc.remove();

// To this:
await doc.deleteOne();
```

Example 3 (css):
```css
// Replace this:
schema.pre('remove', function() {
  /* ... */
});

// With this:
schema.pre('deleteOne', { document: true, query: false }, function() {
  /* ... */
});
```

Example 4 (javascript):
```javascript
// Before
conn.startSession(function(err, session) {
  // ...
});

// After
const session = await conn.startSession();
// Or:
conn.startSession().then(session => { /* ... */ });

// With error handling
try {
  await conn.startSession();
} catch (err) { /* ... */ }
// Or:
const [err, session] = await conn.startSession().then(
  session => ([null, session]),
  err => ([err, null])
);
```

---

## Models

**URL:** https://mongoosejs.com/docs/7.x/docs/models.html

**Contents:**
- Models
- Compiling your first model
- Constructing Documents
- Querying
- Deleting
- Updating
- Change Streams
- Views
- Yet more
- Next Up

Models are fancy constructors compiled from Schema definitions. An instance of a model is called a document. Models are responsible for creating and reading documents from the underlying MongoDB database.

When you call mongoose.model() on a schema, Mongoose compiles a model for you.

The first argument is the singular name of the collection your model is for. Mongoose automatically looks for the plural, lowercased version of your model name. Thus, for the example above, the model Tank is for the tanks collection in the database.

Note: The .model() function makes a copy of schema. Make sure that you've added everything you want to schema, including hooks, before calling .model()!

An instance of a model is called a document. Creating them and saving to the database is easy.

Note that no tanks will be created/removed until the connection your model uses is open. Every model has an associated connection. When you use mongoose.model(), your model will use the default mongoose connection.

If you create a custom connection, use that connection's model() function instead.

Finding documents is easy with Mongoose, which supports the rich query syntax of MongoDB. Documents can be retrieved using a model's find, findById, findOne, or where static functions.

See the chapter on queries for more details on how to use the Query api.

Models have static deleteOne() and deleteMany() functions for removing all documents matching the given filter.

Each model has its own update method for modifying documents in the database without returning them to your application. See the API docs for more detail.

If you want to update a single document in the db and return it to your application, use findOneAndUpdate instead.

Change streams provide a way for you to listen to all inserts and updates going through your MongoDB database. Note that change streams do not work unless you're connected to a MongoDB replica set.

The output from the above async function will look like what you see below.

You can read more about change streams in mongoose in this blog post.

MongoDB Views are essentially read-only collections that contain data computed from other collections using aggregations. In Mongoose, you should define a separate Model for each of your Views. You can also create a View using createCollection().

The following example shows how you can create a new RedactedUser View on a User Model that hides potentially sensitive information, like name and email.

Note that Mongoose does not currently enforce that Views are read-only. If you attempt to save() a document from a View, you will get an error from the MongoDB server.

The API docs cover many additional methods available like count, mapReduce, aggregate, and more.

Now that we've covered Models, let's take a look at Documents.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new mongoose.Schema({ name: String, size: String });
const Tank = mongoose.model('Tank', schema);
```

Example 2 (javascript):
```javascript
const Tank = mongoose.model('Tank', yourSchema);

const small = new Tank({ size: 'small' });
await small.save();

// or

await Tank.create({ size: 'small' });

// or, for inserting large batches of documents
await Tank.insertMany([{ size: 'small' }]);
```

Example 3 (javascript):
```javascript
await mongoose.connect('mongodb://127.0.0.1/gettingstarted');
```

Example 4 (javascript):
```javascript
const connection = mongoose.createConnection('mongodb://127.0.0.1:27017/test');
const Tank = connection.model('Tank', yourSchema);
```

---

## Models

**URL:** https://mongoosejs.com/docs/6.x/docs/models.html

**Contents:**
- Models
- Compiling your first model
- Constructing Documents
- Querying
- Deleting
- Updating
- Change Streams
- Views
- Yet more
- Next Up

Models are fancy constructors compiled from Schema definitions. An instance of a model is called a document. Models are responsible for creating and reading documents from the underlying MongoDB database.

When you call mongoose.model() on a schema, Mongoose compiles a model for you.

The first argument is the singular name of the collection your model is for. Mongoose automatically looks for the plural, lowercased version of your model name. Thus, for the example above, the model Tank is for the tanks collection in the database.

Note: The .model() function makes a copy of schema. Make sure that you've added everything you want to schema, including hooks, before calling .model()!

An instance of a model is called a document. Creating them and saving to the database is easy.

Note that no tanks will be created/removed until the connection your model uses is open. Every model has an associated connection. When you use mongoose.model(), your model will use the default mongoose connection.

If you create a custom connection, use that connection's model() function instead.

Finding documents is easy with Mongoose, which supports the rich query syntax of MongoDB. Documents can be retrieved using a model's find, findById, findOne, or where static functions.

See the chapter on queries for more details on how to use the Query api.

Models have static deleteOne() and deleteMany() functions for removing all documents matching the given filter.

Each model has its own update method for modifying documents in the database without returning them to your application. See the API docs for more detail.

If you want to update a single document in the db and return it to your application, use findOneAndUpdate instead.

Change streams provide a way for you to listen to all inserts and updates going through your MongoDB database. Note that change streams do not work unless you're connected to a MongoDB replica set.

The output from the above async function will look like what you see below.

You can read more about change streams in mongoose in this blog post.

MongoDB Views are essentially read-only collections that contain data computed from other collections using aggregations. In Mongoose, you should define a separate Model for each of your Views. You can also create a View using createCollection().

The following example shows how you can create a new RedactedUser View on a User Model that hides potentially sensitive information, like name and email.

Note that Mongoose does not currently enforce that Views are read-only. If you attempt to save() a document from a View, you will get an error from the MongoDB server.

The API docs cover many additional methods available like count, mapReduce, aggregate, and more.

Now that we've covered Models, let's take a look at Documents.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new mongoose.Schema({ name: 'string', size: 'string' });
const Tank = mongoose.model('Tank', schema);
```

Example 2 (javascript):
```javascript
const Tank = mongoose.model('Tank', yourSchema);

const small = new Tank({ size: 'small' });
small.save(function(err) {
  if (err) return handleError(err);
  // saved!
});

// or

Tank.create({ size: 'small' }, function(err, small) {
  if (err) return handleError(err);
  // saved!
});

// or, for inserting large batches of documents
Tank.insertMany([{ size: 'small' }], function(err) {

});
```

Example 3 (javascript):
```javascript
mongoose.connect('mongodb://127.0.0.1/gettingstarted');
```

Example 4 (javascript):
```javascript
const connection = mongoose.createConnection('mongodb://127.0.0.1:27017/test');
const Tank = connection.model('Tank', yourSchema);
```

---

## Models

**URL:** https://mongoosejs.com/docs/models.html

**Contents:**
- Models
- Compiling your first model
- Constructing Documents
- Querying
- Deleting
- Updating
- Change Streams
- Views
- Yet more
- Next Up

Models are fancy constructors compiled from Schema definitions. An instance of a model is called a document. Models are responsible for creating and reading documents from the underlying MongoDB database.

When you call mongoose.model() on a schema, Mongoose compiles a model for you.

The first argument is the singular name of the collection your model is for. Mongoose automatically looks for the plural, lowercased version of your model name. Thus, for the example above, the model Tank is for the tanks collection in the database.

Note: The .model() function makes a copy of schema. Make sure that you've added everything you want to schema, including hooks, before calling .model()!

An instance of a model is called a document. Creating them and saving to the database is easy.

Note that no tanks will be created/removed until the connection your model uses is open. Every model has an associated connection. When you use mongoose.model(), your model will use the default mongoose connection.

If you create a custom connection, use that connection's model() function instead.

Finding documents is easy with Mongoose, which supports the rich query syntax of MongoDB. Documents can be retrieved using a model's find, findById, findOne, or where static functions.

See the chapter on queries for more details on how to use the Query api.

Models have static deleteOne() and deleteMany() functions for removing all documents matching the given filter.

Each model has its own update method for modifying documents in the database without returning them to your application. See the API docs for more detail.

If you want to update a single document in the db and return it to your application, use findOneAndUpdate instead.

Change streams provide a way for you to listen to all inserts and updates going through your MongoDB database. Note that change streams do not work unless you're connected to a MongoDB replica set.

The output from the above async function will look like what you see below.

You can read more about change streams in mongoose in this blog post.

MongoDB Views are essentially read-only collections that contain data computed from other collections using aggregations. In Mongoose, you should define a separate Model for each of your Views. You can also create a View using createCollection().

The following example shows how you can create a new RedactedUser View on a User Model that hides potentially sensitive information, like name and email.

Note that Mongoose does not currently enforce that Views are read-only. If you attempt to save() a document from a View, you will get an error from the MongoDB server.

The API docs cover many additional methods available like count, mapReduce, aggregate, and more.

Now that we've covered Models, let's take a look at Documents.

**Examples:**

Example 1 (css):
```css
const schema = new mongoose.Schema({ name: String, size: String });
const Tank = mongoose.model('Tank', schema);
```

Example 2 (javascript):
```javascript
const Tank = mongoose.model('Tank', yourSchema);

const small = new Tank({ size: 'small' });
await small.save();

// or

await Tank.create({ size: 'small' });

// or, for inserting large batches of documents
await Tank.insertMany([{ size: 'small' }]);
```

Example 3 (swift):
```swift
await mongoose.connect('mongodb://127.0.0.1/gettingstarted');
```

Example 4 (javascript):
```javascript
const connection = mongoose.createConnection('mongodb://127.0.0.1:27017/test');
const Tank = connection.model('Tank', yourSchema);
```

---
