# Mongoose - Other

**Pages:** 48

---

## Migrating from 7.x to 8.x

**URL:** https://mongoosejs.com/docs/migrating_to_8.html

**Contents:**
- Migrating from 7.x to 8.x
- Removed rawResult option for findOneAndUpdate()
- Document.prototype.deleteOne now returns a query
- MongoDB Node Driver 6
- Removed findOneAndRemove()
- Removed count()
- Removed id Setter
- null is valid for non-required string enums
- Apply minimize when save() updates an existing document
- Apply base schema paths before discriminator paths

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 7.x to Mongoose 8.x.

If you're still on Mongoose 6.x or earlier, please read the Mongoose 6.x to 7.x migration guide and upgrade to Mongoose 7.x first before upgrading to Mongoose 8.

We also recommend reviewing the MongoDB Node.js driver's release notes for v6.0.0 before upgrading to Mongoose 8.

The rawResult option for findOneAndUpdate(), findOneAndReplace(), and findOneAndDelete() has been replaced by the includeResultMetadata option.

includeResultMetadata in Mongoose 8 behaves identically to rawResult.

In Mongoose 7, doc.deleteOne() returned a promise that resolved to doc. In Mongoose 8, doc.deleteOne() returns a query for easier chaining, as well as consistency with doc.updateOne().

Mongoose 8 uses v6.x of the MongoDB Node driver. There's a few noteable changes in MongoDB Node driver v6 that affect Mongoose:

The ObjectId constructor no longer accepts strings of length 12. In Mongoose 7, new mongoose.Types.ObjectId('12charstring') was perfectly valid. In Mongoose 8, new mongoose.Types.ObjectId('12charstring') throws an error.

Deprecated SSL options have been removed

In Mongoose 7, findOneAndRemove() was an alias for findOneAndDelete() that Mongoose supported for backwards compatibility. Mongoose 8 no longer supports findOneAndRemove(). Use findOneAndDelete() instead.

Similarly, Mongoose 8 no longer supports findByIdAndRemove(), which was an alias for findByIdAndDelete(). Please use findByIdAndDelete() instead.

Model.count() and Query.prototype.count() were removed in Mongoose 8. Use Model.countDocuments() and Query.prototype.countDocuments() instead.

In Mongoose 7.4, Mongoose introduced an id setter that made doc.id = '0'.repeat(24) equivalent to doc._id = '0'.repeat(24). In Mongoose 8, that setter is now removed.

Before Mongoose 8, setting a string path with an enum to null would lead to a validation error, even if that path wasn't required. In Mongoose 8, it is valid to set a string path to null if required is not set, even with enum.

In Mongoose 7, Mongoose would only apply minimize when saving a new document, not when updating an existing document.

This means that, in Mongoose 8, getters and setters on discriminator paths run after getters and setters on base paths. In Mongoose 7, getters and setters on discriminator paths ran before getters and setters on base paths.

Mongoose 7 and earlier supported an overwrite option for findOneAndUpdate(), updateOne(), and update(). Before Mongoose 7, overwrite would skip wrapping the update parameter in $set, which meant that findOneAndUpdate() and update() would overwrite the matched document. In Mongoose 7, setting overwrite would convert findOneAndUpdate() to findOneAndReplace() and updateOne() to replaceOne() to retain backwards compatibility.

In Mongoose 8, the overwrite option is no longer supported. If you want to overwrite the entire document, use findOneAndReplace() or replaceOne().

In Mongoose 7, findOneAndUpdate(filter, update, { upsert: true }).orFail() would throw a DocumentNotFoundError if a new document was upserted. In other words, findOneAndUpdate().orFail() always threw an error if no document was found, even if a new document was upserted.

In Mongoose 8, findOneAndUpdate(filter, update, { upsert: true }).orFail() always succeeds. findOneAndUpdate().orFail() now throws a DocumentNotFoundError if there's no document returned, rather than if no document was found.

In Mongoose 7, create() would immediately throw if any save() threw an error by default. Mongoose 8 instead waits for all save() calls to finish before throwing the first error that occurred. So create() will throw the same error in both Mongoose 7 and Mongoose 8, Mongoose 8 just may take longer to throw the error.

In Mongoose 7, Model.validate() would potentially modify the passed in object. Mongoose 8 instead copies the passed in object first.

In Mongoose 8, automatically inferred schema types in TypeScript allow null for optional fields. In Mongoose 7, optional fields only allowed undefined, not null.

In Mongoose 8, no properties are required on model constructors by default.

**Examples:**

Example 1 (javascript):
```javascript
const filter = { name: 'Will Riker' };
const update = { age: 29 };

const res = await Character.findOneAndUpdate(filter, update, {
  new: true,
  upsert: true,
  // Replace `rawResult: true` with `includeResultMetadata: true`
  includeResultMetadata: true
});
```

Example 2 (javascript):
```javascript
const numberOne = await Character.findOne({ name: 'Will Riker' });

// In Mongoose 7, q is a Promise that resolves to `numberOne`
// In Mongoose 8, q is a Query.
const q = numberOne.deleteOne();

// In Mongoose 7, `res === numberOne`
// In Mongoose 8, `res` is a `DeleteResult`.
const res = await q;
```

Example 3 (javascript):
```javascript
const schema = new Schema({
  status: {
    type: String,
    enum: ['on', 'off']
  }
});
const Test = mongoose.model('Test', schema);

// Works fine in Mongoose 8
// Throws a `ValidationError` in Mongoose 7
await Test.create({ status: null });
```

Example 4 (javascript):
```javascript
const schema = new Schema({
  nested: {
    field1: Number
  }
});
const Test = mongoose.model('Test', schema);

// Both Mongoose 7 and Mongoose 8 strip out empty objects when saving
// a new document in MongoDB by default
const { _id } = await Test.create({ nested: {} });
let rawDoc = await Test.findById(_id).lean();
rawDoc.nested; // undefined

// Mongoose 8 will also strip out empty objects when saving an
// existing document in MongoDB
const doc = await Test.findById(_id);
doc.nested = {};
doc.markModified('nested');
await doc.save();

let rawDoc = await Test.findById(_id).lean();
rawDoc.nested; // undefined in Mongoose 8, {} in Mongoose 7
```

---

## Transactions in Mongoose

**URL:** https://mongoosejs.com/docs/7.x/docs/transactions.html

**Contents:**
- Transactions in Mongoose
- Getting Started with Transactions
- With Mongoose Documents and save()
- With the Aggregation Framework
- Using AsyncLocalStorage
- Advanced Usage

Transactions let you execute multiple operations in isolation and potentially undo all the operations if one of them fails. This guide will get you started using transactions with Mongoose.

If you haven't already, import mongoose:

To create a transaction, you first need to create a session using Mongoose#startSession or Connection#startSession().

In practice, you should use either the session.withTransaction() helper or Mongoose's Connection#transaction() function to run a transaction. The session.withTransaction() helper handles:

For more information on the ClientSession#withTransaction() function, please see the MongoDB Node.js driver docs.

Mongoose's Connection#transaction() function is a wrapper around withTransaction() that integrates Mongoose change tracking with transactions. For example, suppose you save() a document in a transaction that later fails. The changes in that document are not persisted to MongoDB. The Connection#transaction() function informs Mongoose change tracking that the save() was rolled back, and marks all fields that were changed in the transaction as modified.

If you get a Mongoose document from findOne() or find() using a session, the document will keep a reference to the session and use that session for save().

To get/set the session associated with a given document, use doc.$session().

The Model.aggregate() function also supports transactions. Mongoose aggregations have a session() helper that sets the session option. Below is an example of executing an aggregation within a transaction.

One major pain point with transactions in Mongoose is that you need to remember to set the session option on every operation. If you don't, your operation will execute outside of the transaction. Mongoose 7.8 is able to set the session operation on all operations within a Connection.prototype.transaction() executor function using Node's AsyncLocalStorage API. Set the transactionAsyncLocalStorage option using mongoose.set('transactionAsyncLocalStorage', true) to enable this feature.

With transactionAsyncLocalStorage, you no longer need to pass sessions to every operation. Mongoose will add the session by default under the hood.

Advanced users who want more fine-grained control over when they commit or abort transactions can use session.startTransaction() to start a transaction:

You can also use session.abortTransaction() to abort a transaction:

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
```

Example 2 (javascript):
```javascript
// Using Mongoose's default connection
const session = await mongoose.startSession();

// Using custom connection
const db = await mongoose.createConnection(mongodbUri).asPromise();
const session = await db.startSession();
```

Example 3 (javascript):
```javascript
let session = null;
return Customer.createCollection().
  then(() => Customer.startSession()).
  // The `withTransaction()` function's first parameter is a function
  // that returns a promise.
  then(_session => {
    session = _session;
    return session.withTransaction(() => {
      return Customer.create([{ name: 'Test' }], { session: session });
    });
  }).
  then(() => Customer.countDocuments()).
  then(count => assert.strictEqual(count, 1)).
  then(() => session.endSession());
```

Example 4 (javascript):
```javascript
const doc = new Person({ name: 'Will Riker' });

await db.transaction(async function setRank(session) {
  doc.name = 'Captain';
  await doc.save({ session });
  doc.isNew; // false

  // Throw an error to abort the transaction
  throw new Error('Oops!');
}, { readPreference: 'primary' }).catch(() => {});

// true, `transaction()` reset the document's state because the
// transaction was aborted.
doc.isNew;
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/3.8.x

---

## FAQ

**URL:** https://mongoosejs.com/docs/6.x/docs/faq.html

**Contents:**
- FAQ

Q. Operation ... timed out after 10000 ms. What gives?

A. At its core, this issue stems from not connecting to MongoDB. You can use Mongoose before connecting to MongoDB, but you must connect at some point. For example:

Q. I am able to connect locally but when I try to connect to MongoDB Atlas I get this error. What gives?

You must ensure that you have whitelisted your ip on mongodb to allow Mongoose to connect. You can allow access from all ips with 0.0.0.0/0.

Q. x.$__y is not a function. What gives?

A. This issue is a result of having multiple versions of mongoose installed that are incompatible with each other. Run npm list | grep "mongoose" to find and remedy the problem. If you're storing schemas or models in a separate npm package, please list Mongoose in peerDependencies rather than dependencies in your separate package.

Q. I declared a schema property as unique but I can still save duplicates. What gives?

A. Mongoose doesn't handle unique on its own: { name: { type: String, unique: true } } is just a shorthand for creating a MongoDB unique index on name. For example, if MongoDB doesn't already have a unique index on name, the below code will not error despite the fact that unique is true.

However, if you wait for the index to build using the Model.on('index') event, attempts to save duplicates will correctly error.

MongoDB persists indexes, so you only need to rebuild indexes if you're starting with a fresh database or you ran db.dropDatabase(). In a production environment, you should create your indexes using the MongoDB shell rather than relying on mongoose to do it for you. The unique option for schemas is convenient for development and documentation, but mongoose is not an index management solution.

Q. When I have a nested property in a schema, mongoose adds empty objects by default. Why?

A. This is a performance optimization. These empty objects are not saved to the database, nor are they in the result toObject(), nor do they show up in JSON.stringify() output unless you turn off the minimize option.

The reason for this behavior is that Mongoose's change detection and getters/setters are based on Object.defineProperty(). In order to support change detection on nested properties without incurring the overhead of running Object.defineProperty() every time a document is created, mongoose defines properties on the Model prototype when the model is compiled. Because mongoose needs to define getters and setters for nested.prop, nested must always be defined as an object on a mongoose document, even if nested is undefined on the underlying POJO.

Q. I'm using an arrow function for a virtual, middleware, getter/setter, or method and the value of this is wrong.

A. Arrow functions handle the this keyword much differently than conventional functions. Mongoose getters/setters depend on this to give you access to the document that you're writing to, but this functionality does not work with arrow functions. Do not use arrow functions for mongoose getters/setters unless do not intend to access the document in the getter/setter.

Q. I have an embedded property named type like this:

But mongoose gives me a CastError telling me that it can't cast an object to a string when I try to save a Holding with an asset object. Why is this?

A. The type property is special in mongoose, so when you say type: String, mongoose interprets it as a type declaration. In the above schema, mongoose thinks asset is a string, not an object. Do this instead:

Q. I'm populating a nested property under an array like the below code:

.populate({ path: 'arr.child', options: { sort: 'name' } }) won't sort by arr.child.name?

A. See this GitHub issue. It's a known issue but one that's exceptionally difficult to fix.

Q. All function calls on my models hang, what am I doing wrong?

A. By default, mongoose will buffer your function calls until it can connect to MongoDB. Read the buffering section of the connection docs for more information.

Q. How can I enable debugging?

A. Set the debug option:

For more debugging options (streams, callbacks), see the 'debug' option under .set().

Q. My save() callback never executes. What am I doing wrong?

A. All collection actions (insert, remove, queries, etc.) are queued until Mongoose successfully connects to MongoDB. It is likely you haven't called Mongoose's connect() or createConnection() function yet.

In Mongoose 5.11, there is a bufferTimeoutMS option (set to 10000 by default) that configures how long Mongoose will allow an operation to stay buffered before throwing an error.

If you want to opt out of Mongoose's buffering mechanism across your entire application, set the global bufferCommands option to false:

Instead of opting out of Mongoose's buffering mechanism, you may want to instead reduce bufferTimeoutMS to make Mongoose only buffer for a short time.

Q. Should I create/destroy a new connection for each database operation?

A. No. Open your connection when your application starts up and leave it open until the application shuts down.

Q. Why do I get "OverwriteModelError: Cannot overwrite .. model once compiled" when I use nodemon / a testing framework?

A. mongoose.model('ModelName', schema) requires 'ModelName' to be unique, so you can access the model by using mongoose.model('ModelName'). If you put mongoose.model('ModelName', schema); in a mocha beforeEach() hook, this code will attempt to create a new model named 'ModelName' before every test, and so you will get an error. Make sure you only create a new model with a given name once. If you need to create multiple models with the same name, create a new connection and bind the model to the connection.

Q. How can I change mongoose's default behavior of initializing an array path to an empty array so that I can require real data on document creation?

A. You can set the default of the array to a function that returns undefined.

Q. How can I initialize an array path to null?

A. You can set the default of the array to a function that returns null.

Q. Why does my aggregate $match fail to return the document that my find query returns when working with dates?

A. Mongoose does not cast aggregation pipeline stages because with $project, $group, etc. the type of a property may change during the aggregation. If you want to query by date using the aggregation framework, you're responsible for ensuring that you're passing in a valid date.

Q. Why don't in-place modifications to date objects (e.g. date.setMonth(1);) get saved?

A. Mongoose currently doesn't watch for in-place updates to date objects. If you have need for this feature, feel free to discuss on this GitHub issue. There are several workarounds:

Q. Why does calling save() multiple times on the same document in parallel only let the first save call succeed and return ParallelSaveErrors for the rest?

A. Due to the asynchronous nature of validation and middleware in general, calling save() multiple times in parallel on the same doc could result in conflicts. For example, validating, and then subsequently invalidating the same path.

Q. Why is any 12 character string successfully cast to an ObjectId?

A. Technically, any 12 character string is a valid ObjectId. Consider using a regex like /^[a-f0-9]{24}$/ to test whether a string is exactly 24 hex characters.

Q. Why do keys in Mongoose Maps have to be strings?

A. Because the Map eventually gets stored in MongoDB where the keys must be strings.

Q. I am using Model.find(...).populate(...) with the limit option, but getting fewer results than the limit. What gives?

A. In order to avoid executing a separate query for each document returned from the find query, Mongoose instead queries using (numDocuments * limit) as the limit. If you need the correct limit, you should use the perDocumentLimit option (new in Mongoose 5.9.0). Just keep in mind that populate() will execute a separate query for each document.

Q. My query/update seems to execute twice. Why is this happening?

A. The most common cause of duplicate queries is mixing callbacks and promises with queries. That's because passing a callback to a query function, like find() or updateOne(), immediately executes the query, and calling then() executes the query again.

Mixing promises and callbacks can lead to duplicate entries in arrays. For example, the below code inserts 2 entries into the tags array, *not just 1.

If you'd like to contribute to this page, please visit it on github and use the Edit button to send a pull request.

**Examples:**

Example 1 (javascript):
```javascript
await mongoose.createConnection(mongodbUri).asPromise();

const Test = mongoose.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because didn't call `mongoose.connect()`
```

Example 2 (javascript):
```javascript
await mongoose.connect(mongodbUri);

const db = mongoose.createConnection();

const Test = db.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because `db` isn't connected to MongoDB
```

Example 3 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

Model.create([{ name: 'Val' }, { name: 'Val' }], function(err) {
  console.log(err); // No error, unless index was already built
});
```

Example 4 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

Model.on('index', function(err) { // <-- Wait for model's indexes to finish
  assert.ifError(err);
  Model.create([{ name: 'Val' }, { name: 'Val' }], function(err) {
    console.log(err);
  });
});

// Promise based alternative. `init()` returns a promise that resolves
// when the indexes have finished building successfully. The `init()`
// function is idempotent, so don't worry about triggering an index rebuild.
Model.init().then(function() {
  Model.create([{ name: 'Val' }, { name: 'Val' }], function(err) {
    console.log(err);
  });
});
```

---

## MongoDB Server Version Compatibility

**URL:** https://mongoosejs.com/docs/compatibility.html

**Contents:**
- MongoDB Server Version Compatibility

Mongoose relies on the MongoDB Node.js Driver to communicate with MongoDB.

You can refer to this table for up-to-date information as to which version of the MongoDB driver supports which version of the MongoDB server.

Below are the SemVer ranges representing which versions of mongoose are compatible with the listed versions of MongoDB server.

Mongoose ^6.5.0 also works with MongoDB server 7.x. But not all new MongoDB server 7.x features are supported by Mongoose 6.x. To verify that your version of Mongoose is compatible based on the table above, use the online SemVer checker.

---

## 

**URL:** https://mongoosejs.com/docs/7.x/docs/jobs.html

**Contents:**
- Mongoose Jobs
- Add Your Own

Localize is looking for a Senior Full Stack Engineer to join our 100% remote team.

As a Localize engineer, you’ll be responsible for implementing new functionality within Localize’s core product. On the frontend you’ll work on our React/Redux/Backbone SPA, and on the backend you’ll build RESTful APIs in Node/Express/MongoDB.

As a key member of our remote engineering team, you’ll lead the development of many high impact product initiatives. We’re looking for a strong engineer who works well on a small team and is excited about the opportunity to have a direct impact on improving customer experience.

Technologies: Experience with these specific technologies a plus, but not strictly required

Experience and Qualifications:

You’ll be joining a close knit and talented team, with plenty of opportunity for professional growth. We offer compensation + benefits that are on par with large companies, while also placing a high value on maintaining a healthy work-life balance.

The Localize platform is used by 500+ companies to translate websites and applications into other languages. We help companies like Trello to translate their help center (https://help.trello.com/), Tinder to translate their blog (https://blog.gotinder.com/), and RocketMiles to translate their web app (https://www.rocketmiles.com/) - to see our product in action, click any of those links and use the website's language switcher to switch from English to another language.

Localize works by providing a code snippet (similar to the Google Analytics javascript snippet) that our customers add to their website / web app. The Localize snippet pulls in content from the page into the Localize dashboard where our customers login to add translations and manage their content. Localize automatically deploys those translations to the customer's production site.

We are a team of 12 people working fully remote. We offer competitive pay, a full benefits package, and a culture with emphasis on work + life balance.

---

## FAQ

**URL:** https://mongoosejs.com/docs/faq.html

**Contents:**
- FAQ

Q. I get an error connect ECONNREFUSED ::1:27017 when connecting to localhost. Why?

The easy solution is to replace localhost with 127.0.0.1.

The reason why this error happens is that Node.js 18 and up prefer IPv6 addresses over IPv4 by default. And, most Linux and OSX machines have a ::1 localhost entry in /etc/hosts by default. That means that Node.js 18 will assume that localhost means the IPv6 ::1 address. And MongoDB doesn't accept IPv6 connections by default.

You can also fix this error by enabling IPv6 support on your MongoDB server.

Q. Operation ... timed out after 10000 ms. What gives?

A. At its core, this issue stems from not connecting to MongoDB. You can use Mongoose before connecting to MongoDB, but you must connect at some point. For example:

Q. I am able to connect locally but when I try to connect to MongoDB Atlas I get this error. What gives?

You must ensure that you have whitelisted your ip on mongodb to allow Mongoose to connect. You can allow access from all ips with 0.0.0.0/0.

Q. x.$__y is not a function. What gives?

A. This issue is a result of having multiple versions of mongoose installed that are incompatible with each other. Run npm list | grep "mongoose" to find and remedy the problem. If you're storing schemas or models in a separate npm package, please list Mongoose in peerDependencies rather than dependencies in your separate package.

Q. I declared a schema property as unique but I can still save duplicates. What gives?

A. Mongoose doesn't handle unique on its own: { name: { type: String, unique: true } } is just a shorthand for creating a MongoDB unique index on name. For example, if MongoDB doesn't already have a unique index on name, the below code will not error despite the fact that unique is true.

However, if you wait for the index to build using the Model.on('index') event, attempts to save duplicates will correctly error.

MongoDB persists indexes, so you only need to rebuild indexes if you're starting with a fresh database or you ran db.dropDatabase(). In a production environment, you should create your indexes using the MongoDB shell rather than relying on mongoose to do it for you. The unique option for schemas is convenient for development and documentation, but mongoose is not an index management solution.

Q. When I have a nested property in a schema, mongoose adds empty objects by default. Why?

A. This is a performance optimization. These empty objects are not saved to the database, nor are they in the result toObject(), nor do they show up in JSON.stringify() output unless you turn off the minimize option.

The reason for this behavior is that Mongoose's change detection and getters/setters are based on Object.defineProperty(). In order to support change detection on nested properties without incurring the overhead of running Object.defineProperty() every time a document is created, mongoose defines properties on the Model prototype when the model is compiled. Because mongoose needs to define getters and setters for nested.prop, nested must always be defined as an object on a mongoose document, even if nested is undefined on the underlying POJO.

Q. I'm using an arrow function for a virtual, middleware, getter/setter, or method and the value of this is wrong.

A. Arrow functions handle the this keyword differently than conventional functions. Mongoose getters/setters depend on this to give you access to the document that you're writing to, but this functionality does not work with arrow functions. Do not use arrow functions for mongoose getters/setters unless do not intend to access the document in the getter/setter.

Q. I have an embedded property named type like this:

But mongoose gives me a CastError telling me that it can't cast an object to a string when I try to save a Holding with an asset object. Why is this?

A. The type property is special in mongoose, so when you say type: String, mongoose interprets it as a type declaration. In the above schema, mongoose thinks asset is a string, not an object. Do this instead:

Q. I'm populating a nested property under an array like the below code:

.populate({ path: 'arr.child', options: { sort: 'name' } }) won't sort by arr.child.name?

A. See this GitHub issue. It's a known issue but one that's exceptionally difficult to fix.

Q. All function calls on my models hang, what am I doing wrong?

A. By default, mongoose will buffer your function calls until it can connect to MongoDB. Read the buffering section of the connection docs for more information.

Q. How can I enable debugging?

A. Set the debug option:

For more debugging options (streams, callbacks), see the 'debug' option under .set().

Q. My save() operation never completes. What am I doing wrong?

A. All collection actions (insert, remove, queries, etc.) are queued until Mongoose successfully connects to MongoDB. It is likely you haven't called Mongoose's connect() or createConnection() function yet.

Mongoose connections support a bufferTimeoutMS option (set to 10000 by default) that configures how long Mongoose will allow an operation to stay buffered before throwing an error.

If you want to opt out of Mongoose's buffering mechanism across your entire application, set the global bufferCommands option to false:

Instead of opting out of Mongoose's buffering mechanism, you may want to instead reduce bufferTimeoutMS to make Mongoose only buffer for a short time.

Q. Should I create/destroy a new connection for each database operation?

A. No. Open your connection when your application starts up and leave it open until the application shuts down.

Q. Why do I get "OverwriteModelError: Cannot overwrite .. model once compiled" when I use nodemon / a testing framework?

A. mongoose.model('ModelName', schema) requires 'ModelName' to be unique, so you can access the model by using mongoose.model('ModelName'). If you put mongoose.model('ModelName', schema); in a mocha beforeEach() hook, this code will attempt to create a new model named 'ModelName' before every test, and so you will get an error. Make sure you only create a new model with a given name once. If you need to create multiple models with the same name, create a new connection and bind the model to the connection.

Q. How can I change mongoose's default behavior of initializing an array path to an empty array so that I can require real data on document creation?

A. You can set the default of the array to undefined.

Q. How can I initialize an array path to null?

A. You can set the default of the array to null.

Q. Why does my aggregate $match fail to return the document that my find query returns when working with dates?

A. Mongoose does not cast aggregation pipeline stages because with $project, $group, etc. the type of a property may change during the aggregation. If you want to query by date using the aggregation framework, you're responsible for ensuring that you're passing in a valid date.

Q. Why don't in-place modifications to date objects (e.g. date.setMonth(1);) get saved?

A. Mongoose currently doesn't watch for in-place updates to date objects. If you have need for this feature, feel free to discuss on this GitHub issue. There are several workarounds:

Q. Why does calling save() multiple times on the same document in parallel only let the first save call succeed and return ParallelSaveErrors for the rest?

A. Due to the asynchronous nature of validation and middleware in general, calling save() multiple times in parallel on the same doc could result in conflicts. For example, validating, and then subsequently invalidating the same path.

Q. Why is any 12 character string successfully cast to an ObjectId?

A. Technically, any 12 character string is a valid ObjectId. Consider using a regex like /^[a-f0-9]{24}$/ to test whether a string is exactly 24 hex characters.

Q. Why do keys in Mongoose Maps have to be strings?

A. Because the Map eventually gets stored in MongoDB where the keys must be strings.

Q. I am using Model.find(...).populate(...) with the limit option, but getting fewer results than the limit. What gives?

A. In order to avoid executing a separate query for each document returned from the find query, Mongoose instead queries using (numDocuments * limit) as the limit. If you need the correct limit, you should use the perDocumentLimit option (new in Mongoose 5.9.0). Just keep in mind that populate() will execute a separate query for each document.

Q. My query/update seems to execute twice. Why is this happening?

A. The most common cause of duplicate queries is executing the same query object twice. Calling then() or await on the same query object multiple times will execute the query multiple times.

Note: Mongoose v7+ no longer supports callbacks. If you're seeing duplicate queries in older code, it may be due to mixing callbacks and promises, which is no longer possible in current versions.

Q. What does DivergentArrayError mean and how do I fix it?

A. Mongoose throws DivergentArrayError when you call document.save() to update an array that was only partially loaded, for example:

Because only part of the array is in memory, Mongoose can't safely reconstruct the full array to send back to MongoDB without risking data loss, so it throws DivergentArrayError instead.

To fix this error, either:

(1) Load the full array before modifying and saving:

(2) Or use updateOne() / updateMany() with positional operators or arrayFilters so MongoDB can update the array atomically without requiring the full array on the document:

The same guidance applies if you populated an array with skip, limit, query conditions, or excluded _id: avoid calling save() to update that partially loaded array; instead, re-query without those options or use an update operation as shown above.

If you'd like to contribute to this page, please visit it on github and use the Edit button to send a pull request.

**Examples:**

Example 1 (javascript):
```javascript
await mongoose.createConnection(mongodbUri).asPromise();

const Test = mongoose.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because didn't call `mongoose.connect()`
```

Example 2 (javascript):
```javascript
await mongoose.connect(mongodbUri);

const db = mongoose.createConnection();

const Test = db.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because `db` isn't connected to MongoDB
```

Example 3 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

// No error, unless index was already built
await Model.create([{ name: 'Val' }, { name: 'Val' }]);
```

Example 4 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

// Wait for model's indexes to finish. The `init()`
// function is idempotent, so don't worry about triggering an index rebuild.
await Model.init();

// Throws a duplicate key error
await Model.create([{ name: 'Val' }, { name: 'Val' }]);
```

---

## Transactions in Mongoose

**URL:** https://mongoosejs.com/docs/5.x/docs/transactions.html

**Contents:**
- Transactions in Mongoose
- Getting Started with Transactions
- With Mongoose Documents and save()
- With the Aggregation Framework
- Advanced Usage

Transactions are new in MongoDB 4.0 and Mongoose 5.2.0. Transactions let you execute multiple operations in isolation and potentially undo all the operations if one of them fails. This guide will get you started using transactions with Mongoose.

If you haven't already, import mongoose:

To create a transaction, you first need to create a session using or Mongoose#startSession or Connection#startSession().

In practice, you should use either the session.withTransaction() helper or Mongoose's Connection#transaction() function to run a transaction. The session.withTransaction() helper handles:

For more information on the ClientSession#withTransaction() function, please see the MongoDB Node.js driver docs.

Mongoose's Connection#transaction() function is a wrapper around withTransaction() that integrates Mongoose change tracking with transactions. For example, the Connection#transaction() function handles resetting a document if you save() that document in a transaction that later fails.

If you get a Mongoose document from findOne() or find() using a session, the document will keep a reference to the session and use that session for save().

To get/set the session associated with a given document, use doc.$session().

The Model.aggregate() function also supports transactions. Mongoose aggregations have a session() helper that sets the session option. Below is an example of executing an aggregation within a transaction.

Advanced users who want more fine-grained control over when they commit or abort transactions can use session.startTransaction() to start a transaction:

You can also use session.abortTransaction() to abort a transaction:

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
```

Example 2 (javascript):
```javascript
// Using Mongoose's default connection
const session = await mongoose.startSession();

// Using custom connection
const db = await mongoose.createConnection(mongodbUri, { useUnifiedTopology: true, useNewUrlParser: true });
const session = await db.startSession();
```

Example 3 (javascript):
```javascript
const session = await Customer.startSession();

// The `withTransaction()` function's first parameter is a function
// that returns a promise.
await session.withTransaction(() => {
  return Customer.create([{ name: 'Test' }], { session: session })
});

const count = await Customer.countDocuments();
assert.strictEqual(count, 1);

session.endSession();
```

Example 4 (javascript):
```javascript
const schema = Schema({ name: String, arr: [String], arr2: [String] });

const Test = db.model('Test', schema);

await Test.createCollection();
let doc = await Test.create({ name: 'foo', arr: ['bar'], arr2: ['foo'] });
doc = await Test.findById(doc);
await db.
  transaction(async (session) => {
    doc.arr.pull('bar');
    doc.arr2.push('bar');

    await doc.save({ session });
    doc.name = 'baz';
    throw new Error('Oops');
  }).
  catch(err => {
    assert.equal(err.message, 'Oops');
  });

const changes = doc.getChanges();
assert.equal(changes.$set.name, 'baz');
assert.deepEqual(changes.$pullAll.arr, ['bar']);
assert.deepEqual(changes.$push.arr2, { $each: ['bar'] });
assert.ok(!changes.$set.arr2);

await doc.save({ session: null });

const newDoc = await Test.findById(doc);
assert.equal(newDoc.name, 'baz');
assert.deepEqual(newDoc.arr, []);
assert.deepEqual(newDoc.arr2, ['foo', 'bar']);
```

---

## Transactions in Mongoose

**URL:** https://mongoosejs.com/docs/transactions.html

**Contents:**
- Transactions in Mongoose
- Getting Started with Transactions
- Note About Parallelism in Transactions
- With Mongoose Documents and save()
- With the Aggregation Framework
- Using AsyncLocalStorage
- Advanced Usage

Transactions let you execute multiple operations in isolation and potentially undo all the operations if one of them fails. This guide will get you started using transactions with Mongoose.

If you haven't already, import mongoose:

To create a transaction, you first need to create a session using Mongoose#startSession or Connection#startSession().

In practice, you should use either the session.withTransaction() helper or Mongoose's Connection#transaction() function to run a transaction. The session.withTransaction() helper handles:

For more information on the ClientSession#withTransaction() function, please see the MongoDB Node.js driver docs.

Mongoose's Connection#transaction() function is a wrapper around withTransaction() that integrates Mongoose change tracking with transactions. For example, suppose you save() a document in a transaction that later fails. The changes in that document are not persisted to MongoDB. The Connection#transaction() function informs Mongoose change tracking that the save() was rolled back, and marks all fields that were changed in the transaction as modified.

Running operations in parallel is not supported during a transaction. The use of Promise.all, Promise.allSettled, Promise.race, etc. to parallelize operations inside a transaction is undefined behaviour and should be avoided.

MongoDB also does not support multiple transactions on the same session in parallel. This also means MongoDB does not support nested transactions on the same session. The following code will throw a Transaction already in progress error.

If you get a Mongoose document from findOne() or find() using a session, the document will keep a reference to the session and use that session for save().

To get/set the session associated with a given document, use doc.$session().

The Model.aggregate() function also supports transactions. Mongoose aggregations have a session() helper that sets the session option. Below is an example of executing an aggregation within a transaction.

One major pain point with transactions in Mongoose is that you need to remember to set the session option on every operation. If you don't, your operation will execute outside of the transaction. Mongoose 8.4 is able to set the session operation on all operations within a Connection.prototype.transaction() executor function using Node's AsyncLocalStorage API. Set the transactionAsyncLocalStorage option using mongoose.set('transactionAsyncLocalStorage', true) to enable this feature.

With transactionAsyncLocalStorage, you no longer need to pass sessions to every operation. Mongoose will add the session by default under the hood.

transactionAsyncLocalStorage creates a new session each time you call connection.transaction(). This means each transaction will have its own session and be independent of other transactions. This also means that nested transactions are also independent of each other.

However, if the nested transaction fails, the top-level transaction will still be rolled back because await mongoose.connection.transaction() throws.

Advanced users who want more fine-grained control over when they commit or abort transactions can use session.startTransaction() to start a transaction:

You can also use session.abortTransaction() to abort a transaction:

**Examples:**

Example 1 (sql):
```sql
import mongoose from 'mongoose';
```

Example 2 (javascript):
```javascript
// Using Mongoose's default connection
const session = await mongoose.startSession();

// Using custom connection
const db = await mongoose.createConnection(mongodbUri).asPromise();
const session = await db.startSession();
```

Example 3 (javascript):
```javascript
let session = null;
return Customer.createCollection().
  then(() => Customer.startSession()).
  // The `withTransaction()` function's first parameter is a function
  // that returns a promise.
  then(_session => {
    session = _session;
    return session.withTransaction(() => {
      return Customer.create([{ name: 'Test' }], { session: session });
    });
  }).
  then(() => Customer.countDocuments()).
  then(count => assert.strictEqual(count, 1)).
  then(() => session.endSession());
```

Example 4 (javascript):
```javascript
const doc = new Person({ name: 'Will Riker' });

await db.transaction(async function setRank(session) {
  doc.name = 'Captain';
  await doc.save({ session });
  doc.isNew; // false

  // Throw an error to abort the transaction
  throw new Error('Oops!');
}, { readPreference: 'primary' }).catch(() => {});

// true, `transaction()` reset the document's state because the
// transaction was aborted.
doc.isNew;
```

---

## Version Support

**URL:** https://mongoosejs.com/docs/7.x/docs/version-support.html

**Contents:**
- Version Support
- Mongoose 6
- Mongoose 5

Mongoose 7.x (released February 27, 2023) is the current Mongoose major version. We ship all new bug fixes and features to 7.x.

Mongoose 6.x (released August 24, 2021) is currently only receiving security fixes and requested bug fixes as of August 24, 2023. Please open a bug report on GitHub to request backporting a fix to Mongoose 6.

We do not currently have a formal end of life (EOL) date for Mongoose 6. However, we will not end support for Mongoose 6 until at least April 1, 2024.

Mongoose 5.x (released January 17, 2018) is currently only receiving security fixes and requested bug fixes. Please open a bug report on GitHub to request backporting a fix to Mongoose 5. We will not backport any new features from Mongoose 6 or Mongoose 7 into Mongoose 5.

Mongoose 5.x end of life (EOL) is March 1, 2024. Mongoose 5.x will no longer receive any updates, security or otherwise, after that date.

---

## Using Mongoose With AWS Lambda

**URL:** https://mongoosejs.com/docs/lambda.html

**Contents:**
- Using Mongoose With AWS Lambda
- Connection Helper
- Using mongoose.connect()

AWS Lambda is a popular service for running arbitrary functions without managing individual servers. Using Mongoose in your AWS Lambda functions is easy. Here's a sample function that connects to a MongoDB instance and finds a single document:

The above code works fine for a single Lambda function, but what if you want to reuse the same connection logic in multiple Lambda functions? You can export the below function.

You can also use mongoose.connect(), so you can use mongoose.model() to create models.

**Examples:**

Example 1 (javascript):
```javascript
const mongoose = require('mongoose');

let conn = null;

const uri = 'YOUR CONNECTION STRING HERE';

exports.handler = async function(event, context) {
  // Make sure to add this so you can re-use `conn` between function calls.
  // See https://www.mongodb.com/blog/post/serverless-development-with-nodejs-aws-lambda-mongodb-atlas
  context.callbackWaitsForEmptyEventLoop = false;

  // Because `conn` is in the global scope, Lambda may retain it between
  // function calls thanks to `callbackWaitsForEmptyEventLoop`.
  // This means your Lambda function doesn't have to go through the
  // potentially expensive process of connecting to MongoDB every time.
  if (conn == null) {
    conn = mongoose.createConnection(uri, {
      // and tell the MongoDB driver to not wait more than 5 seconds
      // before erroring out if it isn't connected
      serverSelectionTimeoutMS: 5000
    });

    // `await`ing connection after assigning to the `conn` variable
    // to avoid multiple function calls creating new connections
    await conn.asPromise();
    conn.model('Test', new mongoose.Schema({ name: String }));
  }

  const M = conn.model('Test');

  const doc = await M.findOne();
  console.log(doc);

  return doc;
};
```

Example 2 (javascript):
```javascript
const mongoose = require('mongoose');

let conn = null;

const uri = 'YOUR CONNECTION STRING HERE';

exports.connect = async function() {
  if (conn == null) {
    conn = mongoose.createConnection(uri, {
      serverSelectionTimeoutMS: 5000
    });

    // `await`ing connection after assigning to the `conn` variable
    // to avoid multiple function calls creating new connections
    await conn.asPromise();
  }

  return conn;
};
```

Example 3 (javascript):
```javascript
const mongoose = require('mongoose');

let conn = null;

const uri = 'YOUR CONNECTION STRING HERE';

exports.connect = async function() {
  if (conn == null) {
    conn = mongoose.connect(uri, {
      serverSelectionTimeoutMS: 5000
    }).then(() => mongoose);

    // `await`ing connection after assigning to the `conn` variable
    // to avoid multiple function calls creating new connections
    await conn;
  }

  return conn;
};
```

---

## Version Support

**URL:** https://mongoosejs.com/docs/6.x/docs/version-support.html

**Contents:**
- Version Support
- Mongoose 6
- Mongoose 5

Mongoose 7.x (released February 27, 2023) is the current Mongoose major version. We ship all new bug fixes and features to 7.x.

Mongoose 6.x (released August 24, 2021) is currently in legacy support. We will continue to ship bug fixes to Mongoose 6 until August 24, 2023. After August 24, 2023, we will only ship security fixes, and backport requested fixes to Mongoose 6. Please open a bug report on GitHub to request backporting a fix to Mongoose 6.

We are not actively backporting any new features from Mongoose 7 into Mongoose 6. Until August 24, 2023, we will backport requested features into Mongoose 6; please open a feature request on GitHub to request backporting a feature into Mongoose 6. After August 24, 2023, we will not backport any new features into Mongoose 6.

We do not currently have a formal end of life (EOL) date for Mongoose 6. However, we will not end support for Mongoose 6 until at least January 1, 2024.

Mongoose 5.x (released January 17, 2018) is currently only receiving security fixes and requested bug fixes. Please open a bug report on GitHub to request backporting a fix to Mongoose 5. We will not backport any new features from Mongoose 6 or Mongoose 7 into Mongoose 5.

Mongoose 5.x end of life (EOL) is March 1, 2024. Mongoose 5.x will no longer receive any updates, security or otherwise, after that date.

---

## Mongoose in the Browser

**URL:** https://mongoosejs.com/docs/browser.html

**Contents:**
- Mongoose in the Browser

As of Mongoose 9, Mongoose's browser build is now in the @mongoosejs/browser npm package. The documentation has been moved to the @mongoosejs/browser README.

---

## MongoDB Server Version Compatibility

**URL:** https://mongoosejs.com/docs/5.x/docs/compatibility.html

**Contents:**
- MongoDB Server Version Compatibility
- Further Reading

Mongoose relies on the MongoDB Node.js Driver to talk to MongoDB. You can refer to this table for up-to-date information as to which version of the MongoDB driver supports which version of MongoDB.

Below are the semver ranges representing which versions of mongoose are compatible with the listed versions of MongoDB server.

Note that Mongoose 5.x dropped support for all versions of MongoDB before 3.0.0. If you need to use MongoDB 2.6 or older, use Mongoose 4.x.

Are you on Mongoose 3.x and looking to migrate to 4.x? Check out Moving Forward with Mongoose.js on Pluralsight. This video course walks you through the new features and backwards breaking changes in Mongoose 4, so you can upgrade with confidence.

---

## Timestamps

**URL:** https://mongoosejs.com/docs/7.x/docs/timestamps.html

**Contents:**
- Timestamps
- Alternate Property Names
- Disabling Timestamps
- Timestamps on Subdocuments
- Under the Hood

Mongoose schemas support a timestamps option. If you set timestamps: true, Mongoose will add two properties of type Date to your schema:

Mongoose will then set createdAt when the document is first inserted, and update updatedAt whenever you update the document using save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), or bulkWrite().

The createdAt property is immutable, and Mongoose overwrites any user-specified updates to updatedAt by default.

For the purposes of these docs, we'll always refer to createdAt and updatedAt. But you can overwrite these property names as shown below.

save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), and bulkWrite() all support a timestamps option. Set timestamps: false to skip setting timestamps for that particular operation.

You can also set the timestamps option to an object to configure createdAt and updatedAt separately. For example, in the below code, Mongoose sets createdAt on save() but skips updatedAt.

Disabling timestamps also lets you set timestamps yourself. For example, suppose you need to correct a document's createdAt or updatedAt property. You can do that by setting timestamps: false and setting createdAt yourself as shown below.

Mongoose also supports setting timestamps on subdocuments. Keep in mind that createdAt and updatedAt for subdocuments represent when the subdocument was created or updated, not the top level document. Overwriting a subdocument will also overwrite createdAt.

For queries with timestamps, Mongoose adds 2 properties to each update query:

For example, if you run the below code:

You'll see the below output from Mongoose debug mode:

Notice the $setOnInsert for createdAt and $set for updatedAt. MongoDB's $setOnInsert operator applies the update only if a new document is upserted. So, for example, if you want to only set updatedAt if a new document is created, you can disable the updatedAt timestamp and set it yourself as shown below:

**Examples:**

Example 1 (javascript):
```javascript
const userSchema = new Schema({ name: String }, { timestamps: true });
const User = mongoose.model('User', userSchema);

let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.244Z

doc.name = 'test2';
await doc.save();
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.307Z

doc = await User.findOneAndUpdate({ _id: doc._id }, { name: 'test3' }, { new: true });
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.366Z
```

Example 2 (javascript):
```javascript
let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.930Z

doc.name = 'test2';
doc.createdAt = new Date(0);
doc.updatedAt = new Date(0);
await doc.save();

// Mongoose blocked changing `createdAt` and set its own `updatedAt`, ignoring
// the attempt to manually set them.
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.991Z

// Mongoose also blocks changing `createdAt` and sets its own `updatedAt`
// on `findOneAndUpdate()`, `updateMany()`, and other query operations
doc = await User.findOneAndUpdate(
  { _id: doc._id },
  { name: 'test3', createdAt: new Date(0), updatedAt: new Date(0) },
  { new: true }
);
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:14.008Z
```

Example 3 (javascript):
```javascript
const userSchema = new Schema({ name: String }, {
  timestamps: {
    createdAt: 'created_at', // Use `created_at` to store the created date
    updatedAt: 'updated_at' // and `updated_at` to store the last updated date
  }
});
```

Example 4 (javascript):
```javascript
let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T23:28:54.264Z
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

doc.name = 'test2';

// Setting `timestamps: false` tells Mongoose to skip updating `updatedAt` on this `save()`
await doc.save({ timestamps: false });
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

// Similarly, setting `timestamps: false` on a query tells Mongoose to skip updating
// `updatedAt`.
doc = await User.findOneAndUpdate({ _id: doc._id }, { name: 'test3' }, {
  new: true,
  timestamps: false
});
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

// Below is how you can disable timestamps on a `bulkWrite()`
await User.bulkWrite([{
  updateOne: {
    filter: { _id: doc._id },
    update: { name: 'test4' },
    timestamps: false
  }
}]);
doc = await User.findOne({ _id: doc._id });
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z
```

---

## 

**URL:** https://mongoosejs.com/docs/jobs.html

**Contents:**
- Mongoose Jobs
- Add Your Own

Localize is looking for a Senior Full Stack Engineer to join our 100% remote team.

As a Localize engineer, you’ll be responsible for implementing new functionality within Localize’s core product. On the frontend you’ll work on our React/Redux/Backbone SPA, and on the backend you’ll build RESTful APIs in Node/Express/MongoDB.

As a key member of our remote engineering team, you’ll lead the development of many high impact product initiatives. We’re looking for a strong engineer who works well on a small team and is excited about the opportunity to have a direct impact on improving customer experience.

Technologies: Experience with these specific technologies a plus, but not strictly required

Experience and Qualifications:

You’ll be joining a close knit and talented team, with plenty of opportunity for professional growth. We offer compensation + benefits that are on par with large companies, while also placing a high value on maintaining a healthy work-life balance.

The Localize platform is used by 500+ companies to translate websites and applications into other languages. We help companies like Trello to translate their help center (https://help.trello.com/), Tinder to translate their blog (https://blog.gotinder.com/), and RocketMiles to translate their web app (https://www.rocketmiles.com/) - to see our product in action, click any of those links and use the website's language switcher to switch from English to another language.

Localize works by providing a code snippet (similar to the Google Analytics javascript snippet) that our customers add to their website / web app. The Localize snippet pulls in content from the page into the Localize dashboard where our customers login to add translations and manage their content. Localize automatically deploys those translations to the customer's production site.

We are a team of 12 people working fully remote. We offer competitive pay, a full benefits package, and a culture with emphasis on work + life balance.

---

## Promises

**URL:** https://mongoosejs.com/docs/promises.html

**Contents:**
- Promises
- Built-in Promises
- Queries are not promises
- Queries are thenable
- Should You Use exec() With await?

Mongoose async operations, like .save() and queries, return thenables. This means that you can do things like MyModel.findOne({}).then() and await MyModel.findOne({}).exec() if you're using async/await.

You can find the return type of specific operations in the api docs You can also read more about promises in Mongoose.

Mongoose queries are not promises. They have a .then() function for co and async/await as a convenience. If you need a fully-fledged promise, use the .exec() function.

Although queries are not promises, queries are thenables. That means they have a .then() function, so you can use queries as promises with either promise chaining or async await

There are two alternatives for using await with queries:

As far as functionality is concerned, these two are equivalent. However, we recommend using .exec() because that gives you better stack traces.

**Examples:**

Example 1 (javascript):
```javascript
const gnr = new Band({
  name: 'Guns N\' Roses',
  members: ['Axl', 'Slash']
});

const promise = gnr.save();
assert.ok(promise instanceof Promise);

promise.then(function(doc) {
  assert.equal(doc.name, 'Guns N\' Roses');
});
```

Example 2 (javascript):
```javascript
const query = Band.findOne({ name: 'Guns N\' Roses' });
assert.ok(!(query instanceof Promise));

// A query is not a fully-fledged promise, but it does have a `.then()`.
query.then(function(doc) {
  // use doc
});

// `.exec()` gives you a fully-fledged promise
const promise = Band.findOne({ name: 'Guns N\' Roses' }).exec();
assert.ok(promise instanceof Promise);

promise.then(function(doc) {
  // use doc
});
```

Example 3 (css):
```css
Band.findOne({ name: 'Guns N\' Roses' }).then(function(doc) {
  // use doc
});
```

Example 4 (javascript):
```javascript
const doc = await Band.findOne({ name: 'Guns N\' Roses' }); // works

const badId = 'this is not a valid id';
try {
  await Band.findOne({ _id: badId });
} catch (err) {
  // Without `exec()`, the stack trace does **not** include the
  // calling code. Below is the stack trace:
  //
  // CastError: Cast to ObjectId failed for value "this is not a valid id" at path "_id" for model "band-promises"
  //   at new CastError (/app/node_modules/mongoose/lib/error/cast.js:29:11)
  //   at model.Query.exec (/app/node_modules/mongoose/lib/query.js:4331:21)
  //   at model.Query.Query.then (/app/node_modules/mongoose/lib/query.js:4423:15)
  //   at process._tickCallback (internal/process/next_tick.js:68:7)
  err.stack;
}

try {
  await Band.findOne({ _id: badId }).exec();
} catch (err) {
  // With `exec()`, the stack trace includes where in your code you
  // called `exec()`. Below is the stack trace:
  //
  // CastError: Cast to ObjectId failed for value "this is not a valid id" at path "_id" for model "band-promises"
  //   at new CastError (/app/node_modules/mongoose/lib/error/cast.js:29:11)
  //   at model.Query.exec (/app/node_modules/mongoose/lib/query.js:4331:21)
  //   at Context.<anonymous> (/app/test/index.test.js:138:42)
  //   at process._tickCallback (internal/process/next_tick.js:68:7)
  err.stack;
}
```

---

## Getting Started

**URL:** https://mongoosejs.com/docs

**Contents:**
- Getting Started
- Congratulations

First be sure you have MongoDB and Node.js installed.

Next install Mongoose from the command line using npm:

Now say we like fuzzy kittens and want to record every kitten we ever meet in MongoDB. The first thing we need to do is include mongoose in our project and open a connection to the test database on our locally running instance of MongoDB.

For brevity, let's assume that all following code is within the main() function.

With Mongoose, everything is derived from a Schema. Let's get a reference to it and define our kittens.

So far so good. We've got a schema with one property, name, which will be a String. The next step is compiling our schema into a Model.

A model is a class with which we construct documents. In this case, each document will be a kitten with properties and behaviors as declared in our schema. Let's create a kitten document representing the little guy we just met on the sidewalk outside:

Kittens can meow, so let's take a look at how to add "speak" functionality to our documents:

Functions added to the methods property of a schema get compiled into the Model prototype and exposed on each document instance:

We have talking kittens! But we still haven't saved anything to MongoDB. Each document can be saved to the database by calling its save method.

Say time goes by and we want to display all the kittens we've seen. We can access all of the kitten documents through our Kitten model.

We just logged all of the kittens in our db to the console. If we want to filter our kittens by name, Mongoose supports MongoDBs rich querying syntax.

This performs a search for all documents with a name property that begins with "fluff" and returns the result as an array of kittens.

That's the end of our quick start. We created a schema, added a custom document method, saved and queried kittens in MongoDB using Mongoose. Head over to the guide, or API docs for more.

**Examples:**

Example 1 (unknown):
```unknown
npm install mongoose
```

Example 2 (javascript):
```javascript
// getting-started.js
const mongoose = require('mongoose');

main().catch(err => console.log(err));

async function main() {
  await mongoose.connect('mongodb://127.0.0.1:27017/test');

  // use `await mongoose.connect('mongodb://user:password@127.0.0.1:27017/test');` if your database has auth enabled
}
```

Example 3 (css):
```css
const kittySchema = new mongoose.Schema({
  name: String
});
```

Example 4 (javascript):
```javascript
const Kitten = mongoose.model('Kitten', kittySchema);
```

---

## Incompatible packages

**URL:** https://mongoosejs.com/docs/incompatible_packages

**Contents:**
- Incompatible packages

The following npm packages are known to be incompatible with Mongoose:

---

## Integrating with MongoDB Client Side Field Level Encryption

**URL:** https://mongoosejs.com/docs/field-level-encryption.html

**Contents:**
- Integrating with MongoDB Client Side Field Level Encryption
- Automatic FLE in Mongoose
  - Encryption types
  - Declaring Encrypted Schemas
  - Registering Models
  - Connecting and configuring encryption options
  - Discriminators
- Managing Data Keys
- Manual FLE in Mongoose

Client Side Field Level Encryption, or CSFLE for short, is a tool for storing your data in an encrypted format in MongoDB. For example, instead of storing the name property as a plain-text string, CSFLE means MongoDB will store your document with name as an encrypted buffer. The resulting document will look similar to the following to a client that doesn't have access to decrypt the data.

You can read more about CSFLE on the MongoDB CSFLE documentation and this blog post about CSFLE in Node.js.

Mongoose supports the declaration of encrypted schemas - schemas that, when connected to a model, utilize MongoDB's Client Side Field Level Encryption or Queryable Encryption under the hood. Mongoose automatically generates either an encryptedFieldsMap or a schemaMap when instantiating a MongoClient and encrypts fields on write and decrypts fields on reads.

MongoDB has two different automatic encryption implementations: client side field level encryption (CSFLE) and queryable encryption (QE).See choosing an in-use encryption approach.

The following schema declares two properties, name and ssn. ssn is encrypted using queryable encryption, and is configured for equality queries:

To declare a field as encrypted, you must:

Not all schematypes are supported for CSFLE and QE. For an overview of supported BSON types, refer to MongoDB's documentation.

Encrypted schemas can be registered on the global mongoose object or on a specific connection, so long as models are registered before the connection is established:

Field level encryption in Mongoose works by generating the encryption schema that the MongoDB driver expects for each encrypted model on the connection. This happens automatically when the model's connection is established.

Queryable encryption and CSFLE require all the same configuration as outlined in the MongoDB encryption in-use documentation, except for the schemaMap or encryptedFieldsMap options.

Once the connection is established, Mongoose's operations will work as usual. Writes are encrypted automatically by the MongoDB driver prior to sending them to the server and reads are decrypted by the driver after fetching documents from the server.

Discriminators are supported for encrypted models as well:

When generating encryption schemas, Mongoose merges all discriminators together for all of the discriminators declared on the same namespace. As a result, discriminators that declare the same key with different types are not supported. Furthermore, all discriminators for the same namespace must share the same encryption type - it is not possible to configure discriminators on the same model for both CSFLE and Queryable Encryption.

Mongoose provides a convenient API to obtain a ClientEncryption object configured to manage data keys in the key vault. A client encryption can be obtained with the Model.clientEncryption() helper:

First, you need to install the mongodb-client-encryption npm package. This is MongoDB's official package for setting up encryption keys.

You also need to make sure you've installed mongocryptd. mongocryptd is a separate process from the MongoDB server that you need to run to work with field level encryption. You can either run mongocryptd yourself, or make sure it is on the system PATH and the MongoDB Node.js driver will run it for you. You can read more about mongocryptd here.

Once you've set up and run mongocryptd, first you need to create a new encryption key as follows. Keep in mind that the following example is a simple example to help you get started. The encryption key in the following example is insecure; MongoDB recommends using a KMS.

Once you have an encryption key, you can create a separate Mongoose connection with a schemaMap that defines which fields are encrypted using JSON schema syntax as follows.

With the above connection, if you create a model named 'Test' that uses the 'tests' collection, any documents will have their name property encrypted.

**Examples:**

Example 1 (json):
```json
{
  "_id" : ObjectId("647a3207661e3a3a1bc3e614"),
  "name" : BinData(6,"ASrIv7XfokKwiCUJEjckOdgCG+u6IqavcOWX8hINz29MLvcKDZ4nnjCnPFZG+0ftVxMdWgzu6Vdh7ys1uIK1WiaPN0SqpmmtL2rPoqT9gfhADpGDmI60+vm0bJepXNY1Gv0="),
  "__v" : 0
}
```

Example 2 (jsx):
```jsx
const encryptedUserSchema = new Schema({ 
  name: String,
  ssn: { 
    type: String, 
    // 1
    encrypt: { 
      keyId: '<uuid string of key id>',
      queries: 'equality'
    }
  }
  // 2
}, { encryptionType: 'queryableEncryption' });
```

Example 3 (javascript):
```javascript
// specific connection
const GlobalUserModel = mongoose.model('User', encryptedUserSchema);

// specific connection
const connection = mongoose.createConnection();
const UserModel = connection.model('User', encryptedUserSchema);
```

Example 4 (javascript):
```javascript
const keyVaultNamespace = 'client.encryption';
const kmsProviders = { local: { key } };
await connection.openUri(`mongodb://localhost:27017`, {
  // Configure auto encryption
  autoEncryption: {
    keyVaultNamespace: 'datakeys.datakeys',
    kmsProviders
  }
});
```

---

## Migrating from 5.x to 6.x

**URL:** https://mongoosejs.com/docs/6.x/docs/migrating_to_6.html

**Contents:**
- Migrating from 5.x to 6.x
- Version Requirements
- MongoDB Driver 4.0
- No More Deprecation Warning Options
- The asPromise() Method for Connections
- mongoose.connect() Returns a Promise
- Duplicate Query Execution
- Model.exists(...) now returns a lean document instead of boolean
- strictQuery is now equal to strict by default
- MongoError is now MongoServerError

Please note: we plan to discontinue Mongoose 5 support on March 1, 2024. Please see our version support guide.

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 5.x to Mongoose 6.x.

If you're still on Mongoose 4.x, please read the Mongoose 4.x to 5.x migration guide and upgrade to Mongoose 5.x first.

Mongoose now requires Node.js >= 12.0.0. Mongoose still supports MongoDB server versions back to 3.0.0.

Mongoose now uses v4.x of the MongoDB Node driver. See the MongoDB Node drivers' migration guide for detailed info. Below are some of the most noteworthy changes:

useNewUrlParser, useUnifiedTopology, useFindAndModify, and useCreateIndex are no longer supported options. Mongoose 6 always behaves as if useNewUrlParser, useUnifiedTopology, and useCreateIndex are true, and useFindAndModify is false. Please remove these options from your code.

Mongoose connections are no longer thenable. This means that await mongoose.createConnection(uri) no longer waits for Mongoose to connect. Use mongoose.createConnection(uri).asPromise() instead. See #8810.

The mongoose.connect() function now always returns a promise, not a Mongoose instance.

Mongoose no longer allows executing the same query object twice. If you do, you'll get a Query was already executed error. Executing the same query instance twice is typically indicative of mixing callbacks and promises, but if you need to execute the same query twice, you can call Query#clone() to clone the query and re-execute it. See gh-7398

Mongoose no longer supports a strictQuery option. You must now use strict. As of Mongoose 6.0.10, we brought back the strictQuery option. In Mongoose 6, strictQuery is set to strict by default. This means that, by default, Mongoose will filter out query filter properties that are not in the schema.

However, this behavior was a source of confusion in some cases, so in Mongoose 7, this default changes back to false. So if you want to retain the default behavior of Mongoose 5 as well as Mongoose 7 and later, you can also disable strictQuery globally to override:

In a test suite, it may be useful to set strictQuery to throw, which will throw exceptions any time a query references schema that doesn't exist, which could help identify a bug in your tests or code.

Here's an example of the effect of strictQuery:

You can also disable strictQuery globally to override:

In MongoDB Node.js Driver v4.x, 'MongoError' is now 'MongoServerError'. Please change any code that depends on the hardcoded string 'MongoError'.

Mongoose now clones discriminator schemas by default. This means you need to pass { clone: false } to discriminator() if you're using recursive embedded discriminators.

In Mongoose 5, mongoose.isValidObjectId() returned false for values like numbers, which was inconsistent with the MongoDB driver's ObjectId.isValid() function. Technically, any JavaScript number can be converted to a MongoDB ObjectId.

In Mongoose 6, mongoose.isValidObjectId() is just a wrapper for mongoose.Types.ObjectId.isValid() for consistency.

Mongoose 6.2.5 now includes a mongoose.isObjectIdOrHexString() function, which does a better job of capturing the more common use case for isValidObjectId(): is the given value an ObjectId instance or a 24 character hex string representing an ObjectId?

Mongoose now saves objects with keys in the order the keys are specified in the schema, not in the user-defined object. So whether Object.keys(new User({ name: String, email: String }).toObject() is ['name', 'email'] or ['email', 'name'] depends on the order name and email are defined in your schema.

Mongoose 6 introduces a new sanitizeFilter option to globals and queries that defends against query selector injection attacks. If you enable sanitizeFilter, Mongoose will wrap any object in the query filter in a $eq:

To explicitly allow a query selector, use mongoose.trusted():

In Mongoose 5.x, setting a key to undefined in an update operation was equivalent to setting it to null.

Mongoose 5.x supported an omitUndefined option to strip out undefined keys. In Mongoose 6.x, the omitUndefined option has been removed, and Mongoose will always strip out undefined keys.

The only workaround is to explicitly set properties to null in your updates:

Mongoose now passes the document as the first parameter to default functions, which is helpful for using arrow functions with defaults.

This may affect you if you pass a function that expects different parameters to default, like default: mongoose.Types.ObjectId. See gh-9633. If you're passing a default function that does not utilize the document, change default: myFunction to default: () => myFunction() to avoid accidentally passing parameters that potentially change the behavior.

Mongoose arrays are now ES6 proxies. You no longer need to markModified() after setting an array index directly.

Schema paths declared with type: { name: String } become single nested subdocs in Mongoose 6, as opposed to Mixed in Mongoose 5. This removes the need for the typePojoToMixed option. See gh-7181.

Mongoose now throws an error if you populate() a path that isn't defined in your schema. This is only for cases when we can infer the local schema, like when you use Query#populate(), not when you call Model.populate() on a POJO. See gh-5124.

When populating a subdocument with a function ref or refPath, this is now the subdocument being populated, not the top-level document. See #8469.

Using save, isNew, and other Mongoose reserved names as schema path names now triggers a warning, not an error. You can suppress the warning by setting the supressReservedKeysWarning in your schema options: new Schema({ save: String }, { supressReservedKeysWarning: true }). Keep in mind that this may break plugins that rely on these reserved names.

Single nested subdocs have been renamed to "subdocument paths". So SchemaSingleNestedOptions is now SchemaSubdocumentOptions and mongoose.Schema.Types.Embedded is now mongoose.Schema.Types.Subdocument. See gh-10419

Aggregate#cursor() now returns an AggregationCursor instance to be consistent with Query#cursor(). You no longer need to do Model.aggregate(pipeline).cursor().exec() to get an aggregation cursor, just Model.aggregate(pipeline).cursor().

autoCreate is true by default unless readPreference is secondary or secondaryPreferred, which means Mongoose will attempt to create every model's underlying collection before creating indexes. If readPreference is secondary or secondaryPreferred, Mongoose will default to false for both autoCreate and autoIndex because both createCollection() and createIndex() will fail when connected to a secondary.

The context option for queries has been removed. Now Mongoose always uses context = 'query'.

Mongoose 6 always calls validators with depopulated paths (that is, with the id rather than the document itself). In Mongoose 5, Mongoose would call validators with the populated doc if the path was populated. See #8042

When connected to a replica set, connections now emit 'disconnected' when connection to the primary is lost. In Mongoose 5, connections only emitted 'disconnected' when losing connection to all members of the replica set.

However, Mongoose 6 does not buffer commands while a connection is disconnected. So you can still successfully execute commands like queries with readPreference = 'secondary', even if the Mongoose connection is in the disconnected state.

Document#populate() now returns a promise and is now no longer chainable.

await Model.create([]) in v6.0 returns an empty array when provided an empty array, in v5.0 it used to return undefined. If any of your code is checking whether the output is undefined or not, you need to modify it with the assumption that await Model.create(...) will always return an array if provided an array.

doc.set({ child: { age: 21 } }) now works the same whether child is a nested path or a subdocument: Mongoose will overwrite the value of child. In Mongoose 5, this operation would merge child if child was a nested path.

Mongoose now adds a valueOf() function to ObjectIds. This means you can now use == to compare an ObjectId against a string.

If you set timestamps: true, Mongoose will now make the createdAt property immutable. See gh-10139

isAsync is no longer an option for validate. Use an async function instead.

safe is no longer an option for schemas, queries, or save(). Use writeConcern instead.

Mongoose now calls setter functions with priorValue as the 2nd parameter, rather than schemaType in Mongoose 5.

This change was technically released with 5.10.5, but caused issues for users migrating from 5.9.x to 6.x. In Mongoose < 5.10.5, toObject() and toJSON() would use the top-level schema's minimize option by default.

As a workaround, you can either explicitly pass minimize to toObject() or toJSON():

Or define the child schema inline (Mongoose 6 only) to inherit the parent's minimize option.

In Mongoose 5, calling populate() on a mixed type or other path with no ref would fall back to using the query's model.

In Mongoose 6, populating a path with no ref, refPath, or model is a no-op.

The Schema class now takes 3 generic params instead of 4. The 3rd generic param, SchemaDefinitionType, is now the same as the 1st generic param DocType. Replace new Schema<UserDocument, UserModel, User>(schemaDefinition) with new Schema<UserDocument, UserModel>(schemaDefinition)

Types.ObjectId is now a class, which means you can no longer omit new when creating a new ObjectId using new mongoose.Types.ObjectId(). Currently, you can still omit new in JavaScript, but you must put new in TypeScript.

The following legacy types have been removed:

Mongoose 6 infers the document's type for this in virtual getters and setters. In Mongoose 5.x, this would be any in the following code.

In Mongoose 6, this will be set to the document type.

The reconnectTries and reconnectInterval options have been removed since they are no longer necessary.

The MongoDB node driver will always attempt to retry any operation for up to serverSelectionTimeoutMS, even if MongoDB is down for a long period of time. So, it will never run out of retries or try to reconnect to MongoDB.

Lodash's isEmpty() function returns true for primitives and primitive wrappers. ObjectId() is an object wrapper that is treated as a primitive by Mongoose. But starting in Mongoose 6, _.isEmpty() will return true for ObjectIds because of Lodash implementation details.

An ObjectId in mongoose is never empty, so if you're using isEmpty() you should check for instanceof ObjectId.

The mongoose.modelSchemas property was removed. This may have been used to delete a model schema.

**Examples:**

Example 1 (javascript):
```javascript
const res = await TestModel.updateMany({}, { someProperty: 'someValue' });

res.matchedCount; // Number of documents that were found that match the filter. Replaces `res.n`
res.modifiedCount; // Number of documents modified. Replaces `res.nModified`
res.upsertedCount; // Number of documents upserted. Replaces `res.upserted`
```

Example 2 (javascript):
```javascript
const res = await TestModel.deleteMany({});

// In Mongoose 6: `{ acknowledged: true, deletedCount: 2 }`
// In Mongoose 5: `{ n: 2, ok: 1, deletedCount: 2 }`
res;

res.deletedCount; // Number of documents that were deleted. Replaces `res.n`
```

Example 3 (javascript):
```javascript
// No longer necessary:
mongoose.set('useFindAndModify', false);

await mongoose.connect('mongodb://127.0.0.1:27017/test', {
  useNewUrlParser: true, // <-- no longer necessary
  useUnifiedTopology: true // <-- no longer necessary
});
```

Example 4 (javascript):
```javascript
// The below no longer works in Mongoose 6
await mongoose.createConnection(uri);

// Do this instead
await mongoose.createConnection(uri).asPromise();
```

---

## 

**URL:** https://mongoosejs.com/docs/6.x/docs/jobs.html

**Contents:**
- Mongoose Jobs
- Add Your Own

Localize is looking for a Senior Full Stack Engineer to join our 100% remote team.

As a Localize engineer, you’ll be responsible for implementing new functionality within Localize’s core product. On the frontend you’ll work on our React/Redux/Backbone SPA, and on the backend you’ll build RESTful APIs in Node/Express/MongoDB.

As a key member of our remote engineering team, you’ll lead the development of many high impact product initiatives. We’re looking for a strong engineer who works well on a small team and is excited about the opportunity to have a direct impact on improving customer experience.

Technologies: Experience with these specific technologies a plus, but not strictly required

Experience and Qualifications:

You’ll be joining a close knit and talented team, with plenty of opportunity for professional growth. We offer compensation + benefits that are on par with large companies, while also placing a high value on maintaining a healthy work-life balance.

The Localize platform is used by 500+ companies to translate websites and applications into other languages. We help companies like Trello to translate their help center (https://help.trello.com/), Tinder to translate their blog (https://blog.gotinder.com/), and RocketMiles to translate their web app (https://www.rocketmiles.com/) - to see our product in action, click any of those links and use the website's language switcher to switch from English to another language.

Localize works by providing a code snippet (similar to the Google Analytics javascript snippet) that our customers add to their website / web app. The Localize snippet pulls in content from the page into the Localize dashboard where our customers login to add translations and manage their content. Localize automatically deploys those translations to the customer's production site.

We are a team of 12 people working fully remote. We offer competitive pay, a full benefits package, and a culture with emphasis on work + life balance.

---

## Timestamps

**URL:** https://mongoosejs.com/docs/timestamps.html

**Contents:**
- Timestamps
- Alternate Property Names
- Disabling Timestamps
- Timestamps on Subdocuments
- Under the Hood
- Updating Timestamps

Mongoose schemas support a timestamps option. If you set timestamps: true, Mongoose will add two properties of type Date to your schema:

Mongoose will then set createdAt when the document is first inserted, and update updatedAt whenever you update the document using save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), or bulkWrite().

The createdAt property is immutable, and Mongoose overwrites any user-specified updates to updatedAt by default.

Keep in mind that replaceOne() and findOneAndReplace() overwrite all non-_id properties, including immutable properties like createdAt. Calling replaceOne() or findOneAndReplace() will update the createdAt timestamp as shown below.

For the purposes of these docs, we'll always refer to createdAt and updatedAt. But you can overwrite these property names as shown below.

save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), and bulkWrite() all support a timestamps option. Set timestamps: false to skip setting timestamps for that particular operation.

You can also set the timestamps option to an object to configure createdAt and updatedAt separately. For example, in the below code, Mongoose sets createdAt on save() but skips updatedAt.

Disabling timestamps also lets you set timestamps yourself. For example, suppose you need to correct a document's createdAt or updatedAt property. You can do that by setting timestamps: false and setting createdAt yourself as shown below.

Mongoose also supports setting timestamps on subdocuments. Keep in mind that createdAt and updatedAt for subdocuments represent when the subdocument was created or updated, not the top level document. Overwriting a subdocument will also overwrite createdAt.

For queries with timestamps, Mongoose adds 2 properties to each update query:

For example, if you run the below code:

You'll see the below output from Mongoose debug mode:

Notice the $setOnInsert for createdAt and $set for updatedAt. MongoDB's $setOnInsert operator applies the update only if a new document is upserted. So, for example, if you want to only set updatedAt if a new document is created, you can disable the updatedAt timestamp and set it yourself as shown below:

If you need to disable Mongoose's timestamps and update a document's timestamps to a different value using updateOne() or findOneAndUpdate(), you need to do the following:

**Examples:**

Example 1 (javascript):
```javascript
const userSchema = new Schema({ name: String }, { timestamps: true });
const User = mongoose.model('User', userSchema);

let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.244Z

doc.name = 'test2';
await doc.save();
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.307Z

doc = await User.findOneAndUpdate({ _id: doc._id }, { name: 'test3' }, { new: true });
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.366Z
```

Example 2 (javascript):
```javascript
let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.930Z

doc.name = 'test2';
doc.createdAt = new Date(0);
doc.updatedAt = new Date(0);
await doc.save();

// Mongoose blocked changing `createdAt` and set its own `updatedAt`, ignoring
// the attempt to manually set them.
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.991Z

// Mongoose also blocks changing `createdAt` and sets its own `updatedAt`
// on `findOneAndUpdate()`, `updateMany()`, and other query operations
// **except** `replaceOne()` and `findOneAndReplace()`.
doc = await User.findOneAndUpdate(
  { _id: doc._id },
  { name: 'test3', createdAt: new Date(0), updatedAt: new Date(0) },
  { new: true }
);
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:14.008Z
```

Example 3 (json):
```json
// `findOneAndReplace()` and `replaceOne()` without timestamps specified in `replacement`
// sets `createdAt` and `updatedAt` to current time.
doc = await User.findOneAndReplace(
  { _id: doc._id },
  { name: 'test3' },
  { new: true }
);
console.log(doc.createdAt); // 2022-02-26T17:08:14.008Z
console.log(doc.updatedAt); // 2022-02-26T17:08:14.008Z

// `findOneAndReplace()` and `replaceOne()` with timestamps specified in `replacement`
// sets `createdAt` and `updatedAt` to the values in `replacement`.
doc = await User.findOneAndReplace(
  { _id: doc._id },
  {
    name: 'test3',
    createdAt: new Date('2022-06-01'),
    updatedAt: new Date('2022-06-01')
  },
  { new: true }
);
console.log(doc.createdAt); // 2022-06-01T00:00:00.000Z
console.log(doc.updatedAt); // 2022-06-01T00:00:00.000Z
```

Example 4 (css):
```css
const userSchema = new Schema({ name: String }, {
  timestamps: {
    createdAt: 'created_at', // Use `created_at` to store the created date
    updatedAt: 'updated_at' // and `updated_at` to store the last updated date
  }
});
```

---

## MongoDB Server Version Compatibility

**URL:** https://mongoosejs.com/docs/6.x/docs/compatibility.html

**Contents:**
- MongoDB Server Version Compatibility

Mongoose relies on the MongoDB Node.js Driver to talk to MongoDB. You can refer to this table for up-to-date information as to which version of the MongoDB driver supports which version of MongoDB.

Below are the semver ranges representing which versions of mongoose are compatible with the listed versions of MongoDB server.

Mongoose ^6.5.0 also works with MongoDB server 7.x. But not all new MongoDB server 7.x features are supported by Mongoose 6.x.

Note that Mongoose 5.x dropped support for all versions of MongoDB before 3.0.0. If you need to use MongoDB 2.6 or older, use Mongoose 4.x.

---

## Timestamps

**URL:** https://mongoosejs.com/docs/6.x/docs/timestamps.html

**Contents:**
- Timestamps
- Alternate Property Names
- Disabling Timestamps
- Timestamps on Subdocuments
- Under the Hood

Mongoose schemas support a timestamps option. If you set timestamps: true, Mongoose will add two properties of type Date to your schema:

Mongoose will then set createdAt when the document is first inserted, and update updatedAt whenever you update the document using save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), or bulkWrite().

The createdAt property is immutable, and Mongoose overwrites any user-specified updates to updatedAt by default.

For the purposes of these docs, we'll always refer to createdAt and updatedAt. But you can overwrite these property names as shown below.

save(), updateOne(), updateMany(), findOneAndUpdate(), update(), replaceOne(), and bulkWrite() all support a timestamps option. Set timestamps: false to skip setting timestamps for that particular operation.

You can also set the timestamps option to an object to configure createdAt and updatedAt separately. For example, in the below code, Mongoose sets createdAt on save() but skips updatedAt.

Disabling timestamps also lets you set timestamps yourself. For example, suppose you need to correct a document's createdAt or updatedAt property. You can do that by setting timestamps: false and setting createdAt yourself as shown below.

Mongoose also supports setting timestamps on subdocuments. Keep in mind that createdAt and updatedAt for subdocuments represent when the subdocument was created or updated, not the top level document. Overwriting a subdocument will also overwrite createdAt.

For queries with timestamps, Mongoose adds 2 properties to each update query:

For example, if you run the below code:

You'll see the below output from Mongoose debug mode:

Notice the $setOnInsert for createdAt and $set for updatedAt. MongoDB's $setOnInsert operator applies the update only if a new document is upserted. So, for example, if you want to only set updatedAt if the document if a new document is created, you can disable the updatedAt timestamp and set it yourself as shown below:

**Examples:**

Example 1 (javascript):
```javascript
const userSchema = new Schema({ name: String }, { timestamps: true });
const User = mongoose.model('User', userSchema);

let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.244Z

doc.name = 'test2';
await doc.save();
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.307Z

doc = await User.findOneAndUpdate({ _id: doc._id }, { name: 'test3' }, { new: true });
console.log(doc.createdAt); // 2022-02-26T16:37:48.244Z
console.log(doc.updatedAt); // 2022-02-26T16:37:48.366Z
```

Example 2 (javascript):
```javascript
let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.930Z

doc.name = 'test2';
doc.createdAt = new Date(0);
doc.updatedAt = new Date(0);
await doc.save();

// Mongoose blocked changing `createdAt` and set its own `updatedAt`, ignoring
// the attempt to manually set them.
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:13.991Z

// Mongoose also blocks changing `createdAt` and sets its own `updatedAt`
// on `findOneAndUpdate()`, `updateMany()`, and other query operations
doc = await User.findOneAndUpdate(
  { _id: doc._id },
  { name: 'test3', createdAt: new Date(0), updatedAt: new Date(0) },
  { new: true }
);
console.log(doc.createdAt); // 2022-02-26T17:08:13.930Z
console.log(doc.updatedAt); // 2022-02-26T17:08:14.008Z
```

Example 3 (javascript):
```javascript
const userSchema = new Schema({ name: String }, {
  timestamps: {
    createdAt: 'created_at', // Use `created_at` to store the created date
    updatedAt: 'updated_at' // and `updated_at` to store the last updated date
  }
});
```

Example 4 (javascript):
```javascript
let doc = await User.create({ name: 'test' });

console.log(doc.createdAt); // 2022-02-26T23:28:54.264Z
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

doc.name = 'test2';

// Setting `timestamps: false` tells Mongoose to skip updating `updatedAt` on this `save()`
await doc.save({ timestamps: false });
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

// Similarly, setting `timestamps: false` on a query tells Mongoose to skip updating
// `updatedAt`.
doc = await User.findOneAndUpdate({ _id: doc._id }, { name: 'test3' }, {
  new: true,
  timestamps: false
});
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z

// Below is how you can disable timestamps on a `bulkWrite()`
await User.bulkWrite([{
  updateOne: {
    filter: { _id: doc._id },
    update: { name: 'test4' },
    timestamps: false
  }
}]);
doc = await User.findOne({ _id: doc._id });
console.log(doc.updatedAt); // 2022-02-26T23:28:54.264Z
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/6.x/

---

## Mongoose for Enterprise

**URL:** https://mongoosejs.com/docs/7.x/docs/enterprise.html

**Contents:**
- Mongoose for Enterprise
- Available as part of the Tidelift Subscription
- Enterprise-ready open source software—managed for you

Tidelift is working with the maintainers of Mongoose and thousands of other open source projects to deliver commercial support and maintenance for the open source dependencies you use to build your applications. Save time, reduce risk, and improve code health, while paying the maintainers of the exact dependencies you use.

The Tidelift Subscription is a managed open source subscription for application dependencies covering millions of open source projects across JavaScript, Python, Java, PHP, Ruby, .NET, and more.

Your subscription includes:

Tidelift’s security response team coordinates patches for new breaking security vulnerabilities and alerts immediately through a private channel, so your software supply chain is always secure.

Licensing verification and indemnification

Tidelift verifies license information to enable easy policy enforcement and adds intellectual property indemnification to cover creators and users in case something goes wrong. You always have a 100% up-to-date bill of materials for your dependencies to share with your legal team, customers, or partners.

Maintenance and code improvement

Tidelift ensures the software you rely on keeps working as long as you need it to work. Your managed dependencies are actively maintained and we recruit additional maintainers where required.

Package selection and version guidance

We help you choose the best open source packages from the start—and then guide you through updates to stay on the best releases as new issues arise.

Take a seat at the table with the creators behind the software you use. Tidelift’s participating maintainers earn more income as their software is used by more subscribers, so they’re interested in knowing what you need.

Tooling and cloud integration

Tidelift works with GitHub, GitLab, BitBucket, and more. We support every cloud platform (and other deployment targets, too).

The end result? All of the capabilities you expect from commercial-grade software, for the full breadth of open source you use. That means less time grappling with esoteric open source trivia, and more time building your own applications—and your business.

---

## FAQ

**URL:** https://mongoosejs.com/docs/7.x/docs/faq.html

**Contents:**
- FAQ

Q. I get an error connect ECONNREFUSED ::1:27017 when connecting to localhost. Why?

The easy solution is to replace localhost with 127.0.0.1.

The reason why this error happens is that Node.js 18 and up prefer IPv6 addresses over IPv4 by default. And, most Linux and OSX machines have a ::1 localhost entry in /etc/hosts by default. That means that Node.js 18 will assume that localhost means the IPv6 ::1 address. And MongoDB doesn't accept IPv6 connections by default.

You can also fix this error by enabling IPv6 support on your MongoDB server.

Q. Operation ... timed out after 10000 ms. What gives?

A. At its core, this issue stems from not connecting to MongoDB. You can use Mongoose before connecting to MongoDB, but you must connect at some point. For example:

Q. I am able to connect locally but when I try to connect to MongoDB Atlas I get this error. What gives?

You must ensure that you have whitelisted your ip on mongodb to allow Mongoose to connect. You can allow access from all ips with 0.0.0.0/0.

Q. x.$__y is not a function. What gives?

A. This issue is a result of having multiple versions of mongoose installed that are incompatible with each other. Run npm list | grep "mongoose" to find and remedy the problem. If you're storing schemas or models in a separate npm package, please list Mongoose in peerDependencies rather than dependencies in your separate package.

Q. I declared a schema property as unique but I can still save duplicates. What gives?

A. Mongoose doesn't handle unique on its own: { name: { type: String, unique: true } } is just a shorthand for creating a MongoDB unique index on name. For example, if MongoDB doesn't already have a unique index on name, the below code will not error despite the fact that unique is true.

However, if you wait for the index to build using the Model.on('index') event, attempts to save duplicates will correctly error.

MongoDB persists indexes, so you only need to rebuild indexes if you're starting with a fresh database or you ran db.dropDatabase(). In a production environment, you should create your indexes using the MongoDB shell rather than relying on mongoose to do it for you. The unique option for schemas is convenient for development and documentation, but mongoose is not an index management solution.

Q. When I have a nested property in a schema, mongoose adds empty objects by default. Why?

A. This is a performance optimization. These empty objects are not saved to the database, nor are they in the result toObject(), nor do they show up in JSON.stringify() output unless you turn off the minimize option.

The reason for this behavior is that Mongoose's change detection and getters/setters are based on Object.defineProperty(). In order to support change detection on nested properties without incurring the overhead of running Object.defineProperty() every time a document is created, mongoose defines properties on the Model prototype when the model is compiled. Because mongoose needs to define getters and setters for nested.prop, nested must always be defined as an object on a mongoose document, even if nested is undefined on the underlying POJO.

Q. I'm using an arrow function for a virtual, middleware, getter/setter, or method and the value of this is wrong.

A. Arrow functions handle the this keyword differently than conventional functions. Mongoose getters/setters depend on this to give you access to the document that you're writing to, but this functionality does not work with arrow functions. Do not use arrow functions for mongoose getters/setters unless do not intend to access the document in the getter/setter.

Q. I have an embedded property named type like this:

But mongoose gives me a CastError telling me that it can't cast an object to a string when I try to save a Holding with an asset object. Why is this?

A. The type property is special in mongoose, so when you say type: String, mongoose interprets it as a type declaration. In the above schema, mongoose thinks asset is a string, not an object. Do this instead:

Q. I'm populating a nested property under an array like the below code:

.populate({ path: 'arr.child', options: { sort: 'name' } }) won't sort by arr.child.name?

A. See this GitHub issue. It's a known issue but one that's exceptionally difficult to fix.

Q. All function calls on my models hang, what am I doing wrong?

A. By default, mongoose will buffer your function calls until it can connect to MongoDB. Read the buffering section of the connection docs for more information.

Q. How can I enable debugging?

A. Set the debug option:

For more debugging options (streams, callbacks), see the 'debug' option under .set().

Q. My save() callback never executes. What am I doing wrong?

A. All collection actions (insert, remove, queries, etc.) are queued until Mongoose successfully connects to MongoDB. It is likely you haven't called Mongoose's connect() or createConnection() function yet.

In Mongoose 5.11, there is a bufferTimeoutMS option (set to 10000 by default) that configures how long Mongoose will allow an operation to stay buffered before throwing an error.

If you want to opt out of Mongoose's buffering mechanism across your entire application, set the global bufferCommands option to false:

Instead of opting out of Mongoose's buffering mechanism, you may want to instead reduce bufferTimeoutMS to make Mongoose only buffer for a short time.

Q. Should I create/destroy a new connection for each database operation?

A. No. Open your connection when your application starts up and leave it open until the application shuts down.

Q. Why do I get "OverwriteModelError: Cannot overwrite .. model once compiled" when I use nodemon / a testing framework?

A. mongoose.model('ModelName', schema) requires 'ModelName' to be unique, so you can access the model by using mongoose.model('ModelName'). If you put mongoose.model('ModelName', schema); in a mocha beforeEach() hook, this code will attempt to create a new model named 'ModelName' before every test, and so you will get an error. Make sure you only create a new model with a given name once. If you need to create multiple models with the same name, create a new connection and bind the model to the connection.

Q. How can I change mongoose's default behavior of initializing an array path to an empty array so that I can require real data on document creation?

A. You can set the default of the array to a function that returns undefined.

Q. How can I initialize an array path to null?

A. You can set the default of the array to a function that returns null.

Q. Why does my aggregate $match fail to return the document that my find query returns when working with dates?

A. Mongoose does not cast aggregation pipeline stages because with $project, $group, etc. the type of a property may change during the aggregation. If you want to query by date using the aggregation framework, you're responsible for ensuring that you're passing in a valid date.

Q. Why don't in-place modifications to date objects (e.g. date.setMonth(1);) get saved?

A. Mongoose currently doesn't watch for in-place updates to date objects. If you have need for this feature, feel free to discuss on this GitHub issue. There are several workarounds:

Q. Why does calling save() multiple times on the same document in parallel only let the first save call succeed and return ParallelSaveErrors for the rest?

A. Due to the asynchronous nature of validation and middleware in general, calling save() multiple times in parallel on the same doc could result in conflicts. For example, validating, and then subsequently invalidating the same path.

Q. Why is any 12 character string successfully cast to an ObjectId?

A. Technically, any 12 character string is a valid ObjectId. Consider using a regex like /^[a-f0-9]{24}$/ to test whether a string is exactly 24 hex characters.

Q. Why do keys in Mongoose Maps have to be strings?

A. Because the Map eventually gets stored in MongoDB where the keys must be strings.

Q. I am using Model.find(...).populate(...) with the limit option, but getting fewer results than the limit. What gives?

A. In order to avoid executing a separate query for each document returned from the find query, Mongoose instead queries using (numDocuments * limit) as the limit. If you need the correct limit, you should use the perDocumentLimit option (new in Mongoose 5.9.0). Just keep in mind that populate() will execute a separate query for each document.

Q. My query/update seems to execute twice. Why is this happening?

A. The most common cause of duplicate queries is mixing callbacks and promises with queries. That's because passing a callback to a query function, like find() or updateOne(), immediately executes the query, and calling then() executes the query again.

Mixing promises and callbacks can lead to duplicate entries in arrays. For example, the below code inserts 2 entries into the tags array, *not just 1.

If you'd like to contribute to this page, please visit it on github and use the Edit button to send a pull request.

**Examples:**

Example 1 (javascript):
```javascript
await mongoose.createConnection(mongodbUri).asPromise();

const Test = mongoose.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because didn't call `mongoose.connect()`
```

Example 2 (javascript):
```javascript
await mongoose.connect(mongodbUri);

const db = mongoose.createConnection();

const Test = db.model('Test', schema);

await Test.findOne(); // Will throw "Operation timed out" error because `db` isn't connected to MongoDB
```

Example 3 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

// No error, unless index was already built
await Model.create([{ name: 'Val' }, { name: 'Val' }]);
```

Example 4 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

// Wait for model's indexes to finish. The `init()`
// function is idempotent, so don't worry about triggering an index rebuild.
await Model.init();

// Throws a duplicate key error
await Model.create([{ name: 'Val' }, { name: 'Val' }]);
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/4.x

---

## Transactions in Mongoose

**URL:** https://mongoosejs.com/docs/6.x/docs/transactions.html

**Contents:**
- Transactions in Mongoose
- Getting Started with Transactions
- With Mongoose Documents and save()
- With the Aggregation Framework
- Advanced Usage

Transactions are new in MongoDB 4.0 and Mongoose 5.2.0. Transactions let you execute multiple operations in isolation and potentially undo all the operations if one of them fails. This guide will get you started using transactions with Mongoose.

If you haven't already, import mongoose:

To create a transaction, you first need to create a session using Mongoose#startSession or Connection#startSession().

In practice, you should use either the session.withTransaction() helper or Mongoose's Connection#transaction() function to run a transaction. The session.withTransaction() helper handles:

For more information on the ClientSession#withTransaction() function, please see the MongoDB Node.js driver docs.

Mongoose's Connection#transaction() function is a wrapper around withTransaction() that integrates Mongoose change tracking with transactions. For example, suppose you save() a document in a transaction that later fails. The changes in that document are not persisted to MongoDB. The Connection#transaction() function informs Mongoose change tracking that the save() was rolled back, and marks all fields that were changed in the transaction as modified.

If you get a Mongoose document from findOne() or find() using a session, the document will keep a reference to the session and use that session for save().

To get/set the session associated with a given document, use doc.$session().

The Model.aggregate() function also supports transactions. Mongoose aggregations have a session() helper that sets the session option. Below is an example of executing an aggregation within a transaction.

Advanced users who want more fine-grained control over when they commit or abort transactions can use session.startTransaction() to start a transaction:

You can also use session.abortTransaction() to abort a transaction:

**Examples:**

Example 1 (javascript):
```javascript
import mongoose from 'mongoose';
```

Example 2 (javascript):
```javascript
// Using Mongoose's default connection
const session = await mongoose.startSession();

// Using custom connection
const db = await mongoose.createConnection(mongodbUri).asPromise();
const session = await db.startSession();
```

Example 3 (javascript):
```javascript
let session = null;
return Customer.createCollection().
  then(() => Customer.startSession()).
  // The `withTransaction()` function's first parameter is a function
  // that returns a promise.
  then(_session => {
    session = _session;
    return session.withTransaction(() => {
      return Customer.create([{ name: 'Test' }], { session: session });
    });
  }).
  then(() => Customer.countDocuments()).
  then(count => assert.strictEqual(count, 1)).
  then(() => session.endSession());
```

Example 4 (javascript):
```javascript
const doc = new Person({ name: 'Will Riker' });

await db.transaction(async function setRank(session) {
  doc.name = 'Captain';
  await doc.save({ session });
  doc.isNew; // false

  // Throw an error to abort the transaction
  throw new Error('Oops!');
}, { readPreference: 'primary' }).catch(() => {});

// true, `transaction()` reset the document's state because the
// transaction was aborted.
doc.isNew;
```

---

## Using Mongoose With Next.js

**URL:** https://mongoosejs.com/docs/nextjs.html

**Contents:**
- Using Mongoose With Next.js
- Quick Start
- Best Practices
  - Connection Management
  - Environment Variables
  - Model Registration
- Common Issues
- TypeError: Cannot read properties of undefined (reading 'prototype')
- Using with Pages Router
- Using with App Router Server Components

Next.js is a popular framework for building full stack applications with React. Mongoose works out of the box with Next.js. If you're looking to get started, please use Next.js' official Mongoose sample app. Furthermore, if you are using Next.js with Vercel Serverless Functions, please review Mongoose's AWS Lambda docs.

Here's a basic example of using Mongoose with Next.js App Router:

Then use it in your API routes or Server Components:

Mongoose handles connection management automatically. Calling mongoose.connect() when Mongoose is already connected is a no-op, so you can safely call dbConnect() in every API route and Server Component without worrying about creating multiple connections.

Store your MongoDB connection string in .env.local:

For production, use environment variables in your hosting platform (Vercel, Netlify, etc.).

Define your models in a separate directory and ensure they're only registered once:

The mongoose.models.User || mongoose.model('User', UserSchema) pattern prevents model recompilation errors during hot reloading in development.

There are a few common issues when working with Next.js that you should be aware of.

You can fix this issue by adding the following to your next.config.js:

This issue is caused by this change in MongoDB's bson parser. MongoDB's bson parser uses top-level await and dynamic import in ESM mode to avoid some Webpack bundling issues. And Next.js forces ESM mode.

If you're using Next.js Pages Router, you can use Mongoose in API routes and getServerSideProps:

Using in getServerSideProps:

Important: Use JSON.parse(JSON.stringify()) to convert Mongoose documents to plain objects, as Next.js requires serializable data.

With Next.js 13+ App Router, you can use Mongoose directly in Server Components:

Mongoose does not currently support Next.js Edge Runtime. There is no way for Mongoose to connect to MongoDB in Edge Runtime, because Edge Runtime currently doesn't support Node.js net API, which is what the MongoDB Node Driver uses to connect to MongoDB.

**Examples:**

Example 1 (javascript):
```javascript
// lib/mongodb.js
import mongoose from 'mongoose';

const MONGODB_URI = process.env.MONGODB_URI;

export default dbConnect;

async function dbConnect() {
  if (!MONGODB_URI) {
    throw new Error('Please define the MONGODB_URI environment variable');
  }
  await mongoose.connect(MONGODB_URI);
  return mongoose;
}
```

Example 2 (javascript):
```javascript
// app/api/users/route.js
import dbConnect from '@/lib/mongodb';
import User from '@/models/User';

export async function GET() {
  await dbConnect();
  const users = await User.find({});
  return Response.json({ users });
}
```

Example 3 (json):
```json
MONGODB_URI=mongodb://localhost:27017/mydb
```

Example 4 (javascript):
```javascript
// models/User.js
import mongoose from 'mongoose';

const UserSchema = new mongoose.Schema({
  name: String,
  email: { type: String, required: true }
}, { timestamps: true });

export default mongoose.models.User || mongoose.model('User', UserSchema);
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/faq.html

**Contents:**
- FAQ

Q. Why don't my changes to arrays get saved when I update an element directly?

A. Mongoose doesn't create getters/setters for array indexes; without them mongoose never gets notified of the change and so doesn't know to persist the new value. There are two workarounds: MongooseArray#set or Document#markModified().

This only affects when you set an array index directly. If you set a path on a document array element, you do not need to use markModified().

Q. I declared a schema property as unique but I can still save duplicates. What gives?

A. Mongoose doesn't handle unique on its own: { name: { type: String, unique: true } } is just a shorthand for creating a MongoDB unique index on name. For example, if MongoDB doesn't already have a unique index on name, the below code will not error despite the fact that unique is true.

However, if you wait for the index to build using the Model.on('index') event, attempts to save duplicates will correctly error.

MongoDB persists indexes, so you only need to rebuild indexes if you're starting with a fresh database or you ran db.dropDatabase(). In a production environment, you should create your indexes using the MongoDB shell rather than relying on mongoose to do it for you. The unique option for schemas is convenient for development and documentation, but mongoose is not an index management solution.

Q. When I have a nested property in a schema, mongoose adds empty objects by default. Why?

A. This is a performance optimization. These empty objects are not saved to the database, nor are they in the result toObject(), nor do they show up in JSON.stringify() output unless you turn off the minimize option.

The reason for this behavior is that Mongoose's change detection and getters/setters are based on Object.defineProperty(). In order to support change detection on nested properties without incurring the overhead of running Object.defineProperty() every time a document is created, mongoose defines properties on the Model prototype when the model is compiled. Because mongoose needs to define getters and setters for nested.prop, nested must always be defined as an object on a mongoose document, even if nested is undefined on the underlying POJO.

Q. When I use named imports like import { set } from 'mongoose', I get a TypeError. What causes this issue and how can I fix it?

A. The only import syntax Mongoose supports is import mongoose from 'mongoose'. Syntaxes like import * from 'mongoose' or import { model } from 'mongoose' do not work. The global Mongoose object stores types, global options, and other important properties that Mongoose needs. When you do import { model } from 'mongoose', the this value in model() is not the Mongoose global.

Q. I'm using an arrow function for a virtual, middleware, getter/setter, or method and the value of this is wrong.

A. Arrow functions handle the this keyword much differently than conventional functions. Mongoose getters/setters depend on this to give you access to the document that you're writing to, but this functionality does not work with arrow functions. Do not use arrow functions for mongoose getters/setters unless do not intend to access the document in the getter/setter.

Q. I have an embedded property named type like this:

But mongoose gives me a CastError telling me that it can't cast an object to a string when I try to save a Holding with an asset object. Why is this?

A. The type property is special in mongoose, so when you say type: String, mongoose interprets it as a type declaration. In the above schema, mongoose thinks asset is a string, not an object. Do this instead:

Q. I'm populating a nested property under an array like the below code:

.populate({ path: 'arr.child', options: { sort: 'name' } }) won't sort by arr.child.name?

A. See this GitHub issue. It's a known issue but one that's exceptionally difficult to fix.

Q. All function calls on my models hang, what am I doing wrong?

A. By default, mongoose will buffer your function calls until it can connect to MongoDB. Read the buffering section of the connection docs for more information.

Q. How can I enable debugging?

A. Set the debug option:

For more debugging options (streams, callbacks), see the 'debug' option under .set().

Q. My save() callback never executes. What am I doing wrong?

A. All collection actions (insert, remove, queries, etc.) are queued until Mongoose successfully connects to MongoDB. It is likely you haven't called Mongoose's connect() or createConnection() function yet.

In Mongoose 5.11, there is a bufferTimeoutMS option (set to 10000 by default) that configures how long Mongoose will allow an operation to stay buffered before throwing an error.

If you want to opt out of Mongoose's buffering mechanism across your entire application, set the global bufferCommands option to false:

Instead of opting out of Mongoose's buffering mechanism, you may want to instead reduce bufferTimeoutMS to make Mongoose only buffer for a short time.

Q. Should I create/destroy a new connection for each database operation?

A. No. Open your connection when your application starts up and leave it open until the application shuts down.

Q. Why do I get "OverwriteModelError: Cannot overwrite .. model once compiled" when I use nodemon / a testing framework?

A. mongoose.model('ModelName', schema) requires 'ModelName' to be unique, so you can access the model by using mongoose.model('ModelName'). If you put mongoose.model('ModelName', schema); in a mocha beforeEach() hook, this code will attempt to create a new model named 'ModelName' before every test, and so you will get an error. Make sure you only create a new model with a given name once. If you need to create multiple models with the same name, create a new connection and bind the model to the connection.

Q. How can I change mongoose's default behavior of initializing an array path to an empty array so that I can require real data on document creation?

A. You can set the default of the array to a function that returns undefined.

Q. How can I initialize an array path to null?

A. You can set the default of the array to a function that returns null.

Q. Why does my aggregate $match fail to return the document that my find query returns when working with dates?

A. Mongoose does not cast aggregation pipeline stages because with $project, $group, etc. the type of a property may change during the aggregation. If you want to query by date using the aggregation framework, you're responsible for ensuring that you're passing in a valid date.

Q. Why don't in-place modifications to date objects (e.g. date.setMonth(1);) get saved?

A. Mongoose currently doesn't watch for in-place updates to date objects. If you have need for this feature, feel free to discuss on this GitHub issue. There are several workarounds:

Q. Why does calling save() multiple times on the same document in parallel only let the first save call succeed and return ParallelSaveErrors for the rest?

A. Due to the asynchronous nature of validation and middleware in general, calling save() multiple times in parallel on the same doc could result in conflicts. For example, validating, and then subsequently invalidating the same path.

Q. Why is any 12 character string successfully cast to an ObjectId?

A. Technically, any 12 character string is a valid ObjectId. Consider using a regex like /^[a-f0-9]{24}$/ to test whether a string is exactly 24 hex characters.

Q. Why do keys in Mongoose Maps have to be strings?

A. Because the Map eventually gets stored in MongoDB where the keys must be strings.

Q. I am using Model.find(...).populate(...) with the limit option, but getting fewer results than the limit. What gives?

A. In order to avoid executing a separate query for each document returned from the find query, Mongoose instead queries using (numDocuments * limit) as the limit. If you need the correct limit, you should use the perDocumentLimit option (new in Mongoose 5.9.0). Just keep in mind that populate() will execute a separate query for each document.

Q. My query/update seems to execute twice. Why is this happening?

A. The most common cause of duplicate queries is mixing callbacks and promises with queries. That's because passing a callback to a query function, like find() or updateOne(), immediately executes the query, and calling then() executes the query again.

Mixing promises and callbacks can lead to duplicate entries in arrays. For example, the below code inserts 2 entries into the tags array, *not just 1.

If you'd like to contribute to this page, please visit it on github and use the Edit button to send a pull request.

**Examples:**

Example 1 (javascript):
```javascript
doc.array[3] = 'changed';
doc.save();
```

Example 2 (javascript):
```javascript
// Saves changes successfully
doc.array.set(3, 'changed');
doc.save();

// Also saves changes successfully
doc.array[3] = 'changed';
doc.markModified('array');
doc.save();
```

Example 3 (javascript):
```javascript
// Saves changes successfully without `markModified()`, because this
// code doesn't set an array index, it sets a path underneath an array index.
doc.docArray[3].name = 'changed';
doc.save();

// Does **not** save changes successfully. You need to use `markModified()`
// or `set()` because this sets an array index.
doc.docArray[3] = { name: 'changed' };
doc.save();
```

Example 4 (javascript):
```javascript
const schema = new mongoose.Schema({
  name: { type: String, unique: true }
});
const Model = db.model('Test', schema);

Model.create([{ name: 'Val' }, { name: 'Val' }], function(err) {
  console.log(err); // No error, unless index was already built
});
```

---

## Version Support

**URL:** https://mongoosejs.com/docs/version-support.html

**Contents:**
- Version Support
- Mongoose 9 (Current Version)
- Mongoose 8 (Prior Version)
- Mongoose 7 (Legacy)
- Mongoose 6 (Limited Maintenance)
- Mongoose 5 (End of Life)

Released on November 21, 2025, Mongoose 9 is the latest major version. All new features, improvements, and bug fixes are delivered to 8.x—so if you want the best experience, this is the version to use.

Released on October 31, 2023, Mongoose 8 was the latest major version until 9.0.0 was released.All new features, improvements, and bug fixes are still delivered to 8.x until at least February 1, 2026.

Mongoose 7.x was released on February 27, 2023 and is now in legacy support.It still receives important fixes when needed, but it’s no longer the primary development focus.

Released on August 24, 2021, Mongoose 6.x now only receives security patches and requested bug fixes.If you find an issue that needs to be backported, you can open a request on GitHub.

End of Life: February 1, 2027 After this date, Mongoose 6 will no longer receive any updates—not even security fixes.

Mongoose 5.x, released January 17, 2018, reached End-of-Life on March 1, 2024.It’s no longer maintained or updated.

---

## How to Check Your Mongoose Version

**URL:** https://mongoosejs.com/docs/check-version.html

**Contents:**
- How to Check Your Mongoose Version
- Using npm list

To check what version of Mongoose you are using in Node.js, print out the mongoose.version property as follows.

We recommend printing the Mongoose version from Node.js, because that better handles cases where you have multiple versions of Mongoose installed. You can also execute the above logic from your terminal using Node.js' -e flag as follows.

You can also get the installed version of the Mongoose npm package using npm list.

npm list is helpful because it can identify if you have multiple versions of Mongoose installed.

Other package managers also support similar functions:

**Examples:**

Example 1 (javascript):
```javascript
const mongoose = require('mongoose');

console.log(mongoose.version); // '7.x.x'
```

Example 2 (javascript):
```javascript
# Prints current Mongoose version, e.g. 7.0.3
node -e "console.log(require('mongoose').version)"
```

Example 3 (python):
```python
$ npm list mongoose
test@ /path/to/test
└── mongoose@7.0.3
```

---

## Using Mongoose with Lodash

**URL:** https://mongoosejs.com/docs/lodash.html

**Contents:**
- Using Mongoose with Lodash
- cloneDeep()

For the most part, Mongoose works well with Lodash. However, there are a few caveats that you should know about.

You should not use Lodash's cloneDeep() function on any Mongoose objects. This includes connections, model classes, and queries, but is especially important for documents. For example, you may be tempted to do the following:

However, the above code will throw the following error if MyModel has any array properties.

This is because Lodash's cloneDeep() function doesn't handle proxies, and Mongoose arrays are proxies as of Mongoose 6. You typically don't have to deep clone Mongoose documents, but, if you have to, use the following alternative to cloneDeep():

**Examples:**

Example 1 (javascript):
```javascript
const _ = require('lodash');

const doc = await MyModel.findOne();

const newDoc = _.cloneDeep(doc);
newDoc.myProperty = 'test';
await newDoc.save();
```

Example 2 (javascript):
```javascript
const doc = await MyModel.findOne();

const newDoc = new MyModel().init(doc.toObject());
newDoc.myProperty = 'test';
await newDoc.save();
```

---

## Change Streams

**URL:** https://mongoosejs.com/docs/change-streams.html

**Contents:**
- Change Streams
- Iterating using next()

Change streams let you listen for updates to documents in a given model's collection, or even documents in an entire database. Unlike middleware, change streams are a MongoDB server construct, which means they pick up changes from anywhere. Even if you update a document from a MongoDB GUI, your Mongoose change stream will be notified.

The watch() function creates a change stream. Change streams emit a 'data' event when a document is updated.

The above script will print output that looks like:

Note that you must be connected to a MongoDB replica set or sharded cluster to use change streams. If you try to call watch() when connected to a standalone MongoDB server, you'll get the below error.

If you're using watch() in production, we recommend using MongoDB Atlas. For local development, we recommend mongodb-memory-server or run-rs to start a replica set locally.

If you want to iterate through a change stream in a AWS Lambda function, do not use event emitters to listen to the change stream. You need to make sure you close your change stream when your Lambda function is done executing, because your change stream may end up in an inconsistent state if Lambda stops your container while the change stream is pulling data from MongoDB.

Change streams also have a next() function that lets you explicitly wait for the next change to come in. Use resumeAfter to track where the last change stream left off, and add a timeout to make sure your handler doesn't wait forever if no changes come in.

**Examples:**

Example 1 (javascript):
```javascript
const Person = mongoose.model('Person', new mongoose.Schema({ name: String }));

// Create a change stream. The 'change' event gets emitted when there's a
// change in the database. Print what the change stream emits.
Person.watch().
  on('change', data => console.log(data));

// Insert a doc, will trigger the change stream handler above
await Person.create({ name: 'Axl Rose' });
```

Example 2 (json):
```json
{
  _id: {
    _data: '8262408DAC000000012B022C0100296E5A10042890851837DB4792BE6B235E8B85489F46645F6964006462408DAC6F5C42FF5EE087A20004'
  },
  operationType: 'insert',
  clusterTime: new Timestamp({ t: 1648397740, i: 1 }),
  fullDocument: {
    _id: new ObjectId("62408dac6f5c42ff5ee087a2"),
    name: 'Axl Rose',
    __v: 0
  },
  ns: { db: 'test', coll: 'people' },
  documentKey: { _id: new ObjectId("62408dac6f5c42ff5ee087a2") }
}
```

Example 3 (javascript):
```javascript
let resumeAfter = undefined;

exports.handler = async(event, context) => {
  // add this so that we can re-use any static/global variables between function calls if Lambda
  // happens to re-use existing containers for the invocation.
  context.callbackWaitsForEmptyEventLoop = false;

  await connectToDatabase();

  const changeStream = await Country.watch([], { resumeAfter });

  // Change stream `next()` will wait forever if there are no changes. So make sure to
  // stop listening to the change stream after a fixed period of time.
  const timeoutPromise = new Promise(resolve => setTimeout(() => resolve(false), 1000));
  let doc = null;
  while (doc = await Promise.race([changeStream.next(), timeoutPromise])) {
    console.log('Got', doc);
  }

  // `resumeToken` tells you where the change stream left off, so next function instance
  // can pick up any changes that happened in the meantime.
  resumeAfter = changeStream.resumeToken;
  await changeStream.close();
};
```

---

## Mongoose for Enterprise

**URL:** https://mongoosejs.com/docs/enterprise.html

**Contents:**
- Mongoose for Enterprise
- Available as part of the Tidelift Subscription
- Enterprise-ready open source software—managed for you

Tidelift is working with the maintainers of Mongoose and thousands of other open source projects to deliver commercial support and maintenance for the open source dependencies you use to build your applications. Save time, reduce risk, and improve code health, while paying the maintainers of the exact dependencies you use.

The Tidelift Subscription is a managed open source subscription for application dependencies covering millions of open source projects across JavaScript, Python, Java, PHP, Ruby, .NET, and more.

Your subscription includes:

Tidelift’s security response team coordinates patches for new breaking security vulnerabilities and alerts immediately through a private channel, so your software supply chain is always secure.

Licensing verification and indemnification

Tidelift verifies license information to enable easy policy enforcement and adds intellectual property indemnification to cover creators and users in case something goes wrong. You always have a 100% up-to-date bill of materials for your dependencies to share with your legal team, customers, or partners.

Maintenance and code improvement

Tidelift ensures the software you rely on keeps working as long as you need it to work. Your managed dependencies are actively maintained and we recruit additional maintainers where required.

Package selection and version guidance

We help you choose the best open source packages from the start—and then guide you through updates to stay on the best releases as new issues arise.

Take a seat at the table with the creators behind the software you use. Tidelift’s participating maintainers earn more income as their software is used by more subscribers, so they’re interested in knowing what you need.

Tooling and cloud integration

Tidelift works with GitHub, GitLab, BitBucket, and more. We support every cloud platform (and other deployment targets, too).

The end result? All of the capabilities you expect from commercial-grade software, for the full breadth of open source you use. That means less time grappling with esoteric open source trivia, and more time building your own applications—and your business.

---

## Migrating from 4.x to 5.x

**URL:** https://mongoosejs.com/docs/migrating_to_5.html

**Contents:**
- Migrating from 4.x to 5.x
- Version Requirements
- Query Middleware
- Promises and Callbacks for mongoose.connect()
- Connection Logic and useMongoClient
- Setter Order
- Checking if a path is populated
- Return Values for remove() and deleteX()
- Aggregation Cursors
- geoNear

Please note: we plan to discontinue Mongoose 5 support on March 1, 2024. Please see our version support guide.

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 4.x to Mongoose 5.x.

If you're still on Mongoose 3.x, please read the Mongoose 3.x to 4.x migration guide.

Mongoose now requires Node.js >= 4.0.0 and MongoDB >= 3.0.0. MongoDB 2.6 and Node.js < 4 where both EOL-ed in 2016.

Query middleware is now compiled when you call mongoose.model() or db.model(). If you add query middleware after calling mongoose.model(), that middleware will not get called.

mongoose.connect() and mongoose.disconnect() now return a promise if no callback specified, or null otherwise. It does not return the mongoose singleton.

The useMongoClient option was removed in Mongoose 5, it is now always true. As a consequence, Mongoose 5 no longer supports several function signatures for mongoose.connect() that worked in Mongoose 4.x if the useMongoClient option was off. Below are some examples of mongoose.connect() calls that do not work in Mongoose 5.x.

In Mongoose 5.x, the first parameter to mongoose.connect() and mongoose.createConnection(), if specified, must be a MongoDB connection string. The connection string and options are then passed down to the MongoDB Node.js driver's MongoClient.connect() function. Mongoose does not modify the connection string, although mongoose.connect() and mongoose.createConnection() support a few additional options in addition to the ones the MongoDB driver supports.

Setters run in reverse order in 4.x:

In 5.x, setters run in the order they're declared.

Mongoose 5.1.0 introduced an _id getter to ObjectIds that lets you get an ObjectId regardless of whether a path is populated.

As a consequence, checking whether blogPost.author._id is no longer viable as a way to check whether author is populated. Use blogPost.populated('author') != null or blogPost.author instanceof mongoose.Types.ObjectId to check whether author is populated instead.

Note that you can call mongoose.set('objectIdGetter', false) to change this behavior.

deleteOne(), deleteMany(), and remove() now resolve to the result object rather than the full driver WriteOpResult object.

The useMongooseAggCursor option from 4.x is now always on. This is the new syntax for aggregation cursors in mongoose 5:

Model.geoNear() has been removed because the MongoDB driver no longer supports it

Due to changes in the MongoDB driver, connection strings must be URI encoded.

If they are not, connections may fail with an illegal character message.

See a full list of affected characters.

If your app is used by a lot of different connection strings, it's possible that your test cases will pass, but production passwords will fail. Encode all your connection strings to be safe.

If you want to continue to use unencoded connection strings, the easiest fix is to use the mongodb-uri module to parse the connection strings, and then produce the properly encoded versions. You can use a function like this:

The function above is safe to use whether the existing string is already encoded or not.

Domain sockets must be URI encoded. For example:

The options parameter to toObject() and toJSON() merge defaults rather than overwriting them.

aggregate() no longer accepts a spread, you must pass your aggregation pipeline as an array. The below code worked in 4.x:

The above code does not work in 5.x, you must wrap the $match and $skip stages in an array.

By default, mongoose 4 would coerce any value to a boolean without error.

Mongoose 5 only casts the following values to true:

And the following values to false:

All other values will cause a CastError

Casting for update(), updateOne(), updateMany(), replaceOne(), remove(), deleteOne(), and deleteMany() doesn't happen until exec(). This makes it easier for hooks and custom query helpers to modify data, because mongoose won't restructure the data you passed in until after your hooks and query helpers have ran. It also makes it possible to set the overwrite option after passing in an update.

Post hooks now get flow control, which means async post save hooks and child document post save hooks execute before your save() callback.

$pushAll is no longer supported and no longer used internally for save(), since it has been deprecated since MongoDB 2.4. Use $push with $each instead.

The retainKeyOrder option was removed, mongoose will now always retain the same key position when cloning objects. If you have queries or indexes that rely on reverse key order, you will have to change them.

Setters now run on queries by default, and the old runSettersOnQuery option has been removed.

We no longer have a pre-compiled version of mongoose for the browser. If you want to use mongoose schemas in the browser, you need to build your own bundle with browserify/webpack.

The saveErrorIfNotFound option was removed, mongoose will now always error out from save() if the underlying document was not found

init hooks are now fully synchronous and do not receive next() as a parameter.

Document.prototype.init() no longer takes a callback as a parameter. It was always synchronous, just had a callback for legacy reasons.

doc.save() no longer passes numAffected as a 3rd param to its callback.

doc.remove() no longer debounces

getPromiseConstructor() is gone, just use mongoose.Promise.

You cannot pass parameters to the next pre middleware in the chain using next() in mongoose 5.x. In mongoose 4, next('Test') in pre middleware would call the next middleware with 'Test' as a parameter. Mongoose 5.x has removed support for this.

In mongoose 5 the required validator only verifies if the value is an array. That is, it will not fail for empty arrays as it would in mongoose 4.

In mongoose 5 the default debug function uses console.info() to display messages instead of console.error().

In Mongoose 4.x, overwriting a filter property that's a primitive with one that is an object would silently fail. For example, the below code would ignore the where() and be equivalent to Sport.find({ name: 'baseball' })

In Mongoose 5.x, the above code will correctly overwrite 'baseball' with { $ne: 'softball' }

Mongoose 5.x uses version 3.x of the MongoDB Node.js driver. MongoDB driver 3.x changed the format of the result of bulkWrite() calls so there is no longer a top-level nInserted, nModified, etc. property. The new result object structure is described here.

In Mongoose 4.x, the above will print:

In Mongoose 5.x, the script will print:

The most recent versions of the MongoDB Node.js driver use strict SSL validation by default, which may lead to errors if you're using self-signed certificates.

If this is blocking you from upgrading, you can set the tlsInsecure option to true.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({ name: String });
const MyModel = mongoose.model('Test', schema);
schema.pre('find', () => { console.log('find!'); });

MyModel.find().exec(function() {
  // In mongoose 4.x, the above `.find()` will print "find!"
  // In mongoose 5.x, "find!" will **not** be printed.
  // Call `pre('find')` **before** calling `mongoose.model()` to make the middleware apply.
});
```

Example 2 (json):
```json
// Worked in mongoose 4. Does **not** work in mongoose 5, `mongoose.connect()`
// now returns a promise consistently. This is to avoid the horrible things
// we've done to allow mongoose to be a thenable that resolves to itself.
mongoose.connect('mongodb://127.0.0.1:27017/test').model('Test', new Schema({}));

// Do this instead
mongoose.connect('mongodb://127.0.0.1:27017/test');
mongoose.model('Test', new Schema({}));
```

Example 3 (javascript):
```javascript
const schema = new Schema({ name: String });
schema.path('name').
  set(() => console.log('This will print 2nd')).
  set(() => console.log('This will print first'));
```

Example 4 (javascript):
```javascript
const schema = new Schema({ name: String });
schema.path('name').
  set(() => console.log('This will print first')).
  set(() => console.log('This will print 2nd'));
```

---

## Getters/Setters in Mongoose

**URL:** https://mongoosejs.com/docs/tutorials/getters-setters.html

**Contents:**
- Getters/Setters in Mongoose
- Getters
- Setters
- Passing Parameters using $locals
- Differences vs ES6 Getters/Setters

Mongoose getters and setters allow you to execute custom logic when getting or setting a property on a Mongoose document. Getters let you transform data in MongoDB into a more user friendly form, and setters let you transform user data before it gets to MongoDB.

Suppose you have a User collection and you want to obfuscate user emails to protect your users' privacy. Below is a basic userSchema that obfuscates the user's email address.

Keep in mind that getters do not impact the underlying data stored in MongoDB. If you save user, the email property will be 'ab@gmail.com' in the database.

By default, Mongoose does not execute getters when converting a document to JSON, including Express' res.json() function.

To run getters when converting a document to JSON, set the toJSON.getters option to true in your schema as shown below.

To skip getters on a one-off basis, use user.get() with the getters option set to false as shown below.

Suppose you want to make sure all user emails in your database are lowercased to make it easy to search without worrying about case. Below is an example userSchema that ensures emails are lowercased.

Mongoose also runs setters on update operations, like updateOne(). Mongoose will upsert a document with a lowercased email in the below example.

In a setter function, this can be either the document being set or the query being run. If you don't want your setter to run when you call updateOne(), you add an if statement that checks if this is a Mongoose document as shown below.

You can't pass parameters to your getter and setter functions like you do to normal function calls. To configure or pass additional properties to your getters and setters, you can use the document's $locals property.

The $locals property is the preferred place to store any program-defined data on your document without conflicting with schema-defined properties. In your getter and setter functions, this is the document being accessed, so you set properties on $locals and then access those properties in your getters examples. For example, the following shows how you can use $locals to configure the language for a custom getter that returns a string in different languages.

Mongoose setters are different from ES6 setters because they allow you to transform the value being set. With ES6 setters, you would need to store an internal _email property to use a setter. With Mongoose, you do not need to define an internal _email property or define a corresponding getter for email.

**Examples:**

Example 1 (javascript):
```javascript
const userSchema = new Schema({
  email: {
    type: String,
    get: obfuscate
  }
});

// Mongoose passes the raw value in MongoDB `email` to the getter
function obfuscate(email) {
  const separatorIndex = email.indexOf('@');
  if (separatorIndex < 3) {
    // 'ab@gmail.com' -> '**@gmail.com'
    return email.slice(0, separatorIndex).replace(/./g, '*') +
      email.slice(separatorIndex);
  }
  // 'test42@gmail.com' -> 'te****@gmail.com'
  return email.slice(0, 2) +
    email.slice(2, separatorIndex).replace(/./g, '*') +
    email.slice(separatorIndex);
}

const User = mongoose.model('User', userSchema);
const user = new User({ email: 'ab@gmail.com' });
user.email; // **@gmail.com
```

Example 2 (css):
```css
app.get(function(req, res) {
  return User.findOne().
    // The `email` getter will NOT run here
    then(doc => res.json(doc)).
    catch(err => res.status(500).json({ message: err.message }));
});
```

Example 3 (javascript):
```javascript
const userSchema = new Schema({
  email: {
    type: String,
    get: obfuscate
  }
}, { toJSON: { getters: true } });

// Or, globally
mongoose.set('toJSON', { getters: true });

// Or, on a one-off basis
app.get(function(req, res) {
  return User.findOne().
    // The `email` getter will run here
    then(doc => res.json(doc.toJSON({ getters: true }))).
    catch(err => res.status(500).json({ message: err.message }));
});
```

Example 4 (css):
```css
user.get('email', null, { getters: false }); // 'ab@gmail.com'
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/7.x/

---

## MongoDB Server Version Compatibility

**URL:** https://mongoosejs.com/docs/7.x/docs/compatibility.html

**Contents:**
- MongoDB Server Version Compatibility

Mongoose relies on the MongoDB Node.js Driver to talk to MongoDB. You can refer to this table for up-to-date information as to which version of the MongoDB driver supports which version of MongoDB.

Below are the semver ranges representing which versions of mongoose are compatible with the listed versions of MongoDB server.

Mongoose ^6.5.0 also works with MongoDB server 7.x. But not all new MongoDB server 7.x features are supported by Mongoose 6.x.

Note that Mongoose 5.x dropped support for all versions of MongoDB before 3.0.0. If you need to use MongoDB 2.6 or older, use Mongoose 4.x.

---

## Migrating from 5.x to 6.x

**URL:** https://mongoosejs.com/docs/migrating_to_6.html

**Contents:**
- Migrating from 5.x to 6.x
- Version Requirements
- MongoDB Driver 4.0
- No More Deprecation Warning Options
- The asPromise() Method for Connections
- mongoose.connect() Returns a Promise
- Duplicate Query Execution
- Model.exists(...) now returns a lean document instead of boolean
- strictQuery is now equal to strict by default
- MongoError is now MongoServerError

Please note: we plan to discontinue Mongoose 5 support on March 1, 2024. Please see our version support guide.

There are several backwards-breaking changes you should be aware of when migrating from Mongoose 5.x to Mongoose 6.x.

If you're still on Mongoose 4.x, please read the Mongoose 4.x to 5.x migration guide and upgrade to Mongoose 5.x first.

Mongoose now requires Node.js >= 12.0.0. Mongoose still supports MongoDB server versions back to 3.0.0.

Mongoose now uses v4.x of the MongoDB Node driver. See the MongoDB Node drivers' migration guide for detailed info. Below are some of the most noteworthy changes:

useNewUrlParser, useUnifiedTopology, useFindAndModify, and useCreateIndex are no longer supported options. Mongoose 6 always behaves as if useNewUrlParser, useUnifiedTopology, and useCreateIndex are true, and useFindAndModify is false. Please remove these options from your code.

Mongoose connections are no longer thenable. This means that await mongoose.createConnection(uri) no longer waits for Mongoose to connect. Use mongoose.createConnection(uri).asPromise() instead. See #8810.

The mongoose.connect() function now always returns a promise, not a Mongoose instance.

Mongoose no longer allows executing the same query object twice. If you do, you'll get a Query was already executed error. Executing the same query instance twice is typically indicative of mixing callbacks and promises, but if you need to execute the same query twice, you can call Query#clone() to clone the query and re-execute it. See gh-7398

Mongoose no longer supports a strictQuery option. You must now use strict. As of Mongoose 6.0.10, we brought back the strictQuery option. In Mongoose 6, strictQuery is set to strict by default. This means that, by default, Mongoose will filter out query filter properties that are not in the schema.

However, this behavior was a source of confusion in some cases, so in Mongoose 7, this default changes back to false. So if you want to retain the default behavior of Mongoose 5 as well as Mongoose 7 and later, you can also disable strictQuery globally to override:

In a test suite, it may be useful to set strictQuery to throw, which will throw exceptions any time a query references schema that doesn't exist, which could help identify a bug in your tests or code.

Here's an example of the effect of strictQuery:

You can also disable strictQuery globally to override:

In MongoDB Node.js Driver v4.x, 'MongoError' is now 'MongoServerError'. Please change any code that depends on the hardcoded string 'MongoError'.

Mongoose now clones discriminator schemas by default. This means you need to pass { clone: false } to discriminator() if you're using recursive embedded discriminators.

In Mongoose 5, mongoose.isValidObjectId() returned false for values like numbers, which was inconsistent with the MongoDB driver's ObjectId.isValid() function. Technically, any JavaScript number can be converted to a MongoDB ObjectId.

In Mongoose 6, mongoose.isValidObjectId() is just a wrapper for mongoose.Types.ObjectId.isValid() for consistency.

Mongoose 6.2.5 now includes a mongoose.isObjectIdOrHexString() function, which does a better job of capturing the more common use case for isValidObjectId(): is the given value an ObjectId instance or a 24 character hex string representing an ObjectId?

Mongoose now saves objects with keys in the order the keys are specified in the schema, not in the user-defined object. So whether Object.keys(new User({ name: String, email: String }).toObject() is ['name', 'email'] or ['email', 'name'] depends on the order name and email are defined in your schema.

Mongoose 6 introduces a new sanitizeFilter option to globals and queries that defends against query selector injection attacks. If you enable sanitizeFilter, Mongoose will wrap any object in the query filter in a $eq:

To explicitly allow a query selector, use mongoose.trusted():

In Mongoose 5.x, setting a key to undefined in an update operation was equivalent to setting it to null.

Mongoose 5.x supported an omitUndefined option to strip out undefined keys. In Mongoose 6.x, the omitUndefined option has been removed, and Mongoose will always strip out undefined keys.

The only workaround is to explicitly set properties to null in your updates:

Mongoose now passes the document as the first parameter to default functions, which is helpful for using arrow functions with defaults.

This may affect you if you pass a function that expects different parameters to default, like default: mongoose.Types.ObjectId. See gh-9633. If you're passing a default function that does not utilize the document, change default: myFunction to default: () => myFunction() to avoid accidentally passing parameters that potentially change the behavior.

Mongoose arrays are now ES6 proxies. You no longer need to markModified() after setting an array index directly.

Schema paths declared with type: { name: String } become single nested subdocs in Mongoose 6, as opposed to Mixed in Mongoose 5. This removes the need for the typePojoToMixed option. See gh-7181.

Mongoose now throws an error if you populate() a path that isn't defined in your schema. This is only for cases when we can infer the local schema, like when you use Query#populate(), not when you call Model.populate() on a POJO. See gh-5124.

When populating a subdocument with a function ref or refPath, this is now the subdocument being populated, not the top-level document. See #8469.

Using save, isNew, and other Mongoose reserved names as schema path names now triggers a warning, not an error. You can suppress the warning by setting the suppressReservedKeysWarning in your schema options: new Schema({ save: String }, { suppressReservedKeysWarning: true }). Keep in mind that this may break plugins that rely on these reserved names.

Single nested subdocs have been renamed to "subdocument paths". So SchemaSingleNestedOptions is now SchemaSubdocumentOptions and mongoose.Schema.Types.Embedded is now mongoose.Schema.Types.Subdocument. See gh-10419

Aggregate#cursor() now returns an AggregationCursor instance to be consistent with Query#cursor(). You no longer need to do Model.aggregate(pipeline).cursor().exec() to get an aggregation cursor, just Model.aggregate(pipeline).cursor().

autoCreate is true by default unless readPreference is secondary or secondaryPreferred, which means Mongoose will attempt to create every model's underlying collection before creating indexes. If readPreference is secondary or secondaryPreferred, Mongoose will default to false for both autoCreate and autoIndex because both createCollection() and createIndex() will fail when connected to a secondary.

The context option for queries has been removed. Now Mongoose always uses context = 'query'.

Mongoose 6 always calls validators with depopulated paths (that is, with the id rather than the document itself). In Mongoose 5, Mongoose would call validators with the populated doc if the path was populated. See #8042

When connected to a replica set, connections now emit 'disconnected' when connection to the primary is lost. In Mongoose 5, connections only emitted 'disconnected' when losing connection to all members of the replica set.

However, Mongoose 6 does not buffer commands while a connection is disconnected. So you can still successfully execute commands like queries with readPreference = 'secondary', even if the Mongoose connection is in the disconnected state.

Document#populate() now returns a promise and is now no longer chainable.

Replace await doc.populate('path1').populate('path2').execPopulate(); with await doc.populate(['path1', 'path2']);

Replace await doc.populate('path1', 'select1').populate('path2', 'select2').execPopulate(); with

await Model.create([]) in v6.0 returns an empty array when provided an empty array, in v5.0 it used to return undefined. If any of your code is checking whether the output is undefined or not, you need to modify it with the assumption that await Model.create(...) will always return an array if provided an array.

doc.set({ child: { age: 21 } }) now works the same whether child is a nested path or a subdocument: Mongoose will overwrite the value of child. In Mongoose 5, this operation would merge child if child was a nested path.

Mongoose now adds a valueOf() function to ObjectIds. This means you can now use == to compare an ObjectId against a string.

If you set timestamps: true, Mongoose will now make the createdAt property immutable. See gh-10139

isAsync is no longer an option for validate. Use an async function instead.

safe is no longer an option for schemas, queries, or save(). Use writeConcern instead.

Mongoose now calls setter functions with priorValue as the 2nd parameter, rather than schemaType in Mongoose 5.

This change was technically released with 5.10.5, but caused issues for users migrating from 5.9.x to 6.x. In Mongoose < 5.10.5, toObject() and toJSON() would use the top-level schema's minimize option by default.

As a workaround, you can either explicitly pass minimize to toObject() or toJSON():

Or define the child schema inline (Mongoose 6 only) to inherit the parent's minimize option.

In Mongoose 5, calling populate() on a mixed type or other path with no ref would fall back to using the query's model.

In Mongoose 6, populating a path with no ref, refPath, or model is a no-op.

The MongoDB Node driver version that Mongoose 6 uses relies on a URL parser module that has several known compatibility issues with other npm packages. This can lead to errors like Invalid URL: mongodb+srv://username:password@development.xyz.mongodb.net/abc if you use one of the incompatible packages. You can find a list of incompatible packages here.

The reconnectTries and reconnectInterval options have been removed since they are no longer necessary.

The MongoDB node driver will always attempt to retry any operation for up to serverSelectionTimeoutMS, even if MongoDB is down for a long period of time. So, it will never run out of retries or try to reconnect to MongoDB.

Lodash's isEmpty() function returns true for primitives and primitive wrappers. ObjectId() is an object wrapper that is treated as a primitive by Mongoose. But starting in Mongoose 6, _.isEmpty() will return true for ObjectIds because of Lodash implementation details.

An ObjectId in mongoose is never empty, so if you're using isEmpty() you should check for instanceof ObjectId.

The mongoose.modelSchemas property was removed. This may have been used to delete a model schema.

The Schema class now takes 3 generic params instead of 4. The 3rd generic param, SchemaDefinitionType, is now the same as the 1st generic param DocType. Replace new Schema<UserDocument, UserModel, User>(schemaDefinition) with new Schema<UserDocument, UserModel>(schemaDefinition)

Types.ObjectId is now a class, which means you can no longer omit new when creating a new ObjectId using new mongoose.Types.ObjectId(). Currently, you can still omit new in JavaScript, but you must put new in TypeScript.

The following legacy types have been removed:

Mongoose 6 infers the document's type for this in virtual getters and setters. In Mongoose 5.x, this would be any in the following code.

In Mongoose 6, this will be set to the document type.

**Examples:**

Example 1 (javascript):
```javascript
const res = await TestModel.updateMany({}, { someProperty: 'someValue' });

res.matchedCount; // Number of documents that were found that match the filter. Replaces `res.n`
res.modifiedCount; // Number of documents modified. Replaces `res.nModified`
res.upsertedCount; // Number of documents upserted. Replaces `res.upserted`
```

Example 2 (javascript):
```javascript
const res = await TestModel.deleteMany({});

// In Mongoose 6: `{ acknowledged: true, deletedCount: 2 }`
// In Mongoose 5: `{ n: 2, ok: 1, deletedCount: 2 }`
res;

res.deletedCount; // Number of documents that were deleted. Replaces `res.n`
```

Example 3 (r):
```r
// No longer necessary:
mongoose.set('useFindAndModify', false);

await mongoose.connect('mongodb://127.0.0.1:27017/test', {
  useNewUrlParser: true, // <-- no longer necessary
  useUnifiedTopology: true // <-- no longer necessary
});
```

Example 4 (swift):
```swift
// The below no longer works in Mongoose 6
await mongoose.createConnection(uri);

// Do this instead
await mongoose.createConnection(uri).asPromise();
```

---

## Mongoose for Enterprise

**URL:** https://mongoosejs.com/docs/6.x/docs/enterprise.html

**Contents:**
- Mongoose for Enterprise
- Available as part of the Tidelift Subscription
- Enterprise-ready open source software—managed for you

Tidelift is working with the maintainers of Mongoose and thousands of other open source projects to deliver commercial support and maintenance for the open source dependencies you use to build your applications. Save time, reduce risk, and improve code health, while paying the maintainers of the exact dependencies you use.

The Tidelift Subscription is a managed open source subscription for application dependencies covering millions of open source projects across JavaScript, Python, Java, PHP, Ruby, .NET, and more.

Your subscription includes:

Tidelift’s security response team coordinates patches for new breaking security vulnerabilities and alerts immediately through a private channel, so your software supply chain is always secure.

Licensing verification and indemnification

Tidelift verifies license information to enable easy policy enforcement and adds intellectual property indemnification to cover creators and users in case something goes wrong. You always have a 100% up-to-date bill of materials for your dependencies to share with your legal team, customers, or partners.

Maintenance and code improvement

Tidelift ensures the software you rely on keeps working as long as you need it to work. Your managed dependencies are actively maintained and we recruit additional maintainers where required.

Package selection and version guidance

We help you choose the best open source packages from the start—and then guide you through updates to stay on the best releases as new issues arise.

Take a seat at the table with the creators behind the software you use. Tidelift’s participating maintainers earn more income as their software is used by more subscribers, so they’re interested in knowing what you need.

Tooling and cloud integration

Tidelift works with GitHub, GitLab, BitBucket, and more. We support every cloud platform (and other deployment targets, too).

The end result? All of the capabilities you expect from commercial-grade software, for the full breadth of open source you use. That means less time grappling with esoteric open source trivia, and more time building your own applications—and your business.

---

## Working With Dates

**URL:** https://mongoosejs.com/docs/tutorials/dates.html

**Contents:**
- Working With Dates
- Validators
- Querying
- Casting Edge Cases
- Timezones

Here's how you declare a path of type Date with a Mongoose schema:

When you create a user document, Mongoose will cast the value to a native JavaScript date using the Date() constructor.

An invalid date will lead to a CastError when you validate the document.

Dates have two built-in validators: min and max. These validators will report a ValidatorError if the given date is strictly less than min or strictly greater than max.

MongoDB supports querying by date ranges and sorting by dates. Here's some examples of querying by dates, date ranges, and sorting by date:

Date casting has a couple small cases where it differs from JavaScript's native date parsing. First, Mongoose looks for a valueOf() function on the given object, and calls valueOf() before casting the date. This means Mongoose can cast moment objects to dates automatically.

By default, if you pass a numeric string to the Date constructor, JavaScript will attempt to convert it to a year.

Mongoose converts numeric strings that contain numbers outside the range of representable dates in JavaScript and converts them to numbers before passing them to the date constructor.

MongoDB stores dates as 64-bit integers, which means that Mongoose does not store timezone information by default. When you call Date#toString(), the JavaScript runtime will use your OS' timezone.

**Examples:**

Example 1 (javascript):
```javascript
const mongoose = require('mongoose');

const userSchema = new mongoose.Schema({
  name: String,
  // `lastActiveAt` is a date
  lastActiveAt: Date
});
const User = mongoose.model('User', userSchema);
```

Example 2 (css):
```css
const user = new User({
  name: 'Jean-Luc Picard',
  lastActiveAt: '2002-12-09'
});
user.lastActiveAt instanceof Date; // true
```

Example 3 (css):
```css
const user = new User({
  name: 'Jean-Luc Picard',
  lastActiveAt: 'not a date'
});
user.lastActiveAt instanceof Date; // false
user.validateSync().errors['lastActiveAt']; // CastError
```

Example 4 (css):
```css
const episodeSchema = new mongoose.Schema({
  title: String,
  airedAt: {
    type: Date,
    // The dates of the first and last episodes of
    // Star Trek: The Next Generation
    min: '1987-09-28',
    max: '1994-05-23'
  }
});
const Episode = mongoose.model('Episode', episodeSchema);

const ok = new Episode({
  title: 'Encounter at Farpoint',
  airedAt: '1987-09-28'
});
ok.validateSync(); // No error

const bad = new Episode({
  title: 'What You Leave Behind',
  airedAt: '1999-06-02'
});
bad.airedAt; // "1999-06-02T00:00:00.000Z"

// Path `airedAt` (Tue Jun 01 1999 20:00:00 GMT-0400 (EDT)) is after
// maximum allowed value (Sun May 22 1994 20:00:00 GMT-0400 (EDT)).
bad.validateSync();
```

---

## Mongoose

**URL:** https://mongoosejs.com/docs/8.x/

---

## Integrating with MongoDB Client Side Field Level Encryption

**URL:** https://mongoosejs.com/docs/field-level-encryption

**Contents:**
- Integrating with MongoDB Client Side Field Level Encryption
- Automatic FLE in Mongoose
  - Encryption types
  - Declaring Encrypted Schemas
  - Registering Models
  - Connecting and configuring encryption options
  - Discriminators
- Managing Data Keys
- Manual FLE in Mongoose

Client Side Field Level Encryption, or CSFLE for short, is a tool for storing your data in an encrypted format in MongoDB. For example, instead of storing the name property as a plain-text string, CSFLE means MongoDB will store your document with name as an encrypted buffer. The resulting document will look similar to the following to a client that doesn't have access to decrypt the data.

You can read more about CSFLE on the MongoDB CSFLE documentation and this blog post about CSFLE in Node.js.

Mongoose supports the declaration of encrypted schemas - schemas that, when connected to a model, utilize MongoDB's Client Side Field Level Encryption or Queryable Encryption under the hood. Mongoose automatically generates either an encryptedFieldsMap or a schemaMap when instantiating a MongoClient and encrypts fields on write and decrypts fields on reads.

MongoDB has two different automatic encryption implementations: client side field level encryption (CSFLE) and queryable encryption (QE).See choosing an in-use encryption approach.

The following schema declares two properties, name and ssn. ssn is encrypted using queryable encryption, and is configured for equality queries:

To declare a field as encrypted, you must:

Not all schematypes are supported for CSFLE and QE. For an overview of supported BSON types, refer to MongoDB's documentation.

Encrypted schemas can be registered on the global mongoose object or on a specific connection, so long as models are registered before the connection is established:

Field level encryption in Mongoose works by generating the encryption schema that the MongoDB driver expects for each encrypted model on the connection. This happens automatically when the model's connection is established.

Queryable encryption and CSFLE require all the same configuration as outlined in the MongoDB encryption in-use documentation, except for the schemaMap or encryptedFieldsMap options.

Once the connection is established, Mongoose's operations will work as usual. Writes are encrypted automatically by the MongoDB driver prior to sending them to the server and reads are decrypted by the driver after fetching documents from the server.

Discriminators are supported for encrypted models as well:

When generating encryption schemas, Mongoose merges all discriminators together for all of the discriminators declared on the same namespace. As a result, discriminators that declare the same key with different types are not supported. Furthermore, all discriminators for the same namespace must share the same encryption type - it is not possible to configure discriminators on the same model for both CSFLE and Queryable Encryption.

Mongoose provides a convenient API to obtain a ClientEncryption object configured to manage data keys in the key vault. A client encryption can be obtained with the Model.clientEncryption() helper:

First, you need to install the mongodb-client-encryption npm package. This is MongoDB's official package for setting up encryption keys.

You also need to make sure you've installed mongocryptd. mongocryptd is a separate process from the MongoDB server that you need to run to work with field level encryption. You can either run mongocryptd yourself, or make sure it is on the system PATH and the MongoDB Node.js driver will run it for you. You can read more about mongocryptd here.

Once you've set up and run mongocryptd, first you need to create a new encryption key as follows. Keep in mind that the following example is a simple example to help you get started. The encryption key in the following example is insecure; MongoDB recommends using a KMS.

Once you have an encryption key, you can create a separate Mongoose connection with a schemaMap that defines which fields are encrypted using JSON schema syntax as follows.

With the above connection, if you create a model named 'Test' that uses the 'tests' collection, any documents will have their name property encrypted.

**Examples:**

Example 1 (json):
```json
{
  "_id" : ObjectId("647a3207661e3a3a1bc3e614"),
  "name" : BinData(6,"ASrIv7XfokKwiCUJEjckOdgCG+u6IqavcOWX8hINz29MLvcKDZ4nnjCnPFZG+0ftVxMdWgzu6Vdh7ys1uIK1WiaPN0SqpmmtL2rPoqT9gfhADpGDmI60+vm0bJepXNY1Gv0="),
  "__v" : 0
}
```

Example 2 (jsx):
```jsx
const encryptedUserSchema = new Schema({ 
  name: String,
  ssn: { 
    type: String, 
    // 1
    encrypt: { 
      keyId: '<uuid string of key id>',
      queries: 'equality'
    }
  }
  // 2
}, { encryptionType: 'queryableEncryption' });
```

Example 3 (javascript):
```javascript
// specific connection
const GlobalUserModel = mongoose.model('User', encryptedUserSchema);

// specific connection
const connection = mongoose.createConnection();
const UserModel = connection.model('User', encryptedUserSchema);
```

Example 4 (javascript):
```javascript
const keyVaultNamespace = 'client.encryption';
const kmsProviders = { local: { key } };
await connection.openUri(`mongodb://localhost:27017`, {
  // Configure auto encryption
  autoEncryption: {
    keyVaultNamespace: 'datakeys.datakeys',
    kmsProviders
  }
});
```

---

## Testing Mongoose with Jest

**URL:** https://mongoosejs.com/docs/jest.html

**Contents:**
- Testing Mongoose with Jest
- Recommended testEnvironment
- Timer Mocks
- globalSetup and globalTeardown
- resetModules
- Further Reading

Jest is a JavaScript runtime developed by Facebook that is usually used for testing. Because Jest is designed primarily for testing React applications, using it to test Node.js server-side applications comes with a lot of caveats. We strongly recommend using a different testing framework, like Mocha.

To suppress any Jest warnings from Mongoose, set the SUPPRESS_JEST_WARNINGS environment variable:

If you choose to delve into dangerous waters and test Mongoose apps with Jest, here's what you need to know:

If you are using Jest <=26, do not use Jest's default jsdom test environment when testing Mongoose apps, unless you are explicitly testing an application that only uses Mongoose's browser library. In Jest >=27, "node" is Jest's default testEnvironment, so this is no longer an issue.

The jsdom test environment attempts to create a browser-like test environment in Node.js, and it comes with numerous nasty surprises like a stubbed setTimeout() function that silently fails after tests are finished. Mongoose does not support jsdom in general and is not expected to function correctly in the jsdom test environment.

To change your testEnvironment to Node.js, add testEnvironment to your jest.config.js file:

Absolutely do not use timer mocks when testing Mongoose apps. This is especially important if you're using Jest >=25, which stubs out process.nextTick().

Fake timers stub out global functions like setTimeout() and setInterval(), which causes problems when an underlying library uses these functions. Mongoose and the MongoDB Node.js driver uses these functions for deferring work until the next tick of the event loop and for monitoring connections to the MongoDB server.

If you absolutely must use timer mocks, make sure you import Mongoose before calling useFakeTimers():

Mongoose devs have already refactored out code to avoid using setImmediate() to defer work to the next tick of the event loop, but we can't reasonably ensure that every library Mongoose depends on doesn't use setImmediate().

A better alternative is to create your own wrapper around setTimeout() and stub that instead using sinon.

Do not use globalSetup to call mongoose.connect() or mongoose.createConnection(). Jest runs globalSetup in a separate environment, so you cannot use any connections you create in globalSetup in your tests.

We recommend setting resetModules to false in your Jest config. resetModules: true can cause issues with internal instanceof checks by creating multiple dangling copies of the Mongoose module.

Want to learn how to test Mongoose apps correctly? The RESTful Web Services with Node.js and Express course on Pluralsight has a great section on testing Mongoose apps with Mocha.

**Examples:**

Example 1 (unknown):
```unknown
env SUPPRESS_JEST_WARNINGS=1 npm test
```

Example 2 (css):
```css
module.exports = {
  testEnvironment: 'node'
};
```

Example 3 (javascript):
```javascript
// Fine for basic cases, but may still cause issues:
const mongoose = require('mongoose');

jest.useFakeTimers();

// Bad:
jest.useFakeTimers();

const mongoose = require('mongoose');
```

Example 4 (javascript):
```javascript
// time.js
exports.setTimeout = function() {
  return global.setTimeout.apply(global, arguments);
};

// Tests
const time = require('../util/time');
const sinon = require('sinon');
sinon.stub(time, 'setTimeout');
```

---

## Custom Casting

**URL:** https://mongoosejs.com/docs/tutorials/custom-casting.html

**Contents:**
- Custom Casting

Mongoose 5.4.0 introduced several ways to configure SchemaTypes globally. One of these new features is the SchemaType.cast() function, which enables you to override Mongoose's built-in casting.

For example, by default Mongoose will throw an error if you attempt to cast a string that contains a Japanese numeral to a number.

You can overwrite the default casting function for numbers to allow converting the string that contains the Japanese numeral "2" to a number as shown below.

**Examples:**

Example 1 (css):
```css
const schema = new mongoose.Schema({
  age: Number
});
const Model = mongoose.model('Test', schema);

const doc = new Model({ age: '二' });
const err = doc.validateSync();
// "Cast to Number failed for value "二" at path "age""
err.message;
```

Example 2 (javascript):
```javascript
// Calling `cast()` on a class that inherits from `SchemaType` returns the
// current casting function.
const originalCast = mongoose.Number.cast();

// Calling `cast()` with a function sets the current function used to
// cast a given schema type, in this cast Numbers.
mongoose.Number.cast(v => {
  if (v === '二') {
    return 2;
  }
  return originalCast(v);
});

const schema = new mongoose.Schema({
  age: Number
});

const Model = mongoose.model('Test', schema);

const doc = new Model({ age: '二' });
const err = doc.validateSync();
err; // null
doc.age; // 2
```

---
