# Mongoose - Queries

**Pages:** 13

---

## How to Use findOneAndUpdate() in Mongoose

**URL:** https://mongoosejs.com/docs/tutorials/findoneandupdate.html

**Contents:**
- How to Use findOneAndUpdate() in Mongoose
- Getting Started
- Atomic Updates
- Upsert
- The includeResultMetadata Option
- Updating Discriminator Keys

The findOneAndUpdate() function in Mongoose has a wide variety of use cases. You should use save() to update documents where possible, for better validation and middleware support. However, there are some cases where you need to use findOneAndUpdate(). In this tutorial, you'll see how to use findOneAndUpdate(), and learn when you need to use it.

As the name implies, findOneAndUpdate() finds the first document that matches a given filter, applies an update, and returns the document. The findOneAndUpdate() function has the following signature:

By default, findOneAndUpdate() returns the document as it was before update was applied. In the following example, doc initially only has name and _id properties. findOneAndUpdate() adds an age property, but the result of findOneAndUpdate() does not have an age property.

You should set the new option to true to return the document after update was applied.

Mongoose's findOneAndUpdate() is slightly different from the MongoDB Node.js driver's findOneAndUpdate() because it returns the document itself, not a result object.

As an alternative to the new option, you can also use the returnOriginal option. returnOriginal: false is equivalent to new: true. The returnOriginal option exists for consistency with the the MongoDB Node.js driver's findOneAndUpdate(), which has the same option.

With the exception of an unindexed upsert, findOneAndUpdate() is atomic. That means you can assume the document doesn't change between when MongoDB finds the document and when it updates the document, unless you're doing an upsert.

For example, if you're using save() to update a document, the document can change in MongoDB in between when you load the document using findOne() and when you save the document using save() as show below. For many use cases, the save() race condition is a non-issue. But you can work around it with findOneAndUpdate() (or transactions) if you need to.

Using the upsert option, you can use findOneAndUpdate() as a find-and-upsert operation. An upsert behaves like a normal findOneAndUpdate() if it finds a document that matches filter. But, if no document matches filter, MongoDB will insert one by combining filter and update as shown below.

Mongoose transforms the result of findOneAndUpdate() by default: it returns the updated document. That makes it difficult to check whether a document was upserted or not. In order to get the updated document and check whether MongoDB upserted a new document in the same operation, you can set the includeResultMetadata flag to make Mongoose return the raw result from MongoDB.

Here's what the res object from the above example looks like:

Mongoose prevents updating the discriminator key using findOneAndUpdate() by default. For example, suppose you have the following discriminator models.

Mongoose will remove __t (the default discriminator key) from the update parameter, if __t is set. This is to prevent unintentional updates to the discriminator key; for example, if you're passing untrusted user input to the update parameter. However, you can tell Mongoose to allow updating the discriminator key by setting the overwriteDiscriminatorKey option to true as shown below.

**Examples:**

Example 1 (javascript):
```javascript
function findOneAndUpdate(filter, update, options) {}
```

Example 2 (javascript):
```javascript
const Character = mongoose.model('Character', new mongoose.Schema({
  name: String,
  age: Number
}));

const _id = new mongoose.Types.ObjectId('0'.repeat(24));
let doc = await Character.create({ _id, name: 'Jean-Luc Picard' });
doc; // { name: 'Jean-Luc Picard', _id: ObjectId('000000000000000000000000') }

const filter = { name: 'Jean-Luc Picard' };
const update = { age: 59 };

// The result of `findOneAndUpdate()` is the document _before_ `update` was applied
doc = await Character.findOneAndUpdate(filter, update);
doc; // { name: 'Jean-Luc Picard', _id: ObjectId('000000000000000000000000') }

doc = await Character.findOne(filter);
doc.age; // 59
```

Example 3 (javascript):
```javascript
const filter = { name: 'Jean-Luc Picard' };
const update = { age: 59 };

// `doc` is the document _after_ `update` was applied because of
// `new: true`
const doc = await Character.findOneAndUpdate(filter, update, {
  new: true
});
doc.name; // 'Jean-Luc Picard'
doc.age; // 59
```

Example 4 (javascript):
```javascript
const filter = { name: 'Jean-Luc Picard' };
const update = { age: 59 };

// `doc` is the document _after_ `update` was applied because of
// `returnOriginal: false`
const doc = await Character.findOneAndUpdate(filter, update, {
  returnOriginal: false
});
doc.name; // 'Jean-Luc Picard'
doc.age; // 59
```

---

## Query

**URL:** https://mongoosejs.com/docs/7.x/docs/api/query.html

**Contents:**
- Query
  - Query()
      - Parameters:
    - Example:
  - Query.prototype.$where()
      - Parameters:
      - Returns:
      - See:
    - Example:
    - Note:

Query constructor used for building queries. You do not need to instantiate a Query directly. Instead use Model functions like Model.find().

Specifies a javascript function or expression to pass to MongoDBs query system.

Only use $where when you have a condition that cannot be met using other MongoDB operators like $lt. Be sure to read about all of its caveats before using.

Specifies an $all query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the allowDiskUse option, which allows the MongoDB server to use more than 100 MB for this query's sort(). This option can let you work around QueryExceededMemoryLimitNoDiskUseAllowed errors from the MongoDB server.

Note that this option requires MongoDB server >= 4.4. Setting this option is a no-op for MongoDB 4.2 and earlier.

Calling query.allowDiskUse(v) is equivalent to query.setOptions({ allowDiskUse: v })

Specifies arguments for a $and condition.

Specifies the batchSize option.

Cannot be used with distinct()

Specifies a $box condition

Casts this query to the schema of model

If obj is present, it is cast instead of this query.

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error. Like .then(), but only takes a rejection handler.

More about Promise catch() in JavaScript.

DEPRECATED Alias for circle

Deprecated. Use circle instead.

DEPRECATED Specifies a $centerSphere condition

Deprecated. Use circle instead.

Specifies a $center or $centerSphere condition.

Make a copy of this query so you can re-execute it.

Adds a collation to this op (MongoDB 3.4 and up)

Specifies the comment option.

Cannot be used with distinct()

Specifies this query as a count query.

This method is deprecated. If you want to count the number of documents in a collection, e.g. count({}), use the estimatedDocumentCount() function instead. Otherwise, use the countDocuments() function instead.

This function triggers the following middleware.

Specifies this query as a countDocuments() query. Behaves like count(), except it always does a full collection scan when passed an empty filter {}.

There are also minor differences in how countDocuments() handles $where and a couple geospatial operators. versus count().

This function triggers the following middleware.

The countDocuments() function is similar to count(), but there are a few operators that countDocuments() does not support. Below are the operators that count() supports but countDocuments() does not, and the suggested replacement:

Returns a wrapper around a mongodb driver cursor. A QueryCursor exposes a Streams3 interface, as well as a .next() function.

The .cursor() function triggers pre find hooks, but not post find hooks.

Declare and/or execute this query as a deleteMany() operation. Works like remove, except it deletes every document that matches filter in the collection, regardless of the value of single.

This function triggers deleteMany middleware.

This function calls the MongoDB driver's Collection#deleteMany() function. The returned promise resolves to an object that contains 3 properties:

Declare and/or execute this query as a deleteOne() operation. Works like remove, except it deletes at most one document regardless of the single option.

This function triggers deleteOne middleware.

This function calls the MongoDB driver's Collection#deleteOne() function. The returned promise resolves to an object that contains 3 properties:

Declares or executes a distinct() operation.

This function does not trigger any middleware.

Specifies an $elemMatch condition

Specifies the complementary comparison value for paths specified with where()

Gets/sets the error flag on this query. If this flag is not null or undefined, the exec() promise will reject without executing.

Note that query casting runs after hooks, so cast errors will override custom errors.

Specifies this query as a estimatedDocumentCount() query. Faster than using countDocuments() for large collections because estimatedDocumentCount() uses collection metadata rather than scanning the entire collection.

estimatedDocumentCount() does not accept a filter. Model.find({ foo: bar }).estimatedDocumentCount() is equivalent to Model.find().estimatedDocumentCount()

This function triggers the following middleware.

Specifies an $exists condition

Sets the explain option, which makes this query return detailed execution stats instead of the actual query result. This method is useful for determining what index your queries use.

Calling query.explain(v) is equivalent to query.setOptions({ explain: v })

Executes the query returning a Promise which will be resolved with .finally() chained.

More about Promise finally() in JavaScript.

Find all documents that match selector. The result will be an array of documents.

If there are too many documents in the result to fit in memory, use Query.prototype.cursor()

Declares the query a findOne operation. When executed, the first found document is passed to the callback.

The result of the query is a single document, or null if no document was found.

This function triggers the following middleware.

Issues a MongoDB findOneAndDelete command.

Finds a matching document, removes it, and returns the found document (if any).

This function triggers the following middleware.

Legacy alias for findOneAndDelete().

Finds a matching document, removes it, returns the found document (if any).

This function triggers the following middleware.

Issues a MongoDB findOneAndReplace command.

Finds a matching document, removes it, and returns the found document (if any).

This function triggers the following middleware.

Issues a mongodb findOneAndUpdate() command.

Finds a matching document, updates it according to the update arg, passing any options, and returns the found document (if any).

This function triggers the following middleware.

Specifies a $geometry condition

The argument is assigned to the most recent path passed to where().

geometry() must come after either intersects() or within().

The object argument must contain type and coordinates properties.

For update operations, returns the value of a path in the update's $set. Useful for writing getters/setters that can work with both update operations and save().

Returns the current query filter (also known as conditions) as a POJO.

Gets a list of paths to be populated by this query

Returns the current query filter. Equivalent to getFilter().

You should use getFilter() instead of getQuery() where possible. getQuery() will likely be deprecated in a future release.

Returns the current update operations as a JSON object.

Specifies a $gt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $gte query condition.

When called with one argument, the most recent path passed to where() is used.

Cannot be used with distinct()

Specifies an $in query condition.

When called with one argument, the most recent path passed to where() is used.

Declares an intersects query for geometry().

MUST be used after where().

In Mongoose 3.7, intersects changed from a getter to a function. If you need the old syntax, use this.

Wrapper function to call isPathSelectedInclusive on a query.

Requests acknowledgement that this operation has been persisted to MongoDB's on-disk journal. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.j option

Sets the lean option.

Documents returned from queries with the lean option enabled are plain javascript objects, not Mongoose Documents. They have no save method, getters/setters, virtuals, or other Mongoose features.

Lean is great for high-performance, read-only cases, especially when combined with cursors.

If you need virtuals, getters/setters, or defaults with lean(), you need to use a plugin. See:

Specifies the maximum number of documents the query will return.

Cannot be used with distinct()

Specifies a $lt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $lte query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a maxDistance query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the maxTimeMS option. This will tell the MongoDB server to abort if the query or write op has been running for more than ms milliseconds.

Calling query.maxTimeMS(v) is equivalent to query.setOptions({ maxTimeMS: v })

Merges another Query or conditions object into this one.

When a Query is passed, conditions, field selection and options are merged.

Specifies a $mod condition, filters documents for documents whose path property is a number that is equal to remainder modulo divisor.

The model this query is associated with.

Getter/setter around the current mongoose-specific options for this query Below are the current Mongoose-specific options.

Mongoose maintains a separate object for internal options because Mongoose sends Query.prototype.options to the MongoDB server, and the above options are not relevant for the MongoDB server.

Specifies a $ne query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $near or $nearSphere condition

These operators return documents sorted by distance.

DEPRECATED Specifies a $nearSphere condition

Deprecated. Use query.near() instead with the spherical option set to true.

Specifies an $nin query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies arguments for a $nor condition.

Specifies arguments for an $or condition.

Make this query throw an error if no documents match the given filter. This is handy for integrating with async/await, because orFail() saves you an extra if statement to check if no document was found.

Specifies a $polygon condition

Specifies paths which should be populated with other documents.

Paths are populated after the query executes and a response is received. A separate query is then executed for each path specified for population. After a response for each query has also been returned, the results are passed to the callback.

Add post middleware to this query instance. Doesn't affect other queries.

Add pre middleware to this query instance. Doesn't affect other queries.

Get/set the current projection (AKA fields). Pass null to remove the current projection.

Unlike projection(), the select() function modifies the current projection in place. This function overwrites the existing projection.

Determines the MongoDB nodes from which to read.

Read more about how to use read preferences here.

Sets the readConcern option for the query.

Read more about how to use read concern here.

Specifies a $regex query condition.

When called with one argument, the most recent path passed to where() is used.

Declare and/or execute this query as a replaceOne() operation. MongoDB will replace the existing document and will not accept any atomic operators ($set, etc.)

Note replaceOne will not fire update middleware. Use pre('replaceOne') and post('replaceOne') instead.

This function triggers the following middleware.

Specifies which document fields to include or exclude (also known as the query "projection")

When using string syntax, prefixing a path with - will flag that path as excluded. When a path does not have the - prefix, it is included. Lastly, if a path is prefixed with +, it forces inclusion of the path, which is useful for paths excluded at the schema level.

A projection must be either inclusive or exclusive. In other words, you must either list the fields to include (which excludes all others), or list the fields to exclude (which implies all other fields are included). The _id field is the only exception because MongoDB includes it by default.

Determines if field selection has been made.

Determines if exclusive field selection has been made.

Determines if inclusive field selection has been made.

Sets the MongoDB session associated with this query. Sessions are how you mark a query as part of a transaction.

Calling session(null) removes the session from this query.

Adds a $set to this query's update without changing the operation. This is useful for query middleware so you can add an update regardless of whether you use updateOne(), updateMany(), findOneAndUpdate(), etc.

Sets query options. Some options only make sense for certain operations.

The following options are only for find():

The following options are only for write operations: updateOne(), updateMany(), replaceOne(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for find(), findOne(), findById(), findOneAndUpdate(), findOneAndReplace(), findOneAndDelete(), and findByIdAndUpdate():

The following options are only for all operations except updateOne(), updateMany(), deleteOne(), and deleteMany():

The following options are for find(), findOne(), findOneAndUpdate(), findOneAndRemove(), findOneAndDelete(), updateOne(), and deleteOne():

The following options are for findOneAndUpdate() and findOneAndRemove()

The following options are for all operations:

Sets the query conditions to the provided JSON object.

Sets the current update operation to new value.

Specifies a $size query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the number of documents to skip.

Cannot be used with distinct()

Specifies a $slice projection for an array.

Note: If the absolute value of the number of elements to be sliced is greater than the number of elements in the array, all array elements will be returned.

Note: If the number of elements to skip is positive and greater than the number of elements in the array, an empty array will be returned.

Note: If the number of elements to skip is negative and its absolute value is greater than the number of elements in the array, the starting position is the start of the array.

If an object is passed, values allowed are asc, desc, ascending, descending, 1, and -1.

If a string is passed, it must be a space delimited list of path names. The sort order of each path is ascending unless the path name is prefixed with - which will be treated as descending.

Cannot be used with distinct()

Sets the tailable option (for use with capped collections).

Cannot be used with distinct()

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error.

More about then() in JavaScript.

Converts this query to a customized, reusable query constructor with all arguments and options retained.

Runs a function fn and treats the return value of fn as the new value for the query to resolve to.

Any functions you pass to transform() will run after any post hooks.

Declare and/or execute this query as an updateMany() operation. MongoDB will update all documents that match filter (as opposed to just the first one).

Note updateMany will not fire update middleware. Use pre('updateMany') and post('updateMany') instead.

This function triggers the following middleware.

Declare and/or execute this query as an updateOne() operation. MongoDB will update only the first document that matches filter.

Note updateOne will not fire update middleware. Use pre('updateOne') and post('updateOne') instead.

This function triggers the following middleware.

Sets the specified number of mongod servers, or tag set of mongod servers, that must acknowledge this write before this write is considered successful. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.w option

Specifies a path for use with chaining.

Defines a $within or $geoWithin argument for geo-spatial queries.

MUST be used after where().

As of Mongoose 3.7, $geoWithin is always used for queries. To change this behavior, see Query.use$geoWithin.

In Mongoose 3.7, within changed from a getter to a function. If you need the old syntax, use this.

Sets the 3 write concern parameters for this query:

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern option

If w > 1, the maximum amount of time to wait for this write to propagate through the replica set before this operation fails. The default is 0, which means no timeout.

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.wtimeout option

Returns an asyncIterator for use with for/await/of loops This function only works for find() queries. You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Node.js 10.x supports async iterators natively without any flags. You can enable async iterators in Node.js 8.x using the --harmony_async_iteration flag.

Note: This function is not if Symbol.asyncIterator is undefined. If Symbol.asyncIterator is undefined, that means your Node.js version does not support async iterators.

Returns a string representation of this query.

More about toString() in JavaScript.

Flag to opt out of using $geoWithin.

MongoDB 2.4 deprecated the use of $within, replacing it with $geoWithin. Mongoose uses $geoWithin by default (which is 100% backward compatible with $within). If you are running an older version of MongoDB, set this flag to false so your within() queries continue to work.

**Examples:**

Example 1 (css):
```css
const query = MyModel.find(); // `query` is an instance of `Query`
query.setOptions({ lean : true });
query.collection(MyModel.collection);
query.where('age').gte(21).exec(callback);

// You can instantiate a query directly. There is no need to do
// this unless you're an advanced user with a very good reason to.
const query = new mongoose.Query();
```

Example 2 (r):
```r
query.$where('this.comments.length === 10 || this.name.length === 5')

// or

query.$where(function () {
  return this.comments.length === 10 || this.name.length === 5;
})
```

Example 3 (csharp):
```csharp
MyModel.find().where('pets').all(['dog', 'cat', 'ferret']);
// Equivalent:
MyModel.find().all('pets', ['dog', 'cat', 'ferret']);
```

Example 4 (python):
```python
await query.find().sort({ name: 1 }).allowDiskUse(true);
// Equivalent:
await query.find().sort({ name: 1 }).allowDiskUse();
```

---

## Query

**URL:** https://mongoosejs.com/docs/5.x/docs/api/query.html

**Contents:**
- Query
  - Query()
      - Parameters
    - Example:
  - Query.prototype.$where()
      - Parameters
      - Returns:
    - Example
    - NOTE:
  - Query.prototype.Symbol.asyncIterator()

Query constructor used for building queries. You do not need to instantiate a Query directly. Instead use Model functions like Model.find().

Specifies a javascript function or expression to pass to MongoDBs query system.

Only use $where when you have a condition that cannot be met using other MongoDB operators like $lt. Be sure to read about all of its caveats before using.

Returns an asyncIterator for use with for/await/of loops This function only works for find() queries. You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Node.js 10.x supports async iterators natively without any flags. You can enable async iterators in Node.js 8.x using the --harmony_async_iteration flag.

Note: This function is not if Symbol.asyncIterator is undefined. If Symbol.asyncIterator is undefined, that means your Node.js version does not support async iterators.

Specifies an $all query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the allowDiskUse option, which allows the MongoDB server to use more than 100 MB for this query's sort(). This option can let you work around QueryExceededMemoryLimitNoDiskUseAllowed errors from the MongoDB server.

Note that this option requires MongoDB server >= 4.4. Setting this option is a no-op for MongoDB 4.2 and earlier.

Calling query.allowDiskUse(v) is equivalent to query.setOptions({ allowDiskUse: v })

Specifies arguments for a $and condition.

Specifies the batchSize option.

Cannot be used with distinct()

Specifies a $box condition

Casts this query to the schema of model

If obj is present, it is cast instead of this query.

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error. Like .then(), but only takes a rejection handler.

More about Promise catch() in JavaScript.

DEPRECATED Alias for circle

Deprecated. Use circle instead.

DEPRECATED Specifies a $centerSphere condition

Deprecated. Use circle instead.

Specifies a $center or $centerSphere condition.

Adds a collation to this op (MongoDB 3.4 and up)

Specifies the comment option.

Cannot be used with distinct()

Specifies this query as a count query.

This method is deprecated. If you want to count the number of documents in a collection, e.g. count({}), use the estimatedDocumentCount() function instead. Otherwise, use the countDocuments() function instead.

Passing a callback executes the query.

This function triggers the following middleware.

Specifies this query as a countDocuments() query. Behaves like count(), except it always does a full collection scan when passed an empty filter {}.

There are also minor differences in how countDocuments() handles $where and a couple geospatial operators. versus count().

Passing a callback executes the query.

This function triggers the following middleware.

The countDocuments() function is similar to count(), but there are a few operators that countDocuments() does not support. Below are the operators that count() supports but countDocuments() does not, and the suggested replacement:

Returns a wrapper around a mongodb driver cursor. A QueryCursor exposes a Streams3 interface, as well as a .next() function.

The .cursor() function triggers pre find hooks, but not post find hooks.

Declare and/or execute this query as a deleteMany() operation. Works like remove, except it deletes every document that matches filter in the collection, regardless of the value of single.

This function triggers deleteMany middleware.

This function calls the MongoDB driver's Collection#deleteMany() function. The returned promise resolves to an object that contains 3 properties:

Declare and/or execute this query as a deleteOne() operation. Works like remove, except it deletes at most one document regardless of the single option.

This function triggers deleteOne middleware.

This function calls the MongoDB driver's Collection#deleteOne() function. The returned promise resolves to an object that contains 3 properties:

Declares or executes a distinct() operation.

Passing a callback executes the query.

This function does not trigger any middleware.

Specifies an $elemMatch condition

Specifies the complementary comparison value for paths specified with where()

Gets/sets the error flag on this query. If this flag is not null or undefined, the exec() promise will reject without executing.

Note that query casting runs after hooks, so cast errors will override custom errors.

Specifies this query as a estimatedDocumentCount() query. Faster than using countDocuments() for large collections because estimatedDocumentCount() uses collection metadata rather than scanning the entire collection.

estimatedDocumentCount() does not accept a filter. Model.find({ foo: bar }).estimatedDocumentCount() is equivalent to Model.find().estimatedDocumentCount()

This function triggers the following middleware.

Specifies an $exists condition

Sets the explain option, which makes this query return detailed execution stats instead of the actual query result. This method is useful for determining what index your queries use.

Calling query.explain(v) is equivalent to query.setOptions({ explain: v })

Find all documents that match selector. The result will be an array of documents.

If there are too many documents in the result to fit in memory, use Query.prototype.cursor()

Declares the query a findOne operation. When executed, the first found document is passed to the callback.

Passing a callback executes the query. The result of the query is a single document.

This function triggers the following middleware.

Issues a MongoDB findOneAndDelete command.

Finds a matching document, removes it, and passes the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

This function differs slightly from Model.findOneAndRemove() in that findOneAndRemove() becomes a MongoDB findAndModify() command, as opposed to a findOneAndDelete() command. For most mongoose use cases, this distinction is purely pedantic. You should use findOneAndDelete() unless you have a good reason not to.

Issues a mongodb findAndModify remove command.

Finds a matching document, removes it, passing the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

Issues a MongoDB findOneAndReplace command.

Finds a matching document, removes it, and passes the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

Issues a mongodb findAndModify update command.

Finds a matching document, updates it according to the update arg, passing any options, and returns the found document (if any) to the callback. The query executes if callback is passed.

This function triggers the following middleware.

Specifies a $geometry condition

The argument is assigned to the most recent path passed to where().

geometry() must come after either intersects() or within().

The object argument must contain type and coordinates properties. - type {String} - coordinates {Array}

For update operations, returns the value of a path in the update's $set. Useful for writing getters/setters that can work with both update operations and save().

Returns the current query filter (also known as conditions) as a POJO.

Gets a list of paths to be populated by this query

Returns the current query filter. Equivalent to getFilter().

You should use getFilter() instead of getQuery() where possible. getQuery() will likely be deprecated in a future release.

Returns the current update operations as a JSON object.

Specifies a $gt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $gte query condition.

When called with one argument, the most recent path passed to where() is used.

Cannot be used with distinct()

Specifies an $in query condition.

When called with one argument, the most recent path passed to where() is used.

Declares an intersects query for geometry().

MUST be used after where().

In Mongoose 3.7, intersects changed from a getter to a function. If you need the old syntax, use this.

Requests acknowledgement that this operation has been persisted to MongoDB's on-disk journal.

Defaults to the schema's writeConcern.j option

Sets the lean option.

Documents returned from queries with the lean option enabled are plain javascript objects, not Mongoose Documents. They have no save method, getters/setters, virtuals, or other Mongoose features.

Lean is great for high-performance, read-only cases, especially when combined with cursors.

If you need virtuals, getters/setters, or defaults with lean(), you need to use a plugin. See:

Specifies the maximum number of documents the query will return.

Cannot be used with distinct()

Specifies a $lt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $lte query condition.

When called with one argument, the most recent path passed to where() is used.

Runs a function fn and treats the return value of fn as the new value for the query to resolve to.

Any functions you pass to map() will run after any post hooks.

Specifies a maxDistance query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the maxScan option.

Cannot be used with distinct()

Sets the maxTimeMS option. This will tell the MongoDB server to abort if the query or write op has been running for more than ms milliseconds.

Calling query.maxTimeMS(v) is equivalent to query.setOptions({ maxTimeMS: v })

DEPRECATED Alias of maxScan

Merges another Query or conditions object into this one.

When a Query is passed, conditions, field selection and options are merged.

Specifies a $mod condition, filters documents for documents whose path property is a number that is equal to remainder modulo divisor.

The model this query is associated with.

Getter/setter around the current mongoose-specific options for this query Below are the current Mongoose-specific options.

Mongoose maintains a separate object for internal options because Mongoose sends Query.prototype.options to the MongoDB server, and the above options are not relevant for the MongoDB server.

Specifies a $ne query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $near or $nearSphere condition

These operators return documents sorted by distance.

DEPRECATED Specifies a $nearSphere condition

Deprecated. Use query.near() instead with the spherical option set to true.

Specifies an $nin query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies arguments for a $nor condition.

Specifies arguments for an $or condition.

Make this query throw an error if no documents match the given filter. This is handy for integrating with async/await, because orFail() saves you an extra if statement to check if no document was found.

Specifies a $polygon condition

Specifies paths which should be populated with other documents.

Paths are populated after the query executes and a response is received. A separate query is then executed for each path specified for population. After a response for each query has also been returned, the results are passed to the callback.

Add post middleware to this query instance. Doesn't affect other queries.

Add pre middleware to this query instance. Doesn't affect other queries.

Get/set the current projection (AKA fields). Pass null to remove the current projection.

Unlike projection(), the select() function modifies the current projection in place. This function overwrites the existing projection.

Determines the MongoDB nodes from which to read.

Read more about how to use read preferences here and here.

Sets the readConcern option for the query.

Read more about how to use read concern here.

Specifies a $regex query condition.

When called with one argument, the most recent path passed to where() is used.

Declare and/or execute this query as a remove() operation. remove() is deprecated, you should use deleteOne() or deleteMany() instead.

This function does not trigger any middleware

This function calls the MongoDB driver's Collection#remove() function. The returned promise resolves to an object that contains 3 properties:

Calling remove() creates a Mongoose query, and a query does not execute until you either pass a callback, call Query#then(), or call Query#exec().

Declare and/or execute this query as a replaceOne() operation. Same as update(), except MongoDB will replace the existing document and will not accept any atomic operators ($set, etc.)

Note replaceOne will not fire update middleware. Use pre('replaceOne') and post('replaceOne') instead.

This function triggers the following middleware.

Specifies which document fields to include or exclude (also known as the query "projection")

When using string syntax, prefixing a path with - will flag that path as excluded. When a path does not have the - prefix, it is included. Lastly, if a path is prefixed with +, it forces inclusion of the path, which is useful for paths excluded at the schema level.

A projection must be either inclusive or exclusive. In other words, you must either list the fields to include (which excludes all others), or list the fields to exclude (which implies all other fields are included). The _id field is the only exception because MongoDB includes it by default.

Determines if field selection has been made.

Determines if exclusive field selection has been made.

Determines if inclusive field selection has been made.

Sets the MongoDB session associated with this query. Sessions are how you mark a query as part of a transaction.

Calling session(null) removes the session from this query.

Adds a $set to this query's update without changing the operation. This is useful for query middleware so you can add an update regardless of whether you use updateOne(), updateMany(), findOneAndUpdate(), etc.

Sets query options. Some options only make sense for certain operations.

The following options are only for find():

The following options are only for write operations: update(), updateOne(), updateMany(), replaceOne(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for find(), findOne(), findById(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for all operations except update(), updateOne(), updateMany(), remove(), deleteOne(), and deleteMany():

The following options are for findOneAndUpdate() and findOneAndRemove()

Sets the query conditions to the provided JSON object.

Sets the current update operation to new value.

Specifies a $size query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the number of documents to skip.

Cannot be used with distinct()

DEPRECATED Sets the slaveOk option.

Deprecated in MongoDB 2.2 in favor of read preferences.

Specifies a $slice projection for an array.

Specifies this query as a snapshot query.

Cannot be used with distinct()

If an object is passed, values allowed are asc, desc, ascending, descending, 1, and -1.

If a string is passed, it must be a space delimited list of path names. The sort order of each path is ascending unless the path name is prefixed with - which will be treated as descending.

Cannot be used with distinct()

Sets the tailable option (for use with capped collections).

Cannot be used with distinct()

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error.

More about then() in JavaScript.

Converts this query to a customized, reusable query constructor with all arguments and options retained.

Declare and/or execute this query as an update() operation.

All paths passed that are not atomic operations will become $set ops.

This function triggers the following middleware.

Passing an empty object {} as the doc will result in a no-op unless the overwrite option is passed. Without the overwrite option set, the update operation will be ignored and the callback executed without sending the command to MongoDB so as to prevent accidently overwritting documents in the collection.

The operation is only executed when a callback is passed. To force execution without a callback, we must first call update() and then execute it by using the exec() method.

Declare and/or execute this query as an updateMany() operation. Same as update(), except MongoDB will update all documents that match filter (as opposed to just the first one) regardless of the value of the multi option.

Note updateMany will not fire update middleware. Use pre('updateMany') and post('updateMany') instead.

This function triggers the following middleware.

Declare and/or execute this query as an updateOne() operation. Same as update(), except it does not support the multi or overwrite options.

Note updateOne will not fire update middleware. Use pre('updateOne') and post('updateOne') instead.

This function triggers the following middleware.

Flag to opt out of using $geoWithin.

MongoDB 2.4 deprecated the use of $within, replacing it with $geoWithin. Mongoose uses $geoWithin by default (which is 100% backward compatible with $within). If you are running an older version of MongoDB, set this flag to false so your within() queries continue to work.

Sets the specified number of mongod servers, or tag set of mongod servers, that must acknowledge this write before this write is considered successful.

Defaults to the schema's writeConcern.w option

Specifies a path for use with chaining.

Defines a $within or $geoWithin argument for geo-spatial queries.

MUST be used after where().

As of Mongoose 3.7, $geoWithin is always used for queries. To change this behavior, see Query.use$geoWithin.

In Mongoose 3.7, within changed from a getter to a function. If you need the old syntax, use this.

Defaults to the schema's writeConcern option

If w > 1, the maximum amount of time to wait for this write to propagate through the replica set before this operation fails. The default is 0, which means no timeout.

Defaults to the schema's writeConcern.wtimeout option

**Examples:**

Example 1 (css):
```css
const query = MyModel.find(); // `query` is an instance of `Query`
query.setOptions({ lean : true });
query.collection(MyModel.collection);
query.where('age').gte(21).exec(callback);

// You can instantiate a query directly. There is no need to do
// this unless you're an advanced user with a very good reason to.
const query = new mongoose.Query();
```

Example 2 (r):
```r
query.$where('this.comments.length === 10 || this.name.length === 5')

// or

query.$where(function () {
  return this.comments.length === 10 || this.name.length === 5;
})
```

Example 3 (javascript):
```javascript
for await (const doc of Model.aggregate([{ $sort: { name: 1 } }])) {
  console.log(doc.name);
}
```

Example 4 (csharp):
```csharp
MyModel.find().where('pets').all(['dog', 'cat', 'ferret']);
// Equivalent:
MyModel.find().all('pets', ['dog', 'cat', 'ferret']);
```

---

## Using GeoJSON

**URL:** https://mongoosejs.com/docs/geojson.html

**Contents:**
- Using GeoJSON
- Point Schema
- Polygon Schema
- Geospatial Queries with Mongoose
- Geospatial Indexes

GeoJSON is a format for storing geographic points and polygons. MongoDB has excellent support for geospatial queries on GeoJSON objects. Let's take a look at how you can use Mongoose to store and query GeoJSON objects.

The most simple structure in GeoJSON is a point. Below is an example point representing the approximate location of San Francisco. Note that longitude comes first in a GeoJSON coordinate array, not latitude.

Below is an example of a Mongoose schema where location is a point.

Using subdocuments, you can define a common pointSchema and reuse it everywhere you want to store a GeoJSON point.

GeoJSON polygons let you define an arbitrary shape on a map. For example, the below polygon is a GeoJSON rectangle that approximates the border of the state of Colorado.

Polygons are tricky because they use triple nested arrays. Below is how you create a Mongoose schema where coordinates is a triple nested array of numbers.

Mongoose queries support the same geospatial query operators that the MongoDB driver does. For example, the below script saves a city document those location property is a GeoJSON point representing the city of Denver, Colorado. It then queries for all documents within a polygon representing the state of Colorado using the MongoDB $geoWithin operator.

Mongoose also has a within() helper that's a shorthand for $geoWithin.

MongoDB supports 2dsphere indexes for speeding up geospatial queries. Here's how you can define a 2dsphere index on a GeoJSON point:

You can also define a geospatial index using the Schema#index() function as shown below.

MongoDB's $near query operator and $geoNear aggregation stage require a 2dsphere index.

**Examples:**

Example 1 (json):
```json
{
  "type" : "Point",
  "coordinates" : [
    -122.5,
    37.7
  ]
}
```

Example 2 (swift):
```swift
const citySchema = new mongoose.Schema({
  name: String,
  location: {
    type: {
      type: String, // Don't do `{ location: { type: String } }`
      enum: ['Point'], // 'location.type' must be 'Point'
      required: true
    },
    coordinates: {
      type: [Number],
      required: true
    }
  }
});
```

Example 3 (swift):
```swift
const pointSchema = new mongoose.Schema({
  type: {
    type: String,
    enum: ['Point'],
    required: true
  },
  coordinates: {
    type: [Number],
    required: true
  }
});

const citySchema = new mongoose.Schema({
  name: String,
  location: {
    type: pointSchema,
    required: true
  }
});
```

Example 4 (json):
```json
{
  "type": "Polygon",
  "coordinates": [[
    [-109, 41],
    [-102, 41],
    [-102, 37],
    [-109, 37],
    [-109, 41]
  ]]
}
```

---

## Faster Mongoose Queries With Lean

**URL:** https://mongoosejs.com/docs/tutorials/lean.html

**Contents:**
- Faster Mongoose Queries With Lean
- Using Lean
- Lean and Populate
- When to Use Lean
- Plugins
- BigInts

The lean option tells Mongoose to skip hydrating the result documents. This makes queries faster and less memory intensive, but the result documents are plain old JavaScript objects (POJOs), not Mongoose documents. In this tutorial, you'll learn more about the tradeoffs of using lean().

By default, Mongoose queries return an instance of the Mongoose Document class. Documents are much heavier than vanilla JavaScript objects, because they have a lot of internal state for change tracking. Enabling the lean option tells Mongoose to skip instantiating a full Mongoose document and just give you the POJO.

How much smaller are lean documents? Here's a comparison.

Under the hood, after executing a query, Mongoose converts the query results from POJOs to Mongoose documents. If you turn on the lean option, Mongoose skips this step.

The downside of enabling lean is that lean docs don't have:

For example, the following code sample shows that the Person model's getters and virtuals don't run if you enable lean.

Populate works with lean(). If you use both populate() and lean(), the lean option propagates to the populated documents as well. In the below example, both the top-level 'Group' documents and the populated 'Person' documents will be lean.

Virtual populate also works with lean.

If you're executing a query and sending the results without modification to, say, an Express response, you should use lean. In general, if you do not modify the query results and do not use custom getters, you should use lean(). If you modify the query results or rely on features like getters or transforms, you should not use lean().

Below is an example of an Express route that is a good candidate for lean(). This route does not modify the person doc and doesn't rely on any Mongoose-specific functionality.

Below is an example of an Express route that should not use lean(). As a general rule of thumb, GET routes are good candidates for lean() in a RESTful API. On the other hand, PUT, POST, etc. routes generally should not use lean().

Remember that virtuals do not end up in lean() query results. Use the mongoose-lean-virtuals plugin to add virtuals to your lean query results.

Using lean() bypasses all Mongoose features, including virtuals, getters/setters, and defaults. If you want to use these features with lean(), you need to use the corresponding plugin:

However, you need to keep in mind that Mongoose does not hydrate lean documents, so this will be a POJO in virtuals, getters, and default functions.

By default, the MongoDB Node driver converts longs stored in MongoDB into JavaScript numbers, not BigInts. Set the useBigInt64 option on your lean() queries to inflate longs into BigInts.

**Examples:**

Example 1 (javascript):
```javascript
const leanDoc = await MyModel.findOne().lean();
```

Example 2 (javascript):
```javascript
const schema = new mongoose.Schema({ name: String });
const MyModel = mongoose.model('Test', schema);

await MyModel.create({ name: 'test' });

const normalDoc = await MyModel.findOne();
// To enable the `lean` option for a query, use the `lean()` function.
const leanDoc = await MyModel.findOne().lean();
v8Serialize(normalDoc).length; // approximately 180
v8Serialize(leanDoc).length; // approximately 55, about 3x smaller!

// In case you were wondering, the JSON form of a Mongoose doc is the same
// as the POJO. This additional memory only affects how much memory your
// Node.js process uses, not how much data is sent over the network.
JSON.stringify(normalDoc).length === JSON.stringify(leanDoc).length; // true
```

Example 3 (javascript):
```javascript
const normalDoc = await MyModel.findOne();
const leanDoc = await MyModel.findOne().lean();

normalDoc instanceof mongoose.Document; // true
normalDoc.constructor.name; // 'model'

leanDoc instanceof mongoose.Document; // false
leanDoc.constructor.name; // 'Object'
```

Example 4 (javascript):
```javascript
// Define a `Person` model. Schema has 2 custom getters and a `fullName`
// virtual. Neither the getters nor the virtuals will run if lean is enabled.
const personSchema = new mongoose.Schema({
  firstName: {
    type: String,
    get: capitalizeFirstLetter
  },
  lastName: {
    type: String,
    get: capitalizeFirstLetter
  }
});
personSchema.virtual('fullName').get(function() {
  return `${this.firstName} ${this.lastName}`;
});
function capitalizeFirstLetter(v) {
  // Convert 'bob' -> 'Bob'
  return v.charAt(0).toUpperCase() + v.substring(1);
}
const Person = mongoose.model('Person', personSchema);

// Create a doc and load it as a lean doc
await Person.create({ firstName: 'benjamin', lastName: 'sisko' });
const normalDoc = await Person.findOne();
const leanDoc = await Person.findOne().lean();

normalDoc.fullName; // 'Benjamin Sisko'
normalDoc.firstName; // 'Benjamin', because of `capitalizeFirstLetter()`
normalDoc.lastName; // 'Sisko', because of `capitalizeFirstLetter()`

leanDoc.fullName; // undefined
leanDoc.firstName; // 'benjamin', custom getter doesn't run
leanDoc.lastName; // 'sisko', custom getter doesn't run
```

---

## Queries

**URL:** https://mongoosejs.com/docs/6.x/docs/queries.html

**Contents:**
- Queries
- Executing
- Queries are Not Promises
- References to other documents
- Streaming
- Versus Aggregation
- Sorting
- Next Up

Mongoose models provide several static helper functions for CRUD operations. Each of these functions returns a mongoose Query object.

A mongoose query can be executed in one of two ways. First, if you pass in a callback function, Mongoose will execute the query asynchronously and pass the results to the callback.

A query also has a .then() function, and thus can be used as a promise.

When executing a query with a callback function, you specify your query as a JSON document. The JSON document's syntax is the same as the MongoDB shell.

Mongoose executed the query and passed the results to callback. All callbacks in Mongoose use the pattern: callback(error, result). If an error occurs executing the query, the error parameter will contain an error document, and result will be null. If the query is successful, the error parameter will be null, and the result will be populated with the results of the query.

Anywhere a callback is passed to a query in Mongoose, the callback follows the pattern callback(error, results). What results is depends on the operation: For findOne() it is a potentially-null single document, find() a list of documents, count() the number of documents, update() the number of documents affected, etc. The API docs for Models provide more detail on what is passed to the callbacks.

Now let's look at what happens when no callback is passed:

In the above code, the query variable is of type Query. A Query enables you to build up a query using chaining syntax, rather than specifying a JSON object. The below 2 examples are equivalent.

A full list of Query helper functions can be found in the API docs.

Mongoose queries are not promises. They have a .then() function for co and async/await as a convenience. However, unlike promises, calling a query's .then() can execute the query multiple times.

For example, the below code will execute 3 updateMany() calls, one because of the callback, and two because .then() is called twice.

Don't mix using callbacks and promises with queries, or you may end up with duplicate operations. That's because passing a callback to a query function immediately executes the query, and calling then() executes the query again.

Mixing promises and callbacks can lead to duplicate entries in arrays. For example, the below code inserts 2 entries into the tags array, not just 1.

There are no joins in MongoDB but sometimes we still want references to documents in other collections. This is where population comes in. Read more about how to include documents from other collections in your query results here.

You can stream query results from MongoDB. You need to call the Query#cursor() function to return an instance of QueryCursor.

Iterating through a Mongoose query using async iterators also creates a cursor.

Cursors are subject to cursor timeouts. By default, MongoDB will close your cursor after 10 minutes and subsequent next() calls will result in a MongoServerError: cursor id 123 not found error. To override this, set the noCursorTimeout option on your cursor.

However, cursors can still time out because of session idle timeouts. So even a cursor with noCursorTimeout set will still time out after 30 minutes of inactivity. You can read more about working around session idle timeouts in the MongoDB documentation.

Aggregation can do many of the same things that queries can. For example, below is how you can use aggregate() to find docs where name.last = 'Ghost':

However, just because you can use aggregate() doesn't mean you should. In general, you should use queries where possible, and only use aggregate() when you absolutely need to.

Unlike query results, Mongoose does not hydrate() aggregation results. Aggregation results are always POJOs, not Mongoose documents.

Also, unlike query filters, Mongoose also doesn't cast aggregation pipelines. That means you're responsible for ensuring the values you pass in to an aggregation pipeline have the correct type.

Sorting is how you can ensure you query results come back in the desired order.

When sorting with mutiple fields, the order of the sort keys determines what key MongoDB server sorts by first.

You can view the output of a single run of this block below. As you can see, age is sorted from 0 to 2 but when age is equal, sorts by weight.

Now that we've covered Queries, let's take a look at Validation.

**Examples:**

Example 1 (javascript):
```javascript
const Person = mongoose.model('Person', yourSchema);

// find each person with a last name matching 'Ghost', selecting the `name` and `occupation` fields
Person.findOne({ 'name.last': 'Ghost' }, 'name occupation', function(err, person) {
  if (err) return handleError(err);
  // Prints "Space Ghost is a talk show host".
  console.log('%s %s is a %s.', person.name.first, person.name.last,
    person.occupation);
});
```

Example 2 (javascript):
```javascript
// find each person with a last name matching 'Ghost'
const query = Person.findOne({ 'name.last': 'Ghost' });

// selecting the `name` and `occupation` fields
query.select('name occupation');

// execute the query at a later time
query.exec(function(err, person) {
  if (err) return handleError(err);
  // Prints "Space Ghost is a talk show host."
  console.log('%s %s is a %s.', person.name.first, person.name.last,
    person.occupation);
});
```

Example 3 (javascript):
```javascript
// With a JSON doc
Person.
  find({
    occupation: /host/,
    'name.last': 'Ghost',
    age: { $gt: 17, $lt: 66 },
    likes: { $in: ['vaporizing', 'talking'] }
  }).
  limit(10).
  sort({ occupation: -1 }).
  select({ name: 1, occupation: 1 }).
  exec(callback);

// Using query builder
Person.
  find({ occupation: /host/ }).
  where('name.last').equals('Ghost').
  where('age').gt(17).lt(66).
  where('likes').in(['vaporizing', 'talking']).
  limit(10).
  sort('-occupation').
  select('name occupation').
  exec(callback);
```

Example 4 (javascript):
```javascript
const q = MyModel.updateMany({}, { isDeleted: true }, function() {
  console.log('Update 1');
});

q.then(() => console.log('Update 2'));
q.then(() => console.log('Update 3'));
```

---

## Query

**URL:** https://mongoosejs.com/docs/6.x/docs/api/query.html

**Contents:**
- Query
  - Query()
      - Parameters:
    - Example:
  - Query.prototype.$where()
      - Parameters:
      - Returns:
      - See:
    - Example:
    - Note:

Query constructor used for building queries. You do not need to instantiate a Query directly. Instead use Model functions like Model.find().

Specifies a javascript function or expression to pass to MongoDBs query system.

Only use $where when you have a condition that cannot be met using other MongoDB operators like $lt. Be sure to read about all of its caveats before using.

Specifies an $all query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the allowDiskUse option, which allows the MongoDB server to use more than 100 MB for this query's sort(). This option can let you work around QueryExceededMemoryLimitNoDiskUseAllowed errors from the MongoDB server.

Note that this option requires MongoDB server >= 4.4. Setting this option is a no-op for MongoDB 4.2 and earlier.

Calling query.allowDiskUse(v) is equivalent to query.setOptions({ allowDiskUse: v })

Specifies arguments for a $and condition.

Specifies the batchSize option.

Cannot be used with distinct()

Specifies a $box condition

Casts this query to the schema of model

If obj is present, it is cast instead of this query.

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error. Like .then(), but only takes a rejection handler.

More about Promise catch() in JavaScript.

DEPRECATED Alias for circle

Deprecated. Use circle instead.

DEPRECATED Specifies a $centerSphere condition

Deprecated. Use circle instead.

Specifies a $center or $centerSphere condition.

Make a copy of this query so you can re-execute it.

Adds a collation to this op (MongoDB 3.4 and up)

Specifies the comment option.

Cannot be used with distinct()

Specifies this query as a count query.

This method is deprecated. If you want to count the number of documents in a collection, e.g. count({}), use the estimatedDocumentCount() function instead. Otherwise, use the countDocuments() function instead.

Passing a callback executes the query.

This function triggers the following middleware.

Specifies this query as a countDocuments() query. Behaves like count(), except it always does a full collection scan when passed an empty filter {}.

There are also minor differences in how countDocuments() handles $where and a couple geospatial operators. versus count().

Passing a callback executes the query.

This function triggers the following middleware.

The countDocuments() function is similar to count(), but there are a few operators that countDocuments() does not support. Below are the operators that count() supports but countDocuments() does not, and the suggested replacement:

Returns a wrapper around a mongodb driver cursor. A QueryCursor exposes a Streams3 interface, as well as a .next() function.

The .cursor() function triggers pre find hooks, but not post find hooks.

Declare and/or execute this query as a deleteMany() operation. Works like remove, except it deletes every document that matches filter in the collection, regardless of the value of single.

This function triggers deleteMany middleware.

This function calls the MongoDB driver's Collection#deleteMany() function. The returned promise resolves to an object that contains 3 properties:

Declare and/or execute this query as a deleteOne() operation. Works like remove, except it deletes at most one document regardless of the single option.

This function triggers deleteOne middleware.

This function calls the MongoDB driver's Collection#deleteOne() function. The returned promise resolves to an object that contains 3 properties:

Declares or executes a distinct() operation.

Passing a callback executes the query.

This function does not trigger any middleware.

Specifies an $elemMatch condition

Specifies the complementary comparison value for paths specified with where()

Gets/sets the error flag on this query. If this flag is not null or undefined, the exec() promise will reject without executing.

Note that query casting runs after hooks, so cast errors will override custom errors.

Specifies this query as a estimatedDocumentCount() query. Faster than using countDocuments() for large collections because estimatedDocumentCount() uses collection metadata rather than scanning the entire collection.

estimatedDocumentCount() does not accept a filter. Model.find({ foo: bar }).estimatedDocumentCount() is equivalent to Model.find().estimatedDocumentCount()

This function triggers the following middleware.

Specifies an $exists condition

Sets the explain option, which makes this query return detailed execution stats instead of the actual query result. This method is useful for determining what index your queries use.

Calling query.explain(v) is equivalent to query.setOptions({ explain: v })

Find all documents that match selector. The result will be an array of documents.

If there are too many documents in the result to fit in memory, use Query.prototype.cursor()

Declares the query a findOne operation. When executed, the first found document is passed to the callback.

Passing a callback executes the query. The result of the query is a single document.

This function triggers the following middleware.

Issues a MongoDB findOneAndDelete command.

Finds a matching document, removes it, and passes the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

This function differs slightly from Model.findOneAndRemove() in that findOneAndRemove() becomes a MongoDB findAndModify() command, as opposed to a findOneAndDelete() command. For most mongoose use cases, this distinction is purely pedantic. You should use findOneAndDelete() unless you have a good reason not to.

Issues a mongodb findAndModify remove command.

Finds a matching document, removes it, passing the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

Issues a MongoDB findOneAndReplace command.

Finds a matching document, removes it, and passes the found document (if any) to the callback. Executes if callback is passed.

This function triggers the following middleware.

Issues a mongodb findAndModify update command.

Finds a matching document, updates it according to the update arg, passing any options, and returns the found document (if any) to the callback. The query executes if callback is passed.

This function triggers the following middleware.

Specifies a $geometry condition

The argument is assigned to the most recent path passed to where().

geometry() must come after either intersects() or within().

The object argument must contain type and coordinates properties.

For update operations, returns the value of a path in the update's $set. Useful for writing getters/setters that can work with both update operations and save().

Returns the current query filter (also known as conditions) as a POJO.

Gets a list of paths to be populated by this query

Returns the current query filter. Equivalent to getFilter().

You should use getFilter() instead of getQuery() where possible. getQuery() will likely be deprecated in a future release.

Returns the current update operations as a JSON object.

Specifies a $gt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $gte query condition.

When called with one argument, the most recent path passed to where() is used.

Cannot be used with distinct()

Specifies an $in query condition.

When called with one argument, the most recent path passed to where() is used.

Declares an intersects query for geometry().

MUST be used after where().

In Mongoose 3.7, intersects changed from a getter to a function. If you need the old syntax, use this.

Requests acknowledgement that this operation has been persisted to MongoDB's on-disk journal. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.j option

Sets the lean option.

Documents returned from queries with the lean option enabled are plain javascript objects, not Mongoose Documents. They have no save method, getters/setters, virtuals, or other Mongoose features.

Lean is great for high-performance, read-only cases, especially when combined with cursors.

If you need virtuals, getters/setters, or defaults with lean(), you need to use a plugin. See:

Specifies the maximum number of documents the query will return.

Cannot be used with distinct()

Specifies a $lt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $lte query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a maxDistance query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the maxScan option.

Cannot be used with distinct()

Sets the maxTimeMS option. This will tell the MongoDB server to abort if the query or write op has been running for more than ms milliseconds.

Calling query.maxTimeMS(v) is equivalent to query.setOptions({ maxTimeMS: v })

DEPRECATED Alias of maxScan

Merges another Query or conditions object into this one.

When a Query is passed, conditions, field selection and options are merged.

Specifies a $mod condition, filters documents for documents whose path property is a number that is equal to remainder modulo divisor.

The model this query is associated with.

Getter/setter around the current mongoose-specific options for this query Below are the current Mongoose-specific options.

Mongoose maintains a separate object for internal options because Mongoose sends Query.prototype.options to the MongoDB server, and the above options are not relevant for the MongoDB server.

Specifies a $ne query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $near or $nearSphere condition

These operators return documents sorted by distance.

DEPRECATED Specifies a $nearSphere condition

Deprecated. Use query.near() instead with the spherical option set to true.

Specifies an $nin query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies arguments for a $nor condition.

Specifies arguments for an $or condition.

Make this query throw an error if no documents match the given filter. This is handy for integrating with async/await, because orFail() saves you an extra if statement to check if no document was found.

Specifies a $polygon condition

Specifies paths which should be populated with other documents.

Paths are populated after the query executes and a response is received. A separate query is then executed for each path specified for population. After a response for each query has also been returned, the results are passed to the callback.

Add post middleware to this query instance. Doesn't affect other queries.

Add pre middleware to this query instance. Doesn't affect other queries.

Get/set the current projection (AKA fields). Pass null to remove the current projection.

Unlike projection(), the select() function modifies the current projection in place. This function overwrites the existing projection.

Determines the MongoDB nodes from which to read.

Read more about how to use read preferences here.

Sets the readConcern option for the query.

Read more about how to use read concern here.

Specifies a $regex query condition.

When called with one argument, the most recent path passed to where() is used.

Declare and/or execute this query as a remove() operation. remove() is deprecated, you should use deleteOne() or deleteMany() instead.

This function does not trigger any middleware

This function calls the MongoDB driver's Collection#remove() function. The returned promise resolves to an object that contains 3 properties:

Calling remove() creates a Mongoose query, and a query does not execute until you either pass a callback, call Query#then(), or call Query#exec().

Declare and/or execute this query as a replaceOne() operation. Same as update(), except MongoDB will replace the existing document and will not accept any atomic operators ($set, etc.)

Note replaceOne will not fire update middleware. Use pre('replaceOne') and post('replaceOne') instead.

This function triggers the following middleware.

Specifies which document fields to include or exclude (also known as the query "projection")

When using string syntax, prefixing a path with - will flag that path as excluded. When a path does not have the - prefix, it is included. Lastly, if a path is prefixed with +, it forces inclusion of the path, which is useful for paths excluded at the schema level.

A projection must be either inclusive or exclusive. In other words, you must either list the fields to include (which excludes all others), or list the fields to exclude (which implies all other fields are included). The _id field is the only exception because MongoDB includes it by default.

Determines if field selection has been made.

Determines if exclusive field selection has been made.

Determines if inclusive field selection has been made.

Sets the MongoDB session associated with this query. Sessions are how you mark a query as part of a transaction.

Calling session(null) removes the session from this query.

Adds a $set to this query's update without changing the operation. This is useful for query middleware so you can add an update regardless of whether you use updateOne(), updateMany(), findOneAndUpdate(), etc.

Sets query options. Some options only make sense for certain operations.

The following options are only for find():

The following options are only for write operations: update(), updateOne(), updateMany(), replaceOne(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for find(), findOne(), findById(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for all operations except update(), updateOne(), updateMany(), remove(), deleteOne(), and deleteMany():

The following options are for findOneAndUpdate() and findOneAndRemove()

The following options are for all operations:

Sets the query conditions to the provided JSON object.

Sets the current update operation to new value.

Specifies a $size query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the number of documents to skip.

Cannot be used with distinct()

Specifies a $slice projection for an array.

Note: If the absolute value of the number of elements to be sliced is greater than the number of elements in the array, all array elements will be returned.

Note: If the number of elements to skip is positive and greater than the number of elements in the array, an empty array will be returned.

Note: If the number of elements to skip is negative and its absolute value is greater than the number of elements in the array, the starting position is the start of the array.

Specifies this query as a snapshot query.

Cannot be used with distinct()

If an object is passed, values allowed are asc, desc, ascending, descending, 1, and -1.

If a string is passed, it must be a space delimited list of path names. The sort order of each path is ascending unless the path name is prefixed with - which will be treated as descending.

Cannot be used with distinct()

Sets the tailable option (for use with capped collections).

Cannot be used with distinct()

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error.

More about then() in JavaScript.

Converts this query to a customized, reusable query constructor with all arguments and options retained.

Runs a function fn and treats the return value of fn as the new value for the query to resolve to.

Any functions you pass to transform() will run after any post hooks.

Declare and/or execute this query as an update() operation.

All paths passed that are not atomic operations will become $set ops.

This function triggers the following middleware.

Passing an empty object {} as the doc will result in a no-op. The update operation will be ignored and the callback executed without sending the command to MongoDB.

The operation is only executed when a callback is passed. To force execution without a callback, we must first call update() and then execute it by using the exec() method.

Declare and/or execute this query as an updateMany() operation. Same as update(), except MongoDB will update all documents that match filter (as opposed to just the first one) regardless of the value of the multi option.

Note updateMany will not fire update middleware. Use pre('updateMany') and post('updateMany') instead.

This function triggers the following middleware.

Declare and/or execute this query as an updateOne() operation. Same as update(), except it does not support the multi option.

Note updateOne will not fire update middleware. Use pre('updateOne') and post('updateOne') instead.

This function triggers the following middleware.

Sets the specified number of mongod servers, or tag set of mongod servers, that must acknowledge this write before this write is considered successful. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.w option

Specifies a path for use with chaining.

Defines a $within or $geoWithin argument for geo-spatial queries.

MUST be used after where().

As of Mongoose 3.7, $geoWithin is always used for queries. To change this behavior, see Query.use$geoWithin.

In Mongoose 3.7, within changed from a getter to a function. If you need the old syntax, use this.

Sets the 3 write concern parameters for this query:

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern option

If w > 1, the maximum amount of time to wait for this write to propagate through the replica set before this operation fails. The default is 0, which means no timeout.

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.wtimeout option

Returns an asyncIterator for use with for/await/of loops This function only works for find() queries. You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Node.js 10.x supports async iterators natively without any flags. You can enable async iterators in Node.js 8.x using the --harmony_async_iteration flag.

Note: This function is not if Symbol.asyncIterator is undefined. If Symbol.asyncIterator is undefined, that means your Node.js version does not support async iterators.

Flag to opt out of using $geoWithin.

MongoDB 2.4 deprecated the use of $within, replacing it with $geoWithin. Mongoose uses $geoWithin by default (which is 100% backward compatible with $within). If you are running an older version of MongoDB, set this flag to false so your within() queries continue to work.

**Examples:**

Example 1 (css):
```css
const query = MyModel.find(); // `query` is an instance of `Query`
query.setOptions({ lean : true });
query.collection(MyModel.collection);
query.where('age').gte(21).exec(callback);

// You can instantiate a query directly. There is no need to do
// this unless you're an advanced user with a very good reason to.
const query = new mongoose.Query();
```

Example 2 (r):
```r
query.$where('this.comments.length === 10 || this.name.length === 5')

// or

query.$where(function () {
  return this.comments.length === 10 || this.name.length === 5;
})
```

Example 3 (csharp):
```csharp
MyModel.find().where('pets').all(['dog', 'cat', 'ferret']);
// Equivalent:
MyModel.find().all('pets', ['dog', 'cat', 'ferret']);
```

Example 4 (python):
```python
await query.find().sort({ name: 1 }).allowDiskUse(true);
// Equivalent:
await query.find().sort({ name: 1 }).allowDiskUse();
```

---

## QueryCursor

**URL:** https://mongoosejs.com/docs/api/querycursor.html

**Contents:**
- QueryCursor
  - QueryCursor()
      - Parameters:
      - Inherits:
  - QueryCursor.prototype.addCursorFlag()
      - Parameters:
      - Returns:
  - QueryCursor.prototype.close()
      - Returns:
      - See:

A QueryCursor is a concurrency primitive for processing query results one document at a time. A QueryCursor fulfills the Node.js streams3 API, in addition to several other mechanisms for loading documents from MongoDB one at a time.

QueryCursors execute the model's pre find hooks before loading any documents from MongoDB, and the model's post find hooks after loading each document.

Unless you're an advanced user, do not instantiate this class directly. Use Query#cursor() instead.

Adds a cursor flag. Useful for setting the noCursorTimeout and tailable flags.

Marks this cursor as closed. Will stop streaming and subsequent calls to next() will error.

Execute fn for every document in the cursor. If fn returns a promise, will wait for the promise to resolve before iterating on to the next one. Returns a promise that resolves when done.

Returns the underlying cursor from the MongoDB Node driver that this cursor uses.

Registers a transform function which subsequently maps documents retrieved via the streams interface or .next()

Get the next document from this cursor. Will return null when there are no documents left.

The options passed in to the QueryCursor constructor.

Rewind this cursor to its uninitialized state. Any options that are present on the cursor will remain in effect. Iterating this cursor will cause new queries to be sent to the server, even if the resultant data has already been retrieved by this cursor.

Returns an asyncIterator for use with for/await/of loops. You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Node.js 10.x supports async iterators natively without any flags. You can enable async iterators in Node.js 8.x using the --harmony_async_iteration flag.

Note: This function is not if Symbol.asyncIterator is undefined. If Symbol.asyncIterator is undefined, that means your Node.js version does not support async iterators.

**Examples:**

Example 1 (swift):
```swift
// Iterate over documents asynchronously
Thing.
  find({ name: /^hello/ }).
  cursor().
  eachAsync(async function (doc, i) {
    doc.foo = doc.bar + i;
    await doc.save();
  })
```

Example 2 (javascript):
```javascript
// Map documents returned by `data` events
Thing.
  find({ name: /^hello/ }).
  cursor().
  map(function (doc) {
   doc.foo = "bar";
   return doc;
  })
  on('data', function(doc) { console.log(doc.foo); });

// Or map documents returned by `.next()`
const cursor = Thing.find({ name: /^hello/ }).
  cursor().
  map(function (doc) {
    doc.foo = "bar";
    return doc;
  });
cursor.next(function(error, doc) {
  console.log(doc.foo);
});
```

Example 3 (javascript):
```javascript
// Works without using `cursor()`
for await (const doc of Model.find([{ $sort: { name: 1 } }])) {
  console.log(doc.name);
}

// Can also use `cursor()`
for await (const doc of Model.find([{ $sort: { name: 1 } }]).cursor()) {
  console.log(doc.name);
}
```

---

## Query

**URL:** https://mongoosejs.com/docs/api/query.html

**Contents:**
- Query
  - Query()
      - Parameters:
    - Example:
  - Query.prototype.$where()
      - Parameters:
      - Returns:
      - See:
    - Example:
    - Note:

Query constructor used for building queries. You do not need to instantiate a Query directly. Instead use Model functions like Model.find().

Specifies a javascript function or expression to pass to MongoDBs query system.

Only use $where when you have a condition that cannot be met using other MongoDB operators like $lt. Be sure to read about all of its caveats before using.

Specifies an $all query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the allowDiskUse option, which allows the MongoDB server to use more than 100 MB for this query's sort(). This option can let you work around QueryExceededMemoryLimitNoDiskUseAllowed errors from the MongoDB server.

Note that this option requires MongoDB server >= 4.4. Setting this option is a no-op for MongoDB 4.2 and earlier.

Calling query.allowDiskUse(v) is equivalent to query.setOptions({ allowDiskUse: v })

Specifies arguments for a $and condition.

Specifies the batchSize option.

Cannot be used with distinct()

Specifies a $box condition

Casts this query to the schema of model

If obj is present, it is cast instead of this query.

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error. Like .then(), but only takes a rejection handler.

More about Promise catch() in JavaScript.

DEPRECATED Alias for circle

Deprecated. Use circle instead.

DEPRECATED Specifies a $centerSphere condition

Deprecated. Use circle instead.

Specifies a $center or $centerSphere condition.

Make a copy of this query so you can re-execute it.

Adds a collation to this op (MongoDB 3.4 and up)

Specifies the comment option.

Cannot be used with distinct()

Specifies this query as a countDocuments() query. Behaves like count(), except it always does a full collection scan when passed an empty filter {}.

There are also minor differences in how countDocuments() handles $where and a couple geospatial operators. versus count().

This function triggers the following middleware.

The countDocuments() function is similar to count(), but there are a few operators that countDocuments() does not support. Below are the operators that count() supports but countDocuments() does not, and the suggested replacement:

Returns a wrapper around a mongodb driver cursor. A QueryCursor exposes a Streams3 interface, as well as a .next() function.

The .cursor() function triggers pre find hooks, but not post find hooks.

Declare and/or execute this query as a deleteMany() operation. Works like remove, except it deletes every document that matches filter in the collection, regardless of the value of single.

This function triggers deleteMany middleware.

This function calls the MongoDB driver's Collection#deleteMany() function. The returned promise resolves to an object that contains 2 properties:

Declare and/or execute this query as a deleteOne() operation. Works like remove, except it deletes at most one document regardless of the single option.

This function triggers deleteOne middleware.

This function calls the MongoDB driver's Collection#deleteOne() function. The returned promise resolves to an object that contains 2 properties:

Declares or executes a distinct() operation.

This function does not trigger any middleware.

Specifies an $elemMatch condition

Specifies the complementary comparison value for paths specified with where()

Gets/sets the error flag on this query. If this flag is not null or undefined, the exec() promise will reject without executing.

Note that query casting runs after hooks, so cast errors will override custom errors.

Specifies this query as a estimatedDocumentCount() query. Faster than using countDocuments() for large collections because estimatedDocumentCount() uses collection metadata rather than scanning the entire collection.

estimatedDocumentCount() does not accept a filter. Model.find({ foo: bar }).estimatedDocumentCount() is equivalent to Model.find().estimatedDocumentCount()

This function triggers the following middleware.

Specifies an $exists condition

Sets the explain option, which makes this query return detailed execution stats instead of the actual query result. This method is useful for determining what index your queries use.

Calling query.explain(v) is equivalent to query.setOptions({ explain: v })

Executes the query returning a Promise which will be resolved with .finally() chained.

More about Promise finally() in JavaScript.

Find all documents that match selector. The result will be an array of documents.

If there are too many documents in the result to fit in memory, use Query.prototype.cursor()

Finds a single document by its _id field. findById(id) is equivalent to findOne({ _id: id }).

The id is cast based on the Schema before sending the command.

This function triggers the following middleware.

Issue a MongoDB findOneAndDelete() command by a document's _id field. In other words, findByIdAndDelete(id) is a shorthand for findOneAndDelete({ _id: id }).

This function triggers the following middleware.

Issues a mongodb findOneAndUpdate command by a document's _id field. findByIdAndUpdate(id, ...) is equivalent to findOneAndUpdate({ _id: id }, ...).

Finds a matching document, updates it according to the update arg, passing any options, and returns the found document (if any).

This function triggers the following middleware.

Declares the query a findOne operation. When executed, the first found document is passed to the callback.

The result of the query is a single document, or null if no document was found.

This function triggers the following middleware.

Issues a MongoDB findOneAndDelete command.

Finds a matching document, removes it, and returns the found document (if any).

This function triggers the following middleware.

Issues a MongoDB findOneAndReplace command.

Finds a matching document, removes it, and returns the found document (if any).

This function triggers the following middleware.

Issues a mongodb findOneAndUpdate() command.

Finds a matching document, updates it according to the update arg, passing any options, and returns the found document (if any).

This function triggers the following middleware.

Specifies a $geometry condition

The argument is assigned to the most recent path passed to where().

geometry() must come after either intersects() or within().

The object argument must contain type and coordinates properties.

For update operations, returns the value of a path in the update's $set. Useful for writing getters/setters that can work with both update operations and save().

Returns the current query filter (also known as conditions) as a POJO.

Gets a list of paths to be populated by this query

Returns the current query filter. Equivalent to getFilter().

You should use getFilter() instead of getQuery() where possible. getQuery() will likely be deprecated in a future release.

Returns the current update operations as a JSON object.

Specifies a $gt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $gte query condition.

When called with one argument, the most recent path passed to where() is used.

Cannot be used with distinct()

Specifies an $in query condition.

When called with one argument, the most recent path passed to where() is used.

Declares an intersects query for geometry().

MUST be used after where().

In Mongoose 3.7, intersects changed from a getter to a function. If you need the old syntax, use this.

Wrapper function to call isPathSelectedInclusive on a query.

Requests acknowledgement that this operation has been persisted to MongoDB's on-disk journal. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.j option

Sets the lean option.

Documents returned from queries with the lean option enabled are plain javascript objects, not Mongoose Documents. They have no save method, getters/setters, virtuals, or other Mongoose features.

Lean is great for high-performance, read-only cases, especially when combined with cursors.

If you need virtuals, getters/setters, or defaults with lean(), you need to use a plugin. See:

Specifies the maximum number of documents the query will return.

Cannot be used with distinct()

Specifies a $lt query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $lte query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a maxDistance query condition.

When called with one argument, the most recent path passed to where() is used.

Sets the maxTimeMS option. This will tell the MongoDB server to abort if the query or write op has been running for more than ms milliseconds.

Calling query.maxTimeMS(v) is equivalent to query.setOptions({ maxTimeMS: v })

Merges another Query or conditions object into this one.

When a Query is passed, conditions, field selection and options are merged.

Specifies a $mod condition, filters documents for documents whose path property is a number that is equal to remainder modulo divisor.

The model this query is associated with.

Getter/setter around the current mongoose-specific options for this query Below are the current Mongoose-specific options.

Mongoose maintains a separate object for internal options because Mongoose sends Query.prototype.options to the MongoDB server, and the above options are not relevant for the MongoDB server.

Specifies a $ne query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies a $near or $nearSphere condition

These operators return documents sorted by distance.

DEPRECATED Specifies a $nearSphere condition

Deprecated. Use query.near() instead with the spherical option set to true.

Specifies an $nin query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies arguments for a $nor condition.

Specifies arguments for an $or condition.

Make this query throw an error if no documents match the given filter. This is handy for integrating with async/await, because orFail() saves you an extra if statement to check if no document was found.

Specifies a $polygon condition

Specifies paths which should be populated with other documents.

Paths are populated after the query executes and a response is received. A separate query is then executed for each path specified for population. After a response for each query has also been returned, the results are passed to the callback.

Add post middleware to this query instance. Doesn't affect other queries.

Add pre middleware to this query instance. Doesn't affect other queries.

Get/set the current projection (AKA fields). Pass null to remove the current projection.

Unlike projection(), the select() function modifies the current projection in place. This function overwrites the existing projection.

Determines the MongoDB nodes from which to read.

Read more about how to use read preferences here.

Sets the readConcern option for the query.

Read more about how to use read concern here.

Specifies a $regex query condition.

When called with one argument, the most recent path passed to where() is used.

Declare and/or execute this query as a replaceOne() operation. MongoDB will replace the existing document and will not accept any atomic operators ($set, etc.)

Note replaceOne will not fire update middleware. Use pre('replaceOne') and post('replaceOne') instead.

This function triggers the following middleware.

Sets this query's sanitizeProjection option. If set, sanitizeProjection does two things:

With sanitizeProjection(), you can pass potentially untrusted user data to .select().

Enable or disable schema level projections for this query. Enabled by default. Set to false to include fields with select: false in the query result by default.

Specifies which document fields to include or exclude (also known as the query "projection")

When using string syntax, prefixing a path with - will flag that path as excluded. When a path does not have the - prefix, it is included. Lastly, if a path is prefixed with +, it forces inclusion of the path, which is useful for paths excluded at the schema level.

A projection must be either inclusive or exclusive. In other words, you must either list the fields to include (which excludes all others), or list the fields to exclude (which implies all other fields are included). The _id field is the only exception because MongoDB includes it by default.

Determines if field selection has been made.

Determines if exclusive field selection has been made.

Determines if inclusive field selection has been made.

Sets the MongoDB session associated with this query. Sessions are how you mark a query as part of a transaction.

Calling session(null) removes the session from this query.

Adds a $set to this query's update without changing the operation. This is useful for query middleware so you can add an update regardless of whether you use updateOne(), updateMany(), findOneAndUpdate(), etc.

Sets query options. Some options only make sense for certain operations.

The following options are only for find():

The following options are only for write operations: updateOne(), updateMany(), replaceOne(), findOneAndUpdate(), and findByIdAndUpdate():

The following options are only for find(), findOne(), findById(), findOneAndUpdate(), findOneAndReplace(), findOneAndDelete(), and findByIdAndUpdate():

The following options are only for all operations except updateOne(), updateMany(), deleteOne(), and deleteMany():

The following options are for find(), findOne(), findOneAndUpdate(), findOneAndDelete(), updateOne(), and deleteOne():

The following options are for findOneAndUpdate() and findOneAndDelete()

The following options are for all operations:

Sets the query conditions to the provided JSON object.

Sets the current update operation to new value.

Specifies a $size query condition.

When called with one argument, the most recent path passed to where() is used.

Specifies the number of documents to skip.

Cannot be used with distinct()

Specifies a $slice projection for an array.

Note: If the absolute value of the number of elements to be sliced is greater than the number of elements in the array, all array elements will be returned.

Note: If the number of elements to skip is positive and greater than the number of elements in the array, an empty array will be returned.

Note: If the number of elements to skip is negative and its absolute value is greater than the number of elements in the array, the starting position is the start of the array.

If an object is passed, values allowed are asc, desc, ascending, descending, 1, and -1.

If a string is passed, it must be a space delimited list of path names. The sort order of each path is ascending unless the path name is prefixed with - which will be treated as descending.

Cannot be used with distinct()

Sets the tailable option (for use with capped collections).

Cannot be used with distinct()

Executes the query returning a Promise which will be resolved with either the doc(s) or rejected with the error.

More about then() in JavaScript.

Converts this query to a customized, reusable query constructor with all arguments and options retained.

Runs a function fn and treats the return value of fn as the new value for the query to resolve to.

Any functions you pass to transform() will run after any post hooks.

Declare and/or execute this query as an updateMany() operation. MongoDB will update all documents that match filter (as opposed to just the first one).

Note updateMany will not fire update middleware. Use pre('updateMany') and post('updateMany') instead.

This function triggers the following middleware.

Declare and/or execute this query as an updateOne() operation. MongoDB will update only the first document that matches filter.

Note updateOne will not fire update middleware. Use pre('updateOne') and post('updateOne') instead.

This function triggers the following middleware.

Sets the specified number of mongod servers, or tag set of mongod servers, that must acknowledge this write before this write is considered successful. This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.w option

Specifies a path for use with chaining.

Defines a $within or $geoWithin argument for geo-spatial queries.

MUST be used after where().

As of Mongoose 3.7, $geoWithin is always used for queries. To change this behavior, see Query.use$geoWithin.

In Mongoose 3.7, within changed from a getter to a function. If you need the old syntax, use this.

Sets the 3 write concern parameters for this query:

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern option

If w > 1, the maximum amount of time to wait for this write to propagate through the replica set before this operation fails. The default is 0, which means no timeout.

This option is only valid for operations that write to the database:

Defaults to the schema's writeConcern.wtimeout option

Returns an asyncIterator for use with for/await/of loops This function only works for find() queries. You do not need to call this function explicitly, the JavaScript runtime will call it for you.

Returns a string representation of this query.

More about toString() in JavaScript.

Flag to opt out of using $geoWithin.

MongoDB 2.4 deprecated the use of $within, replacing it with $geoWithin. Mongoose uses $geoWithin by default (which is 100% backward compatible with $within). If you are running an older version of MongoDB, set this flag to false so your within() queries continue to work.

Determine if we can merge the given value as a query filter. Override for mquery.canMerge() to allow null

**Examples:**

Example 1 (css):
```css
const query = MyModel.find(); // `query` is an instance of `Query`
query.setOptions({ lean : true });
query.collection(MyModel.collection);
query.where('age').gte(21).exec(callback);

// You can instantiate a query directly. There is no need to do
// this unless you're an advanced user with a very good reason to.
const query = new mongoose.Query();
```

Example 2 (r):
```r
query.$where('this.comments.length === 10 || this.name.length === 5')

// or

query.$where(function () {
  return this.comments.length === 10 || this.name.length === 5;
})
```

Example 3 (csharp):
```csharp
MyModel.find().where('pets').all(['dog', 'cat', 'ferret']);
// Equivalent:
MyModel.find().all('pets', ['dog', 'cat', 'ferret']);
```

Example 4 (python):
```python
await query.find().sort({ name: 1 }).allowDiskUse(true);
// Equivalent:
await query.find().sort({ name: 1 }).allowDiskUse();
```

---

## Queries

**URL:** https://mongoosejs.com/docs/queries.html

**Contents:**
- Queries
- Executing
- Queries are Not Promises
- References to other documents
- Streaming
- Versus Aggregation
- Sorting
- Next Up

Mongoose models provide several static helper functions for CRUD operations. Each of these functions returns a mongoose Query object.

Mongoose queries can be executed by using await, or by using .then() to handle the promise returned by the query.

When executing a query, you specify your query as a JSON document. The JSON document's syntax is the same as the MongoDB shell.

What person is depends on the operation: For findOne() it is a potentially-null single document, find() a list of documents, count() the number of documents, update() the number of documents affected, etc. The API docs for Models provide more details.

Now let's look at what happens when no await is used:

In the above code, the query variable is of type Query. A Query enables you to build up a query using chaining syntax, rather than specifying a JSON object. The below 2 examples are equivalent.

A full list of Query helper functions can be found in the API docs.

Mongoose queries are not promises. Queries are thenables, meaning they have a .then() method for async/await as a convenience. However, unlike promises, calling a query's .then() executes the query, so calling then() multiple times will throw an error.

There are no joins in MongoDB but sometimes we still want references to documents in other collections. This is where population comes in. Read more about how to include documents from other collections in your query results in the population documentation.

You can stream query results from MongoDB. You need to call the Query#cursor() function to return an instance of QueryCursor.

Iterating through a Mongoose query using async iterators also creates a cursor.

Cursors are subject to cursor timeouts. By default, MongoDB will close your cursor after 10 minutes and subsequent next() calls will result in a MongoServerError: cursor id 123 not found error. To override this, set the noCursorTimeout option on your cursor.

However, cursors can still time out because of session idle timeouts. So even a cursor with noCursorTimeout set will still time out after 30 minutes of inactivity. You can read more about working around session idle timeouts in the MongoDB documentation.

Aggregation can do many of the same things that queries can. For example, below is how you can use aggregate() to find docs where name.last = 'Ghost':

However, just because you can use aggregate() doesn't mean you should. In general, you should use queries where possible, and only use aggregate() when you absolutely need to.

Unlike query results, Mongoose does not hydrate() aggregation results. Aggregation results are always POJOs, not Mongoose documents.

Also, unlike query filters, Mongoose also doesn't cast aggregation pipelines. That means you're responsible for ensuring the values you pass in to an aggregation pipeline have the correct type.

Sorting is how you can ensure your query results come back in the desired order.

When sorting with mutiple fields, the order of the sort keys determines what key MongoDB server sorts by first.

You can view the output of a single run of this block below. As you can see, age is sorted from 0 to 2 but when age is equal, sorts by weight.

Now that we've covered Queries, let's take a look at Validation.

**Examples:**

Example 1 (javascript):
```javascript
const Person = mongoose.model('Person', yourSchema);

// find each person with a last name matching 'Ghost', selecting the `name` and `occupation` fields
const person = await Person.findOne({ 'name.last': 'Ghost' }, 'name occupation');
// Prints "Space Ghost is a talk show host".
console.log('%s %s is a %s.', person.name.first, person.name.last, person.occupation);
```

Example 2 (javascript):
```javascript
// find each person with a last name matching 'Ghost'
const query = Person.findOne({ 'name.last': 'Ghost' });

// selecting the `name` and `occupation` fields
query.select('name occupation');

// execute the query at a later time
const person = await query.exec();
// Prints "Space Ghost is a talk show host."
console.log('%s %s is a %s.', person.name.first, person.name.last, person.occupation);
```

Example 3 (julia):
```julia
// With a JSON doc
await Person.
  find({
    occupation: /host/,
    'name.last': 'Ghost',
    age: { $gt: 17, $lt: 66 },
    likes: { $in: ['vaporizing', 'talking'] }
  }).
  limit(10).
  sort({ occupation: -1 }).
  select({ name: 1, occupation: 1 }).
  exec();

// Using query builder
await Person.
  find({ occupation: /host/ }).
  where('name.last').equals('Ghost').
  where('age').gt(17).lt(66).
  where('likes').in(['vaporizing', 'talking']).
  limit(10).
  sort('-occupation').
  select('name occupation').
  exec();
```

Example 4 (javascript):
```javascript
const q = MyModel.updateMany({}, { isDeleted: true });

await q.then(() => console.log('Update 2'));
// Throws "Query was already executed: Test.updateMany({}, { isDeleted: true })"
await q.then(() => console.log('Update 3'));
```

---

## Query Casting

**URL:** https://mongoosejs.com/docs/tutorials/query_casting.html

**Contents:**
- Query Casting
- The strictQuery Option
- Implicit $in

The first parameter to Model.find(), Query#find(), Model.findOne(), etc. is called filter. In older content this parameter is sometimes called query or conditions. For example:

When you execute the query using Query#exec() or Query#then(), Mongoose casts the filter to match your schema.

If Mongoose fails to cast the filter to your schema, your query will throw a CastError.

By default, Mongoose does not cast filter properties that aren't in your schema.

You can configure this behavior using the strictQuery option for schemas. This option is analogous to the strict option. Setting strictQuery to true removes non-schema properties from the filter:

To make Mongoose throw an error if your filter has a property that isn't in the schema, set strictQuery to 'throw':

Because of schemas, Mongoose knows what types fields should be, so it can provide some neat syntactic sugar. For example, if you forget to put $in on a non-array field, Mongoose will add $in for you.

**Examples:**

Example 1 (css):
```css
const query = Character.find({ name: 'Jean-Luc Picard' });
query.getFilter(); // `{ name: 'Jean-Luc Picard' }`

// Subsequent chained calls merge new properties into the filter
query.find({ age: { $gt: 50 } });
query.getFilter(); // `{ name: 'Jean-Luc Picard', age: { $gt: 50 } }`
```

Example 2 (javascript):
```javascript
// Note that `_id` and `age` are strings. Mongoose will cast `_id` to
// a MongoDB ObjectId and `age.$gt` to a number.
const query = Character.findOne({
  _id: '5cdc267dd56b5662b7b7cc0c',
  age: { $gt: '50' }
});

// `{ _id: '5cdc267dd56b5662b7b7cc0c', age: { $gt: '50' } }`
// Query hasn't been executed yet, so Mongoose hasn't casted the filter.
query.getFilter();

const doc = await query.exec();
doc.name; // "Jean-Luc Picard"

// Mongoose casted the filter, so `_id` became an ObjectId and `age.$gt`
// became a number.
query.getFilter()._id instanceof mongoose.Types.ObjectId; // true
typeof query.getFilter().age.$gt === 'number'; // true
```

Example 3 (javascript):
```javascript
const query = Character.findOne({ age: { $lt: 'not a number' } });

const err = await query.exec().then(() => null, err => err);
err instanceof mongoose.CastError; // true
// Cast to number failed for value "not a number" at path "age" for
// model "Character"
err.message;
```

Example 4 (javascript):
```javascript
const query = Character.findOne({ notInSchema: { $lt: 'not a number' } });

// No error because `notInSchema` is not defined in the schema
await query.exec();
```

---

## Queries

**URL:** https://mongoosejs.com/docs/7.x/docs/queries.html

**Contents:**
- Queries
- Executing
- Queries are Not Promises
- References to other documents
- Streaming
- Versus Aggregation
- Sorting
- Next Up

Mongoose models provide several static helper functions for CRUD operations. Each of these functions returns a mongoose Query object.

A mongoose query can be executed in one of two ways. First, if you pass in a callback function, Mongoose will execute the query asynchronously and pass the results to the callback.

A query also has a .then() function, and thus can be used as a promise.

When executing a query, you specify your query as a JSON document. The JSON document's syntax is the same as the MongoDB shell.

What person is depends on the operation: For findOne() it is a potentially-null single document, find() a list of documents, count() the number of documents, update() the number of documents affected, etc. The API docs for Models provide more details.

Now let's look at what happens when no await is used:

In the above code, the query variable is of type Query. A Query enables you to build up a query using chaining syntax, rather than specifying a JSON object. The below 2 examples are equivalent.

A full list of Query helper functions can be found in the API docs.

Mongoose queries are not promises. Queries are thenables, meaning they have a .then() method for async/await as a convenience. However, unlike promises, calling a query's .then() executes the query, so calling then() multiple times will throw an error.

There are no joins in MongoDB but sometimes we still want references to documents in other collections. This is where population comes in. Read more about how to include documents from other collections in your query results here.

You can stream query results from MongoDB. You need to call the Query#cursor() function to return an instance of QueryCursor.

Iterating through a Mongoose query using async iterators also creates a cursor.

Cursors are subject to cursor timeouts. By default, MongoDB will close your cursor after 10 minutes and subsequent next() calls will result in a MongoServerError: cursor id 123 not found error. To override this, set the noCursorTimeout option on your cursor.

However, cursors can still time out because of session idle timeouts. So even a cursor with noCursorTimeout set will still time out after 30 minutes of inactivity. You can read more about working around session idle timeouts in the MongoDB documentation.

Aggregation can do many of the same things that queries can. For example, below is how you can use aggregate() to find docs where name.last = 'Ghost':

However, just because you can use aggregate() doesn't mean you should. In general, you should use queries where possible, and only use aggregate() when you absolutely need to.

Unlike query results, Mongoose does not hydrate() aggregation results. Aggregation results are always POJOs, not Mongoose documents.

Also, unlike query filters, Mongoose also doesn't cast aggregation pipelines. That means you're responsible for ensuring the values you pass in to an aggregation pipeline have the correct type.

Sorting is how you can ensure you query results come back in the desired order.

When sorting with mutiple fields, the order of the sort keys determines what key MongoDB server sorts by first.

You can view the output of a single run of this block below. As you can see, age is sorted from 0 to 2 but when age is equal, sorts by weight.

Now that we've covered Queries, let's take a look at Validation.

**Examples:**

Example 1 (javascript):
```javascript
const Person = mongoose.model('Person', yourSchema);

// find each person with a last name matching 'Ghost', selecting the `name` and `occupation` fields
const person = await Person.findOne({ 'name.last': 'Ghost' }, 'name occupation');
// Prints "Space Ghost is a talk show host".
console.log('%s %s is a %s.', person.name.first, person.name.last, person.occupation);
```

Example 2 (javascript):
```javascript
// find each person with a last name matching 'Ghost'
const query = Person.findOne({ 'name.last': 'Ghost' });

// selecting the `name` and `occupation` fields
query.select('name occupation');

// execute the query at a later time
const person = await query.exec();
// Prints "Space Ghost is a talk show host."
console.log('%s %s is a %s.', person.name.first, person.name.last, person.occupation);
```

Example 3 (javascript):
```javascript
// With a JSON doc
await Person.
  find({
    occupation: /host/,
    'name.last': 'Ghost',
    age: { $gt: 17, $lt: 66 },
    likes: { $in: ['vaporizing', 'talking'] }
  }).
  limit(10).
  sort({ occupation: -1 }).
  select({ name: 1, occupation: 1 }).
  exec();

// Using query builder
await Person.
  find({ occupation: /host/ }).
  where('name.last').equals('Ghost').
  where('age').gt(17).lt(66).
  where('likes').in(['vaporizing', 'talking']).
  limit(10).
  sort('-occupation').
  select('name occupation').
  exec();
```

Example 4 (javascript):
```javascript
const q = MyModel.updateMany({}, { isDeleted: true });

await q.then(() => console.log('Update 2'));
// Throws "Query was already executed: Test.updateMany({}, { isDeleted: true })"
await q.then(() => console.log('Update 3'));
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/queries.html

**Contents:**
- Queries
  - Executing
  - Queries are Not Promises
  - References to other documents
  - Streaming
  - Versus Aggregation
  - Next Up

Mongoose models provide several static helper functions for CRUD operations. Each of these functions returns a mongoose Query object.

A mongoose query can be executed in one of two ways. First, if you pass in a callback function, Mongoose will execute the query asynchronously and pass the results to the callback.

A query also has a .then() function, and thus can be used as a promise.

When executing a query with a callback function, you specify your query as a JSON document. The JSON document's syntax is the same as the MongoDB shell.

Mongoose executed the query and passed the results to callback. All callbacks in Mongoose use the pattern: callback(error, result). If an error occurs executing the query, the error parameter will contain an error document, and result will be null. If the query is successful, the error parameter will be null, and the result will be populated with the results of the query.

Anywhere a callback is passed to a query in Mongoose, the callback follows the pattern callback(error, results). What results is depends on the operation: For findOne() it is a potentially-null single document, find() a list of documents, count() the number of documents, update() the number of documents affected, etc. The API docs for Models provide more detail on what is passed to the callbacks.

Now let's look at what happens when no callback is passed:

In the above code, the query variable is of type Query. A Query enables you to build up a query using chaining syntax, rather than specifying a JSON object. The below 2 examples are equivalent.

A full list of Query helper functions can be found in the API docs.

Mongoose queries are not promises. They have a .then() function for co and async/await as a convenience. However, unlike promises, calling a query's .then() can execute the query multiple times.

For example, the below code will execute 3 updateMany() calls, one because of the callback, and two because .then() is called twice.

Don't mix using callbacks and promises with queries, or you may end up with duplicate operations. That's because passing a callback to a query function immediately executes the query, and calling then() executes the query again.

Mixing promises and callbacks can lead to duplicate entries in arrays. For example, the below code inserts 2 entries into the tags array, *not just 1.

There are no joins in MongoDB but sometimes we still want references to documents in other collections. This is where population comes in. Read more about how to include documents from other collections in your query results here.

You can stream query results from MongoDB. You need to call the Query#cursor() function to return an instance of QueryCursor.

Iterating through a Mongoose query using async iterators also creates a cursor.

Cursors are subject to cursor timeouts. By default, MongoDB will close your cursor after 10 minutes and subsequent next() calls will result in a MongoError: cursor id 123 not found error. To override this, set the noCursorTimeout option on your cursor.

However, cursors can still time out because of session idle timeouts. So even a cursor with noCursorTimeout set will still time out after 30 minutes of inactivity. You can read more about working around session idle timeouts in the MongoDB documentation.

Aggregation can do many of the same things that queries can. For example, below is how you can use aggregate() to find docs where name.last = 'Ghost':

However, just because you can use aggregate() doesn't mean you should. In general, you should use queries where possible, and only use aggregate() when you absolutely need to.

Unlike query results, Mongoose does not hydrate() aggregation results. Aggregation results are always POJOs, not Mongoose documents.

Also, unlike query filters, Mongoose also doesn't cast aggregation pipelines. That means you're responsible for ensuring the values you pass in to an aggregation pipeline have the correct type.

Now that we've covered Queries, let's take a look at Validation.

**Examples:**

Example 1 (javascript):
```javascript
const Person = mongoose.model('Person', yourSchema);

// find each person with a last name matching 'Ghost', selecting the `name` and `occupation` fields
Person.findOne({ 'name.last': 'Ghost' }, 'name occupation', function (err, person) {
  if (err) return handleError(err);
  // Prints "Space Ghost is a talk show host".
  console.log('%s %s is a %s.', person.name.first, person.name.last,
    person.occupation);
});
```

Example 2 (javascript):
```javascript
// find each person with a last name matching 'Ghost'
const query = Person.findOne({ 'name.last': 'Ghost' });

// selecting the `name` and `occupation` fields
query.select('name occupation');

// execute the query at a later time
query.exec(function (err, person) {
  if (err) return handleError(err);
  // Prints "Space Ghost is a talk show host."
  console.log('%s %s is a %s.', person.name.first, person.name.last,
    person.occupation);
});
```

Example 3 (javascript):
```javascript
// With a JSON doc
Person.
  find({
    occupation: /host/,
    'name.last': 'Ghost',
    age: { $gt: 17, $lt: 66 },
    likes: { $in: ['vaporizing', 'talking'] }
  }).
  limit(10).
  sort({ occupation: -1 }).
  select({ name: 1, occupation: 1 }).
  exec(callback);

// Using query builder
Person.
  find({ occupation: /host/ }).
  where('name.last').equals('Ghost').
  where('age').gt(17).lt(66).
  where('likes').in(['vaporizing', 'talking']).
  limit(10).
  sort('-occupation').
  select('name occupation').
  exec(callback);
```

Example 4 (javascript):
```javascript
const q = MyModel.updateMany({}, { isDeleted: true }, function() {
  console.log('Update 1');
});

q.then(() => console.log('Update 2'));
q.then(() => console.log('Update 3'));
```

---
