# Mongoose - Schemas

**Pages:** 19

---

## Discriminators

**URL:** https://mongoosejs.com/docs/6.x/docs/discriminators.html

**Contents:**
- Discriminators
- The model.discriminator() function
- Discriminators save to the Event model's collection
- Discriminator keys
- Updating the discriminator key

Discriminators are a schema inheritance mechanism. They enable you to have multiple models with overlapping schemas on top of the same underlying MongoDB collection.

Suppose you wanted to track different types of events in a single collection. Every event will have a timestamp, but events that represent clicked links should have a URL. You can achieve this using the model.discriminator() function. This function takes 3 parameters, a model name, a discriminator schema and an optional key (defaults to the model name). It returns a model whose schema is the union of the base schema and the discriminator schema.

Suppose you created another discriminator to track events where a new user registered. These SignedUpEvent instances will be stored in the same collection as generic events and ClickedLinkEvent instances.

The way Mongoose tells the difference between the different discriminator models is by the 'discriminator key', which is __t by default. Mongoose adds a String path called __t to your schemas that it uses to track which discriminator this document is an instance of.

By default, Mongoose doesn't let you update the discriminator key. save() will throw an error if you attempt to update the discriminator key. And findOneAndUpdate(), updateOne(), etc. will strip out discriminator key updates.

To update a document's discriminator key, use findOneAndUpdate() or updateOne() with the overwriteDiscriminatorKey option set as follows.

**Examples:**

Example 1 (javascript):
```javascript
const options = { discriminatorKey: 'kind' };

const eventSchema = new mongoose.Schema({ time: Date }, options);
const Event = mongoose.model('Event', eventSchema);

// ClickedLinkEvent is a special type of Event that has
// a URL.
const ClickedLinkEvent = Event.discriminator('ClickedLink',
  new mongoose.Schema({ url: String }, options));

// When you create a generic event, it can't have a URL field...
const genericEvent = new Event({ time: Date.now(), url: 'google.com' });
assert.ok(!genericEvent.url);

// But a ClickedLinkEvent can
const clickedEvent = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
assert.ok(clickedEvent.url);
```

Example 2 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });


await Promise.all([event1.save(), event2.save(), event3.save()]);
const count = await Event.countDocuments();
assert.equal(count, 3);
```

Example 3 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });

assert.ok(!event1.__t);
assert.equal(event2.__t, 'ClickedLink');
assert.equal(event3.__t, 'SignedUp');
```

Example 4 (javascript):
```javascript
let event = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
await event.save();

event.__t = 'SignedUp';
// ValidationError: ClickedLink validation failed: __t: Cast to String failed for value "SignedUp" (type string) at path "__t"
  await event.save();

event = await ClickedLinkEvent.findByIdAndUpdate(event._id, { __t: 'SignedUp' }, { new: true });
event.__t; // 'ClickedLink', update was a no-op
```

---

## SchemaTypes

**URL:** https://mongoosejs.com/docs/6.x/docs/schematypes.html

**Contents:**
- SchemaTypes
- What is a SchemaType?
  - Example
- The type Key
- SchemaType Options
  - All Schema Types
  - Indexes
  - String
  - Number
  - Date

SchemaTypes handle definition of path defaults, validation, getters, setters, field selection defaults for queries, and other general characteristics for Mongoose document properties.

You can think of a Mongoose schema as the configuration object for a Mongoose model. A SchemaType is then a configuration object for an individual property. A SchemaType says what type a given path should have, whether it has any getters/setters, and what values are valid for that path.

A SchemaType is different from a type. In other words, mongoose.ObjectId !== mongoose.Types.ObjectId. A SchemaType is just a configuration object for Mongoose. An instance of the mongoose.ObjectId SchemaType doesn't actually create MongoDB ObjectIds, it is just a configuration for a path in a schema.

The following are all the valid SchemaTypes in Mongoose. Mongoose plugins can also add custom SchemaTypes like int32. Check out Mongoose's plugins search to find plugins.

type is a special property in Mongoose schemas. When Mongoose finds a nested property named type in your schema, Mongoose assumes that it needs to define a SchemaType with the given type.

As a consequence, you need a little extra work to define a property named type in your schema. For example, suppose you're building a stock portfolio app, and you want to store the asset's type (stock, bond, ETF, etc.). Naively, you might define your schema as shown below:

However, when Mongoose sees type: String, it assumes that you mean asset should be a string, not an object with a property type. The correct way to define an object with a property type is shown below.

You can declare a schema type using the type directly, or an object with a type property.

In addition to the type property, you can specify additional properties for a path. For example, if you want to lowercase a string before saving:

You can add any property you want to your SchemaType options. Many plugins rely on custom SchemaType options. For example, the mongoose-autopopulate plugin automatically populates paths if you set autopopulate: true in your SchemaType options. Mongoose comes with support for several built-in SchemaType options, like lowercase in the above example.

The lowercase option only works for strings. There are certain options which apply for all schema types, and some that apply for specific schema types.

You can also define MongoDB indexes using schema type options.

To declare a path as a string, you may use either the String global constructor or the string 'String'.

If you pass an element that has a toString() function, Mongoose will call it, unless the element is an array or the toString() function is strictly equal to Object.prototype.toString().

To declare a path as a number, you may use either the Number global constructor or the string 'Number'.

There are several types of values that will be successfully cast to a Number.

If you pass an object with a valueOf() function that returns a Number, Mongoose will call it and assign the returned value to the path.

The values null and undefined are not cast.

NaN, strings that cast to NaN, arrays, and objects that don't have a valueOf() function will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated.

Built-in Date methods are not hooked into the mongoose change tracking logic which in English means that if you use a Date in your document and modify it with a method like setMonth(), mongoose will be unaware of this change and doc.save() will not persist this modification. If you must modify Date types using built-in methods, tell mongoose about the change with doc.markModified('pathToYourDate') before saving.

To declare a path as a Buffer, you may use either the Buffer global constructor or the string 'Buffer'.

Mongoose will successfully cast the below values to buffers.

An "anything goes" SchemaType. Mongoose will not do any casting on mixed paths. You can define a mixed path using Schema.Types.Mixed or by passing an empty object literal. The following are equivalent.

Since Mixed is a schema-less type, you can change the value to anything else you like, but Mongoose loses the ability to auto detect and save those changes. To tell Mongoose that the value of a Mixed type has changed, you need to call doc.markModified(path), passing the path to the Mixed type you just changed.

To avoid these side-effects, a Subdocument path may be used instead.

An ObjectId is a special type typically used for unique identifiers. Here's how you declare a schema with a path driver that is an ObjectId:

ObjectId is a class, and ObjectIds are objects. However, they are often represented as strings. When you convert an ObjectId to a string using toString(), you get a 24-character hexadecimal string:

Booleans in Mongoose are plain JavaScript booleans. By default, Mongoose casts the below values to true:

Mongoose casts the below values to false:

Any other value causes a CastError. You can modify what values Mongoose converts to true or false using the convertToTrue and convertToFalse properties, which are JavaScript sets.

Mongoose supports arrays of SchemaTypes and arrays of subdocuments. Arrays of SchemaTypes are also called primitive arrays, and arrays of subdocuments are also called document arrays.

Arrays are special because they implicitly have a default value of [] (empty array).

To overwrite this default, you need to set the default value to undefined

Note: specifying an empty array is equivalent to Mixed. The following all create arrays of Mixed:

A MongooseMap is a subclass of JavaScript's Map class. In these docs, we'll use the terms 'map' and MongooseMap interchangeably. In Mongoose, maps are how you create a nested document with arbitrary keys.

Note: In Mongoose Maps, keys must be strings in order to store the document in MongoDB.

The above example doesn't explicitly declare github or twitter as paths, but, since socialMediaHandles is a map, you can store arbitrary key/value pairs. However, since socialMediaHandles is a map, you must use .get() to get the value of a key and .set() to set the value of a key.

Map types are stored as BSON objects in MongoDB. Keys in a BSON object are ordered, so this means the insertion order property of maps is maintained.

Mongoose supports a special $* syntax to populate all elements in a map. For example, suppose your socialMediaHandles map contains a ref:

To populate every socialMediaHandles entry's oauth property, you should populate on socialMediaHandles.$*.oauth:

Mongoose also supports a UUID type that stores UUID instances as Node.js buffers. We recommend using ObjectIds rather than UUIDs for unique document ids in Mongoose, but you may use UUIDs if you need to.

In Node.js, a UUID is represented as an instance of bson.Binary type with a getter that converts the binary to a string when you access it. Mongoose stores UUIDs as binary data with subtype 4 in MongoDB.

To create UUIDs, we recommend using Node's built-in UUIDv4 generator.

Getters are like virtuals for paths defined in your schema. For example, let's say you wanted to store user profile pictures as relative paths and then add the hostname in your application. Below is how you would structure your userSchema:

Generally, you only use getters on primitive paths as opposed to arrays or subdocuments. Because getters override what accessing a Mongoose path returns, declaring a getter on an object may remove Mongoose change tracking for that path.

Instead of declaring a getter on the array as shown above, you should declare a getter on the url string as shown below. If you need to declare a getter on a nested document or array, be very careful!

To declare a path as another schema, set type to the sub-schema's instance.

To set a default value based on the sub-schema's shape, simply set a default value, and the value will be cast based on the sub-schema's definition before being set during document creation.

Mongoose can also be extended with custom SchemaTypes. Search the plugins site for compatible types like mongoose-long, mongoose-int32, and mongoose-function.

Read more about creating custom SchemaTypes here.

The schema.path() function returns the instantiated schema type for a given path.

You can use this function to inspect the schema type for a given path, including what validators it has and what the type is.

Now that we've covered SchemaTypes, let's take a look at Connections.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({ name: String });
schema.path('name') instanceof mongoose.SchemaType; // true
schema.path('name') instanceof mongoose.Schema.Types.String; // true
schema.path('name').instance; // 'String'
```

Example 2 (javascript):
```javascript
const schema = new Schema({
  name: String,
  binary: Buffer,
  living: Boolean,
  updated: { type: Date, default: Date.now },
  age: { type: Number, min: 18, max: 65 },
  mixed: Schema.Types.Mixed,
  _someId: Schema.Types.ObjectId,
  decimal: Schema.Types.Decimal128,
  array: [],
  ofString: [String],
  ofNumber: [Number],
  ofDates: [Date],
  ofBuffer: [Buffer],
  ofBoolean: [Boolean],
  ofMixed: [Schema.Types.Mixed],
  ofObjectId: [Schema.Types.ObjectId],
  ofArrays: [[]],
  ofArrayOfNumbers: [[Number]],
  nested: {
    stuff: { type: String, lowercase: true, trim: true }
  },
  map: Map,
  mapOfString: {
    type: Map,
    of: String
  }
});

// example use

const Thing = mongoose.model('Thing', schema);

const m = new Thing;
m.name = 'Statue of Liberty';
m.age = 125;
m.updated = new Date;
m.binary = Buffer.alloc(0);
m.living = false;
m.mixed = { any: { thing: 'i want' } };
m.markModified('mixed');
m._someId = new mongoose.Types.ObjectId;
m.array.push(1);
m.ofString.push('strings!');
m.ofNumber.unshift(1, 2, 3, 4);
m.ofDates.addToSet(new Date);
m.ofBuffer.pop();
m.ofMixed = [1, [], 'three', { four: 5 }];
m.nested.stuff = 'good';
m.map = new Map([['key', 'value']]);
m.save(callback);
```

Example 3 (javascript):
```javascript
// 3 string SchemaTypes: 'name', 'nested.firstName', 'nested.lastName'
const schema = new Schema({
  name: { type: String },
  nested: {
    firstName: { type: String },
    lastName: { type: String }
  }
});
```

Example 4 (javascript):
```javascript
const holdingSchema = new Schema({
  // You might expect `asset` to be an object that has 2 properties,
  // but unfortunately `type` is special in Mongoose so mongoose
  // interprets this schema to mean that `asset` is a string
  asset: {
    type: String,
    ticker: String
  }
});
```

---

## SchemaTypes

**URL:** https://mongoosejs.com/docs/schematypes.html

**Contents:**
- SchemaTypes
- What is a SchemaType?
  - Example
- The type Key
- SchemaType Options
  - All Schema Types
  - Indexes
  - String
  - Number
  - Date

SchemaTypes handle definition of path defaults, validation, getters, setters, field selection defaults for queries, and other general characteristics for Mongoose document properties.

You can think of a Mongoose schema as the configuration object for a Mongoose model. A SchemaType is then a configuration object for an individual property. A SchemaType says what type a given path should have, whether it has any getters/setters, and what values are valid for that path.

A SchemaType is different from a type. In other words, mongoose.ObjectId !== mongoose.Types.ObjectId. A SchemaType is just a configuration object for Mongoose. An instance of the mongoose.ObjectId SchemaType doesn't actually create MongoDB ObjectIds, it is just a configuration for a path in a schema.

The following are all the valid SchemaTypes in Mongoose. Mongoose plugins can also add custom SchemaTypes like int32. Check out Mongoose's plugins search to find plugins.

type is a special property in Mongoose schemas. When Mongoose finds a nested property named type in your schema, Mongoose assumes that it needs to define a SchemaType with the given type.

As a consequence, you need a little extra work to define a property named type in your schema. For example, suppose you're building a stock portfolio app, and you want to store the asset's type (stock, bond, ETF, etc.). Naively, you might define your schema as shown below:

However, when Mongoose sees type: String, it assumes that you mean asset should be a string, not an object with a property type. The correct way to define an object with a property type is shown below.

You can declare a schema type using the type directly, or an object with a type property.

In addition to the type property, you can specify additional properties for a path. For example, if you want to lowercase a string before saving:

You can add any property you want to your SchemaType options. Many plugins rely on custom SchemaType options. For example, the mongoose-autopopulate plugin automatically populates paths if you set autopopulate: true in your SchemaType options. Mongoose comes with support for several built-in SchemaType options, like lowercase in the above example.

The lowercase option only works for strings. There are certain options which apply for all schema types, and some that apply for specific schema types.

You can also define MongoDB indexes using schema type options.

To declare a path as a string, you may use either the String global constructor or the string 'String'.

If you pass an element that has a toString() function, Mongoose will call it, unless the element is an array or the toString() function is strictly equal to Object.prototype.toString().

To declare a path as a number, you may use either the Number global constructor or the string 'Number'.

There are several types of values that will be successfully cast to a Number.

If you pass an object with a valueOf() function that returns a Number, Mongoose will call it and assign the returned value to the path.

The values null and undefined are not cast.

NaN, strings that cast to NaN, arrays, and objects that don't have a valueOf() function will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated.

Built-in Date methods are not hooked into the mongoose change tracking logic which in English means that if you use a Date in your document and modify it with a method like setMonth(), mongoose will be unaware of this change and doc.save() will not persist this modification. If you must modify Date types using built-in methods, tell mongoose about the change with doc.markModified('pathToYourDate') before saving.

To declare a path as a Buffer, you may use either the Buffer global constructor or the string 'Buffer'.

Mongoose will successfully cast the below values to buffers.

An "anything goes" SchemaType. Mongoose will not do any casting on mixed paths. You can define a mixed path using Schema.Types.Mixed or by passing an empty object literal. The following are equivalent.

Since Mixed is a schema-less type, you can change the value to anything else you like, but Mongoose loses the ability to auto detect and save those changes. To tell Mongoose that the value of a Mixed type has changed, you need to call doc.markModified(path), passing the path to the Mixed type you just changed.

To avoid these side-effects, a Subdocument path may be used instead.

An ObjectId is a special type typically used for unique identifiers. Here's how you declare a schema with a path driver that is an ObjectId:

ObjectId is a class, and ObjectIds are objects. However, they are often represented as strings. When you convert an ObjectId to a string using toString(), you get a 24-character hexadecimal string:

Booleans in Mongoose are plain JavaScript booleans. By default, Mongoose casts the below values to true:

Mongoose casts the below values to false:

Any other value causes a CastError. You can modify what values Mongoose converts to true or false using the convertToTrue and convertToFalse properties, which are JavaScript sets.

Mongoose supports arrays of SchemaTypes and arrays of subdocuments. Arrays of SchemaTypes are also called primitive arrays, and arrays of subdocuments are also called document arrays.

Arrays are special because they implicitly have a default value of [] (empty array).

To overwrite this default, you need to set the default value to undefined

Note: specifying an empty array is equivalent to Mixed. The following all create arrays of Mixed:

A MongooseMap is a subclass of JavaScript's Map class. In these docs, we'll use the terms 'map' and MongooseMap interchangeably. In Mongoose, maps are how you create a nested document with arbitrary keys.

Note: In Mongoose Maps, keys must be strings in order to store the document in MongoDB.

The above example doesn't explicitly declare github or twitter as paths, but, since socialMediaHandles is a map, you can store arbitrary key/value pairs. However, since socialMediaHandles is a map, you must use .get() to get the value of a key and .set() to set the value of a key.

Map types are stored as BSON objects in MongoDB. Keys in a BSON object are ordered, so this means the insertion order property of maps is maintained.

Mongoose supports a special $* syntax to populate all elements in a map. For example, suppose your socialMediaHandles map contains a ref:

To populate every socialMediaHandles entry's oauth property, you should populate on socialMediaHandles.$*.oauth:

Mongoose also supports a UUID type that stores UUID instances as Node.js buffers. We recommend using ObjectIds rather than UUIDs for unique document ids in Mongoose, but you may use UUIDs if you need to.

In Node.js, a UUID is represented as an instance of bson.Binary type with a getter that converts the binary to a string when you access it. Mongoose stores UUIDs as binary data with subtype 4 in MongoDB.

To create UUIDs, we recommend using Node's built-in UUIDv4 generator.

Mongoose supports JavaScript BigInts as a SchemaType. BigInts are stored as 64-bit integers in MongoDB (BSON type "long").

Mongoose supports 64-bit IEEE 754-2008 floating point numbers as a SchemaType. Doubles are stored as BSON type "double" in MongoDB.

There are several types of values that will be successfully cast to a Double.

The following inputs will result will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated:

Mongoose supports 32-bit integers as a SchemaType. Int32s are stored as 32-bit integers in MongoDB (BSON type "int").

There are several types of values that will be successfully cast to a Number.

If you pass an object with a valueOf() function that returns a Number, Mongoose will call it and assign the returned value to the path.

The values null and undefined are not cast.

The following inputs will result will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated:

The Union SchemaType allows a path to accept multiple types. Mongoose will attempt to cast the value to one of the specified types.

When you set a value on a Union path, Mongoose tries to cast it to each type in the of array in order. If the value matches one of the types exactly (using ===), Mongoose uses that value. Otherwise, Mongoose uses the first type that successfully casts the value.

If Mongoose cannot cast the value to any of the specified types, it throws the error from the last type in the union.

You can specify options for individual types in the union, such as trim for strings.

Union types work with queries and updates. Mongoose casts query filters and update operations according to the union types.

Getters are like virtuals for paths defined in your schema. For example, let's say you wanted to store user profile pictures as relative paths and then add the hostname in your application. Below is how you would structure your userSchema:

Generally, you only use getters on primitive paths as opposed to arrays or subdocuments. Because getters override what accessing a Mongoose path returns, declaring a getter on an object may remove Mongoose change tracking for that path.

Instead of declaring a getter on the array as shown above, you should declare a getter on the url string as shown below. If you need to declare a getter on a nested document or array, be very careful!

To declare a path as another schema, set type to the sub-schema's instance.

To set a default value based on the sub-schema's shape, simply set a default value, and the value will be cast based on the sub-schema's definition before being set during document creation.

Mongoose can also be extended with custom SchemaTypes. Search the plugins site for compatible types like mongoose-long, mongoose-int32, and mongoose-function.

Read more about creating custom SchemaTypes in our Custom SchemaTypes guide.

The schema.path() function returns the instantiated schema type for a given path.

You can use this function to inspect the schema type for a given path, including what validators it has and what the type is.

Now that we've covered SchemaTypes, let's take a look at Connections.

**Examples:**

Example 1 (css):
```css
const schema = new Schema({ name: String });
schema.path('name') instanceof mongoose.SchemaType; // true
schema.path('name') instanceof mongoose.Schema.Types.String; // true
schema.path('name').instance; // 'String'
```

Example 2 (swift):
```swift
const schema = new Schema({
  name: String,
  binary: Buffer,
  living: Boolean,
  updated: { type: Date, default: Date.now },
  age: { type: Number, min: 18, max: 65 },
  mixed: Schema.Types.Mixed,
  union: { type: Schema.Types.Union, of: [String, Number] },
  _someId: Schema.Types.ObjectId,
  decimal: Schema.Types.Decimal128,
  double: Schema.Types.Double,
  int32bit: Schema.Types.Int32,
  array: [],
  ofString: [String],
  ofNumber: [Number],
  ofDates: [Date],
  ofBuffer: [Buffer],
  ofBoolean: [Boolean],
  ofMixed: [Schema.Types.Mixed],
  ofObjectId: [Schema.Types.ObjectId],
  ofArrays: [[]],
  ofArrayOfNumbers: [[Number]],
  nested: {
    stuff: { type: String, lowercase: true, trim: true }
  },
  map: Map,
  mapOfString: {
    type: Map,
    of: String
  }
});

// example use

const Thing = mongoose.model('Thing', schema);

const m = new Thing;
m.name = 'Statue of Liberty';
m.age = 125;
m.updated = new Date;
m.binary = Buffer.alloc(0);
m.living = false;
m.mixed = { any: { thing: 'i want' } };
m.markModified('mixed');
m._someId = new mongoose.Types.ObjectId;
m.array.push(1);
m.ofString.push('strings!');
m.ofNumber.unshift(1, 2, 3, 4);
m.ofDates.addToSet(new Date);
m.ofBuffer.pop();
m.ofMixed = [1, [], 'three', { four: 5 }];
m.nested.stuff = 'good';
m.map = new Map([['key', 'value']]);
m.save(callback);
```

Example 3 (css):
```css
// 3 string SchemaTypes: 'name', 'nested.firstName', 'nested.lastName'
const schema = new Schema({
  name: { type: String },
  nested: {
    firstName: { type: String },
    lastName: { type: String }
  }
});
```

Example 4 (css):
```css
const holdingSchema = new Schema({
  // You might expect `asset` to be an object that has 2 properties,
  // but unfortunately `type` is special in Mongoose so mongoose
  // interprets this schema to mean that `asset` is a string
  asset: {
    type: String,
    ticker: String
  }
});
```

---

## Subdocuments

**URL:** https://mongoosejs.com/docs/6.x/docs/subdocs.html

**Contents:**
- Subdocuments
- What is a Subdocument?
- Subdocuments versus Nested Paths
- Subdocument Defaults
- Finding a Subdocument
- Adding Subdocs to Arrays
- Removing Subdocs
- Parents of Subdocs
  - Alternate declaration syntax for arrays
  - Next Up

Subdocuments are documents embedded in other documents. In Mongoose, this means you can nest schemas in other schemas. Mongoose has two distinct notions of subdocuments: arrays of subdocuments and single nested subdocuments.

Note that populated documents are not subdocuments in Mongoose. Subdocument data is embedded in the top-level document. Referenced documents are separate top-level documents.

Subdocuments are similar to normal documents. Nested schemas can have middleware, custom validation logic, virtuals, and any other feature top-level schemas can use. The major difference is that subdocuments are not saved individually, they are saved whenever their top-level parent document is saved.

Subdocuments have save and validate middleware just like top-level documents. Calling save() on the parent document triggers the save() middleware for all its subdocuments, and the same for validate() middleware.

Subdocuments' pre('save') and pre('validate') middleware execute before the top-level document's pre('save') but after the top-level document's pre('validate') middleware. This is because validating before save() is actually a piece of built-in middleware.

In Mongoose, nested paths are subtly different from subdocuments. For example, below are two schemas: one with child as a subdocument, and one with child as a nested path.

These two schemas look similar, and the documents in MongoDB will have the same structure with both schemas. But there are a few Mongoose-specific differences:

First, instances of Nested never have child === undefined. You can always set subproperties of child, even if you don't set the child property. But instances of Subdoc can have child === undefined.

Subdocument paths are undefined by default, and Mongoose does not apply subdocument defaults unless you set the subdocument path to a non-nullish value.

However, if you set doc.child to any object, Mongoose will apply the age default if necessary.

Mongoose applies defaults recursively, which means there's a nice workaround if you want to make sure Mongoose applies subdocument defaults: make the subdocument path default to an empty object.

Each subdocument has an _id by default. Mongoose document arrays have a special id method for searching a document array to find a document with a given _id.

MongooseArray methods such as push, unshift, addToSet, and others cast arguments to their proper types transparently:

You can also create a subdocument without adding it to an array by using the create() method of Document Arrays.

Each subdocument has its own remove method. For an array subdocument, this is equivalent to calling .pull() on the subdocument. For a single nested subdocument, remove() is equivalent to setting the subdocument to null.

Sometimes, you need to get the parent of a subdoc. You can access the parent using the parent() function.

If you have a deeply nested subdoc, you can access the top-level document using the ownerDocument() function.

If you create a schema with an array of objects, Mongoose will automatically convert the object to a schema for you:

Now that we've covered Subdocuments, let's take a look at querying.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });

const parentSchema = new Schema({
  // Array of subdocuments
  children: [childSchema],
  // Single nested subdocuments
  child: childSchema
});
```

Example 2 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });
const Child = mongoose.model('Child', childSchema);

const parentSchema = new Schema({
  child: {
    type: mongoose.ObjectId,
    ref: 'Child'
  }
});
const Parent = mongoose.model('Parent', parentSchema);

const doc = await Parent.findOne().populate('child');
// NOT a subdocument. `doc.child` is a separate top-level document.
doc.child;
```

Example 3 (javascript):
```javascript
const Parent = mongoose.model('Parent', parentSchema);
const parent = new Parent({ children: [{ name: 'Matt' }, { name: 'Sarah' }] });
parent.children[0].name = 'Matthew';

// `parent.children[0].save()` is a no-op, it triggers middleware but
// does **not** actually save the subdocument. You need to save the parent
// doc.
parent.save(callback);
```

Example 4 (javascript):
```javascript
childSchema.pre('save', function(next) {
  if ('invalid' == this.name) {
    return next(new Error('#sadpanda'));
  }
  next();
});

const parent = new Parent({ children: [{ name: 'invalid' }] });
parent.save(function(err) {
  console.log(err.message); // #sadpanda
});
```

---

## Schema

**URL:** https://mongoosejs.com/docs/5.x/docs/api/schema.html

**Contents:**
- Schema
  - Schema()
      - Parameters
      - Inherits:
    - Example:
    - Options:
    - Options for Nested Schemas:
    - Note:
  - Schema.Types
      - Type:

When nesting schemas, (children in the example above), always declare the child schema first before passing it into its parent.

The various built-in Mongoose Schema Types.

Using this exposed access to the Mixed SchemaType, we can use them in our schema.

The allowed index types

Adds key path / schema type pairs to this schema.

Array of child schemas (from document arrays and single nested subdocs) and their corresponding compiled models. Each element of the array is an object with 2 properties: schema and model.

This property is typically only useful for plugin authors and advanced users. You do not need to interact with this property at all to use mongoose.

Returns a deep copy of the schema

Iterates the schemas paths similar to Array#forEach.

The callback is passed the pathname and the schemaType instance.

Gets a schema option.

Defines an index (most likely compound) for this schema.

Returns a list of indexes that this schema declares, via schema.index() or by index: true in a path's options. Indexes are expressed as an array [spec, options].

Plugins can use the return value of this function to modify a schema's indexes. For example, the below plugin makes every index unique by default.

Loads an ES6 class into a schema. Maps setters + getters, static methods, and instance methods to schema virtuals, statics, and methods.

Adds an instance method to documents constructed from Models compiled from this schema.

If a hash of name/fn pairs is passed as the only argument, each name/fn pair will be added as methods.

NOTE: Schema.method() adds instance methods to the Schema.methods object. You can also add instance methods directly to the Schema.methods object as seen in the guide

The original object passed to the schema constructor

Gets/sets schema paths.

Sets a path (if arity 2) Gets a path (if arity 1)

Returns the pathType of path for this schema.

Given a path, returns whether it is a real, virtual, nested, or ad-hoc/undefined path.

The paths defined on this schema. The keys are the top-level paths in this schema, and the values are instances of the SchemaType class.

Returns a new schema that has the picked paths from this schema.

This method is analagous to Lodash's pick() function for Mongoose schemas.

Registers a plugin for this schema.

Defines a post hook for the document

Defines a pre hook for the model.

Adds a method call to the queue.

Removes the given path (or [paths]).

Returns an Array of path strings that are required by this schema.

Sets a schema option.

Adds static "class" methods to Models compiled from this schema.

If a hash of name/fn pairs is passed as the only argument, each name/fn pair will be added as statics.

Creates a virtual type with the given name.

Returns the virtual type with the given name.

Reserved document keys.

Keys in this object are names that are rejected in schema declarations because they conflict with Mongoose functionality. If you create a schema using new Schema() with one of these property names, Mongoose will throw an error.

NOTE: Use of these terms as method names is permitted, but play at your own risk, as they may be existing mongoose document methods you are stomping on.

**Examples:**

Example 1 (swift):
```swift
const child = new Schema({ name: String });
const schema = new Schema({ name: String, age: Number, children: [child] });
const Tree = mongoose.model('Tree', schema);

// setting schema options
new Schema({ name: String }, { _id: false, autoIndex: false })
```

Example 2 (javascript):
```javascript
const mongoose = require('mongoose');
const ObjectId = mongoose.Schema.Types.ObjectId;
```

Example 3 (css):
```css
const Mixed = mongoose.Schema.Types.Mixed;
new mongoose.Schema({ _user: Mixed })
```

Example 4 (css):
```css
const ToySchema = new Schema();
ToySchema.add({ name: 'string', color: 'string', price: 'number' });

const TurboManSchema = new Schema();
// You can also `add()` another schema and copy over all paths, virtuals,
// getters, setters, indexes, methods, and statics.
TurboManSchema.add(ToySchema).add({ year: Number });
```

---

## Discriminators

**URL:** https://mongoosejs.com/docs/discriminators.html

**Contents:**
- Discriminators
- The model.discriminator() function
- Discriminators save to the Event model's collection
- Discriminator keys
- Updating the discriminator key
- Embedded discriminators in arrays
- Single nested discriminators

Discriminators are a schema inheritance mechanism. They enable you to have multiple models with overlapping schemas on top of the same underlying MongoDB collection.

Suppose you wanted to track different types of events in a single collection. Every event will have a timestamp, but events that represent clicked links should have a URL. You can achieve this using the model.discriminator() function. This function takes 3 parameters, a model name, a discriminator schema and an optional key (defaults to the model name). It returns a model whose schema is the union of the base schema and the discriminator schema.

Suppose you created another discriminator to track events where a new user registered. These SignedUpEvent instances will be stored in the same collection as generic events and ClickedLinkEvent instances.

The way Mongoose tells the difference between the different discriminator models is by the 'discriminator key', which is __t by default. Mongoose adds a String path called __t to your schemas that it uses to track which discriminator this document is an instance of.

By default, Mongoose doesn't let you update the discriminator key. save() will throw an error if you attempt to update the discriminator key. And findOneAndUpdate(), updateOne(), etc. will strip out discriminator key updates.

To update a document's discriminator key, use findOneAndUpdate() or updateOne() with the overwriteDiscriminatorKey option set as follows.

You can also define discriminators on embedded document arrays. Embedded discriminators are different because the different discriminator types are stored in the same document array (within a document) rather than the same collection. In other words, embedded discriminators let you store subdocuments matching different schemas in the same array.

As a general best practice, make sure you declare any hooks on your schemas before you use them. You should not call pre() or post() after calling discriminator().

You can also define discriminators on single nested subdocuments, similar to how you can define discriminators on arrays of subdocuments.

As a general best practice, make sure you declare any hooks on your schemas before you use them. You should not call pre() or post() after calling discriminator().

**Examples:**

Example 1 (css):
```css
const options = { discriminatorKey: 'kind' };

const eventSchema = new mongoose.Schema({ time: Date }, options);
const Event = mongoose.model('Event', eventSchema);

// ClickedLinkEvent is a special type of Event that has
// a URL.
const ClickedLinkEvent = Event.discriminator('ClickedLink',
  new mongoose.Schema({ url: String }, options));

// When you create a generic event, it can't have a URL field...
const genericEvent = new Event({ time: Date.now(), url: 'google.com' });
assert.ok(!genericEvent.url);

// But a ClickedLinkEvent can
const clickedEvent = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
assert.ok(clickedEvent.url);
```

Example 2 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });


await Promise.all([event1.save(), event2.save(), event3.save()]);
const count = await Event.countDocuments();
assert.equal(count, 3);
```

Example 3 (css):
```css
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });

assert.ok(!event1.__t);
assert.equal(event2.__t, 'ClickedLink');
assert.equal(event3.__t, 'SignedUp');
```

Example 4 (swift):
```swift
let event = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
await event.save();

event.__t = 'SignedUp';
// ValidationError: ClickedLink validation failed: __t: Cast to String failed for value "SignedUp" (type string) at path "__t"
  await event.save();

event = await ClickedLinkEvent.findByIdAndUpdate(event._id, { __t: 'SignedUp' }, { new: true });
event.__t; // 'ClickedLink', update was a no-op
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/schematypes.html

**Contents:**
- SchemaTypes
  - What is a SchemaType?
    - Example
  - The `type` Key
  - SchemaType Options
      - All Schema Types
      - Indexes
      - String
      - Number
      - Date

SchemaTypes handle definition of path defaults, validation, getters, setters, field selection defaults for queries, and other general characteristics for Mongoose document properties.

You can think of a Mongoose schema as the configuration object for a Mongoose model. A SchemaType is then a configuration object for an individual property. A SchemaType says what type a given path should have, whether it has any getters/setters, and what values are valid for that path.

A SchemaType is different from a type. In other words, mongoose.ObjectId !== mongoose.Types.ObjectId. A SchemaType is just a configuration object for Mongoose. An instance of the mongoose.ObjectId SchemaType doesn't actually create MongoDB ObjectIds, it is just a configuration for a path in a schema.

The following are all the valid SchemaTypes in Mongoose. Mongoose plugins can also add custom SchemaTypes like int32. Check out Mongoose's plugins search to find plugins.

type is a special property in Mongoose schemas. When Mongoose finds a nested property named type in your schema, Mongoose assumes that it needs to define a SchemaType with the given type.

As a consequence, you need a little extra work to define a property named type in your schema. For example, suppose you're building a stock portfolio app, and you want to store the asset's type (stock, bond, ETF, etc.). Naively, you might define your schema as shown below:

However, when Mongoose sees type: String, it assumes that you mean asset should be a string, not an object with a property type. The correct way to define an object with a property type is shown below.

You can declare a schema type using the type directly, or an object with a type property.

In addition to the type property, you can specify additional properties for a path. For example, if you want to lowercase a string before saving:

You can add any property you want to your SchemaType options. Many plugins rely on custom SchemaType options. For example, the mongoose-autopopulate plugin automatically populates paths if you set autopopulate: true in your SchemaType options. Mongoose comes with support for several built-in SchemaType options, like lowercase in the above example.

The lowercase option only works for strings. There are certain options which apply for all schema types, and some that apply for specific schema types.

You can also define MongoDB indexes using schema type options.

To declare a path as a string, you may use either the String global constructor or the string 'String'.

If you pass an element that has a toString() function, Mongoose will call it, unless the element is an array or the toString() function is strictly equal to Object.prototype.toString().

To declare a path as a number, you may use either the Number global constructor or the string 'Number'.

There are several types of values that will be successfully cast to a Number.

If you pass an object with a valueOf() function that returns a Number, Mongoose will call it and assign the returned value to the path.

The values null and undefined are not cast.

NaN, strings that cast to NaN, arrays, and objects that don't have a valueOf() function will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated.

Built-in Date methods are not hooked into the mongoose change tracking logic which in English means that if you use a Date in your document and modify it with a method like setMonth(), mongoose will be unaware of this change and doc.save() will not persist this modification. If you must modify Date types using built-in methods, tell mongoose about the change with doc.markModified('pathToYourDate') before saving.

To declare a path as a Buffer, you may use either the Buffer global constructor or the string 'Buffer'.

Mongoose will successfully cast the below values to buffers.

An "anything goes" SchemaType. Mongoose will not do any casting on mixed paths. You can define a mixed path using Schema.Types.Mixed or by passing an empty object literal. The following are equivalent.

Since Mixed is a schema-less type, you can change the value to anything else you like, but Mongoose loses the ability to auto detect and save those changes. To tell Mongoose that the value of a Mixed type has changed, you need to call doc.markModified(path), passing the path to the Mixed type you just changed.

To avoid these side-effects, a Subdocument path may be used instead.

An ObjectId is a special type typically used for unique identifiers. Here's how you declare a schema with a path driver that is an ObjectId:

ObjectId is a class, and ObjectIds are objects. However, they are often represented as strings. When you convert an ObjectId to a string using toString(), you get a 24-character hexadecimal string:

Booleans in Mongoose are plain JavaScript booleans. By default, Mongoose casts the below values to true:

Mongoose casts the below values to false:

Any other value causes a CastError. You can modify what values Mongoose converts to true or false using the convertToTrue and convertToFalse properties, which are JavaScript sets.

Mongoose supports arrays of SchemaTypes and arrays of subdocuments. Arrays of SchemaTypes are also called primitive arrays, and arrays of subdocuments are also called document arrays.

Arrays are special because they implicitly have a default value of [] (empty array).

To overwrite this default, you need to set the default value to undefined

Note: specifying an empty array is equivalent to Mixed. The following all create arrays of Mixed:

New in Mongoose 5.1.0

A MongooseMap is a subclass of JavaScript's Map class. In these docs, we'll use the terms 'map' and MongooseMap interchangeably. In Mongoose, maps are how you create a nested document with arbitrary keys.

Note: In Mongoose Maps, keys must be strings in order to store the document in MongoDB.

The above example doesn't explicitly declare github or twitter as paths, but, since socialMediaHandles is a map, you can store arbitrary key/value pairs. However, since socialMediaHandles is a map, you must use .get() to get the value of a key and .set() to set the value of a key.

Map types are stored as BSON objects in MongoDB. Keys in a BSON object are ordered, so this means the insertion order property of maps is maintained.

Mongoose supports a special $* syntax to populate all elements in a map. For example, suppose your socialMediaHandles map contains a ref:

To populate every socialMediaHandles entry's oauth property, you should populate on socialMediaHandles.$*.oauth:

Getters are like virtuals for paths defined in your schema. For example, let's say you wanted to store user profile pictures as relative paths and then add the hostname in your application. Below is how you would structure your userSchema:

Generally, you only use getters on primitive paths as opposed to arrays or subdocuments. Because getters override what accessing a Mongoose path returns, declaring a getter on an object may remove Mongoose change tracking for that path.

Instead of declaring a getter on the array as shown above, you should declare a getter on the url string as shown below. If you need to declare a getter on a nested document or array, be very careful!

To declare a path as another schema, set type to the sub-schema's instance.

To set a default value based on the sub-schema's shape, simply set a default value, and the value will be cast based on the sub-schema's definition before being set during document creation.

Mongoose can also be extended with custom SchemaTypes. Search the plugins site for compatible types like mongoose-long, mongoose-int32, and other types.

Read more about creating custom SchemaTypes here.

The schema.path() function returns the instantiated schema type for a given path.

You can use this function to inspect the schema type for a given path, including what validators it has and what the type is.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({ name: String });
schema.path('name') instanceof mongoose.SchemaType; // true
schema.path('name') instanceof mongoose.Schema.Types.String; // true
schema.path('name').instance; // 'String'
```

Example 2 (javascript):
```javascript
const schema = new Schema({
  name:    String,
  binary:  Buffer,
  living:  Boolean,
  updated: { type: Date, default: Date.now },
  age:     { type: Number, min: 18, max: 65 },
  mixed:   Schema.Types.Mixed,
  _someId: Schema.Types.ObjectId,
  decimal: Schema.Types.Decimal128,
  array: [],
  ofString: [String],
  ofNumber: [Number],
  ofDates: [Date],
  ofBuffer: [Buffer],
  ofBoolean: [Boolean],
  ofMixed: [Schema.Types.Mixed],
  ofObjectId: [Schema.Types.ObjectId],
  ofArrays: [[]],
  ofArrayOfNumbers: [[Number]],
  nested: {
    stuff: { type: String, lowercase: true, trim: true }
  },
  map: Map,
  mapOfString: {
    type: Map,
    of: String
  }
})

// example use

const Thing = mongoose.model('Thing', schema);

const m = new Thing;
m.name = 'Statue of Liberty';
m.age = 125;
m.updated = new Date;
m.binary = Buffer.alloc(0);
m.living = false;
m.mixed = { any: { thing: 'i want' } };
m.markModified('mixed');
m._someId = new mongoose.Types.ObjectId;
m.array.push(1);
m.ofString.push("strings!");
m.ofNumber.unshift(1,2,3,4);
m.ofDates.addToSet(new Date);
m.ofBuffer.pop();
m.ofMixed = [1, [], 'three', { four: 5 }];
m.nested.stuff = 'good';
m.map = new Map([['key', 'value']]);
m.save(callback);
```

Example 3 (javascript):
```javascript
// 3 string SchemaTypes: 'name', 'nested.firstName', 'nested.lastName'
const schema = new Schema({
  name: { type: String },
  nested: {
    firstName: { type: String },
    lastName: { type: String }
  }
});
```

Example 4 (javascript):
```javascript
const holdingSchema = new Schema({
  // You might expect `asset` to be an object that has 2 properties,
  // but unfortunately `type` is special in Mongoose so mongoose
  // interprets this schema to mean that `asset` is a string
  asset: {
    type: String,
    ticker: String
  }
});
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/discriminators.html

**Contents:**
- Discriminators
  - The model.discriminator() function
  - Discriminators save to the Event model's collection
  - Discriminator keys

Discriminators are a schema inheritance mechanism. They enable you to have multiple models with overlapping schemas on top of the same underlying MongoDB collection.

Suppose you wanted to track different types of events in a single collection. Every event will have a timestamp, but events that represent clicked links should have a URL. You can achieve this using the model.discriminator() function. This function takes 3 parameters, a model name, a discriminator schema and an optional key (defaults to the model name). It returns a model whose schema is the union of the base schema and the discriminator schema.

Suppose you created another discriminator to track events where a new user registered. These SignedUpEvent instances will be stored in the same collection as generic events and ClickedLinkEvent instances.

The way mongoose tells the difference between the different discriminator models is by the 'discriminator key', which is __t by default. Mongoose adds a String path called __t to your schemas that it uses to track which discriminator this document is an instance of.

**Examples:**

Example 1 (javascript):
```javascript
const options = { discriminatorKey: 'kind' };

const eventSchema = new mongoose.Schema({ time: Date }, options);
const Event = mongoose.model('Event', eventSchema);

// ClickedLinkEvent is a special type of Event that has
// a URL.
const ClickedLinkEvent = Event.discriminator('ClickedLink',
  new mongoose.Schema({ url: String }, options));

// When you create a generic event, it can't have a URL field...
const genericEvent = new Event({ time: Date.now(), url: 'google.com' });
assert.ok(!genericEvent.url);

// But a ClickedLinkEvent can
const clickedEvent =
  new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
assert.ok(clickedEvent.url);
```

Example 2 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });

/*
const save = function(doc, callback) {
  doc.save(function(error, doc) {
    callback(error, doc);
  });
}; */

Promise.all([event1.save(), event2.save(), event3.save()]).
  then(() => Event.countDocuments()).
  then(count => {
    assert.equal(count, 3);
  });
```

Example 3 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });

assert.ok(!event1.__t);
assert.equal(event2.__t, 'ClickedLink');
assert.equal(event3.__t, 'SignedUp');
```

---

## 

**URL:** https://mongoosejs.com/docs/5.x/docs/subdocs.html

**Contents:**
- Subdocuments
  - What is a Subdocument?
  - Subdocuments versus Nested Paths
  - Subdocument Defaults
  - Finding a Subdocument
  - Adding Subdocs to Arrays
  - Removing Subdocs
  - Parents of Subdocs
    - Alternate declaration syntax for arrays
    - Alternate declaration syntax for single nested subdocuments

Subdocuments are documents embedded in other documents. In Mongoose, this means you can nest schemas in other schemas. Mongoose has two distinct notions of subdocuments: arrays of subdocuments and single nested subdocuments.

Aside from code reuse, one important reason to use subdocuments is to create a path where there would otherwise not be one to allow for validation over a group of fields (e.g. dateRange.fromDate <= dateRange.toDate).

Subdocuments are similar to normal documents. Nested schemas can have middleware, custom validation logic, virtuals, and any other feature top-level schemas can use. The major difference is that subdocuments are not saved individually, they are saved whenever their top-level parent document is saved.

Subdocuments have save and validate middleware just like top-level documents. Calling save() on the parent document triggers the save() middleware for all its subdocuments, and the same for validate() middleware.

Subdocuments' pre('save') and pre('validate') middleware execute before the top-level document's pre('save') but after the top-level document's pre('validate') middleware. This is because validating before save() is actually a piece of built-in middleware.

In Mongoose, nested paths are subtly different from subdocuments. For example, below are two schemas: one with child as a subdocument, and one with child as a nested path.

These two schemas look similar, and the documents in MongoDB will have the same structure with both schemas. But there are a few Mongoose-specific differences:

First, instances of Nested never have child === undefined. You can always set subproperties of child, even if you don't set the child property. But instances of Subdoc can have child === undefined.

Secondly, in Mongoose 5, Document#set() merges when you call it on a nested path, but overwrites when you call it on a subdocument.

Subdocument paths are undefined by default, and Mongoose does not apply subdocument defaults unless you set the subdocument path to a non-nullish value.

However, if you set doc.child to any object, Mongoose will apply the age default if necessary.

Mongoose applies defaults recursively, which means there's a nice workaround if you want to make sure Mongoose applies subdocument defaults: make the subdocument path default to an empty object.

Each subdocument has an _id by default. Mongoose document arrays have a special id method for searching a document array to find a document with a given _id.

MongooseArray methods such as push, unshift, addToSet, and others cast arguments to their proper types transparently:

Subdocs may also be created without adding them to the array by using the create method of MongooseArrays.

Each subdocument has it's own remove method. For an array subdocument, this is equivalent to calling .pull() on the subdocument. For a single nested subdocument, remove() is equivalent to setting the subdocument to null.

Sometimes, you need to get the parent of a subdoc. You can access the parent using the parent() function.

If you have a deeply nested subdoc, you can access the top-level document using the ownerDocument() function.

If you create a schema with an array of objects, Mongoose will automatically convert the object to a schema for you:

Unlike document arrays, Mongoose 5 does not convert an objects in schemas into nested schemas. In the below example, nested is a nested path rather than a subdocument.

This leads to some surprising behavior when you attempt to define a nested path with validators or getters/setters.

Surprisingly, declaring nested with an object type makes nested into a path of type Mixed. To instead make Mongoose automatically convert type: { prop: String } into type: new Schema({ prop: String }), set the typePojoToMixed option to false.

Now that we've covered Subdocuments, let's take a look at querying.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });

const parentSchema = new Schema({
  // Array of subdocuments
  children: [childSchema],
  // Single nested subdocuments. Caveat: single nested subdocs only work
  // in mongoose >= 4.2.0
  child: childSchema
});
```

Example 2 (javascript):
```javascript
const Parent = mongoose.model('Parent', parentSchema);
const parent = new Parent({ children: [{ name: 'Matt' }, { name: 'Sarah' }] })
parent.children[0].name = 'Matthew';

// `parent.children[0].save()` is a no-op, it triggers middleware but
// does **not** actually save the subdocument. You need to save the parent
// doc.
parent.save(callback);
```

Example 3 (javascript):
```javascript
childSchema.pre('save', function (next) {
  if ('invalid' == this.name) {
    return next(new Error('#sadpanda'));
  }
  next();
});

const parent = new Parent({ children: [{ name: 'invalid' }] });
parent.save(function (err) {
  console.log(err.message) // #sadpanda
});
```

Example 4 (javascript):
```javascript
// Below code will print out 1-4 in order
const childSchema = new mongoose.Schema({ name: 'string' });

childSchema.pre('validate', function(next) {
  console.log('2');
  next();
});

childSchema.pre('save', function(next) {
  console.log('3');
  next();
});

const parentSchema = new mongoose.Schema({
  child: childSchema
});

parentSchema.pre('validate', function(next) {
  console.log('1');
  next();
});

parentSchema.pre('save', function(next) {
  console.log('4');
  next();
});
```

---

## Custom Schema Types

**URL:** https://mongoosejs.com/docs/customschematypes.html

**Contents:**
- Custom Schema Types
- Creating a Basic Custom Schema Type

New in Mongoose 4.4.0: Mongoose supports custom types. Before you reach for a custom type, however, know that a custom type is overkill for most use cases. You can do most basic tasks with custom getters/setters, virtuals, and single embedded docs.

Let's take a look at an example of a basic schema type: a 1-byte integer. To create a new schema type, you need to inherit from mongoose.SchemaType and add the corresponding property to mongoose.Schema.Types. The one method you need to implement is the cast() method.

**Examples:**

Example 1 (css):
```css
class Int8 extends mongoose.SchemaType {
  constructor(key, options) {
    super(key, options, 'Int8');
  }

  // `cast()` takes a parameter that can be anything. You need to
  // validate the provided `val` and throw a `CastError` if you
  // can't convert it.
  cast(val) {
    let _val = Number(val);
    if (isNaN(_val)) {
      throw new Error('Int8: ' + val + ' is not a number');
    }
    _val = Math.round(_val);
    if (_val < -0x80 || _val > 0x7F) {
      throw new Error('Int8: ' + val +
        ' is outside of the range of valid 8-bit ints');
    }
    return _val;
  }
}

// Don't forget to add `Int8` to the type registry
mongoose.Schema.Types.Int8 = Int8;

const testSchema = new Schema({ test: Int8 });
const Test = mongoose.model('CustomTypeExample', testSchema);

const t = new Test();
t.test = 'abc';
assert.ok(t.validateSync());
assert.equal(t.validateSync().errors['test'].name, 'CastError');
assert.equal(t.validateSync().errors['test'].message,
  'Cast to Int8 failed for value "abc" (type string) at path "test"');
assert.equal(t.validateSync().errors['test'].reason.message,
  'Int8: abc is not a number');
```

---

## Discriminators

**URL:** https://mongoosejs.com/docs/7.x/docs/discriminators.html

**Contents:**
- Discriminators
- The model.discriminator() function
- Discriminators save to the Event model's collection
- Discriminator keys
- Updating the discriminator key
- Embedded discriminators in arrays
- Single nested discriminators

Discriminators are a schema inheritance mechanism. They enable you to have multiple models with overlapping schemas on top of the same underlying MongoDB collection.

Suppose you wanted to track different types of events in a single collection. Every event will have a timestamp, but events that represent clicked links should have a URL. You can achieve this using the model.discriminator() function. This function takes 3 parameters, a model name, a discriminator schema and an optional key (defaults to the model name). It returns a model whose schema is the union of the base schema and the discriminator schema.

Suppose you created another discriminator to track events where a new user registered. These SignedUpEvent instances will be stored in the same collection as generic events and ClickedLinkEvent instances.

The way Mongoose tells the difference between the different discriminator models is by the 'discriminator key', which is __t by default. Mongoose adds a String path called __t to your schemas that it uses to track which discriminator this document is an instance of.

By default, Mongoose doesn't let you update the discriminator key. save() will throw an error if you attempt to update the discriminator key. And findOneAndUpdate(), updateOne(), etc. will strip out discriminator key updates.

To update a document's discriminator key, use findOneAndUpdate() or updateOne() with the overwriteDiscriminatorKey option set as follows.

You can also define discriminators on embedded document arrays. Embedded discriminators are different because the different discriminator types are stored in the same document array (within a document) rather than the same collection. In other words, embedded discriminators let you store subdocuments matching different schemas in the same array.

As a general best practice, make sure you declare any hooks on your schemas before you use them. You should not call pre() or post() after calling discriminator().

You can also define discriminators on single nested subdocuments, similar to how you can define discriminators on arrays of subdocuments.

As a general best practice, make sure you declare any hooks on your schemas before you use them. You should not call pre() or post() after calling discriminator().

**Examples:**

Example 1 (javascript):
```javascript
const options = { discriminatorKey: 'kind' };

const eventSchema = new mongoose.Schema({ time: Date }, options);
const Event = mongoose.model('Event', eventSchema);

// ClickedLinkEvent is a special type of Event that has
// a URL.
const ClickedLinkEvent = Event.discriminator('ClickedLink',
  new mongoose.Schema({ url: String }, options));

// When you create a generic event, it can't have a URL field...
const genericEvent = new Event({ time: Date.now(), url: 'google.com' });
assert.ok(!genericEvent.url);

// But a ClickedLinkEvent can
const clickedEvent = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
assert.ok(clickedEvent.url);
```

Example 2 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });


await Promise.all([event1.save(), event2.save(), event3.save()]);
const count = await Event.countDocuments();
assert.equal(count, 3);
```

Example 3 (javascript):
```javascript
const event1 = new Event({ time: Date.now() });
const event2 = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
const event3 = new SignedUpEvent({ time: Date.now(), user: 'testuser' });

assert.ok(!event1.__t);
assert.equal(event2.__t, 'ClickedLink');
assert.equal(event3.__t, 'SignedUp');
```

Example 4 (javascript):
```javascript
let event = new ClickedLinkEvent({ time: Date.now(), url: 'google.com' });
await event.save();

event.__t = 'SignedUp';
// ValidationError: ClickedLink validation failed: __t: Cast to String failed for value "SignedUp" (type string) at path "__t"
  await event.save();

event = await ClickedLinkEvent.findByIdAndUpdate(event._id, { __t: 'SignedUp' }, { new: true });
event.__t; // 'ClickedLink', update was a no-op
```

---

## SchemaSubdocument

**URL:** https://mongoosejs.com/docs/api/schemasubdocument.html

**Contents:**
- SchemaSubdocument
  - SchemaSubdocument()
      - Parameters:
      - Inherits:
  - SchemaSubdocument.get()
      - Parameters:
      - Returns:
      - Type:
  - SchemaSubdocument.prototype.$conditionalHandlers
      - Type:

Single nested subdocument SchemaType constructor.

Attaches a getter for all Subdocument instances

Contains the handlers for different query operators for this schema type. For example, $conditionalHandlers.$exists is the function Mongoose calls to cast $exists filter operators.

Adds a discriminator to this single nested subdocument.

Async validation on this single nested doc.

Returns this schema type's representation in a JSON schema.

Sets a default option for all Subdocument instances.

**Examples:**

Example 1 (css):
```css
const shapeSchema = Schema({ name: String }, { discriminatorKey: 'kind' });
const schema = Schema({ shape: shapeSchema });

const singleNestedPath = parentSchema.path('shape');
singleNestedPath.discriminator('Circle', Schema({ radius: Number }));
```

Example 2 (unknown):
```unknown
// Make all numbers have option `min` equal to 0.
mongoose.Schema.Subdocument.set('required', true);
```

---

## Subdocuments

**URL:** https://mongoosejs.com/docs/7.x/docs/subdocs.html

**Contents:**
- Subdocuments
- What is a Subdocument?
- Subdocuments versus Nested Paths
- Subdocument Defaults
- Finding a Subdocument
- Adding Subdocs to Arrays
- Removing Subdocs
- Parents of Subdocs
  - Alternate declaration syntax for arrays
  - Next Up

Subdocuments are documents embedded in other documents. In Mongoose, this means you can nest schemas in other schemas. Mongoose has two distinct notions of subdocuments: arrays of subdocuments and single nested subdocuments.

Note that populated documents are not subdocuments in Mongoose. Subdocument data is embedded in the top-level document. Referenced documents are separate top-level documents.

Subdocuments are similar to normal documents. Nested schemas can have middleware, custom validation logic, virtuals, and any other feature top-level schemas can use. The major difference is that subdocuments are not saved individually, they are saved whenever their top-level parent document is saved.

Subdocuments have save and validate middleware just like top-level documents. Calling save() on the parent document triggers the save() middleware for all its subdocuments, and the same for validate() middleware.

Subdocuments' pre('save') and pre('validate') middleware execute before the top-level document's pre('save') but after the top-level document's pre('validate') middleware. This is because validating before save() is actually a piece of built-in middleware.

In Mongoose, nested paths are subtly different from subdocuments. For example, below are two schemas: one with child as a subdocument, and one with child as a nested path.

These two schemas look similar, and the documents in MongoDB will have the same structure with both schemas. But there are a few Mongoose-specific differences:

First, instances of Nested never have child === undefined. You can always set subproperties of child, even if you don't set the child property. But instances of Subdoc can have child === undefined.

Subdocument paths are undefined by default, and Mongoose does not apply subdocument defaults unless you set the subdocument path to a non-nullish value.

However, if you set doc.child to any object, Mongoose will apply the age default if necessary.

Mongoose applies defaults recursively, which means there's a nice workaround if you want to make sure Mongoose applies subdocument defaults: make the subdocument path default to an empty object.

Each subdocument has an _id by default. Mongoose document arrays have a special id method for searching a document array to find a document with a given _id.

MongooseArray methods such as push, unshift, addToSet, and others cast arguments to their proper types transparently:

You can also create a subdocument without adding it to an array by using the create() method of Document Arrays.

Each subdocument has its own deleteOne method. For an array subdocument, this is equivalent to calling .pull() on the subdocument. For a single nested subdocument, deleteOne() is equivalent to setting the subdocument to null.

Sometimes, you need to get the parent of a subdoc. You can access the parent using the parent() function.

If you have a deeply nested subdoc, you can access the top-level document using the ownerDocument() function.

If you create a schema with an array of objects, Mongoose will automatically convert the object to a schema for you:

Now that we've covered Subdocuments, let's take a look at querying.

**Examples:**

Example 1 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });

const parentSchema = new Schema({
  // Array of subdocuments
  children: [childSchema],
  // Single nested subdocuments
  child: childSchema
});
```

Example 2 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });
const Child = mongoose.model('Child', childSchema);

const parentSchema = new Schema({
  child: {
    type: mongoose.ObjectId,
    ref: 'Child'
  }
});
const Parent = mongoose.model('Parent', parentSchema);

const doc = await Parent.findOne().populate('child');
// NOT a subdocument. `doc.child` is a separate top-level document.
doc.child;
```

Example 3 (javascript):
```javascript
const Parent = mongoose.model('Parent', parentSchema);
const parent = new Parent({ children: [{ name: 'Matt' }, { name: 'Sarah' }] });
parent.children[0].name = 'Matthew';

// `parent.children[0].save()` is a no-op, it triggers middleware but
// does **not** actually save the subdocument. You need to save the parent
// doc.
await parent.save();
```

Example 4 (javascript):
```javascript
childSchema.pre('save', function(next) {
  if ('invalid' == this.name) {
    return next(new Error('#sadpanda'));
  }
  next();
});

const parent = new Parent({ children: [{ name: 'invalid' }] });
try {
  await parent.save();
} catch (err) {
  err.message; // '#sadpanda'
}
```

---

## SchemaStringOptions

**URL:** https://mongoosejs.com/docs/api/schemastringoptions.html

**Contents:**
- SchemaStringOptions
  - SchemaStringOptions()
      - Type:
      - Inherits:
    - Example:
  - SchemaStringOptions.prototype.enum
      - Type:
  - SchemaStringOptions.prototype.lowercase
      - Type:
  - SchemaStringOptions.prototype.match

The options defined on a string schematype.

Array of allowed values for this path

If truthy, Mongoose will add a custom setter that lowercases this string using JavaScript's built-in String#toLowerCase().

Attach a validator that succeeds if the data string matches the given regular expression, and fails otherwise.

If set, Mongoose will add a custom validator that ensures the given string's length is at most the given number.

Mongoose supports two different spellings for this option: maxLength and maxlength. maxLength is the recommended way to specify this option, but Mongoose also supports maxlength (lowercase "l").

If set, Mongoose will add a custom validator that ensures the given string's length is at least the given number.

Mongoose supports two different spellings for this option: minLength and minlength. minLength is the recommended way to specify this option, but Mongoose also supports minlength (lowercase "l").

Sets default populate options.

If truthy, Mongoose will add a custom setter that removes leading and trailing whitespace using JavaScript's built-in String#trim().

If truthy, Mongoose will add a custom setter that uppercases this string using JavaScript's built-in String#toUpperCase().

**Examples:**

Example 1 (css):
```css
const schema = new Schema({ name: String });
schema.path('name').options; // SchemaStringOptions instance
```

---

## SchemaString

**URL:** https://mongoosejs.com/docs/api/schemastring.html

**Contents:**
- SchemaString
  - SchemaString()
      - Parameters:
      - Inherits:
  - SchemaString.cast()
      - Parameters:
      - Returns:
      - Type:
    - Example:
  - SchemaString.checkRequired()

String SchemaType constructor.

Get/set the function used to cast arbitrary values to strings.

Override the function the required validator uses to check whether a string passes the required check.

Attaches a getter for all String instances.

Contains the handlers for different query operators for this schema type. For example, $conditionalHandlers.$exists is the function Mongoose calls to cast $exists filter operators.

Check if the given value satisfies the required validator. The value is considered valid if it is a string (that is, not null or undefined) and has positive length. The required validator will fail for empty strings.

Adds an enum validator

Adds a lowercase setter.

Note that lowercase does not affect regular expression queries:

Sets a regexp validator.

Any value that does not pass regExp.test(val) will fail validation.

Empty strings, undefined, and null values always pass the match validator. If you require these values, enable the required validator also.

Sets a maximum length validator.

Sets a minimum length validator.

Returns this schema type's representation in a JSON schema.

The string value will be trimmed when set.

Note that trim does not affect regular expression queries:

Adds an uppercase setter.

Note that uppercase does not affect regular expression queries:

This schema type's name, to defend against minifiers that mangle function names.

Sets a default option for all String instances.

**Examples:**

Example 1 (javascript):
```javascript
// Throw an error if you pass in an object. Normally, Mongoose allows
// objects with custom `toString()` functions.
const original = mongoose.Schema.Types.String.cast();
mongoose.Schema.Types.String.cast(v => {
  assert.ok(v == null || typeof v !== 'object');
  return original(v);
});

// Or disable casting entirely
mongoose.Schema.Types.String.cast(false);
```

Example 2 (javascript):
```javascript
// Allow empty strings to pass `required` check
mongoose.Schema.Types.String.checkRequired(v => v != null);

const M = mongoose.model({ str: { type: String, required: true } });
new M({ str: '' }).validateSync(); // `null`, validation passes!
```

Example 3 (javascript):
```javascript
// Make all numbers round down
mongoose.Schema.String.get(v => v.toLowerCase());

const Model = mongoose.model('Test', new Schema({ test: String }));
new Model({ test: 'FOO' }).test; // 'foo'
```

Example 4 (javascript):
```javascript
const states = ['opening', 'open', 'closing', 'closed']
const s = new Schema({ state: { type: String, enum: states }})
const M = db.model('M', s)
const m = new M({ state: 'invalid' })
await m.save()
  .catch((err) => console.error(err)); // ValidationError: `invalid` is not a valid enum value for path `state`.
m.state = 'open';
await m.save();
// success

// or with custom error messages
const enum = {
  values: ['opening', 'open', 'closing', 'closed'],
  message: 'enum validator failed for path `{PATH}` with value `{VALUE}`'
}
const s = new Schema({ state: { type: String, enum: enum })
const M = db.model('M', s)
const m = new M({ state: 'invalid' })
await m.save()
  .catch((err) => console.error(err)); // ValidationError: enum validator failed for path `state` with value `invalid`
m.state = 'open';
await m.save();
// success
```

---

## Subdocuments

**URL:** https://mongoosejs.com/docs/subdocs.html

**Contents:**
- Subdocuments
- What is a Subdocument?
- Subdocuments versus Nested Paths
- Subdocument Defaults
- Finding a Subdocument
- Adding Subdocs to Arrays
- Removing Subdocs
- Parents of Subdocs
  - Alternate declaration syntax for arrays
  - Next Up

Subdocuments are documents embedded in other documents. In Mongoose, this means you can nest schemas in other schemas. Mongoose has two distinct notions of subdocuments: arrays of subdocuments and single nested subdocuments.

Note that populated documents are not subdocuments in Mongoose. Subdocument data is embedded in the top-level document. Referenced documents are separate top-level documents.

Subdocuments are similar to normal documents. Nested schemas can have middleware, custom validation logic, virtuals, and any other feature top-level schemas can use. The major difference is that subdocuments are not saved individually, they are saved whenever their top-level parent document is saved.

Subdocuments have save and validate middleware just like top-level documents. Calling save() on the parent document triggers the save() middleware for all its subdocuments, and the same for validate() middleware.

Subdocuments' pre('save') and pre('validate') middleware execute before the top-level document's pre('save') but after the top-level document's pre('validate') middleware. This is because validating before save() is actually a piece of built-in middleware.

In Mongoose, nested paths are subtly different from subdocuments. For example, below are two schemas: one with child as a subdocument, and one with child as a nested path.

These two schemas look similar, and the documents in MongoDB will have the same structure with both schemas. But there are a few Mongoose-specific differences:

First, instances of Nested never have child === undefined. You can always set subproperties of child, even if you don't set the child property. But instances of Subdoc can have child === undefined.

Subdocument paths are undefined by default, and Mongoose does not apply subdocument defaults unless you set the subdocument path to a non-nullish value.

However, if you set doc.child to any object, Mongoose will apply the age default if necessary.

Mongoose applies defaults recursively, which means there's a nice workaround if you want to make sure Mongoose applies subdocument defaults: make the subdocument path default to an empty object.

Each subdocument has an _id by default. Mongoose document arrays have a special id method for searching a document array to find a document with a given _id.

MongooseArray methods such as push, unshift, addToSet, and others cast arguments to their proper types transparently:

You can also create a subdocument without adding it to an array by using the create() method of Document Arrays.

Each subdocument has its own deleteOne method. For an array subdocument, this is equivalent to calling .pull() on the subdocument. For a single nested subdocument, deleteOne() is equivalent to setting the subdocument to null.

Sometimes, you need to get the parent of a subdoc. You can access the parent using the parent() function.

If you have a deeply nested subdoc, you can access the top-level document using the ownerDocument() function.

If you create a schema with an array of objects, Mongoose will automatically convert the object to a schema for you:

Now that we've covered Subdocuments, let's take a look at querying.

**Examples:**

Example 1 (swift):
```swift
const childSchema = new Schema({ name: 'string' });

const parentSchema = new Schema({
  // Array of subdocuments
  children: [childSchema],
  // Single nested subdocuments
  child: childSchema
});
```

Example 2 (javascript):
```javascript
const childSchema = new Schema({ name: 'string' });
const Child = mongoose.model('Child', childSchema);

const parentSchema = new Schema({
  child: {
    type: mongoose.ObjectId,
    ref: 'Child'
  }
});
const Parent = mongoose.model('Parent', parentSchema);

const doc = await Parent.findOne().populate('child');
// NOT a subdocument. `doc.child` is a separate top-level document.
doc.child;
```

Example 3 (javascript):
```javascript
const Parent = mongoose.model('Parent', parentSchema);
const parent = new Parent({ children: [{ name: 'Matt' }, { name: 'Sarah' }] });
parent.children[0].name = 'Matthew';

// `parent.children[0].save()` is a no-op, it triggers middleware but
// does **not** actually save the subdocument. You need to save the parent
// doc.
await parent.save();
```

Example 4 (javascript):
```javascript
childSchema.pre('save', function(next) {
  if ('invalid' == this.name) {
    return next(new Error('#sadpanda'));
  }
  next();
});

const parent = new Parent({ children: [{ name: 'invalid' }] });
try {
  await parent.save();
} catch (err) {
  err.message; // '#sadpanda'
}
```

---

## SchemaTypes

**URL:** https://mongoosejs.com/docs/7.x/docs/schematypes.html

**Contents:**
- SchemaTypes
- What is a SchemaType?
  - Example
- The type Key
- SchemaType Options
  - All Schema Types
  - Indexes
  - String
  - Number
  - Date

SchemaTypes handle definition of path defaults, validation, getters, setters, field selection defaults for queries, and other general characteristics for Mongoose document properties.

You can think of a Mongoose schema as the configuration object for a Mongoose model. A SchemaType is then a configuration object for an individual property. A SchemaType says what type a given path should have, whether it has any getters/setters, and what values are valid for that path.

A SchemaType is different from a type. In other words, mongoose.ObjectId !== mongoose.Types.ObjectId. A SchemaType is just a configuration object for Mongoose. An instance of the mongoose.ObjectId SchemaType doesn't actually create MongoDB ObjectIds, it is just a configuration for a path in a schema.

The following are all the valid SchemaTypes in Mongoose. Mongoose plugins can also add custom SchemaTypes like int32. Check out Mongoose's plugins search to find plugins.

type is a special property in Mongoose schemas. When Mongoose finds a nested property named type in your schema, Mongoose assumes that it needs to define a SchemaType with the given type.

As a consequence, you need a little extra work to define a property named type in your schema. For example, suppose you're building a stock portfolio app, and you want to store the asset's type (stock, bond, ETF, etc.). Naively, you might define your schema as shown below:

However, when Mongoose sees type: String, it assumes that you mean asset should be a string, not an object with a property type. The correct way to define an object with a property type is shown below.

You can declare a schema type using the type directly, or an object with a type property.

In addition to the type property, you can specify additional properties for a path. For example, if you want to lowercase a string before saving:

You can add any property you want to your SchemaType options. Many plugins rely on custom SchemaType options. For example, the mongoose-autopopulate plugin automatically populates paths if you set autopopulate: true in your SchemaType options. Mongoose comes with support for several built-in SchemaType options, like lowercase in the above example.

The lowercase option only works for strings. There are certain options which apply for all schema types, and some that apply for specific schema types.

You can also define MongoDB indexes using schema type options.

To declare a path as a string, you may use either the String global constructor or the string 'String'.

If you pass an element that has a toString() function, Mongoose will call it, unless the element is an array or the toString() function is strictly equal to Object.prototype.toString().

To declare a path as a number, you may use either the Number global constructor or the string 'Number'.

There are several types of values that will be successfully cast to a Number.

If you pass an object with a valueOf() function that returns a Number, Mongoose will call it and assign the returned value to the path.

The values null and undefined are not cast.

NaN, strings that cast to NaN, arrays, and objects that don't have a valueOf() function will all result in a CastError once validated, meaning that it will not throw on initialization, only when validated.

Built-in Date methods are not hooked into the mongoose change tracking logic which in English means that if you use a Date in your document and modify it with a method like setMonth(), mongoose will be unaware of this change and doc.save() will not persist this modification. If you must modify Date types using built-in methods, tell mongoose about the change with doc.markModified('pathToYourDate') before saving.

To declare a path as a Buffer, you may use either the Buffer global constructor or the string 'Buffer'.

Mongoose will successfully cast the below values to buffers.

An "anything goes" SchemaType. Mongoose will not do any casting on mixed paths. You can define a mixed path using Schema.Types.Mixed or by passing an empty object literal. The following are equivalent.

Since Mixed is a schema-less type, you can change the value to anything else you like, but Mongoose loses the ability to auto detect and save those changes. To tell Mongoose that the value of a Mixed type has changed, you need to call doc.markModified(path), passing the path to the Mixed type you just changed.

To avoid these side-effects, a Subdocument path may be used instead.

An ObjectId is a special type typically used for unique identifiers. Here's how you declare a schema with a path driver that is an ObjectId:

ObjectId is a class, and ObjectIds are objects. However, they are often represented as strings. When you convert an ObjectId to a string using toString(), you get a 24-character hexadecimal string:

Booleans in Mongoose are plain JavaScript booleans. By default, Mongoose casts the below values to true:

Mongoose casts the below values to false:

Any other value causes a CastError. You can modify what values Mongoose converts to true or false using the convertToTrue and convertToFalse properties, which are JavaScript sets.

Mongoose supports arrays of SchemaTypes and arrays of subdocuments. Arrays of SchemaTypes are also called primitive arrays, and arrays of subdocuments are also called document arrays.

Arrays are special because they implicitly have a default value of [] (empty array).

To overwrite this default, you need to set the default value to undefined

Note: specifying an empty array is equivalent to Mixed. The following all create arrays of Mixed:

A MongooseMap is a subclass of JavaScript's Map class. In these docs, we'll use the terms 'map' and MongooseMap interchangeably. In Mongoose, maps are how you create a nested document with arbitrary keys.

Note: In Mongoose Maps, keys must be strings in order to store the document in MongoDB.

The above example doesn't explicitly declare github or twitter as paths, but, since socialMediaHandles is a map, you can store arbitrary key/value pairs. However, since socialMediaHandles is a map, you must use .get() to get the value of a key and .set() to set the value of a key.

Map types are stored as BSON objects in MongoDB. Keys in a BSON object are ordered, so this means the insertion order property of maps is maintained.

Mongoose supports a special $* syntax to populate all elements in a map. For example, suppose your socialMediaHandles map contains a ref:

To populate every socialMediaHandles entry's oauth property, you should populate on socialMediaHandles.$*.oauth:

Mongoose also supports a UUID type that stores UUID instances as Node.js buffers. We recommend using ObjectIds rather than UUIDs for unique document ids in Mongoose, but you may use UUIDs if you need to.

In Node.js, a UUID is represented as an instance of bson.Binary type with a getter that converts the binary to a string when you access it. Mongoose stores UUIDs as binary data with subtype 4 in MongoDB.

To create UUIDs, we recommend using Node's built-in UUIDv4 generator.

Mongoose supports JavaScript BigInts as a SchemaType. BigInts are stored as 64-bit integers in MongoDB (BSON type "long").

Getters are like virtuals for paths defined in your schema. For example, let's say you wanted to store user profile pictures as relative paths and then add the hostname in your application. Below is how you would structure your userSchema:

Generally, you only use getters on primitive paths as opposed to arrays or subdocuments. Because getters override what accessing a Mongoose path returns, declaring a getter on an object may remove Mongoose change tracking for that path.

Instead of declaring a getter on the array as shown above, you should declare a getter on the url string as shown below. If you need to declare a getter on a nested document or array, be very careful!

To declare a path as another schema, set type to the sub-schema's instance.

To set a default value based on the sub-schema's shape, simply set a default value, and the value will be cast based on the sub-schema's definition before being set during document creation.

Mongoose can also be extended with custom SchemaTypes. Search the plugins site for compatible types like mongoose-long, mongoose-int32, and mongoose-function.

Read more about creating custom SchemaTypes here.

The schema.path() function returns the instantiated schema type for a given path.

You can use this function to inspect the schema type for a given path, including what validators it has and what the type is.

Now that we've covered SchemaTypes, let's take a look at Connections.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({ name: String });
schema.path('name') instanceof mongoose.SchemaType; // true
schema.path('name') instanceof mongoose.Schema.Types.String; // true
schema.path('name').instance; // 'String'
```

Example 2 (javascript):
```javascript
const schema = new Schema({
  name: String,
  binary: Buffer,
  living: Boolean,
  updated: { type: Date, default: Date.now },
  age: { type: Number, min: 18, max: 65 },
  mixed: Schema.Types.Mixed,
  _someId: Schema.Types.ObjectId,
  decimal: Schema.Types.Decimal128,
  array: [],
  ofString: [String],
  ofNumber: [Number],
  ofDates: [Date],
  ofBuffer: [Buffer],
  ofBoolean: [Boolean],
  ofMixed: [Schema.Types.Mixed],
  ofObjectId: [Schema.Types.ObjectId],
  ofArrays: [[]],
  ofArrayOfNumbers: [[Number]],
  nested: {
    stuff: { type: String, lowercase: true, trim: true }
  },
  map: Map,
  mapOfString: {
    type: Map,
    of: String
  }
});

// example use

const Thing = mongoose.model('Thing', schema);

const m = new Thing;
m.name = 'Statue of Liberty';
m.age = 125;
m.updated = new Date;
m.binary = Buffer.alloc(0);
m.living = false;
m.mixed = { any: { thing: 'i want' } };
m.markModified('mixed');
m._someId = new mongoose.Types.ObjectId;
m.array.push(1);
m.ofString.push('strings!');
m.ofNumber.unshift(1, 2, 3, 4);
m.ofDates.addToSet(new Date);
m.ofBuffer.pop();
m.ofMixed = [1, [], 'three', { four: 5 }];
m.nested.stuff = 'good';
m.map = new Map([['key', 'value']]);
m.save(callback);
```

Example 3 (javascript):
```javascript
// 3 string SchemaTypes: 'name', 'nested.firstName', 'nested.lastName'
const schema = new Schema({
  name: { type: String },
  nested: {
    firstName: { type: String },
    lastName: { type: String }
  }
});
```

Example 4 (javascript):
```javascript
const holdingSchema = new Schema({
  // You might expect `asset` to be an object that has 2 properties,
  // but unfortunately `type` is special in Mongoose so mongoose
  // interprets this schema to mean that `asset` is a string
  asset: {
    type: String,
    ticker: String
  }
});
```

---

## Advanced Schemas

**URL:** https://mongoosejs.com/docs/advanced_schemas.html

**Contents:**
- Advanced Schemas
- Creating from ES6 Classes Using loadClass()

Mongoose allows creating schemas from ES6 classes. The loadClass() function lets you pull in methods, statics, and virtuals from an ES6 class. A class method maps to a schema method, a static method maps to a schema static, and getters/setters map to virtuals.

**Examples:**

Example 1 (javascript):
```javascript
const schema = new Schema({ firstName: String, lastName: String });

class HumanClass {
  get fullName() {
    return 'My name';
  }
}

class PersonClass extends HumanClass {
  // `fullName` becomes a virtual
  get fullName() {
    return `${super.fullName} is ${this.firstName} ${this.lastName}`;
  }

  set fullName(v) {
    const firstSpace = v.indexOf(' ');
    this.firstName = v.split(' ')[0];
    this.lastName = firstSpace === -1 ? '' : v.substring(firstSpace + 1);
  }

  // `getFullName()` becomes a document method
  getFullName() {
    return `${this.firstName} ${this.lastName}`;
  }

  // `findByFullName()` becomes a static
  static findByFullName(name) {
    const firstSpace = name.indexOf(' ');
    const firstName = name.split(' ')[0];
    const lastName = firstSpace === -1 ? '' : name.substring(firstSpace + 1);
    return this.findOne({ firstName, lastName });
  }
}

schema.loadClass(PersonClass);
const Person = db.model('Person', schema);

const doc = await Person.create({
  firstName: 'Jon',
  lastName: 'Snow'
});

assert.equal(doc.fullName, 'My name is Jon Snow');
doc.fullName = 'Jon Stark';
assert.equal(doc.firstName, 'Jon');
assert.equal(doc.lastName, 'Stark');
const foundPerson = await Person.findByFullName('Jon Snow');

assert.equal(foundPerson.fullName, 'My name is Jon Snow');
```

---

## Sharing Schemas Between Mongoose Projects

**URL:** https://mongoosejs.com/docs/shared-schemas.html

**Contents:**
- Sharing Schemas Between Mongoose Projects
- Put Mongoose as a Peer Dependency
- Export Schemas, Not Models
- Workaround: Export a POJO

In larger organizations, it is common to have a project that contains schemas which are shared between multiple projects. For example, suppose your company has an @initech/shared-schemas private npm package, and npm list looks like the following:

In the above output, @initech/web-app1 is a client project and @initech/shared-schemas is the shared library.

First, and most importantly, we recommend that @initech/shared-schemas list Mongoose in your shared-schema's peerDependencies, not as a top-level dependency. For example, @initech/shared-schemas's package.json should look like the following.

We recommend this approach for the following reasons:

We recommend that @initech/shared-schemas export Mongoose schemas, not models. This approach is more flexible and allows client projects to instantiate models using their preferred patterns. In particular, if @initech/shared-schemas exports a model that is registered using mongoose.model(), there is no way to transfer that model to a different connection.

Sometimes, existing shared libraries don't follow the above best practices. If you find yourself with a shared library that depends on an old version of Mongoose, a helpful workaround is to export a POJO rather than a schema or model. This will remove any conflicts between the shared library's version of Mongoose and the client project's version of Mongoose.

And update your client project to do the following:

**Examples:**

Example 1 (python):
```python
@initech/web-app1@1.0.0
├── @initech/shared-schemas@1.0.0
├── mongoose@8.0.1
```

Example 2 (json):
```json
{
  "name": "@initech/shared-schemas",
  "peerDependencies": {
    "mongoose": "8.x"
  }
}
```

Example 3 (python):
```python
// `userSchema.js` in `@initech/shared-schemas`
const userSchema = new mongoose.Schema({ name: String });

// Do this:
module.exports = userSchema;

// Not this:
module.exports = mongoose.model('User', userSchema);
```

Example 4 (css):
```css
// Replace this:
module.exports = new mongoose.Schema({ name: String });

// With this:
module.exports = { name: String };
```

---
