---
name: mvv-synthesis
description: Synthesize Mission, Vision, Values from ontological analysis output
version: 1.0.0
author: multi-agent-os
---

# MVV Synthesis Skill

## Purpose

Transforms the structured output from ontological-analysis into actionable Mission, Vision, and Values (MVV) statements. This skill extracts the essential identity elements from deep codebase analysis and formats them for direct insertion into CLAUDE.md project configuration files.

## When to Use

- After running `/ontological-analysis` on a repository
- When establishing or updating project identity in CLAUDE.md
- During project onboarding to capture existing organizational DNA
- When refactoring or realigning project direction based on codebase evidence

## Input Parameters

This skill receives the JSON output from the `ontological-analysis` skill:

```json
{
  "project_name": "string",
  "analysis_date": "ISO-8601",
  "ontological_layers": {
    "purpose": { "core_function": "...", "value_proposition": "...", "target_users": "..." },
    "principles": { "design_patterns": [...], "quality_attributes": [...], "constraints": [...] },
    "practices": { "workflows": [...], "conventions": [...], "integrations": [...] },
    "artifacts": { "primary_outputs": [...], "documentation": [...], "configurations": [...] }
  },
  "identity_signals": {
    "recurring_themes": [...],
    "emphasized_concepts": [...],
    "cultural_markers": [...]
  },
  "confidence_score": 0.0-1.0
}
```

## Synthesis Rules

### Mission Extraction

The Mission answers: **"Why do we exist? What do we do?"**

**Sources** (priority order):
1. `ontological_layers.purpose.core_function` - Primary source
2. `ontological_layers.purpose.value_proposition` - Supporting context
3. `identity_signals.recurring_themes` - Reinforcing patterns

**Rules**:
- Must be 1-2 sentences maximum
- Use imperative/active voice ("We build...", "Deliver...", "Enable...")
- Focus on action and impact, not features
- Include the "who" (target users) when clear from analysis
- Avoid jargon unless domain-specific and necessary

**Confidence Threshold**: Require >= 0.6 confidence to auto-generate; below that, output draft with `[REVIEW REQUIRED]` marker.

### Vision Extraction

The Vision answers: **"What future state are we creating?"**

**Sources** (priority order):
1. `ontological_layers.purpose.value_proposition` - Future impact
2. `identity_signals.emphasized_concepts` - Aspirational patterns
3. `ontological_layers.principles.quality_attributes` - Desired end state

**Rules**:
- Describe a desirable future state, not current state
- Use present tense to describe the envisioned future ("A world where...")
- Should be ambitious but achievable
- Timeframe implicit (typically 3-5 years horizon)
- Must connect to Mission as the "why it matters"

**Pattern Templates**:
```
"A [domain] where [transformation] enables [benefit]"
"[Target users] [achieving/experiencing] [desired outcome] through [mechanism]"
"The leading [category] that [differentiator]"
```

### Values Extraction

Values answer: **"What principles guide our decisions?"**

**Sources** (priority order):
1. `ontological_layers.principles.design_patterns` - Embedded beliefs
2. `identity_signals.cultural_markers` - Team culture signals
3. `ontological_layers.practices.conventions` - Behavioral evidence

**Rules**:
- Extract 3-7 values (optimal: 5)
- Each value: single word or short phrase (2-4 words max)
- Must be actionable (can guide decisions)
- Order by prominence in codebase evidence
- Include brief descriptor (1 sentence) explaining the value
- Avoid generic values unless strongly evidenced (e.g., "Quality" needs specific context)

**Anti-Patterns to Avoid**:
- Values that contradict observed practices
- Aspirational values with no codebase evidence
- Redundant values (merge similar concepts)
- More than 7 values (dilutes focus)

## Output Format

### Standard Output (for CLAUDE.md insertion)

```markdown
## Project Identity (MVV)

<!-- Generated by mvv-synthesis v1.0.0 | Source: ontological-analysis | Date: {analysis_date} -->
<!-- Confidence: {confidence_score} | Review Status: {AUTO-APPROVED|REVIEW REQUIRED} -->

### Mission

{mission_statement}

### Vision

{vision_statement}

### Values

| Value | Description |
|-------|-------------|
| **{value_1}** | {description_1} |
| **{value_2}** | {description_2} |
| **{value_3}** | {description_3} |
| **{value_4}** | {description_4} |
| **{value_5}** | {description_5} |

---
*MVV synthesized from codebase ontological analysis. Review and adjust to align with organizational intent.*
```

### Extended Output (with evidence)

```markdown
## Project Identity (MVV) — Extended Report

### Mission

**Statement**: {mission_statement}

**Evidence**:
- Core function: `{ontological_layers.purpose.core_function}`
- Value proposition: `{ontological_layers.purpose.value_proposition}`
- Recurring themes: {identity_signals.recurring_themes}

**Confidence**: {mission_confidence}/1.0

---

### Vision

**Statement**: {vision_statement}

**Evidence**:
- Emphasized concepts: {identity_signals.emphasized_concepts}
- Quality aspirations: {ontological_layers.principles.quality_attributes}

**Confidence**: {vision_confidence}/1.0

---

### Values

| # | Value | Description | Evidence Source | Strength |
|---|-------|-------------|-----------------|----------|
| 1 | **{value}** | {description} | {source} | {high/medium/low} |
| 2 | **{value}** | {description} | {source} | {high/medium/low} |
| ... | ... | ... | ... | ... |

**Values Confidence**: {values_confidence}/1.0

---

### Synthesis Metadata

| Field | Value |
|-------|-------|
| Analysis Date | {analysis_date} |
| Overall Confidence | {confidence_score} |
| Ontological Depth | {layers_analyzed}/4 |
| Identity Signals Found | {signal_count} |
| Synthesis Duration | {duration_ms}ms |
```

## Invocation Syntax

```
/mvv-synthesis [options]
```

### Options

| Option | Description | Default |
|--------|-------------|---------|
| `--input <file>` | Path to ontological-analysis output JSON | stdin/last analysis |
| `--format <type>` | Output format: `standard`, `extended`, `json` | `standard` |
| `--confidence <threshold>` | Minimum confidence for auto-approval | `0.6` |
| `--values-count <n>` | Target number of values to extract | `5` |
| `--output <file>` | Write output to file instead of stdout | stdout |
| `--dry-run` | Show synthesis plan without generating output | false |

## Example Usage

### Example 1: Basic Synthesis (Pipeline)

**Input** (from ontological-analysis):
```json
{
  "project_name": "multi-agent-os",
  "analysis_date": "2026-01-08T12:00:00-03:00",
  "ontological_layers": {
    "purpose": {
      "core_function": "Coordinate multiple AI agents working on shared codebases",
      "value_proposition": "Prevent conflicts and enable parallel AI development",
      "target_users": "Development teams using AI coding assistants"
    },
    "principles": {
      "design_patterns": ["Isolation via worktrees", "Hierarchical merge", "Protocol discipline"],
      "quality_attributes": ["Conflict-free", "Observable", "Scalable"],
      "constraints": ["Git-based workflows", "No automatic hooks"]
    },
    "practices": {
      "workflows": ["Worktree per agent", "Lock file coordination", "Session registry"],
      "conventions": ["ISO 8601 dates", "Semantic naming", "PROV tags"],
      "integrations": ["Claude Code", "Sentinel Protocol", "Status Maps"]
    },
    "artifacts": {
      "primary_outputs": ["Protocols", "Skills", "Templates"],
      "documentation": ["CLAUDE.md", "README.md", "Skill files"],
      "configurations": ["sessions.json", "protected_files.json"]
    }
  },
  "identity_signals": {
    "recurring_themes": ["Isolation", "Observability", "Coordination"],
    "emphasized_concepts": ["Multi-agent", "Conflict prevention", "Protocol discipline"],
    "cultural_markers": ["Documentation-first", "Explicit over implicit", "Human oversight"]
  },
  "confidence_score": 0.85
}
```

**Command**:
```
/mvv-synthesis --format standard
```

**Output**:
```markdown
## Project Identity (MVV)

<!-- Generated by mvv-synthesis v1.0.0 | Source: ontological-analysis | Date: 2026-01-08T12:00:00-03:00 -->
<!-- Confidence: 0.85 | Review Status: AUTO-APPROVED -->

### Mission

Enable development teams to coordinate multiple AI agents working on shared codebases, preventing conflicts and maximizing parallel productivity through protocol-driven isolation.

### Vision

A development ecosystem where AI agents collaborate seamlessly on any codebase, with human developers maintaining full observability and control over the orchestrated workflow.

### Values

| Value | Description |
|-------|-------------|
| **Isolation First** | Every agent works in its own context; conflicts are prevented, not resolved. |
| **Observable Operations** | All agent activity is traceable, auditable, and transparent to humans. |
| **Protocol Discipline** | Explicit protocols over implicit behaviors; documented standards over tribal knowledge. |
| **Human Oversight** | AI agents augment human capability; humans retain decision authority. |
| **Documentation-Driven** | If it's not documented, it doesn't exist; docs are the source of truth. |

---
*MVV synthesized from codebase ontological analysis. Review and adjust to align with organizational intent.*
```

### Example 2: Extended Format with Evidence

**Command**:
```
/mvv-synthesis --format extended --confidence 0.7
```

**Output**: (See Extended Output format above with full evidence mapping)

### Example 3: JSON Output for Automation

**Command**:
```
/mvv-synthesis --format json --output mvv.json
```

**Output**:
```json
{
  "version": "1.0.0",
  "generated_at": "2026-01-08T12:30:00-03:00",
  "source": "ontological-analysis",
  "confidence": 0.85,
  "review_status": "AUTO-APPROVED",
  "mvv": {
    "mission": {
      "statement": "Enable development teams to coordinate multiple AI agents...",
      "confidence": 0.88
    },
    "vision": {
      "statement": "A development ecosystem where AI agents collaborate seamlessly...",
      "confidence": 0.82
    },
    "values": [
      { "name": "Isolation First", "description": "...", "confidence": 0.90 },
      { "name": "Observable Operations", "description": "...", "confidence": 0.85 },
      ...
    ]
  }
}
```

## Integration

### With /mvv Command

This skill is the second step in the `/mvv` pipeline:

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│  /ontological-  │────►│  /mvv-synthesis │────►│  /mvv-inject    │
│    analysis     │     │                 │     │                 │
└─────────────────┘     └─────────────────┘     └─────────────────┘
     (Step 1)                (Step 2)               (Step 3)
   Extract data          Synthesize MVV         Insert in CLAUDE.md
```

### With CLAUDE.md

The standard output format is designed for direct insertion into CLAUDE.md under a `## Project Identity (MVV)` section. The skill:
- Preserves existing CLAUDE.md content
- Inserts MVV after the header/overview section
- Uses HTML comments for metadata (non-intrusive)
- Includes review markers for low-confidence outputs

### With Sentinel Protocol

MVV synthesis operations are logged to Sentinel for observability:
- Trace type: `MVV_SYNTHESIS`
- Includes: input hash, output hash, confidence scores, duration

## Validation Rules

Before generating output, the skill validates:

| Check | Failure Action |
|-------|----------------|
| Input has required ontological layers | Abort with error |
| Confidence >= threshold | Mark `[REVIEW REQUIRED]` |
| Mission < 3 sentences | Pass |
| Mission > 3 sentences | Truncate with warning |
| Vision describes future (not present) | Warn if present-tense dominant |
| Values count in 3-7 range | Warn if outside range |
| Values have evidence | Exclude unsubstantiated values |
| No contradictory values | Flag conflicts for human review |

## Limitations

- Requires ontological-analysis output as input (cannot synthesize from raw code)
- Confidence scoring is heuristic-based, not ML-derived
- Cultural markers detection limited to explicit documentation signals
- Cannot infer unstated organizational values
- Best results with well-documented codebases

## Error Handling

| Error | Response |
|-------|----------|
| Missing input | `ERROR: No ontological-analysis output provided. Run /ontological-analysis first.` |
| Invalid JSON | `ERROR: Input is not valid JSON. Expected ontological-analysis output format.` |
| Missing layers | `WARNING: Ontological layer '{layer}' missing. Synthesis will be partial.` |
| Low confidence | `INFO: Confidence {score} below threshold {threshold}. Output marked for review.` |

---

## Changelog

### v1.0.0 (2026-01-08)
- Initial release
- Core synthesis rules for Mission, Vision, Values
- Standard and extended output formats
- JSON output for automation
- Integration with /mvv pipeline

---

*MVV Synthesis Skill v1.0.0 | Part of multi-agent-os | Claude-Dev-eed7-002*
