#!/usr/bin/env python3
"""
generate_report.py - Create visualization and report for event impact analysis.

This script generates plots and markdown reports summarizing the causal impact
of events on contract prices.
"""

import argparse
import logging
import sys
from datetime import datetime
from pathlib import Path

import matplotlib.pyplot as plt
import pandas as pd

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


def generate_visualization(
    adjusted_forecast: pd.DataFrame,
    output_path: str = "impact_plot.png",
    title: str = "Event Impact on Contract Prices",
) -> None:
    """
    Generate a visualization of event impact over time.

    Args:
        adjusted_forecast: DataFrame with 'ds', 'y', and 'mean' columns.
        output_path: Path to save the plot (default: impact_plot.png).
        title: Plot title (default: Event Impact on Contract Prices).
    """
    try:
        plt.figure(figsize=(12, 6))

        # Plot actual prices
        plt.plot(
            adjusted_forecast["ds"],
            adjusted_forecast["y"],
            label="Actual Price",
            linewidth=2,
            color="blue",
        )

        # Plot adjusted forecast if available
        if "mean" in adjusted_forecast.columns:
            plt.plot(
                adjusted_forecast["ds"],
                adjusted_forecast["mean"],
                label="Adjusted Forecast",
                linestyle="--",
                linewidth=2,
                color="red",
            )

        # Mark treatment periods if available
        if "treatment" in adjusted_forecast.columns:
            treatment_mask = adjusted_forecast["treatment"] == 1
            plt.fill_between(
                adjusted_forecast["ds"],
                adjusted_forecast["y"].min(),
                adjusted_forecast["y"].max(),
                where=treatment_mask,
                alpha=0.2,
                color="orange",
                label="Treatment Period",
            )

        plt.xlabel("Date", fontsize=12)
        plt.ylabel("Price", fontsize=12)
        plt.title(title, fontsize=14, fontweight="bold")
        plt.legend(loc="best", fontsize=10)
        plt.grid(True, alpha=0.3)
        plt.tight_layout()
        plt.savefig(output_path, dpi=300, bbox_inches="tight")

        logger.info(f"Visualization saved to {output_path}")

    except Exception as e:
        logger.error(f"Visualization error: {e}")
        raise


def generate_markdown_report(
    impact_results: pd.DataFrame,
    adjusted_forecast: pd.DataFrame,
    causal_summary: str,
    output_path: str = "impact_report.md",
) -> None:
    """
    Generate a markdown report summarizing the analysis.

    Args:
        impact_results: DataFrame with event impact metrics.
        adjusted_forecast: DataFrame with forecast data.
        causal_summary: CausalImpact text summary.
        output_path: Path to save the report (default: impact_report.md).
    """
    try:
        report_lines = [
            f"# Event Impact Analysis Report",
            f"",
            f"**Generated**: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}",
            f"",
            f"## Executive Summary",
            f"",
            f"This report quantifies the causal impact of events on contract prices using TimeGPT forecasting and CausalImpact analysis.",
            f"",
            f"## Impact Results",
            f"",
            impact_results.to_markdown(index=False),
            f"",
            f"## CausalImpact Summary",
            f"",
            f"```",
            causal_summary,
            f"```",
            f"",
            f"## Forecast Statistics",
            f"",
            f"- **Observations**: {len(adjusted_forecast)}",
            f"- **Date Range**: {adjusted_forecast['ds'].min()} to {adjusted_forecast['ds'].max()}",
            f"- **Mean Actual Price**: {adjusted_forecast['y'].mean():.2f}",
        ]

        if "mean" in adjusted_forecast.columns:
            report_lines.append(
                f"- **Mean Forecast Price**: {adjusted_forecast['mean'].mean():.2f}"
            )

        report_lines.extend(
            [
                f"",
                f"## Methodology",
                f"",
                f"1. **Data Preparation**: Contract prices and event data loaded and validated",
                f"2. **Model Configuration**: Event periods defined with treatment/control windows",
                f"3. **Causal Analysis**: CausalImpact MCMC to estimate counterfactual",
                f"4. **Forecasting**: TimeGPT used to generate adjusted forecasts",
                f"",
                f"## Visualization",
                f"",
                f"See `impact_plot.png` for time series visualization.",
                f"",
                f"---",
                f"*Report generated by nixtla-event-impact-modeler skill*",
            ]
        )

        with open(output_path, "w") as f:
            f.write("\n".join(report_lines))

        logger.info(f"Markdown report saved to {output_path}")

    except Exception as e:
        logger.error(f"Report generation error: {e}")
        raise


def main():
    """CLI entry point for report generation."""
    parser = argparse.ArgumentParser(
        description="Generate visualization and markdown report for event impact analysis."
    )
    parser.add_argument(
        "--impact-results", type=str, required=True, help="Path to impact results CSV"
    )
    parser.add_argument(
        "--adjusted-forecast", type=str, required=True, help="Path to adjusted forecast CSV"
    )
    parser.add_argument(
        "--causal-summary",
        type=str,
        default="causal_summary.txt",
        help="Path to CausalImpact summary text (default: causal_summary.txt)",
    )
    parser.add_argument(
        "--output-plot",
        type=str,
        default="impact_plot.png",
        help="Path to save visualization (default: impact_plot.png)",
    )
    parser.add_argument(
        "--output-report",
        type=str,
        default="impact_report.md",
        help="Path to save markdown report (default: impact_report.md)",
    )
    parser.add_argument(
        "--title", type=str, default="Event Impact on Contract Prices", help="Plot title"
    )

    args = parser.parse_args()

    try:
        logger.info(f"Loading results from {args.impact_results} and {args.adjusted_forecast}")
        impact_results = pd.read_csv(args.impact_results)
        adjusted_forecast = pd.read_csv(args.adjusted_forecast, parse_dates=["ds"])

        # Load causal summary if available
        causal_summary = ""
        if Path(args.causal_summary).exists():
            with open(args.causal_summary, "r") as f:
                causal_summary = f.read()

        # Generate visualization
        generate_visualization(adjusted_forecast, args.output_plot, args.title)

        # Generate markdown report
        generate_markdown_report(
            impact_results, adjusted_forecast, causal_summary, args.output_report
        )

        logger.info("Report generation completed successfully")

        print(f"\nGenerated outputs:")
        print(f"  - Visualization: {args.output_plot}")
        print(f"  - Report: {args.output_report}")

        return 0

    except Exception as e:
        logger.error(f"Error in report generation: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
