# Icon Automation Patterns

Strategies for automatically applying icons in Notion using templates, the API, and database properties.

## Template-Based Automation

### Database Templates with Preset Icons

When creating database templates in Notion, you can set default icons that apply to all new items created from that template.

**Setup:**
1. Open database in full-page view
2. Click "New" dropdown → "New template"
3. Set the icon on the template page
4. All pages created from this template inherit the icon

**Example: Resource Types**
```
Template: "Spec"       → Icon: 📋
Template: "Guide"      → Icon: 📖
Template: "Playbook"   → Icon: 📒
Template: "Reference"  → Icon: 🏷️
```

### Multiple Templates per Database

Create different templates for different item types within the same database:

```
Resources Database:
├── Template: Spec       📋
├── Template: Guide      📖
├── Template: Playbook   📒
├── Template: Concept    💡
└── Template: Reference  🏷️
```

Users select the appropriate template when creating items, and icons apply automatically.

## API-Based Automation

### Setting Icons via Notion API

When creating pages programmatically, include the icon in the request:

```json
POST /v1/pages
{
  "parent": { "page_id": "..." },
  "icon": {
    "type": "emoji",
    "emoji": "📋"
  },
  "properties": {
    "title": [{ "text": { "content": "New Spec Document" } }],
    "Type": { "select": { "name": "Spec" } }
  }
}
```

### Type-to-Icon Mapping Function

```javascript
function getIconForType(type) {
  const iconMap = {
    'Spec': '📋',
    'Guide': '📖',
    'Playbook': '📒',
    'Reference': '🏷️',
    'Template': '📝',
    'Concept': '💡',
    'Process': '🔄',
    'Policy': '📜',
    'Report': '📈',
    'Archive': '📦'
  };
  return iconMap[type] || '📄'; // Default fallback
}
```

### Status-to-Icon Mapping

```javascript
function getIconForStatus(status) {
  const iconMap = {
    'Not Started': '⚪',
    'Active': '🟢',
    'On Hold': '🟡',
    'Blocked': '🔴',
    'Complete': '✅'
  };
  return iconMap[status] || '⚪';
}
```

### Bulk Icon Update Script

```javascript
async function updateDatabaseIcons(databaseId) {
  // Query all pages in database
  const response = await notion.databases.query({
    database_id: databaseId
  });

  for (const page of response.results) {
    const type = page.properties.Type?.select?.name;
    const icon = getIconForType(type);

    // Update page icon
    await notion.pages.update({
      page_id: page.id,
      icon: { type: 'emoji', emoji: icon }
    });
  }
}
```

## Property-Based Icon Strategy

### Using Select Property Colors

While icons can't be set by formula, you can use the **Select property color** as a visual indicator alongside icons:

| Type | Icon | Select Color |
|------|------|--------------|
| Spec | 📋 | Blue |
| Guide | 📖 | Green |
| Playbook | 📒 | Yellow |
| Reference | 🏷️ | Gray |

This creates a dual visual system: icon + color badge.

### Icon in Title Convention

For databases where you can't control icons, embed the icon in the title:

```
📋 API Specification v2
📖 Getting Started Guide
📒 Incident Response Playbook
```

**Pros:**
- Works in all views
- Sortable/searchable
- No API needed

**Cons:**
- Redundant with actual icon
- Manual to maintain

## Webhook-Based Automation

### Notion → Automation Platform → Icon Update

Use Make, Zapier, or n8n to update icons when properties change:

```
Trigger: Page property "Status" changed
Action: Update page icon based on new status value

If Status = "Active" → Set icon to 🟢
If Status = "Blocked" → Set icon to 🔴
If Status = "Complete" → Set icon to ✅
```

### Example: Make.com Workflow

1. **Watch Notion Database** - Monitor for property changes
2. **Router** - Branch based on property value
3. **Update Page** - Set appropriate icon

## Claude Code Automation

### Using km-notion MCP Server

When working with Notion via Claude Code, apply icons programmatically:

```javascript
// When creating a new page
const page = await notion.pages.create({
  parent: { page_id: parentId },
  icon: { type: 'emoji', emoji: getIconForType('Guide') },
  properties: {
    title: [{ text: { content: 'New Guide' } }],
    Type: { select: { name: 'Guide' } }
  }
});
```

### Icon Audit Script

```javascript
async function auditDatabaseIcons(databaseId) {
  const response = await notion.databases.query({ database_id: databaseId });
  const issues = [];

  for (const page of response.results) {
    const type = page.properties.Type?.select?.name;
    const expectedIcon = getIconForType(type);
    const actualIcon = page.icon?.emoji;

    if (actualIcon !== expectedIcon) {
      issues.push({
        pageId: page.id,
        title: page.properties.Name?.title?.[0]?.plain_text,
        type,
        expected: expectedIcon,
        actual: actualIcon || 'none'
      });
    }
  }

  return issues;
}
```

## Best Practices

### 1. Template First
Always set up database templates before adding content. It's easier to maintain consistency from the start.

### 2. Document Your Mappings
Keep a reference document (like this skill) that maps types/statuses to icons. Update it as patterns evolve.

### 3. Periodic Audits
Run icon audits monthly to catch drift. Automated scripts make this easy.

### 4. Single Source of Truth
Define icon mappings in one place (code constant, config file, or reference doc) and derive all implementations from it.

### 5. Graceful Fallbacks
Always have a default icon for unknown types. `📄` (generic document) is a safe fallback.

## Migration Strategy

### For Existing Databases

1. **Audit current state** - List all pages and their current icons
2. **Define target state** - Map types to icons
3. **Script the update** - Use API to bulk update
4. **Set up templates** - Prevent drift for new pages
5. **Document** - Record the icon system for team reference

### Incremental Rollout

If bulk update is risky:
1. Update templates first (affects new pages only)
2. Update high-visibility pages manually
3. Schedule gradual API updates for remaining pages
4. Monitor for issues

---

## Reference Implementation

For a complete working example of icon automation, see the Vouchline workspace implementation in:
- `docs/notion-migration/complete-seci-migration-plan.md`
- `docs/notion-migration/refactoring-proposal.md`
