#!/usr/bin/env python3
"""Create a two-column Word document from extracted text."""

import argparse
import subprocess
import sys

def main():
    parser = argparse.ArgumentParser(description='Create two-column Word document from text')
    parser.add_argument('text_file', help='Path to the OCR text file')
    parser.add_argument('output_path', help='Output path for the Word document')
    parser.add_argument('--title', '-t', help='Document title')
    parser.add_argument('--author', '-a', help='Document author')
    parser.add_argument('--page-number', '-p', type=int, default=1, help='Page number for header/footer')
    parser.add_argument('--total-pages', type=int, default=1, help='Total pages for footer')
    parser.add_argument('--header-left', help='Left header text (e.g., date)')
    parser.add_argument('--header-center', help='Center header text')
    parser.add_argument('--footer-left', help='Left footer text (e.g., URL)')
    args = parser.parse_args()

    try:
        from docx import Document
        from docx.shared import Pt, Cm
        from docx.enum.text import WD_ALIGN_PARAGRAPH
        from docx.enum.table import WD_TABLE_ALIGNMENT
        from docx.oxml.ns import qn
        from docx.oxml import OxmlElement
    except ImportError:
        print("Installing python-docx...")
        subprocess.run([sys.executable, '-m', 'pip', 'install', 'python-docx', '-q'], check=True)
        from docx import Document
        from docx.shared import Pt, Cm
        from docx.enum.text import WD_ALIGN_PARAGRAPH
        from docx.enum.table import WD_TABLE_ALIGNMENT
        from docx.oxml.ns import qn
        from docx.oxml import OxmlElement

    # Read text file
    with open(args.text_file, 'r') as f:
        text_content = f.read()

    # Create document
    doc = Document()

    # Set page margins
    for section in doc.sections:
        section.top_margin = Cm(1.5)
        section.bottom_margin = Cm(1.5)
        section.left_margin = Cm(1.5)
        section.right_margin = Cm(1.5)

    # Add header if provided
    if args.header_left or args.header_center:
        header = doc.sections[0].header
        header_para = header.paragraphs[0]
        header_text = ""
        if args.header_left:
            header_text += args.header_left
        if args.header_center:
            header_text += f"\t\t{args.header_center}"
        header_para.text = header_text

    # Add footer if provided
    if args.footer_left:
        footer = doc.sections[0].footer
        footer_para = footer.paragraphs[0]
        footer_para.text = f"{args.footer_left}\t\t{args.page_number}/{args.total_pages}"

    # Add title if provided
    if args.title:
        title_p = doc.add_paragraph()
        title_p.alignment = WD_ALIGN_PARAGRAPH.CENTER
        title_run = title_p.add_run(args.title)
        title_run.bold = True
        title_run.font.size = Pt(16)

    # Add author if provided
    if args.author:
        author_p = doc.add_paragraph()
        author_p.alignment = WD_ALIGN_PARAGRAPH.CENTER
        author_run = author_p.add_run(args.author)
        author_run.italic = True
        author_run.font.size = Pt(10)

    # Helper function to remove table borders
    def set_cell_border(cell):
        tc = cell._tc
        tcPr = tc.get_or_add_tcPr()
        tcBorders = OxmlElement('w:tcBorders')
        for edge in ('left', 'top', 'right', 'bottom'):
            edge_element = OxmlElement(f'w:{edge}')
            edge_element.set(qn('w:val'), 'nil')
            tcBorders.append(edge_element)
        tcPr.append(tcBorders)

    # Split content roughly in half for two columns
    lines = text_content.strip().split('\n')
    mid_point = len(lines) // 2
    left_content = '\n'.join(lines[:mid_point])
    right_content = '\n'.join(lines[mid_point:])

    # Create two-column table
    table = doc.add_table(rows=1, cols=2)
    table.alignment = WD_TABLE_ALIGNMENT.CENTER

    for row in table.rows:
        row.cells[0].width = Cm(8.5)
        row.cells[1].width = Cm(8.5)

    left_cell = table.rows[0].cells[0]
    right_cell = table.rows[0].cells[1]
    set_cell_border(left_cell)
    set_cell_border(right_cell)

    # Add content to cells
    def add_content_to_cell(cell, content):
        cell.paragraphs[0].text = ""
        for para_text in content.split('\n\n'):
            if para_text.strip():
                p = cell.add_paragraph()
                p.alignment = WD_ALIGN_PARAGRAPH.JUSTIFY
                run = p.add_run(para_text.strip())
                run.font.size = Pt(9)

    add_content_to_cell(left_cell, left_content)
    add_content_to_cell(right_cell, right_content)

    # Save document
    doc.save(args.output_path)
    print(f"Document saved: {args.output_path}")

if __name__ == '__main__':
    main()
