# Simple REST API - Example Project

此範例展示如何使用 **Project Development Planner Skill** 來開發一個模組化的 REST API 專案。

## 專案概述

一個簡單的任務管理 REST API，包含：
- 使用者認證（JWT）
- 任務 CRUD 操作
- 使用者管理

## 技術堆疊

- **Runtime:** Node.js 18+
- **Framework:** Express.js
- **Language:** TypeScript
- **Database:** PostgreSQL
- **Testing:** Jest
- **Documentation:** OpenAPI 3.0

## 模組化架構範例

此專案展示了推薦的模組化結構，以優化 AI token 使用：

### ✅ 優秀的模組結構

```
src/features/auth/
  auth.types.ts          # 45 行 - 公開介面定義
  auth.service.ts        # 180 行 - 核心業務邏輯
  auth.controller.ts     # 95 行 - Express API 控制器
  auth.middleware.ts     # 60 行 - JWT 驗證中介軟體
  auth.validator.ts      # 40 行 - 輸入驗證
  auth.test.ts           # 210 行 - 測試
  README.md              # 40 行 - 使用說明
  internals/             # 內部實作細節
    jwt.helper.ts        # 80 行
    password.helper.ts   # 60 行
```

### 為什麼這樣好？

1. **Interface First (auth.types.ts)**
   - 其他模組僅需讀取 45 行的介面定義
   - 不需要載入 180 行的實作細節
   - **Token 節省：76%**

2. **Small, Focused Files**
   - 每個檔案 < 220 行
   - AI 可以僅載入需要的部分
   - 易於理解和維護

3. **Clear Documentation (README.md)**
   - 40 行的 README 快速說明模組用途
   - 包含使用範例
   - 不需要閱讀程式碼就能理解模組

4. **Isolated Internals**
   - 內部實作細節放在 `internals/` 目錄
   - 其他模組不直接存取
   - 降低耦合度

## 使用 Skill 的工作流程

### Step 1: 檢視專案狀態

```bash
# 在 Claude Code 中
go status
```

輸出：
```
📍 當前階段：Backend Development (Phase 2) 🔄

進度：5/12 任務完成（42%）

✅ 已完成：
- Project setup
- Auth module (完整實作 + 測試)

⏳ 待辦：
- Tasks module (50% 完成)
- Users module
- API documentation
- Module dependency diagram

下一個任務：完成 tasks.controller.ts
```

### Step 2: 繼續開發

```bash
# 在 Claude Code 中
go
```

AI 會：
1. 檢查 Git 狀態
2. 讀取當前階段的 TODO.md
3. **僅載入相關檔案**（token 優化）：
   ```
   載入檔案：
   ✅ docs/architecture.md（僅 tasks 部分）
   ✅ src/features/tasks/tasks.types.ts（介面）
   ✅ src/features/tasks/tasks.controller.ts（當前實作）
   ✅ src/features/auth/auth.types.ts（依賴的介面）

   不載入：
   ❌ src/features/auth/auth.service.ts（不需要實作細節）
   ❌ src/features/users/*（無關模組）
   ❌ 測試檔案（當前任務不需要）
   ```
4. 開始實作任務
5. 完成後提示 commit

### Step 3: 任務完成與 Commit

AI 完成任務後會提示：

```
✅ 任務完成！已完成 tasks.controller.ts 實作

變更的檔案：
  src/features/tasks/tasks.controller.ts（新增，120 行）

建議的 commit：
───────────────────────────────────
feat(tasks): implement task CRUD endpoints

- Add GET /api/tasks (list with filtering)
- Add POST /api/tasks (create new task)
- Add PUT /api/tasks/:id (update task)
- Add DELETE /api/tasks/:id (delete task)
- Integrate with auth middleware
- Add input validation

Total: 120 lines

Closes #15
───────────────────────────────────

繼續提交？（yes/edit/skip）
```

選擇 `yes` 後，AI 會：
1. 執行 `git commit`
2. 更新 TODO.md 並標記任務為完成
3. 在 Development Log 中記錄
4. 單獨 commit TODO.md 的變更

### Step 4: 階段轉換

當 Phase 2 所有任務完成後：

```
🎉 Phase 2: Backend Development 完成！

所有任務已完成：
  ✅ Project setup
  ✅ Auth module
  ✅ Tasks module
  ✅ Users module
  ✅ API documentation
  ✅ Module dependency diagram

📊 階段統計：
  - 總 commits：12
  - 模組建立：3（auth, tasks, users）
  - 程式碼行數：~1,800 行
  - 測試覆蓋率：88%

模組化品質報告：
  ✅ 所有模組有清楚的介面定義
  ✅ 平均檔案大小：165 行
  ✅ 跨模組耦合度：低

🚀 準備進入 Phase 3: Testing

將卸載：Backend 實作細節（~1,400 行）
將載入：測試配置 + 模組介面（~400 行）
預估 token 節省：71%

繼續進入 Phase 3？（yes/no）
```

## Token 優化實例

### 情境：實作 Users Module

#### ❌ 傳統做法（未優化）

AI 載入：
```
✅ src/features/auth/auth.service.ts        180 行
✅ src/features/auth/auth.controller.ts      95 行
✅ src/features/auth/auth.middleware.ts      60 行
✅ src/features/auth/internals/*.ts         140 行
✅ src/features/tasks/tasks.service.ts      150 行
✅ src/features/tasks/tasks.controller.ts   120 行
✅ docs/architecture.md                     200 行
✅ docs/api-spec.md                         300 行
───────────────────────────────────────────────
總計：1,245 行
```

**問題：**
- 載入了 auth 和 tasks 的實作細節（不需要）
- 載入了完整的 API 規格（僅需 users 部分）
- 浪費大量 token

#### ✅ 使用 Skill（優化後）

AI 載入：
```
✅ src/features/auth/auth.types.ts           45 行（僅介面）
✅ src/features/tasks/tasks.types.ts         60 行（僅介面）
✅ docs/architecture.md#users-module         40 行（僅相關部分）
✅ docs/api-spec.md#users-endpoints          80 行（僅 users API）
───────────────────────────────────────────────
總計：225 行
```

**Token 節省：82%** 🎉

## 檔案大小分析

### 模組化前 vs 模組化後

#### Before（單一大檔案）
```
src/
  services.ts           2,100 行 ❌
  controllers.ts        1,500 行 ❌
  types.ts                800 行 ❌
```

**問題：**
- 修改一個小功能也要載入 2,100 行
- AI 需要讀取大量無關程式碼
- 難以維護

#### After（模組化）
```
src/features/
  auth/
    auth.types.ts          45 行 ✅
    auth.service.ts       180 行 ✅
    auth.controller.ts     95 行 ✅
    README.md              40 行 ✅

  tasks/
    tasks.types.ts         60 行 ✅
    tasks.service.ts      150 行 ✅
    tasks.controller.ts   120 行 ✅
    README.md              35 行 ✅

  users/
    users.types.ts         50 行 ✅
    users.service.ts      140 行 ✅
    users.controller.ts   110 行 ✅
    README.md              30 行 ✅
```

**優點：**
- 每個檔案 < 200 行
- 可以僅載入需要的模組
- 介面定義清楚（.types.ts）
- 每個模組有使用說明（README.md）

## 模組化健康度

使用 `go optimize` 指令可以檢查：

```
📊 模組化健康度報告

檔案大小分布：
  ✅ 95% 檔案 < 300 行
  ✅ 平均檔案大小：165 行

模組耦合度：
  ✅ auth: 完全獨立（僅依賴 core 模組）
  ✅ tasks: 僅透過介面依賴 auth
  ✅ users: 僅透過介面依賴 auth

介面覆蓋率：
  ✅ 3/3 模組有 .types.ts（100%）
  ✅ 3/3 模組有 README.md（100%）

Token 效率：
  ✅ 平均每任務載入：~250 行
  vs. 未優化前：~1,200 行
  節省：79%

建議：
  無需改進，模組化結構優秀！✨
```

## 學到的經驗

### 1. Interface First 是關鍵

先定義 `.types.ts`，再實作：
- AI 僅需讀 45 行介面就能理解整個模組
- 其他模組僅依賴介面，不需要載入實作
- 重構時僅需改實作，介面保持不變

### 2. 保持檔案小而專注

目標：每個檔案 < 300 行
- 更容易理解
- AI 可以僅載入需要的部分
- 降低認知負擔

### 3. README 是投資報酬率最高的文件

40 行的 README 可以：
- 讓 AI 快速理解模組用途
- 提供使用範例
- 說明依賴關係
- 避免載入實作程式碼

### 4. 善用 `internals/` 隔離實作細節

將內部實作放在 `internals/` 目錄：
- 清楚區分公開 API 和內部細節
- 其他模組不會誤用內部函數
- 降低耦合度

## 開始使用

### 1. 複製 TODO.md 範本

```bash
cp TODO.md /path/to/your/project/TODO.md
```

### 2. 調整階段和任務

根據你的專案需求修改 TODO.md

### 3. 啟動 Skill

```bash
cd /path/to/your/project

# 在 Claude Code 中
go
```

### 4. 遵循工作流程

讓 Skill 引導你完成開發，自動優化 token 使用！

## 相關資源

- [Skill 完整定義](../../project-dev-planner.md)
- [TODO.md 範本](../../TODO.template.md)
- [使用指南](../../README.md)

## 授權

MIT
