# Auth Module

## Purpose

處理使用者身份驗證和授權，包括：
- 使用者註冊和登入
- JWT token 產生和驗證
- Session 管理
- 密碼安全性檢查

## Public Interface

完整的型別定義請參見 `auth.types.ts`

### Main Service: AuthService

```typescript
interface AuthService {
  login(credentials: LoginRequest): Promise<AuthSession>
  register(data: RegisterRequest): Promise<UserInfo>
  logout(sessionId: string): Promise<boolean>
  refreshToken(refreshToken: string): Promise<AuthTokens>
  verifyToken(accessToken: string): Promise<UserInfo>
}
```

## Usage Examples

### Example 1: User Registration

```typescript
import { createAuthService } from './auth.service'
import { RegisterRequest } from './auth.types'

const authService = createAuthService()

const registerData: RegisterRequest = {
  email: 'user@example.com',
  password: 'SecurePass123',
  name: 'John Doe',
}

try {
  const user = await authService.register(registerData)
  console.log('User created:', user.id)
} catch (error) {
  if (error instanceof EmailAlreadyExistsError) {
    console.error('Email already registered')
  } else if (error instanceof WeakPasswordError) {
    console.error('Password too weak')
  }
}
```

### Example 2: User Login

```typescript
import { LoginRequest } from './auth.types'

const loginData: LoginRequest = {
  email: 'user@example.com',
  password: 'SecurePass123',
  rememberMe: true,
}

try {
  const session = await authService.login(loginData)

  // Store tokens securely
  localStorage.setItem('accessToken', session.tokens.accessToken)
  localStorage.setItem('refreshToken', session.tokens.refreshToken)

  console.log('Logged in as:', session.user.name)
  console.log('Session expires:', session.expiresAt)
} catch (error) {
  if (error instanceof InvalidCredentialsError) {
    console.error('Invalid email or password')
  } else if (error instanceof EmailNotVerifiedError) {
    console.error('Please verify your email first')
  }
}
```

### Example 3: Token Refresh

```typescript
const refreshToken = localStorage.getItem('refreshToken')

try {
  const newTokens = await authService.refreshToken(refreshToken)

  // Update stored tokens
  localStorage.setItem('accessToken', newTokens.accessToken)
  localStorage.setItem('refreshToken', newTokens.refreshToken)
} catch (error) {
  // Refresh token invalid/expired - redirect to login
  window.location.href = '/login'
}
```

### Example 4: Protected Route Middleware

```typescript
import { Request, Response, NextFunction } from 'express'
import { authMiddleware } from './auth.middleware'

// Apply to protected routes
app.get('/api/profile', authMiddleware, async (req, res) => {
  // req.user is populated by middleware
  res.json({ user: req.user })
})

// Require specific role
app.delete('/api/users/:id', authMiddleware, (req, res) => {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ error: 'Forbidden' })
  }
  // Delete user...
})
```

## Architecture

### Module Structure

```
auth/
  auth.types.ts          # 公開介面定義 (此檔案)
  auth.service.ts        # 核心業務邏輯
  auth.controller.ts     # Express API 控制器
  auth.middleware.ts     # JWT 驗證中介軟體
  auth.validator.ts      # 輸入驗證規則
  auth.test.ts           # 單元測試
  README.md              # 本文件
  internals/             # 內部實作細節（非公開）
    jwt.helper.ts        # JWT token 操作
    password.helper.ts   # 密碼雜湊和驗證
```

### Dependencies

#### External Dependencies
- `jsonwebtoken` - JWT token 產生和驗證
- `bcrypt` - 密碼雜湊
- `validator` - Email 和輸入驗證

#### Internal Dependencies
- `src/core/config` - 應用程式配置 (JWT secret, token expiry)
- `src/core/logger` - 日誌記錄
- `src/core/database` - 資料庫連線（透過 UserRepository）

### Data Flow

```
Client Request
    ↓
auth.controller.ts (API endpoint)
    ↓
auth.validator.ts (validate input)
    ↓
auth.service.ts (business logic)
    ↓
internals/password.helper.ts (hash/verify password)
    ↓
internals/jwt.helper.ts (generate/verify JWT)
    ↓
Database (via UserRepository)
    ↓
Response to Client
```

## Configuration

此模組需要以下環境變數：

```env
# JWT Settings
JWT_SECRET=your-secret-key-here
JWT_ACCESS_TOKEN_EXPIRY=15m
JWT_REFRESH_TOKEN_EXPIRY=7d

# Password Rules
PASSWORD_MIN_LENGTH=8
PASSWORD_REQUIRE_UPPERCASE=true
PASSWORD_REQUIRE_NUMBER=true
```

## Error Handling

所有錯誤都繼承自 `AuthError` 基礎類別，並包含錯誤代碼：

| Error Class | Code | HTTP Status | Description |
|-------------|------|-------------|-------------|
| `InvalidCredentialsError` | INVALID_CREDENTIALS | 401 | 錯誤的 email 或密碼 |
| `UserLockedError` | USER_LOCKED | 403 | 帳號被鎖定 |
| `EmailNotVerifiedError` | EMAIL_NOT_VERIFIED | 403 | Email 未驗證 |
| `EmailAlreadyExistsError` | EMAIL_ALREADY_EXISTS | 409 | Email 已被註冊 |
| `WeakPasswordError` | WEAK_PASSWORD | 400 | 密碼不符合安全要求 |
| `InvalidTokenError` | INVALID_TOKEN | 401 | Token 無效或過期 |

## Security Considerations

### Password Security
- 使用 bcrypt 進行密碼雜湊（cost factor: 10）
- 強制密碼複雜度要求（最少 8 字元，包含大小寫和數字）
- 密碼永遠不會以明文儲存或記錄

### Token Security
- Access tokens 短期有效（15 分鐘）
- Refresh tokens 長期有效但可被撤銷
- Tokens 包含 session ID 用於撤銷
- 使用 HS256 演算法（對稱式加密）

### Rate Limiting
建議在 API 層級加入 rate limiting：
- 登入端點：5 次/分鐘
- 註冊端點：3 次/小時
- Token refresh：10 次/分鐘

## Testing

### Run Tests
```bash
npm test -- auth.test
```

### Test Coverage
Current coverage: **92%**

- ✅ Unit tests for all service methods
- ✅ Error cases and edge cases
- ✅ JWT token generation and verification
- ✅ Password hashing and verification
- ✅ Middleware authentication flow

### Example Test
```typescript
describe('AuthService.login', () => {
  it('should return session with valid credentials', async () => {
    const session = await authService.login({
      email: 'test@example.com',
      password: 'ValidPass123',
    })

    expect(session.user.email).toBe('test@example.com')
    expect(session.tokens.accessToken).toBeDefined()
    expect(session.sessionId).toBeDefined()
  })

  it('should throw InvalidCredentialsError with wrong password', async () => {
    await expect(
      authService.login({
        email: 'test@example.com',
        password: 'WrongPassword',
      })
    ).rejects.toThrow(InvalidCredentialsError)
  })
})
```

## Performance

### Token Generation
- Average: ~10ms per token
- Bcrypt hashing: ~100ms (intentionally slow for security)

### Recommendations
- Cache user data to reduce database queries
- Use Redis for session storage (faster than database)
- Implement token blacklist in Redis for revoked tokens

## Known Issues / Limitations

1. **Session Storage** - Currently using database, should migrate to Redis for better performance
2. **Token Revocation** - No real-time token revocation (tokens remain valid until expiry)
3. **Multi-factor Auth** - Not yet implemented (planned for future)

## Future Enhancements

- [ ] OAuth 2.0 integration (Google, GitHub login)
- [ ] Two-factor authentication (2FA)
- [ ] Email verification flow
- [ ] Password reset functionality
- [ ] Session management dashboard
- [ ] Audit logging for authentication events

## Owned By

**Team:** Backend Team
**Maintainer:** [@backend-dev](mailto:backend-dev@example.com)
**Created:** 2025-01-08
**Last Updated:** 2025-01-08

## Related Documentation

- [API Documentation](../../../docs/api/auth.md)
- [Security Best Practices](../../../docs/security.md)
- [JWT Token Guide](../../../docs/guides/jwt-tokens.md)
