/**
 * Authentication Module - Type Definitions
 *
 * This file defines the public interface for the authentication module.
 * Other modules should only depend on these types, not the implementation.
 *
 * @module auth.types
 */

/**
 * Authentication Service Interface
 *
 * Handles user authentication, session management, and token operations.
 */
export interface AuthService {
  /**
   * Authenticate user with email and password
   *
   * @param credentials - User login credentials
   * @returns Authentication session with user info and tokens
   * @throws {InvalidCredentialsError} If email or password is incorrect
   * @throws {UserLockedError} If user account is locked
   * @throws {EmailNotVerifiedError} If email is not verified
   */
  login(credentials: LoginRequest): Promise<AuthSession>

  /**
   * Register a new user account
   *
   * @param data - User registration data
   * @returns Created user info (without password)
   * @throws {EmailAlreadyExistsError} If email is already registered
   * @throws {WeakPasswordError} If password doesn't meet requirements
   */
  register(data: RegisterRequest): Promise<UserInfo>

  /**
   * End user session and invalidate tokens
   *
   * @param sessionId - Session identifier
   * @returns true if session was active, false if already expired
   */
  logout(sessionId: string): Promise<boolean>

  /**
   * Refresh access token using refresh token
   *
   * @param refreshToken - Valid refresh token
   * @returns New token pair
   * @throws {InvalidTokenError} If refresh token is invalid or expired
   */
  refreshToken(refreshToken: string): Promise<AuthTokens>

  /**
   * Verify access token and return user info
   *
   * @param accessToken - JWT access token
   * @returns User information if token is valid
   * @throws {InvalidTokenError} If token is invalid or expired
   */
  verifyToken(accessToken: string): Promise<UserInfo>
}

/**
 * Login request payload
 */
export interface LoginRequest {
  /** User email address */
  email: string

  /** User password (plain text, will be hashed) */
  password: string

  /** Remember user session (extends token expiry) */
  rememberMe?: boolean
}

/**
 * Registration request payload
 */
export interface RegisterRequest {
  /** User email address (must be unique) */
  email: string

  /** User password (min 8 chars, must include uppercase, lowercase, number) */
  password: string

  /** User's full name */
  name: string

  /** Optional: User role (defaults to 'user') */
  role?: UserRole
}

/**
 * Authentication session data
 */
export interface AuthSession {
  /** Authenticated user information */
  user: UserInfo

  /** Access and refresh tokens */
  tokens: AuthTokens

  /** Session expiration timestamp */
  expiresAt: Date

  /** Unique session identifier */
  sessionId: string
}

/**
 * JWT token pair
 */
export interface AuthTokens {
  /** Short-lived access token (15 min) */
  accessToken: string

  /** Long-lived refresh token (7 days) */
  refreshToken: string

  /** Token type (always 'Bearer') */
  tokenType: 'Bearer'

  /** Access token expiry in seconds */
  expiresIn: number
}

/**
 * User information (safe for client)
 */
export interface UserInfo {
  /** Unique user identifier */
  id: string

  /** User email address */
  email: string

  /** User's full name */
  name: string

  /** User role for authorization */
  role: UserRole

  /** Account creation timestamp */
  createdAt: Date

  /** Email verification status */
  emailVerified: boolean
}

/**
 * User role enumeration
 */
export type UserRole = 'admin' | 'user' | 'guest'

/**
 * JWT token payload (internal use)
 */
export interface JWTPayload {
  /** User ID */
  sub: string

  /** User email */
  email: string

  /** User role */
  role: UserRole

  /** Issued at timestamp */
  iat: number

  /** Expiration timestamp */
  exp: number

  /** Session ID */
  sessionId: string
}

// ============================================================================
// Error Classes
// ============================================================================

/**
 * Base authentication error
 */
export class AuthError extends Error {
  constructor(message: string, public code: string) {
    super(message)
    this.name = 'AuthError'
  }
}

/**
 * Invalid email or password error
 */
export class InvalidCredentialsError extends AuthError {
  constructor() {
    super('Invalid email or password', 'INVALID_CREDENTIALS')
    this.name = 'InvalidCredentialsError'
  }
}

/**
 * User account locked error
 */
export class UserLockedError extends AuthError {
  constructor() {
    super('User account is locked. Please contact support.', 'USER_LOCKED')
    this.name = 'UserLockedError'
  }
}

/**
 * Email not verified error
 */
export class EmailNotVerifiedError extends AuthError {
  constructor() {
    super('Email address is not verified', 'EMAIL_NOT_VERIFIED')
    this.name = 'EmailNotVerifiedError'
  }
}

/**
 * Email already exists error
 */
export class EmailAlreadyExistsError extends AuthError {
  constructor() {
    super('Email address is already registered', 'EMAIL_ALREADY_EXISTS')
    this.name = 'EmailAlreadyExistsError'
  }
}

/**
 * Weak password error
 */
export class WeakPasswordError extends AuthError {
  constructor(message: string = 'Password does not meet security requirements') {
    super(message, 'WEAK_PASSWORD')
    this.name = 'WeakPasswordError'
  }
}

/**
 * Invalid or expired token error
 */
export class InvalidTokenError extends AuthError {
  constructor() {
    super('Invalid or expired token', 'INVALID_TOKEN')
    this.name = 'InvalidTokenError'
  }
}

// ============================================================================
// Constants
// ============================================================================

/**
 * Password validation rules
 */
export const PASSWORD_RULES = {
  minLength: 8,
  requireUppercase: true,
  requireLowercase: true,
  requireNumber: true,
  requireSpecialChar: false,
} as const

/**
 * Token expiry times (in seconds)
 */
export const TOKEN_EXPIRY = {
  accessToken: 15 * 60, // 15 minutes
  refreshToken: 7 * 24 * 60 * 60, // 7 days
  rememberMe: 30 * 24 * 60 * 60, // 30 days
} as const
