# 模組化設計指南

## 目的

本指南說明如何設計模組化的專案結構，以最大化 AI token 使用效率並提升程式碼品質。

## 核心原則

### 1. Interface First（介面優先）

**原則**：在實作前先定義清楚的介面

**為什麼重要**：
- AI 僅需讀取介面（30-50 行）即可理解模組
- 不需要載入實作細節（200+ 行）
- **Token 節省：70-90%**

**範例**：

```typescript
// ✅ 好的做法：auth.types.ts（僅 45 行）

/**
 * Authentication Service Interface
 */
export interface AuthService {
  login(credentials: LoginRequest): Promise<AuthSession>
  logout(sessionId: string): Promise<boolean>
  refreshToken(token: string): Promise<AuthTokens>
  verifyToken(token: string): Promise<UserInfo>
}

export interface LoginRequest {
  email: string
  password: string
  rememberMe?: boolean
}

export interface AuthSession {
  user: UserInfo
  tokens: AuthTokens
  expiresAt: Date
  sessionId: string
}

// ... 其他型別定義
```

**效益**：
- 規劃時：僅讀介面（45 行）
- 實作時：載入介面 + 實作（45 + 180 = 225 行）
- 使用時：僅讀介面（45 行）
- vs 傳統：總是載入實作（500+ 行）

### 2. Small Files（小檔案）

**原則**：每個檔案 < 300 行

**為什麼重要**：
- AI 可以僅載入需要的部分
- 更容易理解和維護
- 降低認知負擔

**檔案大小建議**：

| 檔案類型 | 建議行數 | 說明 |
|---------|---------|-----|
| 介面定義 (.types.ts) | 30-100 行 | 型別和介面定義 |
| 核心邏輯 (.service.ts) | 100-250 行 | 業務邏輯實作 |
| API 控制器 (.controller.ts) | 50-150 行 | 端點處理 |
| 測試檔案 (.test.ts) | 150-300 行 | 測試案例 |
| README.md | 30-100 行 | 模組說明 |

**範例**：

```
✅ 好的結構（token 友善）：

src/features/payment/
  payment.types.ts       # 45 行 - 介面定義
  payment.service.ts     # 180 行 - 核心邏輯
  payment.controller.ts  # 95 行 - API 端點
  payment.validator.ts   # 60 行 - 輸入驗證
  payment.test.ts        # 220 行 - 測試
  README.md              # 50 行 - 說明
  internals/             # 內部實作細節
    stripe.adapter.ts    # 120 行
    paypal.adapter.ts    # 100 行

總計：~870 行，但可以分別載入

❌ 不好的結構（token 浪費）：

src/
  payment.ts             # 1200 行 - 所有邏輯混在一起

總計：1200 行，必須整個載入
```

### 3. Single Responsibility（單一職責）

**原則**：每個模組/檔案只負責一件事

**範例**：

```
✅ 好的模組劃分：

src/features/
  auth/              # 僅負責身份驗證
    auth.types.ts
    auth.service.ts
    auth.controller.ts

  user/              # 僅負責使用者管理
    user.types.ts
    user.service.ts
    user.controller.ts

  notification/      # 僅負責通知
    notification.types.ts
    notification.service.ts

❌ 不好的模組劃分：

src/
  user-auth-notification.ts  # 混合多個職責
```

### 4. Clear Boundaries（清楚的邊界）

**原則**：模組間僅透過介面溝通

**範例**：

```typescript
// ✅ 好的做法：透過介面依賴

// order.service.ts
import { PaymentService } from '../payment/payment.types'  // 僅介面

class OrderService {
  constructor(private paymentService: PaymentService) {}

  async createOrder(data: OrderData) {
    // 使用 PaymentService 介面，不知道實作細節
    const payment = await this.paymentService.processPayment(...)
  }
}

// ❌ 不好的做法：直接依賴實作

// order.service.ts
import { PaymentServiceImpl } from '../payment/payment.service'  // 實作

class OrderService {
  constructor(private paymentService: PaymentServiceImpl) {
    // 直接依賴實作，緊密耦合
  }
}
```

### 5. Internals Isolation（內部實作隔離）

**原則**：將內部實作細節放在 `internals/` 目錄

**範例**：

```
src/features/auth/
  auth.types.ts          # 公開介面
  auth.service.ts        # 公開 API
  auth.controller.ts     # 公開 API
  README.md              # 公開文件
  internals/             # 內部細節（非公開）
    jwt.helper.ts        # JWT 內部邏輯
    password.helper.ts   # 密碼處理內部邏輯
    session.store.ts     # Session 儲存內部邏輯
```

**好處**：
- 清楚區分公開 API 和內部實作
- 其他模組不會誤用內部函數
- 更容易重構內部實作

## 推薦的專案結構

### 後端專案（Node.js/TypeScript）

```
src/
  core/                      # 核心共用模組
    config/
      config.types.ts        # 配置介面（30 行）
      config.service.ts      # 配置實作（80 行）
      README.md              # 使用說明（20 行）

    logger/
      logger.types.ts        # Logger 介面（25 行）
      logger.service.ts      # Logger 實作（120 行）
      README.md              # 使用說明（25 行）

    database/
      database.types.ts      # DB 連線介面（40 行）
      database.service.ts    # DB 連線實作（150 行）
      README.md              # 使用說明（30 行）

  features/                  # 功能模組（業務邏輯）
    auth/
      auth.types.ts          # 認證介面（45 行）
      auth.service.ts        # 認證邏輯（180 行）
      auth.controller.ts     # API 控制器（95 行）
      auth.middleware.ts     # 中介軟體（60 行）
      auth.validator.ts      # 輸入驗證（50 行）
      auth.test.ts           # 測試（210 行）
      README.md              # 模組說明（40 行）
      internals/
        jwt.helper.ts        # JWT 內部邏輯（80 行）
        password.helper.ts   # 密碼處理（60 行）

    [feature-name]/          # 其他功能模組
      [feature].types.ts     # 介面定義
      [feature].service.ts   # 核心邏輯
      [feature].controller.ts # API 端點
      [feature].test.ts      # 測試
      README.md              # 模組說明
      internals/             # 內部實作

  models/                    # 資料模型（共用）
    base.model.ts            # 基礎模型（50 行）
    user.model.ts            # 使用者模型（80 行）
    ...

  utils/                     # 工具函數（共用）
    date.utils.ts            # 日期工具（60 行）
    string.utils.ts          # 字串工具（40 行）
    validation.utils.ts      # 驗證工具（70 行）
```

### 前端專案（React/TypeScript）

```
src/
  components/                # 共用 UI 組件
    Button/
      Button.tsx             # 按鈕組件（60 行）
      Button.types.ts        # Props 介面（25 行）
      Button.styles.ts       # 樣式（40 行）
      Button.test.tsx        # 測試（90 行）
      Button.stories.tsx     # Storybook（50 行）
      README.md              # 使用說明（20 行）

    Input/
      Input.tsx
      Input.types.ts
      Input.styles.ts
      Input.test.tsx
      README.md

    Modal/
      ...

  features/                  # 功能模組
    auth/
      components/            # 功能專用組件
        LoginForm/
          LoginForm.tsx      # 登入表單（80 行）
          LoginForm.types.ts # Props 介面（20 行）
          LoginForm.test.tsx # 測試（70 行）

        RegisterForm/
          ...

      hooks/                 # 功能專用 hooks
        useAuth.ts           # 認證 hook（60 行）
        useAuth.types.ts     # Hook 介面（15 行）
        useAuth.test.ts      # 測試（50 行）

      store/                 # 功能專用狀態
        auth.slice.ts        # Redux slice（120 行）
        auth.types.ts        # State 介面（30 行）
        auth.slice.test.ts   # 測試（80 行）

      pages/                 # 頁面組件
        LoginPage.tsx        # 登入頁（50 行）
        RegisterPage.tsx     # 註冊頁（60 行）

      README.md              # 功能說明（35 行）

    dashboard/
      components/
      hooks/
      store/
      pages/
      README.md

  hooks/                     # 共用 hooks
    useApi.ts                # API hook（80 行）
    useDebounce.ts           # Debounce hook（30 行）
    useLocalStorage.ts       # LocalStorage hook（40 行）

  store/                     # 全域狀態管理
    index.ts                 # Store 配置（50 行）
    store.types.ts           # Store 介面（40 行）
    slices/
      app.slice.ts           # App slice（70 行）
      theme.slice.ts         # Theme slice（60 行）

  utils/                     # 工具函數
    format.ts                # 格式化工具（50 行）
    validation.ts            # 驗證工具（60 行）
    api.ts                   # API 工具（80 行）

  styles/                    # 共用樣式
    theme.ts                 # 主題定義（100 行）
    tokens.ts                # Design tokens（80 行）
    global.css               # 全域樣式（150 行）
```

## 模組介面定義規範

### TypeScript 範例

```typescript
// feature.types.ts

/**
 * [Feature] Service Interface
 *
 * [簡短說明模組用途]
 *
 * @example
 * const service = createFeatureService()
 * const result = await service.doSomething({ ... })
 */
export interface FeatureService {
  /**
   * [方法說明]
   *
   * @param request - [參數說明]
   * @returns [回傳值說明]
   * @throws {ErrorType} [錯誤情況說明]
   */
  doSomething(request: FeatureRequest): Promise<FeatureResult>

  /**
   * [另一個方法說明]
   */
  doAnotherThing(id: string): Promise<void>
}

/**
 * [Request 型別說明]
 */
export interface FeatureRequest {
  /** [欄位說明] */
  field1: string

  /** [欄位說明] */
  field2: number

  /** [可選欄位說明] */
  field3?: boolean
}

/**
 * [Result 型別說明]
 */
export interface FeatureResult {
  id: string
  status: FeatureStatus
  data: FeatureData
}

/**
 * [Status 列舉說明]
 */
export type FeatureStatus = 'pending' | 'completed' | 'failed'

/**
 * [Data 介面說明]
 */
export interface FeatureData {
  // ...
}

// ============================================================================
// 錯誤類別
// ============================================================================

/**
 * [錯誤類別說明]
 */
export class FeatureError extends Error {
  constructor(message: string, public code: string) {
    super(message)
    this.name = 'FeatureError'
  }
}

export class SpecificError extends FeatureError {
  constructor() {
    super('Specific error message', 'SPECIFIC_ERROR_CODE')
    this.name = 'SpecificError'
  }
}

// ============================================================================
// 常數
// ============================================================================

/**
 * [常數說明]
 */
export const FEATURE_CONSTANTS = {
  maxRetries: 3,
  timeout: 5000,
  // ...
} as const
```

### Python 範例

```python
# feature/types.py

from typing import Protocol, TypedDict, Literal
from dataclasses import dataclass

class FeatureService(Protocol):
    """
    [Feature] Service Interface

    [簡短說明模組用途]

    Example:
        service = create_feature_service()
        result = service.do_something(request)
    """

    def do_something(self, request: 'FeatureRequest') -> 'FeatureResult':
        """
        [方法說明]

        Args:
            request: [參數說明]

        Returns:
            [回傳值說明]

        Raises:
            FeatureError: [錯誤情況說明]
        """
        ...

    def do_another_thing(self, id: str) -> None:
        """[另一個方法說明]"""
        ...

class FeatureRequest(TypedDict):
    """[Request 型別說明]"""
    field1: str
    field2: int
    field3: bool  # optional

@dataclass
class FeatureResult:
    """[Result 型別說明]"""
    id: str
    status: Literal['pending', 'completed', 'failed']
    data: 'FeatureData'

@dataclass
class FeatureData:
    """[Data 類別說明]"""
    # ...

# 錯誤類別
class FeatureError(Exception):
    """[錯誤類別說明]"""
    def __init__(self, message: str, code: str):
        super().__init__(message)
        self.code = code

# 常數
FEATURE_CONSTANTS = {
    'max_retries': 3,
    'timeout': 5000,
}
```

## 模組 README 範本

```markdown
# [Module Name]

## Purpose

[1-2 句話說明模組用途和職責]

## Public Interface

完整的型別定義請參見 `[module].types.ts`（或 `types.py`）

### Main Service: [ServiceName]

\`\`\`typescript
interface [ServiceName] {
  method1(param: Type): Promise<Result>
  method2(id: string): Promise<void>
}
\`\`\`

## Usage Examples

### Example 1: [常見使用案例名稱]

\`\`\`typescript
import { createService } from './service'

const service = createService()
const result = await service.method1({ ... })
console.log(result)
\`\`\`

### Example 2: [另一個常見案例]

\`\`\`typescript
// 使用範例程式碼
\`\`\`

## Architecture

### Module Structure

\`\`\`
[module]/
  [module].types.ts      # 公開介面定義
  [module].service.ts    # 核心業務邏輯
  [module].controller.ts # API 控制器（如果適用）
  [module].test.ts       # 測試
  README.md              # 本文件
  internals/             # 內部實作細節
    helper1.ts
    helper2.ts
\`\`\`

### Dependencies

#### External Dependencies
- `package-name` - [用途說明]
- `another-package` - [用途說明]

#### Internal Dependencies
- `src/core/logger` - [用途說明]
- `src/core/config` - [用途說明]

### Data Flow

\`\`\`
[簡單的資料流程說明或圖表]
\`\`\`

## Configuration

[如果模組需要配置，說明配置選項]

\`\`\`typescript
// 配置範例
\`\`\`

## Error Handling

[說明模組可能拋出的錯誤]

| Error Class | Code | Description |
|-------------|------|-------------|
| ErrorType1 | CODE1 | [說明] |
| ErrorType2 | CODE2 | [說明] |

## Testing

\`\`\`bash
# 執行測試
npm test -- [module].test

# 查看覆蓋率
npm run test:coverage -- [module]
\`\`\`

Current coverage: [XX%]

## Performance

[如果有效能考量，說明效能特性]

- Average response time: [XX ms]
- Memory usage: [XX MB]
- Recommendations: [優化建議]

## Known Issues / Limitations

[列出已知問題或限制]

1. [問題 1]
2. [問題 2]

## Future Enhancements

[未來可能的改進]

- [ ] [改進項目 1]
- [ ] [改進項目 2]

## Owned By

**Team:** [團隊名稱]
**Maintainer:** [@username](mailto:email@example.com)
**Created:** YYYY-MM-DD
**Last Updated:** YYYY-MM-DD

## Related Documentation

- [相關文件連結 1]
- [相關文件連結 2]
```

## Token 優化檢查清單

建立或審查模組時，使用此檢查清單：

### ✅ 介面定義
- [ ] 模組有獨立的介面定義檔（.types.ts 或類似）
- [ ] 介面檔案 < 100 行
- [ ] 所有公開 API 都有 JSDoc 註解
- [ ] 包含使用範例

### ✅ 檔案大小
- [ ] 所有檔案 < 300 行
- [ ] 介面定義 < 100 行
- [ ] 核心邏輯 < 250 行
- [ ] 測試檔案 < 300 行

### ✅ 模組邊界
- [ ] 僅透過介面依賴其他模組
- [ ] 無直接引用其他模組的實作檔案
- [ ] 內部實作放在 internals/ 目錄

### ✅ 文件
- [ ] 有 README.md 說明模組用途
- [ ] 包含至少 2 個使用範例
- [ ] 說明依賴關係
- [ ] 列出可能的錯誤

### ✅ 測試
- [ ] 有測試檔案
- [ ] 測試覆蓋率 > 80%
- [ ] Mock 所有外部依賴
- [ ] 透過介面 mock 其他模組

## 實際案例分析

### 案例 1：Auth 模組優化

#### Before（未優化）

```
src/auth/
  auth.ts                    # 850 行 - 所有邏輯混在一起

Token 使用：
- 實作時：載入 850 行
- 使用時：載入 850 行
- 測試時：載入 850 行
```

#### After（優化後）

```
src/auth/
  auth.types.ts              # 45 行 - 介面定義
  auth.service.ts            # 180 行 - 核心邏輯
  auth.controller.ts         # 95 行 - API 端點
  auth.middleware.ts         # 60 行 - 中介軟體
  auth.test.ts               # 210 行 - 測試
  README.md                  # 40 行 - 說明
  internals/
    jwt.helper.ts            # 80 行
    password.helper.ts       # 60 行

Token 使用：
- 規劃時：載入 auth.types.ts（45 行）
- 實作 service：載入 types + service（45 + 180 = 225 行）
- 實作 controller：載入 types + controller（45 + 95 = 140 行）
- 使用時：載入 auth.types.ts（45 行）
- 測試時：載入 types + test（45 + 210 = 255 行）

節省：47-89%
```

### 案例 2：跨模組依賴優化

#### Before（未優化）

```typescript
// order.service.ts
import { PaymentService } from '../payment/payment.service'  // 載入 500 行實作
import { UserService } from '../user/user.service'          // 載入 400 行實作
import { NotificationService } from '../notification/notification.service'  // 載入 300 行實作

// 總計需要載入：500 + 400 + 300 = 1200 行的依賴實作
```

#### After（優化後）

```typescript
// order.service.ts
import { PaymentService } from '../payment/payment.types'      // 載入 50 行介面
import { UserService } from '../user/user.types'              // 載入 40 行介面
import { NotificationService } from '../notification/notification.types'  // 載入 35 行介面

// 總計需要載入：50 + 40 + 35 = 125 行的介面定義

// Token 節省：(1200 - 125) / 1200 = 89.6%
```

## 總結

### 關鍵要點

1. **Interface First** - 先定義介面，再實作
2. **Small Files** - 保持檔案小於 300 行
3. **Single Responsibility** - 每個模組只做一件事
4. **Clear Boundaries** - 透過介面溝通
5. **Internals Isolation** - 隔離內部實作

### Token 節省效益

| 策略 | 節省幅度 | 說明 |
|------|---------|-----|
| Interface First | 70-90% | 僅載入介面而非實作 |
| Small Files | 60-80% | 僅載入需要的檔案 |
| Module README | 50-70% | 快速了解用途無需讀程式碼 |
| Internals Isolation | 40-60% | 不載入內部實作細節 |

### 下一步

1. 審查現有專案的模組結構
2. 使用檢查清單評估每個模組
3. 優先重構最常用的模組
4. 使用 `go optimize` 追蹤改進
