# 開發階段詳細說明

完整的 5 階段開發流程詳解，包含每個階段的目標、任務、產出和最佳實踐。

## 目錄

- [階段概覽](#階段概覽)
- [Phase 1: Planning](#phase-1-planning)
- [Phase 2: Backend Development](#phase-2-backend-development)
- [Phase 3: Frontend Development](#phase-3-frontend-development)
- [Phase 4: Testing](#phase-4-testing)
- [Phase 5: Deployment](#phase-5-deployment)
- [階段轉換](#階段轉換)

## 階段概覽

### 5 階段開發流程

```
Phase 1: Planning (10-20% 時間)
  ↓ 定義清楚的架構和介面
Phase 2: Backend Development (30-35% 時間)
  ↓ 穩固的後端基礎
Phase 3: Frontend Development (25-30% 時間)
  ↓ 使用者介面完成
Phase 4: Testing (15-20% 時間)
  ↓ 品質保證
Phase 5: Deployment (5-10% 時間)
  ↓ 產品上線
```

### 階段特性

| 階段 | 主要活動 | 載入檔案類型 | Token 使用 |
|------|---------|-------------|-----------|
| Planning | 架構設計、介面定義 | 文件、spec | ~200 行 |
| Backend | 實作後端邏輯 | 介面 + 當前模組 | ~300 行 |
| Frontend | 實作前端介面 | API spec + 元件 | ~250 行 |
| Testing | 測試撰寫 | 測試 + 被測模組 | ~400 行 |
| Deployment | 部署設定 | 配置檔 + 文件 | ~300 行 |

## Phase 1: Planning

### 目標

建立清楚的技術架構和模組邊界，為後續開發奠定基礎。

### 核心原則

1. **Interface First** - 先定義介面，再實作
2. **模組化思維** - 清楚的模組邊界和職責
3. **最小化設計** - 避免過度設計
4. **Token 優化** - 為後續階段優化上下文載入

### 主要任務

#### 1. Requirements Analysis（需求分析）

**目標**：確保團隊理解產品需求

**產出**：
```markdown
# docs/requirements.md

## MVP 功能範圍
- [ ] 功能 1：[描述]
- [ ] 功能 2：[描述]
- [ ] 功能 3：[描述]

## 明確排除
- ❌ 功能 X（v1.1）
- ❌ 功能 Y（v2.0）

## 使用者故事
- 作為 [使用者]，我想要 [功能]，以便 [價值]
```

#### 2. Architecture Design（架構設計）

**目標**：定義系統的高層架構

**產出**：
```markdown
# docs/architecture.md

## 系統架構

### 後端架構
- API Layer：處理 HTTP 請求
- Service Layer：業務邏輯
- Data Layer：資料存取

### 前端架構
- Component Layer：UI 元件
- State Management：應用狀態
- API Client：後端通訊

### 資料流
Client → API → Service → Data → Database
```

**最佳實踐**：
```
✅ 使用圖表（如 Mermaid）視覺化架構
✅ 說明每一層的職責
✅ 定義層與層之間的介面
❌ 避免過度細節（實作細節在後續階段）
```

#### 3. Module Boundaries Definition（模組邊界定義）

**目標**：定義模組職責和依賴關係

**產出**：
```markdown
# docs/module-boundaries.md

## 模組列表

### auth（認證模組）
- 職責：使用者認證、授權
- 公開介面：auth.types.ts
- 依賴：database
- 被依賴：api, user

### task（任務模組）
- 職責：任務 CRUD、狀態管理
- 公開介面：task.types.ts
- 依賴：database, auth
- 被依賴：api

## 模組依賴圖
```mermaid
graph TD
  api --> auth
  api --> task
  task --> database
  auth --> database
```
```

**最佳實踐**：
```
✅ 每個模組單一職責
✅ 避免循環依賴
✅ 依賴關係清楚明確
✅ 3-5 個核心模組（MVP）
```

#### 4. Interface Definition（介面定義）

**目標**：定義所有模組的公開介面

**產出**（每個模組）：
```typescript
// src/modules/task/task.types.ts

/**
 * 任務服務介面
 * 提供任務的 CRUD 操作
 */
export interface TaskService {
  /**
   * 建立新任務
   * @param data - 任務資料
   * @returns 建立的任務
   */
  createTask(data: CreateTaskRequest): Promise<Task>

  /**
   * 取得任務清單
   * @param userId - 使用者 ID
   * @returns 任務陣列
   */
  getTasks(userId: string): Promise<Task[]>

  /**
   * 更新任務狀態
   * @param taskId - 任務 ID
   * @param completed - 是否完成
   */
  updateTaskStatus(taskId: string, completed: boolean): Promise<Task>

  /**
   * 刪除任務
   * @param taskId - 任務 ID
   */
  deleteTask(taskId: string): Promise<boolean>
}

export interface Task {
  id: string
  title: string
  description?: string
  completed: boolean
  userId: string
  createdAt: Date
  updatedAt: Date
}

export interface CreateTaskRequest {
  title: string
  description?: string
  userId: string
}
```

**最佳實踐**：
```
✅ 詳細的 JSDoc 註解
✅ 明確的型別定義
✅ 介面檔案保持簡潔（< 100 行）
✅ 避免實作細節
```

#### 5. Database Schema Design（資料庫設計）

**目標**：定義資料庫結構

**產出**：
```sql
-- docs/schema.sql

-- 使用者表
CREATE TABLE users (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  email VARCHAR(255) UNIQUE NOT NULL,
  password_hash VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 任務表
CREATE TABLE tasks (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title VARCHAR(255) NOT NULL,
  description TEXT,
  completed BOOLEAN DEFAULT FALSE,
  user_id UUID REFERENCES users(id) ON DELETE CASCADE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 索引
CREATE INDEX idx_tasks_user_id ON tasks(user_id);
CREATE INDEX idx_tasks_completed ON tasks(completed);
```

### 智慧上下文載入

**載入檔案**（~200 行）：
```
✅ docs/requirements.md（僅摘要）
✅ docs/architecture.md
✅ docs/module-boundaries.md
✅ docs/schema.sql
```

**不載入**：
```
❌ 實作檔案（尚未建立）
❌ 測試檔案
❌ 配置檔案
```

### 階段完成標準

```
✅ 所有模組都有介面定義（.types.ts）
✅ 模組依賴關係清楚無循環
✅ 資料庫 schema 已定義
✅ 架構文件完整
✅ 團隊理解技術決策
```

### 常見錯誤

```
❌ 過度設計：定義太多模組或層次
❌ 缺少介面：直接開始寫實作
❌ 忽略依賴：沒有規劃模組依賴關係
❌ 文件不足：口頭討論但沒有記錄
```

## Phase 2: Backend Development

### 目標

實作後端業務邏輯和 API 端點，確保模組化和可測試性。

### 核心原則

1. **按模組開發** - 一次完成一個模組
2. **Interface First** - 依照 Phase 1 定義的介面實作
3. **測試就緒** - 易於測試的程式碼結構
4. **Token 優化** - 僅載入當前模組及其介面

### 開發順序

```
1. 基礎設施模組（database, logger, config）
   ↓
2. 核心業務模組（按依賴順序）
   ↓
3. API 層（routes, controllers）
   ↓
4. 中介軟體（auth, error handling）
```

### 主要任務

#### 1. Database Module（資料庫模組）

**目標**：封裝資料庫存取邏輯

**檔案結構**：
```
src/modules/database/
├── db.types.ts          # 介面定義（Phase 1 已建立）
├── db.service.ts        # 資料庫連接和基本操作
├── migrations/          # 資料庫遷移
│   └── 001_initial.sql
└── README.md            # 使用說明
```

**實作範例**（db.service.ts）：
```typescript
import { Pool } from 'pg'
import { DatabaseService } from './db.types'

export class PostgresService implements DatabaseService {
  private pool: Pool

  constructor(connectionString: string) {
    this.pool = new Pool({ connectionString })
  }

  async query<T>(sql: string, params: any[]): Promise<T[]> {
    const result = await this.pool.query(sql, params)
    return result.rows
  }

  async transaction<T>(callback: (client: any) => Promise<T>): Promise<T> {
    const client = await this.pool.connect()
    try {
      await client.query('BEGIN')
      const result = await callback(client)
      await client.query('COMMIT')
      return result
    } catch (error) {
      await client.query('ROLLBACK')
      throw error
    } finally {
      client.release()
    }
  }
}
```

#### 2. Business Logic Modules（業務邏輯模組）

**目標**：實作核心業務邏輯

**範例：Task Module**

```
src/modules/task/
├── task.types.ts        # 介面定義（Phase 1）
├── task.service.ts      # 業務邏輯實作
├── task.repository.ts   # 資料存取層
└── README.md
```

**task.service.ts**：
```typescript
import { TaskService, Task, CreateTaskRequest } from './task.types'
import { TaskRepository } from './task.repository'

export class TaskServiceImpl implements TaskService {
  constructor(private repository: TaskRepository) {}

  async createTask(data: CreateTaskRequest): Promise<Task> {
    // 業務驗證
    if (!data.title.trim()) {
      throw new Error('Task title cannot be empty')
    }

    // 呼叫資料層
    return this.repository.create(data)
  }

  async getTasks(userId: string): Promise<Task[]> {
    return this.repository.findByUserId(userId)
  }

  async updateTaskStatus(taskId: string, completed: boolean): Promise<Task> {
    const task = await this.repository.findById(taskId)
    if (!task) {
      throw new Error('Task not found')
    }

    return this.repository.update(taskId, { completed })
  }

  async deleteTask(taskId: string): Promise<boolean> {
    return this.repository.delete(taskId)
  }
}
```

**最佳實踐**：
```
✅ 業務邏輯在 Service 層
✅ 資料存取在 Repository 層
✅ 清楚的錯誤處理
✅ 單一檔案 < 200 行
```

#### 3. API Layer（API 層）

**目標**：暴露 HTTP 端點

**檔案結構**：
```
src/api/
├── routes/
│   ├── task.routes.ts
│   └── auth.routes.ts
├── controllers/
│   ├── task.controller.ts
│   └── auth.controller.ts
├── middleware/
│   ├── auth.middleware.ts
│   └── error.middleware.ts
└── app.ts               # Express 應用設定
```

**task.routes.ts**：
```typescript
import express from 'express'
import { TaskController } from '../controllers/task.controller'
import { authMiddleware } from '../middleware/auth.middleware'

export function createTaskRoutes(controller: TaskController) {
  const router = express.Router()

  router.post('/tasks', authMiddleware, (req, res, next) => {
    controller.createTask(req, res).catch(next)
  })

  router.get('/tasks', authMiddleware, (req, res, next) => {
    controller.getTasks(req, res).catch(next)
  })

  router.patch('/tasks/:id/status', authMiddleware, (req, res, next) => {
    controller.updateTaskStatus(req, res).catch(next)
  })

  router.delete('/tasks/:id', authMiddleware, (req, res, next) => {
    controller.deleteTask(req, res).catch(next)
  })

  return router
}
```

**task.controller.ts**：
```typescript
import { Request, Response } from 'express'
import { TaskService } from '../../modules/task/task.types'

export class TaskController {
  constructor(private taskService: TaskService) {}

  async createTask(req: Request, res: Response) {
    const { title, description } = req.body
    const userId = req.user.id // 從 auth middleware 取得

    const task = await this.taskService.createTask({
      title,
      description,
      userId,
    })

    res.status(201).json(task)
  }

  async getTasks(req: Request, res: Response) {
    const userId = req.user.id
    const tasks = await this.taskService.getTasks(userId)
    res.json(tasks)
  }

  async updateTaskStatus(req: Request, res: Response) {
    const { id } = req.params
    const { completed } = req.body

    const task = await this.taskService.updateTaskStatus(id, completed)
    res.json(task)
  }

  async deleteTask(req: Request, res: Response) {
    const { id } = req.params
    await this.taskService.deleteTask(id)
    res.status(204).send()
  }
}
```

#### 4. Error Handling（錯誤處理）

**error.middleware.ts**：
```typescript
import { Request, Response, NextFunction } from 'express'

export class AppError extends Error {
  constructor(
    public statusCode: number,
    public message: string
  ) {
    super(message)
  }
}

export function errorHandler(
  err: Error,
  req: Request,
  res: Response,
  next: NextFunction
) {
  if (err instanceof AppError) {
    return res.status(err.statusCode).json({
      error: err.message,
    })
  }

  console.error('Unexpected error:', err)
  res.status(500).json({
    error: 'Internal server error',
  })
}
```

### 智慧上下文載入

**開發單一模組時載入**（~300 行）：
```
✅ docs/architecture.md（當前模組相關部分）
✅ docs/module-boundaries.md（依賴關係）
✅ src/modules/[current]/[module].types.ts（介面定義）
✅ src/modules/[dependency]/[module].types.ts（依賴模組介面）
✅ src/modules/[current]/[module].service.ts（當前實作）
```

**不載入**：
```
❌ 其他模組的實作細節
❌ 前端檔案
❌ 測試檔案（除非需要參考）
❌ 完整的文件檔案
```

### 階段完成標準

```
✅ 所有模組實作完成
✅ API 端點可正常呼叫
✅ 錯誤處理完整
✅ 模組 README 完成
✅ 每個模組 < 200 行
✅ 無跨模組直接依賴實作（僅依賴介面）
```

### 開發流程範例

```
1. 開始 database 模組
   └─ 載入：db.types.ts, docs/schema.sql
   └─ 實作：db.service.ts
   └─ 提交：feat(database): implement PostgreSQL service

2. 開始 task 模組
   └─ 載入：task.types.ts, db.types.ts（不載入 db.service.ts）
   └─ 實作：task.repository.ts, task.service.ts
   └─ 提交：feat(task): implement task CRUD logic

3. 開始 API 層
   └─ 載入：task.types.ts, auth.types.ts（不載入實作）
   └─ 實作：task.routes.ts, task.controller.ts
   └─ 提交：feat(api): add task API endpoints

4. 錯誤處理
   └─ 實作：error.middleware.ts
   └─ 提交：feat(api): add error handling middleware
```

## Phase 3: Frontend Development

### 目標

實作使用者介面，整合後端 API，提供良好的使用者體驗。

### 核心原則

1. **元件化開發** - 可重用的小元件
2. **API 整合** - 正確處理非同步請求
3. **狀態管理** - 清楚的資料流
4. **Token 優化** - 僅載入當前元件及相關 API

### 開發順序

```
1. API Client（API 客戶端）
   ↓
2. 核心元件（按頁面/功能）
   ↓
3. 狀態管理（Context/Redux/等）
   ↓
4. 樣式和 UI 優化
```

### 主要任務

#### 1. API Client（API 客戶端）

**目標**：封裝後端 API 呼叫

**檔案結構**：
```
src/api/
├── client.ts            # HTTP 客戶端設定
├── task.api.ts          # Task API
└── auth.api.ts          # Auth API
```

**client.ts**：
```typescript
import axios, { AxiosInstance } from 'axios'

export class ApiClient {
  private client: AxiosInstance

  constructor(baseURL: string) {
    this.client = axios.create({
      baseURL,
      headers: {
        'Content-Type': 'application/json',
      },
    })

    // 自動加入 auth token
    this.client.interceptors.request.use((config) => {
      const token = localStorage.getItem('auth_token')
      if (token) {
        config.headers.Authorization = `Bearer ${token}`
      }
      return config
    })
  }

  async get<T>(url: string): Promise<T> {
    const response = await this.client.get<T>(url)
    return response.data
  }

  async post<T>(url: string, data: any): Promise<T> {
    const response = await this.client.post<T>(url, data)
    return response.data
  }

  // ... patch, delete
}
```

**task.api.ts**：
```typescript
import { ApiClient } from './client'
import { Task, CreateTaskRequest } from '../types/task.types'

export class TaskApi {
  constructor(private client: ApiClient) {}

  async getTasks(): Promise<Task[]> {
    return this.client.get<Task[]>('/tasks')
  }

  async createTask(data: CreateTaskRequest): Promise<Task> {
    return this.client.post<Task>('/tasks', data)
  }

  async updateTaskStatus(taskId: string, completed: boolean): Promise<Task> {
    return this.client.patch<Task>(`/tasks/${taskId}/status`, { completed })
  }

  async deleteTask(taskId: string): Promise<void> {
    await this.client.delete(`/tasks/${taskId}`)
  }
}
```

#### 2. Core Components（核心元件）

**目標**：實作 UI 元件

**範例：TaskList Component**

```
src/components/TaskList/
├── TaskList.tsx
├── TaskList.types.ts
├── TaskList.module.css
└── TaskItem.tsx
```

**TaskList.types.ts**：
```typescript
export interface TaskListProps {
  tasks: Task[]
  onToggleTask: (taskId: string) => void
  onDeleteTask: (taskId: string) => void
}
```

**TaskList.tsx**：
```typescript
import React from 'react'
import { TaskListProps } from './TaskList.types'
import { TaskItem } from './TaskItem'
import styles from './TaskList.module.css'

export const TaskList: React.FC<TaskListProps> = ({
  tasks,
  onToggleTask,
  onDeleteTask,
}) => {
  if (tasks.length === 0) {
    return <p className={styles.empty}>沒有任務</p>
  }

  return (
    <ul className={styles.list}>
      {tasks.map((task) => (
        <TaskItem
          key={task.id}
          task={task}
          onToggle={() => onToggleTask(task.id)}
          onDelete={() => onDeleteTask(task.id)}
        />
      ))}
    </ul>
  )
}
```

#### 3. State Management（狀態管理）

**範例：使用 Custom Hook**

**useTasks.ts**：
```typescript
import { useState, useEffect } from 'react'
import { Task } from '../types/task.types'
import { taskApi } from '../api'

export function useTasks() {
  const [tasks, setTasks] = useState<Task[]>([])
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    loadTasks()
  }, [])

  async function loadTasks() {
    setLoading(true)
    setError(null)
    try {
      const data = await taskApi.getTasks()
      setTasks(data)
    } catch (err) {
      setError('Failed to load tasks')
    } finally {
      setLoading(false)
    }
  }

  async function createTask(title: string, description?: string) {
    try {
      const newTask = await taskApi.createTask({ title, description })
      setTasks([...tasks, newTask])
    } catch (err) {
      setError('Failed to create task')
    }
  }

  async function toggleTask(taskId: string) {
    const task = tasks.find((t) => t.id === taskId)
    if (!task) return

    try {
      const updated = await taskApi.updateTaskStatus(taskId, !task.completed)
      setTasks(tasks.map((t) => (t.id === taskId ? updated : t)))
    } catch (err) {
      setError('Failed to update task')
    }
  }

  async function deleteTask(taskId: string) {
    try {
      await taskApi.deleteTask(taskId)
      setTasks(tasks.filter((t) => t.id !== taskId))
    } catch (err) {
      setError('Failed to delete task')
    }
  }

  return {
    tasks,
    loading,
    error,
    createTask,
    toggleTask,
    deleteTask,
    reload: loadTasks,
  }
}
```

### 智慧上下文載入

**開發單一元件時載入**（~250 行）：
```
✅ docs/api-spec.md（相關 API 端點）
✅ src/api/task.api.ts（API 客戶端）
✅ src/components/TaskList/TaskList.types.ts（介面定義）
✅ src/components/TaskList/TaskList.tsx（當前元件）
✅ src/hooks/useTasks.ts（相關狀態管理）
```

**不載入**：
```
❌ 後端實作檔案
❌ 其他頁面/元件
❌ 完整的 API 文件
```

### 階段完成標準

```
✅ 所有核心元件實作完成
✅ API 整合正確
✅ 錯誤處理完整
✅ 載入狀態顯示
✅ 基本樣式完成
```

## Phase 4: Testing

### 目標

確保程式碼品質和正確性，建立信心部署到正式環境。

### 核心原則

1. **測試金字塔** - 單元測試為主，整合測試為輔
2. **MVP 測試策略** - 專注於核心功能測試
3. **測試可讀性** - 測試即文件
4. **Token 優化** - 僅載入測試和被測模組

### 測試優先順序（MVP）

```
1. 核心業務邏輯單元測試（必須）
2. API 端點整合測試（必須）
3. 關鍵元件測試（重要）
4. E2E 測試（可選，MVP 可省略）
```

### 主要任務

#### 1. Backend Unit Tests（後端單元測試）

**目標**：測試業務邏輯

**task.service.test.ts**：
```typescript
import { TaskServiceImpl } from '../task.service'
import { TaskRepository } from '../task.repository'

describe('TaskService', () => {
  let service: TaskServiceImpl
  let mockRepository: jest.Mocked<TaskRepository>

  beforeEach(() => {
    mockRepository = {
      create: jest.fn(),
      findByUserId: jest.fn(),
      findById: jest.fn(),
      update: jest.fn(),
      delete: jest.fn(),
    } as any

    service = new TaskServiceImpl(mockRepository)
  })

  describe('createTask', () => {
    it('should create a task with valid data', async () => {
      const taskData = {
        title: 'Test Task',
        description: 'Test Description',
        userId: 'user-1',
      }

      const expectedTask = {
        id: 'task-1',
        ...taskData,
        completed: false,
        createdAt: new Date(),
        updatedAt: new Date(),
      }

      mockRepository.create.mockResolvedValue(expectedTask)

      const result = await service.createTask(taskData)

      expect(result).toEqual(expectedTask)
      expect(mockRepository.create).toHaveBeenCalledWith(taskData)
    })

    it('should throw error when title is empty', async () => {
      const taskData = {
        title: '   ',
        userId: 'user-1',
      }

      await expect(service.createTask(taskData)).rejects.toThrow(
        'Task title cannot be empty'
      )

      expect(mockRepository.create).not.toHaveBeenCalled()
    })
  })

  describe('updateTaskStatus', () => {
    it('should update task status', async () => {
      const existingTask = {
        id: 'task-1',
        title: 'Test',
        completed: false,
        userId: 'user-1',
        createdAt: new Date(),
        updatedAt: new Date(),
      }

      const updatedTask = { ...existingTask, completed: true }

      mockRepository.findById.mockResolvedValue(existingTask)
      mockRepository.update.mockResolvedValue(updatedTask)

      const result = await service.updateTaskStatus('task-1', true)

      expect(result.completed).toBe(true)
      expect(mockRepository.update).toHaveBeenCalledWith('task-1', {
        completed: true,
      })
    })

    it('should throw error when task not found', async () => {
      mockRepository.findById.mockResolvedValue(null)

      await expect(service.updateTaskStatus('invalid', true)).rejects.toThrow(
        'Task not found'
      )
    })
  })
})
```

#### 2. API Integration Tests（API 整合測試）

**task.routes.test.ts**：
```typescript
import request from 'supertest'
import { app } from '../app'
import { setupTestDatabase, cleanupTestDatabase } from './test-helpers'

describe('Task API', () => {
  let authToken: string

  beforeAll(async () => {
    await setupTestDatabase()
    // 建立測試使用者並取得 token
    const response = await request(app)
      .post('/auth/login')
      .send({ email: 'test@example.com', password: 'password' })
    authToken = response.body.token
  })

  afterAll(async () => {
    await cleanupTestDatabase()
  })

  describe('POST /tasks', () => {
    it('should create a new task', async () => {
      const response = await request(app)
        .post('/tasks')
        .set('Authorization', `Bearer ${authToken}`)
        .send({
          title: 'Test Task',
          description: 'Test Description',
        })

      expect(response.status).toBe(201)
      expect(response.body).toMatchObject({
        title: 'Test Task',
        description: 'Test Description',
        completed: false,
      })
      expect(response.body.id).toBeDefined()
    })

    it('should return 401 without auth token', async () => {
      const response = await request(app).post('/tasks').send({
        title: 'Test Task',
      })

      expect(response.status).toBe(401)
    })

    it('should return 400 with invalid data', async () => {
      const response = await request(app)
        .post('/tasks')
        .set('Authorization', `Bearer ${authToken}`)
        .send({
          title: '',
        })

      expect(response.status).toBe(400)
    })
  })

  describe('GET /tasks', () => {
    it('should return user tasks', async () => {
      // 先建立幾個任務
      await request(app)
        .post('/tasks')
        .set('Authorization', `Bearer ${authToken}`)
        .send({ title: 'Task 1' })

      await request(app)
        .post('/tasks')
        .set('Authorization', `Bearer ${authToken}`)
        .send({ title: 'Task 2' })

      const response = await request(app)
        .get('/tasks')
        .set('Authorization', `Bearer ${authToken}`)

      expect(response.status).toBe(200)
      expect(response.body.length).toBeGreaterThanOrEqual(2)
    })
  })
})
```

#### 3. Frontend Component Tests（前端元件測試）

**TaskList.test.tsx**：
```typescript
import { render, screen, fireEvent } from '@testing-library/react'
import { TaskList } from './TaskList'
import { Task } from '../../types/task.types'

describe('TaskList', () => {
  const mockTasks: Task[] = [
    {
      id: '1',
      title: 'Task 1',
      completed: false,
      userId: 'user-1',
      createdAt: new Date(),
      updatedAt: new Date(),
    },
    {
      id: '2',
      title: 'Task 2',
      completed: true,
      userId: 'user-1',
      createdAt: new Date(),
      updatedAt: new Date(),
    },
  ]

  const mockHandlers = {
    onToggleTask: jest.fn(),
    onDeleteTask: jest.fn(),
  }

  it('should render all tasks', () => {
    render(<TaskList tasks={mockTasks} {...mockHandlers} />)

    expect(screen.getByText('Task 1')).toBeInTheDocument()
    expect(screen.getByText('Task 2')).toBeInTheDocument()
  })

  it('should show empty message when no tasks', () => {
    render(<TaskList tasks={[]} {...mockHandlers} />)

    expect(screen.getByText('沒有任務')).toBeInTheDocument()
  })

  it('should call onToggleTask when task is clicked', () => {
    render(<TaskList tasks={mockTasks} {...mockHandlers} />)

    const checkbox = screen.getAllByRole('checkbox')[0]
    fireEvent.click(checkbox)

    expect(mockHandlers.onToggleTask).toHaveBeenCalledWith('1')
  })

  it('should call onDeleteTask when delete button is clicked', () => {
    render(<TaskList tasks={mockTasks} {...mockHandlers} />)

    const deleteButtons = screen.getAllByRole('button', { name: /delete/i })
    fireEvent.click(deleteButtons[0])

    expect(mockHandlers.onDeleteTask).toHaveBeenCalledWith('1')
  })
})
```

### 智慧上下文載入

**測試單一模組時載入**（~400 行）：
```
✅ 測試配置（jest.config.js）
✅ 測試工具（test-helpers.ts）
✅ 當前測試檔案
✅ 被測模組（task.service.ts）
✅ 被測模組介面（task.types.ts）
```

**不載入**：
```
❌ 其他模組實作
❌ 前端檔案（測試後端時）
❌ 後端檔案（測試前端時）
```

### 階段完成標準（MVP）

```
✅ 核心業務邏輯有單元測試
✅ 主要 API 端點有整合測試
✅ 關鍵元件有基本測試
✅ 測試涵蓋率 > 60%（核心功能 > 80%）
✅ 所有測試通過
```

## Phase 5: Deployment

### 目標

將應用程式部署到可存取的環境，準備交付使用者。

### 核心原則

1. **容器化** - 使用 Docker 確保環境一致
2. **自動化** - CI/CD 自動部署
3. **文件化** - 清楚的部署和使用說明
4. **MVP 部署** - 簡單可靠優於複雜完美

### 主要任務

#### 1. Docker Setup（容器化）

**Dockerfile**：
```dockerfile
# 後端 Dockerfile
FROM node:18-alpine

WORKDIR /app

COPY package*.json ./
RUN npm ci --only=production

COPY dist ./dist

EXPOSE 3000

CMD ["node", "dist/index.js"]
```

**docker-compose.yml**：
```yaml
version: '3.8'

services:
  database:
    image: postgres:15-alpine
    environment:
      POSTGRES_DB: taskapp
      POSTGRES_USER: user
      POSTGRES_PASSWORD: password
    volumes:
      - db-data:/var/lib/postgresql/data
    ports:
      - "5432:5432"

  backend:
    build: ./backend
    environment:
      DATABASE_URL: postgresql://user:password@database:5432/taskapp
      JWT_SECRET: your-secret-key
    ports:
      - "3000:3000"
    depends_on:
      - database

  frontend:
    build: ./frontend
    ports:
      - "8080:80"
    depends_on:
      - backend

volumes:
  db-data:
```

#### 2. CI/CD Pipeline（持續整合/部署）

**.github/workflows/deploy.yml**（MVP 簡化版）：
```yaml
name: Deploy

on:
  push:
    branches: [main]

jobs:
  test:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - uses: actions/setup-node@v3
        with:
          node-version: '18'
      - run: npm ci
      - run: npm test

  deploy:
    needs: test
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - name: Deploy to production
        run: |
          # MVP: 簡單的部署腳本
          echo "Deploying to production..."
          # 實際部署指令
```

#### 3. Documentation（文件）

**README.md**（專案根目錄）：
```markdown
# Task Management App

簡單的任務管理應用。

## 功能

- ✅ 建立任務
- ✅ 標記完成
- ✅ 任務清單

## 技術堆疊

- Backend: Node.js + Express + PostgreSQL
- Frontend: React
- Deployment: Docker

## 快速開始

### 使用 Docker（推薦）

\`\`\`bash
# 啟動所有服務
docker-compose up

# 存取應用
# Frontend: http://localhost:8080
# Backend API: http://localhost:3000
\`\`\`

### 本地開發

\`\`\`bash
# 後端
cd backend
npm install
npm run dev

# 前端
cd frontend
npm install
npm start
\`\`\`

## API 文件

參見 [docs/api.md](docs/api.md)

## 授權

MIT
```

### 智慧上下文載入

**部署設定時載入**（~300 行）：
```
✅ Dockerfile, docker-compose.yml
✅ CI/CD 配置
✅ 環境變數範本
✅ 部署文件
✅ 模組依賴圖（了解服務依賴）
```

**不載入**：
```
❌ 實作細節
❌ 測試檔案
```

### 階段完成標準（MVP）

```
✅ Docker 容器可正常啟動
✅ 所有服務可互相通訊
✅ 基本的 CI/CD 設定
✅ README 文件完整
✅ 應用可正常運作
```

## 階段轉換

### 轉換流程

```
1. 確認當前階段所有任務完成
2. 進行模組化健康度檢查
3. 建立階段轉換 commit
4. 更新 TODO.md
5. 卸載前階段實作檔案
6. 保留前階段介面定義
7. 載入新階段相關檔案
8. 顯示新階段概覽
```

### Token 節省統計

階段轉換時的 token 優化：

| 轉換 | 卸載 | 保留 | 載入 | 節省 |
|------|------|------|------|------|
| Planning → Backend | 200 行 | 150 行 | 300 行 | -30% |
| Backend → Frontend | 1800 行 | 300 行 | 250 行 | 83% |
| Frontend → Testing | 800 行 | 200 行 | 400 行| 33% |
| Testing → Deployment | 1200 行 | 150 行 | 300 行| 63% |

### 最佳實踐

```
✅ 每個階段結束時進行品質檢查
✅ 確保文件與程式碼同步
✅ 使用 conventional commits
✅ 階段轉換時 commit 分離
✅ 定期執行 'go optimize' 檢查
```

## 總結

5 階段開發流程透過結構化的方式和智慧上下文載入，實現：

- **高效開發**：清楚的階段目標和任務
- **Token 優化**：平均節省 75-85% token
- **品質保證**：每個階段都有完成標準
- **MVP 導向**：專注於核心功能，快速交付

更多詳情參考：
- [Getting Started Guide](getting-started.md)
- [Token Optimization Guide](token-optimization.md)
- [Modularity Guide](modularity-guide.md)
