# Token 優化策略

## 概述

本文件詳細說明如何透過智慧上下文載入和模組化設計，將 AI token 使用量減少 70-85%。

## Token 使用現況分析

### 傳統開發方式的 Token 浪費

**問題情境**：實作一個新的支付功能

```
載入內容：
✅ 整個專案的所有檔案
✅ 所有模組的實作細節
✅ 完整的文件
✅ 所有測試檔案

總計：~5,000-10,000 行程式碼
```

**問題**：
- 90% 的載入內容與當前任務無關
- 浪費大量 token 在無關的程式碼上
- AI 需要處理過多的干擾資訊
- 降低回應品質和速度

### 優化後的方式

**同樣情境**：實作一個新的支付功能

```
載入內容：
✅ 支付模組介面定義（payment.types.ts）     50 行
✅ 相關依賴模組的介面（auth.types.ts）      45 行
✅ API 規格（僅支付部分）                    80 行
✅ 架構文件（僅支付模組部分）                40 行

總計：~215 行

Token 節省：(5000 - 215) / 5000 = 95.7%
```

## 六大 Token 優化策略

### 策略 1：Interface First（介面優先）

#### 原理

在實作前先定義清楚的介面，讓 AI 僅需讀取介面即可理解模組。

#### 實作方式

**Step 1：建立介面定義檔**

```typescript
// payment.types.ts（僅 50 行）

export interface PaymentService {
  processPayment(request: PaymentRequest): Promise<PaymentResult>
  refundPayment(paymentId: string): Promise<RefundResult>
  getPaymentStatus(paymentId: string): Promise<PaymentStatus>
}

export interface PaymentRequest {
  amount: number
  currency: string
  method: 'card' | 'paypal' | 'crypto'
  metadata?: Record<string, unknown>
}

// ... 其他型別定義
```

**Step 2：其他模組僅依賴介面**

```typescript
// order.service.ts
import { PaymentService } from '../payment/payment.types'  // 僅載入 50 行

class OrderService {
  constructor(private paymentService: PaymentService) {}
  // 使用介面，不需要知道實作細節
}
```

#### Token 節省計算

| 情境 | 傳統方式 | Interface First | 節省 |
|------|---------|----------------|------|
| 理解模組 API | 載入實作（500 行）| 載入介面（50 行）| 90% |
| 使用模組 | 載入實作（500 行）| 載入介面（50 行）| 90% |
| 規劃整合 | 載入所有實作（2000 行）| 載入所有介面（200 行）| 90% |

**平均節省：90%**

### 策略 2：選擇性載入（Selective Loading）

#### 原理

根據當前開發階段和任務，僅載入相關的檔案。

#### 實作方式

**按階段載入**

```
Phase 1: Planning（規劃階段）
載入：
  ✅ docs/requirements.md（摘要）          50 行
  ✅ docs/architecture.md                 150 行
  ✅ docs/module-boundaries.md            60 行
  ❌ src/* （所有實作）                   0 行
總計：260 行

vs 傳統：載入整個專案（5000 行）
節省：94.8%
```

```
Phase 2: Backend Development（後端開發）
當前任務：實作 auth 模組

載入：
  ✅ docs/architecture.md（auth 部分）    40 行
  ✅ src/auth/auth.types.ts               45 行
  ✅ src/auth/auth.service.ts             180 行
  ✅ src/core/logger.types.ts（依賴）     30 行
  ❌ src/payment/*（無關模組）            0 行
  ❌ src/user/*（無關模組）               0 行
  ❌ src/*/README.md（無關文件）          0 行
總計：295 行

vs 傳統：載入所有後端模組（3000 行）
節省：90.2%
```

#### 階段載入對照表

| 階段 | 傳統載入 | 選擇性載入 | 節省 |
|------|---------|-----------|------|
| Planning | 5000 行 | 260 行 | 94.8% |
| Backend Dev | 3000 行 | 295 行 | 90.2% |
| Frontend Dev | 4000 行 | 320 行 | 92.0% |
| Testing | 2000 行 | 400 行 | 80.0% |
| Deployment | 1500 行 | 280 行 | 81.3% |

**平均節省：87.7%**

### 策略 3：模組化檔案（Modular Files）

#### 原理

將大檔案拆分成小檔案，每個檔案 < 300 行，讓 AI 可以僅載入需要的部分。

#### 對比範例

**Before（未模組化）**

```
src/
  services.ts              2100 行
  controllers.ts           1500 行
  models.ts                800 行

問題：
- 需要任何功能都要載入 2100 行
- 無法局部載入
- AI 被大量無關程式碼干擾
```

**After（模組化）**

```
src/features/
  auth/
    auth.types.ts          45 行
    auth.service.ts        180 行
    auth.controller.ts     95 行

  payment/
    payment.types.ts       60 行
    payment.service.ts     200 行
    payment.controller.ts  120 行

  user/
    user.types.ts          50 行
    user.service.ts        160 行
    user.controller.ts     110 行

優勢：
- 實作 auth 時僅載入 auth/* （320 行）
- 使用 payment 時僅載入 payment.types.ts（60 行）
- Token 節省：85-95%
```

#### 檔案大小建議

| 檔案類型 | 目標行數 | 最大行數 | Token 估計 |
|---------|---------|---------|-----------|
| .types.ts | 30-80 | 100 | ~100-300 |
| .service.ts | 100-200 | 250 | ~300-800 |
| .controller.ts | 50-120 | 150 | ~150-450 |
| .test.ts | 100-250 | 300 | ~300-900 |
| README.md | 30-80 | 100 | ~100-300 |

### 策略 4：文件模組化（Documentation Modularity）

#### 原理

將大型文件拆分成多個小檔案，用連結組織，而非單一大檔案。

#### 對比範例

**Before（單一大檔案）**

```
docs/
  project-documentation.md        2500 行

問題：
- 需要任何資訊都要載入 2500 行
- 難以維護
- 更新一小部分也影響整個檔案
```

**After（模組化文件）**

```
docs/
  README.md                       80 行（概覽 + 連結）
  architecture/
    overview.md                   120 行
    data-flow.md                  80 行
    module-map.md                 60 行
  modules/
    auth.md                       100 行
    payment.md                    120 行
    user.md                       90 行
  api/
    auth-endpoints.md             130 行
    payment-endpoints.md          150 行
    user-endpoints.md             110 行

優勢：
- 規劃時僅讀 README + architecture/overview（200 行）
- 實作 auth 時僅讀 modules/auth + api/auth-endpoints（230 行）
- Token 節省：80-92%
```

#### 文件組織建議

```
docs/
  README.md                    # 專案概覽 + 導航
  architecture/                # 架構文件
    overview.md                # 系統架構概覽
    module-boundaries.md       # 模組邊界定義
    data-flow.md               # 資料流程
    tech-stack.md              # 技術堆疊
  modules/                     # 各模組詳細說明
    [module-name].md           # 每個模組一個檔案
  api/                         # API 文件
    [feature]-endpoints.md     # 按功能分組
  guides/                      # 操作指南
    getting-started.md
    deployment.md
    testing.md
```

### 策略 5：README 驅動說明（README-Driven Documentation）

#### 原理

每個模組都有簡短的 README（30-80 行），讓 AI 快速了解用途，無需讀程式碼。

#### README 範本（精簡版）

```markdown
# [Module Name]

## Purpose
[1 句話說明用途]

## Public Interface
參見 `[module].types.ts`

## Quick Example
\`\`\`typescript
const result = await service.method({ ... })
\`\`\`

## Dependencies
- External: package-name
- Internal: core/logger, core/config

## Testing
\`\`\`bash
npm test -- [module].test
\`\`\`
```

#### Token 節省計算

| 情境 | 無 README | 有 README | 節省 |
|------|----------|----------|------|
| 了解模組用途 | 讀實作（200 行）| 讀 README（40 行）| 80% |
| 了解如何使用 | 讀實作 + 測試（400 行）| 讀 README（40 行）| 90% |
| 了解依賴關係 | 搜尋所有 import（1000 行）| 讀 README（40 行）| 96% |

**平均節省：88.7%**

### 策略 6：Internals 隔離（Internals Isolation）

#### 原理

將內部實作細節放在 `internals/` 目錄，其他模組不直接存取，減少載入範圍。

#### 範例

```
src/features/auth/
  auth.types.ts              45 行（公開介面）
  auth.service.ts            180 行（公開 API）
  auth.controller.ts         95 行（公開 API）
  README.md                  40 行（公開文件）
  internals/                 （內部細節，外部不載入）
    jwt.helper.ts            80 行
    password.helper.ts       60 行
    session.store.ts         70 行
```

#### Token 節省

**情境：其他模組需要使用 auth**

```
傳統方式（無隔離）：
載入所有 auth 檔案：45 + 180 + 95 + 80 + 60 + 70 = 530 行

Internals 隔離：
僅載入公開介面：45 行（auth.types.ts）

節省：(530 - 45) / 530 = 91.5%
```

## 綜合優化案例

### 案例：實作新的 Order 模組

#### Before（未優化）

```
AI 需要載入的內容：

1. 理解系統架構：
   - docs/project-documentation.md         2500 行

2. 了解相關模組：
   - src/payment/payment.service.ts        500 行
   - src/user/user.service.ts              400 行
   - src/product/product.service.ts        350 行
   - src/notification/notification.service.ts 300 行

3. 實作 order 模組：
   - src/order/order.service.ts            250 行（新建）

總計：2500 + 500 + 400 + 350 + 300 = 4050 行
```

#### After（全面優化）

```
AI 需要載入的內容：

1. 理解系統架構：
   - docs/README.md                        80 行
   - docs/architecture/overview.md         120 行
   - docs/modules/order.md                 90 行

2. 了解相關模組（僅介面）：
   - src/payment/payment.types.ts          60 行
   - src/user/user.types.ts                50 行
   - src/product/product.types.ts          55 行
   - src/notification/notification.types.ts 40 行

3. 實作 order 模組：
   - src/order/order.types.ts              70 行（先定義介面）
   - src/order/order.service.ts            250 行（實作）

總計：80 + 120 + 90 + 60 + 50 + 55 + 40 + 70 = 565 行
（實作時僅載入 types + service = 320 行）

Token 節省：
- 規劃階段：(4050 - 290) / 4050 = 92.8%
- 實作階段：(4050 - 320) / 4050 = 92.1%
- 平均節省：92.5%
```

## 階段性載入策略詳解

### Phase 1: Planning（規劃階段）

**目標**：理解需求、設計架構、定義模組邊界

**載入內容**（總計 ~200-300 行）：

1. **需求文件**（僅摘要）
   - `docs/requirements.md`（前 100 行或摘要章節）

2. **架構文件**
   - `docs/architecture/overview.md`（120 行）
   - `docs/architecture/module-boundaries.md`（60 行）

3. **技術選型**
   - `docs/tech-stack.md`（80 行）

**不載入**：
- ❌ 任何實作程式碼
- ❌ 詳細的 API 規格
- ❌ 測試檔案

**Token 節省**：vs 載入整個專案（5000 行）= 94%

### Phase 2: Backend Development（後端開發）

**目標**：實作後端模組，按模組逐一完成

**當前任務**：實作 Authentication 模組

**載入內容**（總計 ~250-350 行）：

1. **架構參考**
   - `docs/architecture/overview.md`（僅 auth 相關部分，~40 行）
   - `docs/modules/auth.md`（80 行）

2. **當前模組**
   - `src/auth/auth.types.ts`（45 行，先定義）
   - `src/auth/auth.service.ts`（180 行，實作）
   - `docs/api/auth-endpoints.md`（80 行）

3. **依賴模組（僅介面）**
   - `src/core/logger.types.ts`（30 行）
   - `src/core/config.types.ts`（25 行）

**不載入**：
- ❌ 其他 features（payment, user, product 等）
- ❌ 前端程式碼
- ❌ 測試（實作完成後再載入）

**Token 節省**：vs 載入所有後端（3000 行）= 88%

### Phase 3: Frontend Development（前端開發）

**目標**：實作前端組件和功能

**當前任務**：實作 Login 組件

**載入內容**（總計 ~280-380 行）：

1. **設計規格**
   - `docs/component-specs/login.md`（60 行）
   - `src/styles/theme.ts`（80 行，design tokens）

2. **當前組件**
   - `src/features/auth/components/LoginForm/LoginForm.types.ts`（25 行）
   - `src/features/auth/components/LoginForm/LoginForm.tsx`（120 行）

3. **相關 hooks 和 store**
   - `src/features/auth/hooks/useAuth.types.ts`（20 行）
   - `src/features/auth/store/auth.types.ts`（30 行）

4. **API 介面**
   - `docs/api/auth-endpoints.md`（僅 login 部分，~40 行）

**不載入**：
- ❌ 後端實作（僅需 API 規格）
- ❌ 其他組件
- ❌ 其他功能的 store

**Token 節省**：vs 載入所有前端（4000 行）= 92%

### Phase 4: Testing（測試階段）

**目標**：為各模組撰寫測試

**當前任務**：測試 auth 模組

**載入內容**（總計 ~350-450 行）：

1. **測試配置**
   - `jest.config.js` 或 `vitest.config.ts`（50 行）
   - `tests/helpers/setup.ts`（40 行）

2. **被測試模組**
   - `src/auth/auth.types.ts`（45 行）
   - `src/auth/auth.service.ts`（180 行）
   - `src/auth/auth.test.ts`（200 行）

3. **Mock 定義**
   - `src/auth/__mocks__/dependencies.ts`（30 行）

**不載入**：
- ❌ 依賴模組的實作（使用 mock）
- ❌ 其他模組的測試
- ❌ E2E 測試（另外處理）

**Token 節省**：vs 載入所有測試 + 實作（2000 行）= 78%

### Phase 5: Deployment（部署階段）

**目標**：設定 CI/CD、部署環境

**載入內容**（總計 ~250-350 行）：

1. **部署配置**
   - `Dockerfile`（60 行）
   - `docker-compose.yml`（80 行）
   - `.github/workflows/deploy.yml`（100 行）

2. **環境設定**
   - `.env.example`（40 行）
   - `docs/deployment.md`（90 行）

3. **依賴檢查**
   - `docs/architecture/module-dependency-map.md`（60 行）

**不載入**：
- ❌ 實作程式碼
- ❌ 測試檔案
- ❌ 開發工具配置

**Token 節省**：vs 載入整個專案（5000 行）= 93%

## Token 使用追蹤

### 建議的 Token 使用指標

| 階段 | 目標 Token 使用 | 警告線 | 說明 |
|------|---------------|-------|-----|
| Planning | < 1000 tokens | > 1500 | 應僅載入架構和需求摘要 |
| Backend Dev（per module）| < 1200 tokens | > 2000 | 僅載入當前模組 + 依賴介面 |
| Frontend Dev（per component）| < 1000 tokens | > 1800 | 僅載入當前組件 + API 介面 |
| Testing（per module）| < 1500 tokens | > 2500 | 僅載入被測模組 + mock |
| Deployment | < 1200 tokens | > 2000 | 僅載入部署配置 |

### 如何追蹤

在 TODO.md 中記錄（可選）：

```markdown
## Token Usage Log

- 2025-01-08 15:30: Phase 2 - Auth module implementation
  - Loaded: auth.types.ts (45), auth.service.ts (180), logger.types.ts (30)
  - Total: 255 lines (~800 tokens)
  - Status: ✅ Within target

- 2025-01-08 14:00: Phase 1 - Planning
  - Loaded: architecture docs (200), requirements summary (80)
  - Total: 280 lines (~900 tokens)
  - Status: ✅ Within target
```

## 最佳化檢查清單

在每個階段開始前，檢查：

### ✅ 載入前檢查
- [ ] 確認當前階段和任務
- [ ] 列出真正需要的檔案（< 10 個）
- [ ] 檢查是否可以僅載入介面而非實作
- [ ] 檢查是否可以僅載入文件的部分章節

### ✅ 載入中檢查
- [ ] 載入的檔案都與當前任務直接相關
- [ ] 沒有載入整個目錄
- [ ] 依賴模組僅載入介面定義
- [ ] 文件僅載入相關章節

### ✅ 載入後檢查
- [ ] 總行數 < 500 行（理想）或 < 1000 行（可接受）
- [ ] 沒有載入測試檔案（除非在測試階段）
- [ ] 沒有載入無關的功能模組
- [ ] 記錄載入的內容（可選，用於優化）

## 工具和自動化

### 使用 `go optimize` 指令

定期檢查專案的模組化健康度：

```bash
# 在 Claude Code 中執行
go optimize
```

輸出範例：

```
📊 模組化健康度報告

Token 效率分析：
  ✅ 平均每任務載入：280 行（~900 tokens）
  📊 vs 未優化：1500 行（~4800 tokens）
  🎉 節省：81.3%

檔案大小分析：
  ✅ 95% 檔案 < 300 行
  ⚠️  2 個檔案 > 500 行
     - src/legacy/old-service.ts（820 行）
     - src/utils/helpers.ts（650 行）

建議優化：
  1. 拆分 old-service.ts 為 3 個模組
     預估節省：~400 tokens/使用
  2. 拆分 helpers.ts 按功能分類
     預估節省：~250 tokens/使用

總預估節省：~650 tokens/使用（額外 15%）
```

## 總結

### Token 優化公式

```
總 Token 節省 =
  Interface First 節省（90%）×
  選擇性載入節省（87%）×
  模組化檔案節省（85%）×
  文件模組化節省（80%）×
  README 驅動節省（89%）×
  Internals 隔離節省（92%）

實際綜合節省：70-85%（考慮重疊）
```

### 關鍵要點

1. **永遠 Interface First** - 先定義介面，節省 90% token
2. **按階段載入** - 僅載入當前階段相關內容，節省 87% token
3. **保持檔案小** - 每個檔案 < 300 行，可選擇性載入
4. **文件模組化** - 拆分大文件為小檔案
5. **每個模組有 README** - 快速了解無需讀程式碼
6. **隔離內部實作** - 減少載入範圍

### 實施順序建議

1. ✅ **立即實施**：Interface First（最高投資報酬率）
2. ✅ **第一週**：模組化現有大檔案
3. ✅ **第二週**：為每個模組新增 README
4. ✅ **第三週**：拆分文件為模組化結構
5. ✅ **持續**：使用 `go optimize` 監控和改進
