#!/usr/bin/env python3
"""
分析專案的模組化健康度
"""

import os
import sys
import re
from pathlib import Path
from collections import defaultdict

def count_lines(file_path):
    """計算檔案行數（排除空行和註解）"""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            lines = f.readlines()
            return len([l for l in lines if l.strip() and not l.strip().startswith(('#', '//'))])
    except:
        return 0

def find_source_files(directory, extensions):
    """尋找所有原始碼檔案"""
    files = []
    for ext in extensions:
        files.extend(Path(directory).rglob(f'*.{ext}'))
    return files

def analyze_file_sizes(files):
    """分析檔案大小分布"""
    sizes = []
    large_files = []

    for f in files:
        lines = count_lines(f)
        sizes.append(lines)
        if lines > 300:
            large_files.append((f, lines))

    if not sizes:
        return None, []

    avg = sum(sizes) / len(sizes)
    under_300 = len([s for s in sizes if s < 300])
    percentage = (under_300 / len(sizes)) * 100

    return {
        'total_files': len(sizes),
        'avg_lines': int(avg),
        'max_lines': max(sizes),
        'min_lines': min(sizes),
        'under_300_pct': percentage,
    }, large_files

def find_interface_files(directory):
    """尋找介面定義檔案"""
    patterns = ['*.types.ts', '*.types.js', '*.interface.ts', '*.d.ts']
    interface_files = []

    for pattern in patterns:
        interface_files.extend(Path(directory).rglob(pattern))

    return interface_files

def analyze_interface_coverage(src_dir):
    """分析介面覆蓋率"""
    # 尋找所有模組目錄（包含多個檔案的目錄）
    modules = defaultdict(list)

    for root, dirs, files in os.walk(src_dir):
        source_files = [f for f in files if f.endswith(('.ts', '.js', '.py'))]
        if len(source_files) > 0:
            modules[root] = source_files

    # 檢查每個模組是否有介面定義
    modules_with_interface = 0
    modules_without_interface = []

    for module_path, files in modules.items():
        has_interface = any(
            'types' in f or 'interface' in f or f.endswith('.d.ts')
            for f in files
        )

        if has_interface:
            modules_with_interface += 1
        else:
            # 排除只有一個檔案的目錄
            if len(files) > 1:
                modules_without_interface.append(module_path)

    total_modules = len([m for m, files in modules.items() if len(files) > 1])

    return {
        'total_modules': total_modules,
        'with_interface': modules_with_interface,
        'coverage_pct': (modules_with_interface / total_modules * 100) if total_modules > 0 else 0,
    }, modules_without_interface

def main():
    print("📊 模組化健康度分析")
    print("=" * 60)

    # 檢查是否在專案根目錄
    if not os.path.exists('package.json') and not os.path.exists('pyproject.toml'):
        print("⚠️  警告：未偵測到 package.json 或 pyproject.toml")
        print("   請在專案根目錄執行此腳本")

    # 偵測原始碼目錄
    src_dirs = []
    for candidate in ['src', 'lib', 'app', '.']:
        if os.path.exists(candidate):
            src_dirs.append(candidate)
            break

    if not src_dirs:
        print("❌ 找不到原始碼目錄")
        sys.exit(1)

    src_dir = src_dirs[0]
    print(f"📁 分析目錄：{src_dir}")
    print()

    # 偵測程式語言
    extensions = ['ts', 'js', 'py', 'java', 'go']
    files = find_source_files(src_dir, extensions)

    if not files:
        print("❌ 找不到原始碼檔案")
        sys.exit(1)

    print(f"📄 找到 {len(files)} 個原始碼檔案")
    print()

    # 1. 檔案大小分析
    print("## 檔案大小分布")
    size_stats, large_files = analyze_file_sizes(files)

    if size_stats:
        status = "✅ 優秀" if size_stats['under_300_pct'] > 80 else "⚠️  需注意"
        print(f"  {status}")
        print(f"  - {size_stats['under_300_pct']:.1f}% 檔案 < 300 行")
        print(f"  - 平均檔案大小：{size_stats['avg_lines']} 行")
        print(f"  - 最大檔案：{size_stats['max_lines']} 行")

        if large_files:
            print()
            print(f"  ⚠️  {len(large_files)} 個檔案過大（> 300 行）：")
            for f, lines in sorted(large_files, key=lambda x: x[1], reverse=True)[:5]:
                print(f"     - {f}: {lines} 行（建議拆分）")
    print()

    # 2. 介面覆蓋率分析
    print("## 介面覆蓋率")
    interface_stats, missing_interfaces = analyze_interface_coverage(src_dir)

    if interface_stats['total_modules'] > 0:
        coverage = interface_stats['coverage_pct']
        status = "✅ 優秀" if coverage >= 80 else "⚠️  需改進"
        print(f"  {status}")
        print(f"  - {interface_stats['with_interface']}/{interface_stats['total_modules']} 模組有介面定義（{coverage:.1f}%）")

        if missing_interfaces:
            print()
            print(f"  ❌ 缺少介面定義的模組：")
            for module in missing_interfaces[:5]:
                print(f"     - {module}")
    else:
        print("  ℹ️  未偵測到模組結構")
    print()

    # 3. Token 效率預估
    print("## Token 效率預估")

    if size_stats:
        # 計算介面檔案的平均大小
        interface_files = find_interface_files(src_dir)
        interface_sizes = [count_lines(f) for f in interface_files]
        avg_interface_size = sum(interface_sizes) / len(interface_sizes) if interface_sizes else 50

        # 預估 Token 使用
        current_avg = size_stats['avg_lines'] * 1.5  # 假設需要載入 1.5 個檔案
        optimized_avg = avg_interface_size + size_stats['avg_lines'] * 0.5  # 介面 + 半個實作檔案
        improvement = (1 - optimized_avg / current_avg) * 100 if current_avg > 0 else 0

        print(f"  當前平均：~{int(current_avg)} 行/任務")
        print(f"  優化後預估：~{int(optimized_avg)} 行/任務")
        print(f"  提升：{improvement:.1f}%")
    print()

    # 4. 建議
    print("## 建議")

    recommendations = []

    if large_files:
        recommendations.append(f"1. 🔴 高優先：拆分 {len(large_files)} 個大檔案")

    if missing_interfaces:
        recommendations.append(f"2. 🔴 高優先：為 {len(missing_interfaces)} 個模組建立介面定義")

    if size_stats and size_stats['under_300_pct'] < 80:
        recommendations.append("3. 🟡 中優先：提升小檔案比例到 80% 以上")

    if not recommendations:
        print("  ✅ 專案模組化健康度良好！")
    else:
        for rec in recommendations:
            print(f"  {rec}")

    print()
    print("=" * 60)
    print("✅ 分析完成")

if __name__ == '__main__':
    main()
