#!/bin/bash
# 檢查 TODO.md 格式是否符合規範

set -e

TODO_FILE="TODO.md"

echo "🔍 檢查 TODO.md 格式..."

# 檢查檔案是否存在
if [ ! -f "$TODO_FILE" ]; then
    echo "❌ 錯誤：找不到 TODO.md"
    echo "   請在專案根目錄執行此腳本"
    exit 1
fi

# 檢查必要標記
echo "檢查必要標記..."

if ! grep -q "## Current Phase:" "$TODO_FILE"; then
    echo "❌ 缺少 'Current Phase' 標記"
    echo "   格式：## Current Phase: [phase-name]"
    exit 1
fi

if ! grep -q "### Phase" "$TODO_FILE"; then
    echo "❌ 找不到階段定義"
    echo "   格式：### Phase N: [Name]"
    exit 1
fi

# 檢查階段標記（✅ 或 🔄）
PHASES_WITH_MARKS=$(grep "### Phase" "$TODO_FILE" | grep -E "(✅|🔄)" | wc -l || echo "0")

if [ "$PHASES_WITH_MARKS" -eq 0 ]; then
    echo "⚠️  警告：沒有任何階段標記為完成或進行中"
    echo "   使用 ✅ 標記已完成階段"
    echo "   使用 🔄 標記進行中階段"
fi

# 檢查任務格式
TASKS=$(grep -E "^- \[[ x]\]" "$TODO_FILE" | wc -l || echo "0")

if [ "$TASKS" -eq 0 ]; then
    echo "⚠️  警告：找不到任務清單"
    echo "   格式：- [ ] Task name 或 - [x] Completed task"
fi

# 統計資訊
TOTAL_PHASES=$(grep "### Phase" "$TODO_FILE" | wc -l || echo "0")
COMPLETED_PHASES=$(grep "### Phase" "$TODO_FILE" | grep "✅" | wc -l || echo "0")
IN_PROGRESS_PHASES=$(grep "### Phase" "$TODO_FILE" | grep "🔄" | wc -l || echo "0")
COMPLETED_TASKS=$(grep -E "^- \[x\]" "$TODO_FILE" | wc -l || echo "0")
PENDING_TASKS=$(grep -E "^- \[ \]" "$TODO_FILE" | wc -l || echo "0")

echo ""
echo "📊 統計資訊："
echo "   階段總數：$TOTAL_PHASES"
echo "   已完成：$COMPLETED_PHASES ✅"
echo "   進行中：$IN_PROGRESS_PHASES 🔄"
echo "   任務完成：$COMPLETED_TASKS"
echo "   任務待辦：$PENDING_TASKS"
echo "   完成率：$((COMPLETED_TASKS * 100 / (COMPLETED_TASKS + PENDING_TASKS)))%"

echo ""
echo "✅ TODO.md 格式正確"
exit 0
