#!/bin/bash
# 分析 git diff 並產生 conventional commit 訊息建議

set -e

echo "📝 產生 Conventional Commit 訊息..."
echo ""

# 檢查是否在 git repository
if ! git rev-parse --git-dir > /dev/null 2>&1; then
    echo "❌ 錯誤：不在 Git 倉庫中"
    exit 1
fi

# 檢查是否有變更
if git diff --cached --quiet && git diff --quiet; then
    echo "⚠️  沒有變更需要提交"
    echo "   使用 'git add' 暫存變更"
    exit 0
fi

# 分析變更的檔案
echo "變更的檔案："
CHANGED_FILES=$(git diff --cached --name-only 2>/dev/null || git diff --name-only)

if [ -z "$CHANGED_FILES" ]; then
    echo "   （沒有暫存的變更，顯示未暫存的變更）"
    CHANGED_FILES=$(git diff --name-only)
fi

echo "$CHANGED_FILES" | head -10
TOTAL_FILES=$(echo "$CHANGED_FILES" | wc -l | tr -d ' ')

if [ "$TOTAL_FILES" -gt 10 ]; then
    echo "   ... 和其他 $((TOTAL_FILES - 10)) 個檔案"
fi

echo ""

# 推測 commit type
TYPE="feat"
SCOPE=""

# 檢查是否為測試檔案
if echo "$CHANGED_FILES" | grep -q "test\|spec"; then
    TYPE="test"
fi

# 檢查是否為文件
if echo "$CHANGED_FILES" | grep -qi "README\|\.md$\|docs/"; then
    TYPE="docs"
fi

# 檢查是否為配置檔案
if echo "$CHANGED_FILES" | grep -qi "package.json\|tsconfig\|\.config\|Dockerfile\|docker-compose"; then
    TYPE="chore"
fi

# 推測 scope（從第一個變更檔案的目錄）
FIRST_FILE=$(echo "$CHANGED_FILES" | head -1)
if [[ "$FIRST_FILE" == *"/"* ]]; then
    SCOPE=$(echo "$FIRST_FILE" | cut -d'/' -f1-2 | rev | cut -d'/' -f1 | rev)
fi

# 顯示統計
INSERTIONS=$(git diff --cached --numstat 2>/dev/null || git diff --numstat | awk '{sum += $1} END {print sum}')
DELETIONS=$(git diff --cached --numstat 2>/dev/null || git diff --numstat | awk '{sum += $2} END {print sum}')

echo "變更統計："
echo "   檔案：$TOTAL_FILES 個"
echo "   新增：${INSERTIONS:-0} 行"
echo "   刪除：${DELETIONS:-0} 行"
echo ""

# 產生建議的 commit 訊息
echo "建議的 Commit 訊息："
echo "───────────────────────────────────"

if [ -n "$SCOPE" ]; then
    echo "$TYPE($SCOPE): <簡短描述變更內容>"
else
    echo "$TYPE: <簡短描述變更內容>"
fi

echo ""
echo "<詳細說明（可選）>"
echo "- 變更 1"
echo "- 變更 2"
echo ""
echo "Closes #<issue-number>"
echo "───────────────────────────────────"
echo ""

# Type 說明
echo "Commit Types:"
echo "  feat:     新功能"
echo "  fix:      錯誤修復"
echo "  docs:     僅文件變更"
echo "  style:    程式碼格式（不影響功能）"
echo "  refactor: 重構"
echo "  test:     測試"
echo "  chore:    建置過程或輔助工具"
echo ""

echo "💡 提示：編輯後執行 'git commit'"
echo "   或使用：git commit -m \"$TYPE: <訊息>\""
