# Advanced Pytest Patterns and Best Practices

## Overview

This document covers advanced pytest patterns and best practices that go beyond basic testing, including complex fixture arrangements, parametrization strategies, hook implementations, and performance optimization techniques.

## Advanced Fixture Patterns

### Factory Fixtures
```python
import pytest

@pytest.fixture
def user_factory():
    """Factory fixture to create users with different properties."""
    def _create_user(name="default", email=None, active=True):
        email = email or f"{name}@example.com"
        return User(name=name, email=email, active=active)
    return _create_user

def test_user_factory(user_factory):
    user = user_factory("john", "john@example.com", active=True)
    assert user.name == "john"
    assert user.active is True

def test_user_factory_defaults(user_factory):
    user = user_factory()
    assert user.name == "default"
    assert user.active is True
```

### Indirect Parametrization with Fixtures
```python
import pytest

@pytest.fixture
def user_data(request):
    """Parametrized fixture based on test parameters."""
    return {
        "admin": {"name": "admin", "role": "administrator", "permissions": ["read", "write", "delete"]},
        "editor": {"name": "editor", "role": "editor", "permissions": ["read", "write"]},
        "viewer": {"name": "viewer", "role": "viewer", "permissions": ["read"]}
    }[request.param]

@pytest.mark.parametrize("user_data", ["admin", "editor", "viewer"], indirect=True)
def test_user_permissions(user_data):
    user = User(**user_data)
    assert user.role == user_data["role"]
```

### Cross-Session Fixture Dependencies
```python
import pytest

@pytest.fixture(scope="session")
def database_url():
    """Provide database URL for the entire test session."""
    return "postgresql://localhost/test_db"

@pytest.fixture(scope="session")
def database_engine(database_url):
    """Create database engine once per session."""
    from sqlalchemy import create_engine
    engine = create_engine(database_url)
    yield engine
    engine.dispose()

@pytest.fixture(scope="session")
def database_schema(database_engine):
    """Create database schema once per session."""
    # Create all tables
    Base.metadata.create_all(database_engine)
    yield
    # Optionally drop tables after session
    Base.metadata.drop_all(database_engine)
```

## Complex Parametrization Strategies

### Parametrization with Complex Data Structures
```python
import pytest

test_cases = [
    pytest.param(
        {"username": "valid_user", "password": "ValidPass123!"},
        {"status": "success", "error": None},
        id="valid_credentials"
    ),
    pytest.param(
        {"username": "invalid", "password": "short"},
        {"status": "failure", "error": "Password too short"},
        id="invalid_password"
    ),
    pytest.param(
        {"username": "", "password": "ValidPass123!"},
        {"status": "failure", "error": "Username required"},
        id="empty_username"
    )
]

@pytest.mark.parametrize("credentials,expected", test_cases)
def test_login(credentials, expected):
    result = login(**credentials)
    assert result.status == expected["status"]
    if expected["error"]:
        assert expected["error"] in result.error_message
```

### Dynamic Parametrization from External Sources
```python
import pytest
import csv
import json

def load_test_data_from_csv(filename):
    """Load test data from CSV file."""
    test_data = []
    with open(filename, newline='') as csvfile:
        reader = csv.DictReader(csvfile)
        for row in reader:
            test_data.append((row['input'], row['expected']))
    return test_data

def pytest_generate_tests(metafunc):
    """Dynamically parametrize tests from external data."""
    if "csv_data" in metafunc.fixturenames:
        test_data = load_test_data_from_csv("test_data.csv")
        metafunc.parametrize("csv_data", test_data)
```

## Hook Implementations

### Custom Collection Hook
```python
# conftest.py
def pytest_collection_modifyitems(config, items):
    """Modify collected test items."""
    for item in items:
        # Add markers based on test names
        if "slow" in item.nodeid:
            item.add_marker(pytest.mark.slow)

        if "integration" in item.nodeid:
            item.add_marker(pytest.mark.integration)
```

### Result Reporting Hook
```python
# conftest.py
def pytest_runtest_logreport(report):
    """Log detailed test results."""
    if report.when == "call":
        if report.failed:
            print(f"FAILED: {report.nodeid}")
            print(f"Duration: {report.duration:.2f}s")
            print(f"Longrepr: {report.longrepr}")
```

### Configuration Hook
```python
# conftest.py
def pytest_configure(config):
    """Configure pytest settings."""
    config.addinivalue_line("markers", "critical: mark test as critical priority")
    config.addinivalue_line("markers", "flaky: mark test as flaky and allow retries")

def pytest_configure_node(node):
    """Configure worker node (for distributed testing)."""
    node.workerinput["env"] = "test"
```

## Performance Optimization Patterns

### Lazy Fixture Evaluation
```python
import pytest

@pytest.fixture(scope="session")
def expensive_resource():
    """Expensive resource that's only created if needed."""
    print("Creating expensive resource...")
    resource = ExpensiveResource()
    yield resource
    print("Cleaning up expensive resource...")
    resource.cleanup()

@pytest.mark.usefixtures("expensive_resource")
def test_that_uses_expensive_resource(expensive_resource):
    """This test will trigger the fixture creation."""
    assert expensive_resource.is_ready()
```

### Conditional Fixture Activation
```python
import pytest

@pytest.fixture
def db_connection(request):
    """Conditionally use real or mock DB based on marker."""
    if request.node.get_closest_marker('integration'):
        # Use real database for integration tests
        return RealDatabaseConnection()
    else:
        # Use mock for unit tests
        return MockDatabaseConnection()

@pytest.mark.integration
def test_real_db_integration(db_connection):
    """Integration test using real database."""
    assert db_connection.is_connected()

def test_mock_db_unit(db_connection):
    """Unit test using mock database."""
    assert db_connection.is_mocked()
```

## Test Organization and Structure

### Hierarchical Test Organization
```
tests/
├── conftest.py                 # Global fixtures
├── unit/
│   ├── conftest.py            # Unit test specific fixtures
│   ├── test_models/
│   │   ├── test_user.py
│   │   └── test_product.py
│   └── test_services/
│       ├── test_auth.py
│       └── test_payment.py
├── integration/
│   ├── conftest.py            # Integration test fixtures
│   ├── test_api/
│   │   ├── test_user_api.py
│   │   └── test_product_api.py
│   └── test_e2e/
│       └── test_checkout_flow.py
└── performance/
    ├── conftest.py            # Performance test fixtures
    └── test_load.py
```

### Parametrized Test Classes
```python
import pytest

@pytest.mark.parametrize("database_type", ["sqlite", "postgresql", "mysql"])
class TestDatabaseOperations:
    """Test database operations across different database types."""

    def test_create_table(self, database_type):
        db = create_database(database_type)
        table = db.create_table("users", {"id": int, "name": str})
        assert table.exists

    def test_insert_data(self, database_type):
        db = create_database(database_type)
        result = db.insert("users", {"name": "John Doe"})
        assert result.success
```

## Error Handling and Debugging

### Custom Exception Handling
```python
import pytest

class CustomTestError(Exception):
    """Custom exception for test-specific errors."""
    pass

@pytest.fixture
def error_handler():
    """Fixture to handle custom test errors."""
    def handle_error(error_type, message):
        if error_type == "custom":
            raise CustomTestError(message)
        elif error_type == "value":
            raise ValueError(message)
    return handle_error

def test_custom_error_handling(error_handler):
    """Test custom error handling."""
    with pytest.raises(CustomTestError):
        error_handler("custom", "This is a custom error")
```

### Conditional Test Skipping
```python
import pytest
import sys
import platform

def is_platform_supported():
    """Check if current platform is supported for test."""
    return platform.system() in ["Linux", "Darwin"]

@pytest.mark.skipif(not is_platform_supported(), reason="Platform not supported")
def test_platform_specific_feature():
    """Test that only runs on supported platforms."""
    assert True

@pytest.mark.skipif(sys.version_info < (3, 8), reason="Requires Python 3.8+")
def test_python_version_specific():
    """Test that requires newer Python version."""
    # Use features only available in Python 3.8+
    pass
```

## Plugin Integration Patterns

### Coverage Configuration
```python
# pytest.ini
[tool:pytest]
addopts =
    --cov=myproject
    --cov-report=html
    --cov-report=term-missing
    --cov-fail-under=80
```

### Parallel Test Execution
```python
# For pytest-xdist
def pytest_configure(config):
    if config.option.numprocesses:
        # Configure for parallel execution
        config.addinivalue_line("markers", "serial: mark test to run in serial (not parallel)")
```

### Retry Failed Tests
```python
# conftest.py
import pytest

def pytest_configure(config):
    config.addinivalue_line("markers", "retry: mark test to retry on failure")

@pytest.hookimpl(hookwrapper=True)
def pytest_runtest_makereport(item, call):
    """Hook to implement retry functionality."""
    outcome = yield
    rep = outcome.get_result()

    if rep.failed and item.get_closest_marker("retry"):
        # Retry the test
        for i in range(2):  # Retry twice
            rep = yield
            if not rep.failed:
                break
```

These advanced patterns provide solutions for complex testing scenarios, performance optimization, and sophisticated test organization that goes beyond basic pytest usage.