#!/usr/bin/env python3
"""
Pytest Helper Scripts

This module contains utility functions for common pytest operations:
- Test scaffold generation
- Fixture creation
- Configuration file generation
- Test runner utilities
"""

import argparse
import os
from pathlib import Path


def create_test_scaffold(module_name: str, test_functions: list = None):
    """
    Generate a basic test scaffold with common pytest patterns.

    Args:
        module_name: Name of the module being tested (without .py extension)
        test_functions: List of function names to create tests for
    """
    if not test_functions:
        test_functions = ["function1", "function2"]

    test_content = f'''import pytest
from {module_name} import *  # Import what you need from your module


class Test{module_name.capitalize()}:
    """Test cases for {module_name} module."""

    def setup_method(self):
        """Setup method called before each test method."""
        # Add common setup code here
        pass

    def teardown_method(self):
        """Teardown method called after each test method."""
        # Add common teardown code here
        pass

'''

    for func_name in test_functions:
        test_content += f'''
    def test_{func_name}_basic(self):
        """Test basic functionality of {func_name}."""
        # Arrange
        # Add setup code here

        # Act
        # result = {func_name}()

        # Assert
        # assert result == expected_value
        pass

    def test_{func_name}_edge_cases(self):
        """Test edge cases for {func_name}."""
        # Test with edge case inputs
        pass

    def test_{func_name}_exceptions(self):
        """Test exception handling for {func_name}."""
        # Test that exceptions are properly raised
        # with pytest.raises(ExpectedException):
        #     {func_name}(invalid_input)
        pass
'''

    return test_content


def create_fixture_scaffold(fixture_name: str, scope: str = "function", params: list = None):
    """
    Generate a fixture scaffold with different configurations.

    Args:
        fixture_name: Name of the fixture
        scope: Scope of the fixture (function, class, module, session)
        params: List of parameters for parametrized fixture
    """
    fixture_content = f'''import pytest
# Add imports for your fixture needs


@pytest.fixture(scope="{scope}")
def {fixture_name}(request):
    """
    Fixture for {fixture_name}.

    Scope: {scope}
    '''

    if params:
        fixture_content += f'''Params: {params}
    """
    if request.param in {params}:
        # Setup based on param
        value = request.param
        # Add setup code here
        yield value
        # Add teardown code here
    else:
        # Default setup
        yield None
'''
    else:
        fixture_content += f'''    """
    # Add setup code here
    value = None  # Replace with actual setup
    yield value
    # Add teardown code here
'''

    return fixture_content


def create_pytest_ini():
    """
    Generate a basic pytest.ini configuration file.
    """
    config_content = '''[tool:pytest]
minversion = 6.0
testpaths = tests
python_files = test_*.py *_test.py
python_classes = Test*
python_functions = test_*
addopts =
    -ra
    --strict-markers
    --strict-config
    --verbose
markers =
    slow: marks tests as slow
    fast: marks tests as fast
    integration: marks tests as integration tests
    unit: marks tests as unit tests
xfail_strict = true
'''
    return config_content


def create_conftest_py(fixtures: list = None):
    """
    Generate a conftest.py file with common fixtures.

    Args:
        fixtures: List of fixture names to include
    """
    if not fixtures:
        fixtures = ["sample_data", "temp_directory"]

    conftest_content = '''import pytest
import tempfile
import shutil
from pathlib import Path


'''

    for fixture_name in fixtures:
        conftest_content += f'''@pytest.fixture
def {fixture_name}():
    """Common fixture: {fixture_name}."""
    # Add fixture implementation here
    yield None


'''

    # Add some common built-in style fixtures
    conftest_content += '''@pytest.fixture
def temp_dir():
    """Create a temporary directory that gets cleaned up."""
    temp_path = Path(tempfile.mkdtemp())
    yield temp_path
    # Cleanup
    shutil.rmtree(temp_path)


@pytest.fixture
def sample_data():
    """Provide sample test data."""
    return {
        "users": [
            {"id": 1, "name": "Alice", "email": "alice@example.com"},
            {"id": 2, "name": "Bob", "email": "bob@example.com"}
        ],
        "config": {
            "debug": True,
            "timeout": 30
        }
    }

'''

    return conftest_content


def main():
    import sys
    # Only run examples if no command line arguments provided
    if len(sys.argv) == 1:
        # Example usage
        print("# Example test scaffold:")
        example_scaffold = create_test_scaffold("calculator", ["add", "subtract"])
        print(example_scaffold)

        print("\n# Example fixture:")
        example_fixture = create_fixture_scaffold("database_connection", "session")
        print(example_fixture)

        print("\n# Example pytest.ini:")
        example_config = create_pytest_ini()
        print(example_config)

        print("\n# Example conftest.py:")
        example_conftest = create_conftest_py(["db_session", "api_client"])
        print(example_conftest)
    else:
        # Parse and execute command line arguments
        parser = argparse.ArgumentParser(
            description="Pytest Helper Scripts: Generate test scaffolds, fixtures, and configurations"
        )
        parser.add_argument(
            "action",
            choices=["scaffold", "fixture", "config", "conftest"],
            help="Action to perform: 'scaffold' for test files, 'fixture' for fixtures, 'config' for ini files, 'conftest' for conftest.py"
        )
        parser.add_argument("--module-name", help="Name of the module being tested (for scaffold generation)")
        parser.add_argument("--fixture-name", help="Name of the fixture (for fixture generation)")
        parser.add_argument("--scope", default="function", help="Scope for fixture (function, class, module, session)")
        parser.add_argument("--functions", nargs="+", help="Function names to create tests for")
        parser.add_argument("--fixtures", nargs="+", help="Fixture names to include in conftest.py")
        parser.add_argument("--output", "-o", help="Output file path (default: print to stdout)")

        args = parser.parse_args()

        result = ""

        if args.action == "scaffold":
            if not args.module_name:
                print("Error: --module-name is required for 'scaffold' action")
                return

            result = create_test_scaffold(args.module_name, args.functions)

        elif args.action == "fixture":
            if not args.fixture_name:
                print("Error: --fixture-name is required for 'fixture' action")
                return

            result = create_fixture_scaffold(args.fixture_name, args.scope)

        elif args.action == "config":
            result = create_pytest_ini()

        elif args.action == "conftest":
            result = create_conftest_py(args.fixtures)

        if args.output:
            with open(args.output, 'w') as f:
                f.write(result)
            print(f"Generated code saved to {args.output}")
        else:
            print(result)


if __name__ == "__main__":
    main()
