# Secrets Management for Railway Deployments

## Overview

This guide covers secrets management for the simplified Railway deployment model, using GitHub Secrets for CI/CD automation and Railway Shared Variables for application secrets that PR environments need to inherit.

## Secret Storage Strategy

### Simplified Two-Location Approach

1. **GitHub Secrets** - Only for CI/CD automation (`RAILWAY_TOKEN_PROD`)
2. **Railway Shared Variables** - For application secrets that PR environments need (`YOTO_CLIENT_ID`)

### Secret Types and Locations

**CI/CD Secrets (GitHub):**
- `RAILWAY_TOKEN_PROD` - Railway API token for production deployments

**Application Secrets (Railway Shared Variables):**
- `YOTO_CLIENT_ID` - Yoto API client identifier (set as Shared Variable in production)
- Other secrets as needed - Set directly in Railway production environment

**Auto-Generated by Railway:**
- `DATABASE_URL` - PostgreSQL connection string (if using database)
- `REDIS_URL` - Redis connection string (if using Redis)
- `PORT` - Application port
- `RAILWAY_ENVIRONMENT_NAME` - Environment name (production, pr-123, etc.)

## GitHub Secrets Setup

### Required GitHub Secret

Only ONE GitHub Secret is needed for automated deployments:

#### Via GitHub Web UI

1. Navigate to repository on GitHub
2. Go to **Settings** → **Secrets and variables** → **Actions**
3. Click **New repository secret**
4. Add the secret:

```
Name: RAILWAY_TOKEN_PROD
Value: [your Railway API token for production]
```

#### Via GitHub CLI

```bash
# Install GitHub CLI if not already installed
brew install gh  # macOS
# or: sudo apt install gh  # Ubuntu/Debian

# Authenticate
gh auth login

# Add the production Railway token
gh secret set RAILWAY_TOKEN_PROD --body "your_railway_prod_token"

# Verify secret was added
gh secret list
```

### Obtaining Required Secrets

#### Railway Token

```bash
# Generate Railway token via dashboard:
# 1. Go to https://railway.app/account/tokens
# 2. Click "Create Token"
# 3. Name it "GitHub Actions Production"
# 4. Copy token and add to GitHub Secrets as RAILWAY_TOKEN_PROD
```

```bash
# Install Railway CLI
npm i -g @railway/cli

# Login
railway login

# Link to project
railway link

# Set variables for production
railway variables set YOTO_CLIENT_ID="your_client_id" -e production
railway variables set YOTO_CLIENT_SECRET="your_secret" -e production
railway variables set LOG_LEVEL="info" -e production
railway variables set DEBUG="false" -e production

# Set variables for staging
railway variables set YOTO_CLIENT_ID="your_client_id" -e staging
railway variables set YOTO_CLIENT_SECRET="your_secret" -e staging
## Railway Shared Variables

### What are Shared Variables?

Railway Shared Variables allow you to define a variable in one environment (production) and have other environments (like PR environments) reference it. This is perfect for secrets that should be consistent across environments but shouldn't be duplicated.

### Setting Up Shared Variables

**Via Railway Dashboard:**

1. Go to https://railway.app/dashboard
2. Select your project → **production** environment
3. Click on **Variables** tab
4. Find or create the variable (e.g., `YOTO_CLIENT_ID`)
5. Set the variable **type** to **"Shared Variable"**
6. Save

**Key Benefits:**
- ✅ Single source of truth for secrets
- ✅ PR environments automatically inherit the value
- ✅ No need to store in GitHub Secrets
- ✅ Easy to update in one place

### Referencing Shared Variables

In PR environments, reference shared variables using Railway's template syntax:

```bash
# In production environment
YOTO_CLIENT_ID=abc123xyz  # Type: Shared Variable

# In PR environment (set by GitHub Actions)
YOTO_CLIENT_ID=${{shared.YOTO_CLIENT_ID}}  # References production's value
```

**How it works:**
1. Production defines `YOTO_CLIENT_ID` as a Shared Variable
2. GitHub Actions workflow sets PR environment's `YOTO_CLIENT_ID` to `${{shared.YOTO_CLIENT_ID}}`
3. Railway resolves the reference at runtime to production's value
4. PR environment uses the same value as production

## Railway Variables Configuration

### Setting Variables via CLI

```bash
# Set variables in production
railway variables set DEBUG="false" -e production
railway variables set LOG_LEVEL="warning" -e production

# Note: Shared Variable type must be set via Dashboard
# CLI doesn't support setting variable types

# View variables for environment
railway variables -e production
railway variables -e pr-123  # View PR environment variables
```

### Setting Variables via Railway Dashboard

1. Go to https://railway.app/dashboard
2. Select your project
3. Select environment (production or pr-{number})
4. Go to **Variables** tab
5. Add/edit variables:
   - **Shared Variable**: `YOTO_CLIENT_ID` (in production)
   - **Plain text**: Other configuration values
   - **Reference Variables**: `DATABASE_URL=${{Postgres.DATABASE_URL}}`

### Variable Syntax

```bash
# Plain text variable
DEBUG=false

# Reference another service's variable
DATABASE_URL=${{Postgres.DATABASE_URL}}
REDIS_URL=${{Redis.REDIS_URL}}

# Reference a shared variable (for PR environments)
YOTO_CLIENT_ID=${{shared.YOTO_CLIENT_ID}}

# Railway auto-set variables
PORT=${{PORT}}
RAILWAY_ENVIRONMENT_NAME=${{RAILWAY_ENVIRONMENT_NAME}}
```

## CI/CD Integration

### GitHub Actions Workflow with Secrets

The simplified workflow only needs `RAILWAY_TOKEN_PROD`:

```yaml
# .github/workflows/railway-deploy.yml
name: Deploy to Railway

on:
  push:
    branches:
      - main  # Only production deployments
  pull_request:
    branches:
      - main

jobs:
  deploy:
    runs-on: ubuntu-latest
    
    steps:
      - name: Checkout code
        uses: actions/checkout@v3
      
      - name: Install Railway CLI
        run: npm i -g @railway/cli
      
      - name: Deploy to Railway
        run: railway up --environment ${{ env.ENVIRONMENT }}
        env:
          RAILWAY_TOKEN: ${{ secrets.RAILWAY_TOKEN }}
      
      - name: Sync Secrets to Railway
        run: |
          # Set Yoto credentials in Railway
          # Note: In GitHub Actions workflows, ${{ secrets.SECRET_NAME }} is correct syntax
          # and is safely interpolated by GitHub (not exposed in logs)
          railway variables set YOTO_CLIENT_ID="${{ secrets.YOTO_CLIENT_ID }}" \
            --environment ${{ env.ENVIRONMENT }}
          railway variables set YOTO_CLIENT_SECRET="${{ secrets.YOTO_CLIENT_SECRET }}" \
            --environment ${{ env.ENVIRONMENT }}
        env:
          RAILWAY_TOKEN: ${{ secrets.RAILWAY_TOKEN }}
      
      - name: Verify Deployment
        run: |
          # Wait for deployment to complete
          sleep 10
          
          # Check health endpoint
          DEPLOY_URL=$(railway status --environment ${{ env.ENVIRONMENT }} | grep -oP 'https?://[^\s]+')
          curl -f "$DEPLOY_URL/health" || exit 1
        env:
          RAILWAY_TOKEN: ${{ secrets.RAILWAY_TOKEN }}
```

### Testing Secrets in CI/CD

```yaml
# .github/workflows/test.yml
name: Test

on:
  push:
    branches: [main, develop, 'feature/**']
  pull_request:
    branches: [main, develop]

jobs:
  test:
    runs-on: ubuntu-latest
    
    steps:
      - uses: actions/checkout@v3
      
      - name: Set up Python
        uses: actions/setup-python@v4
        with:
          python-version: '3.11'
      
      - name: Install dependencies
        run: |
          pip install -r requirements.txt
          pip install -r requirements-dev.txt
      
      - name: Run tests with secrets
        env:
          YOTO_CLIENT_ID: ${{ secrets.YOTO_CLIENT_ID }}
          YOTO_CLIENT_SECRET: ${{ secrets.YOTO_CLIENT_SECRET }}
        run: |
          pytest tests/ -v --cov=yoto_smart_stream
```

## Local Development

### Using .env Files (Never Commit!)

```bash
# .env (add to .gitignore!)
YOTO_CLIENT_ID=dev_client_id_abc123
YOTO_CLIENT_SECRET=dev_secret_xyz789
DATABASE_URL=postgresql://localhost:5432/yoto_dev
REDIS_URL=redis://localhost:6379
DEBUG=true
LOG_LEVEL=debug

# Load in Python with python-dotenv
# pip install python-dotenv
```

```python
# Load environment variables
from dotenv import load_dotenv
import os

load_dotenv()  # Load .env file

YOTO_CLIENT_ID = os.getenv("YOTO_CLIENT_ID")
YOTO_CLIENT_SECRET = os.getenv("YOTO_CLIENT_SECRET")
```

### .env.example Template

```bash
# .env.example (commit this to repo)
# Copy this to .env and fill in your values

# Yoto API Configuration
YOTO_CLIENT_ID=your_client_id_here
YOTO_CLIENT_SECRET=your_client_secret_here

# Database (provided by Railway in production)
DATABASE_URL=postgresql://localhost:5432/yoto_dev

# Redis (provided by Railway in production)
REDIS_URL=redis://localhost:6379

# Application Configuration
DEBUG=true
LOG_LEVEL=debug
PORT=8000
```

## Environment-Specific Configuration

### Development Environment

```bash
# Development - use local values or test credentials
YOTO_CLIENT_ID=dev_test_id
YOTO_CLIENT_SECRET=dev_test_secret
DEBUG=true
LOG_LEVEL=debug
ENVIRONMENT=development
```

### Staging Environment

```bash
# Staging - use production-like credentials
# Set via Railway CLI or GitHub Actions (secrets synced from GitHub)
YOTO_CLIENT_ID=<from GitHub Secrets>
YOTO_CLIENT_SECRET=<from GitHub Secrets>
DEBUG=true
LOG_LEVEL=info
ENVIRONMENT=staging
DATABASE_URL=${{Postgres.DATABASE_URL}}  # Railway reference variable
```

### Production Environment

```bash
# Production - use production credentials
# Set via Railway CLI or GitHub Actions (secrets synced from GitHub)
YOTO_CLIENT_ID=<from GitHub Secrets>
YOTO_CLIENT_SECRET=<from GitHub Secrets>
DEBUG=false
LOG_LEVEL=warning
ENVIRONMENT=production
DATABASE_URL=${{Postgres.DATABASE_URL}}  # Railway reference variable
REDIS_URL=${{Redis.REDIS_URL}}  # Railway reference variable
```

## Accessing Secrets in Application Code

### Python/FastAPI Example

```python
# config.py - Centralized configuration
import os
from typing import Optional

class Config:
    """Application configuration from environment variables"""
    
    # Required secrets
    YOTO_CLIENT_ID: str = os.getenv("YOTO_CLIENT_ID", "")
    YOTO_CLIENT_SECRET: str = os.getenv("YOTO_CLIENT_SECRET", "")
    
    # Database
    DATABASE_URL: str = os.getenv("DATABASE_URL", "postgresql://localhost:5432/yoto")
    REDIS_URL: Optional[str] = os.getenv("REDIS_URL")
    
    # Application settings
    DEBUG: bool = os.getenv("DEBUG", "false").lower() == "true"
    LOG_LEVEL: str = os.getenv("LOG_LEVEL", "info")
    ENVIRONMENT: str = os.getenv("ENVIRONMENT", "development")
    PORT: int = int(os.getenv("PORT", "8000"))
    
    @classmethod
    def validate(cls) -> None:
        """Validate required configuration"""
        if not cls.YOTO_CLIENT_ID:
            raise ValueError("YOTO_CLIENT_ID environment variable is required")
        if not cls.YOTO_CLIENT_SECRET:
            raise ValueError("YOTO_CLIENT_SECRET environment variable is required")
        if not cls.DATABASE_URL:
            raise ValueError("DATABASE_URL environment variable is required")

# Validate on import
Config.validate()
```

```python
# main.py - Use configuration
from config import Config
from yoto_api import YotoManager

# Initialize Yoto client with secrets
yoto_manager = YotoManager(
    client_id=Config.YOTO_CLIENT_ID,
    client_secret=Config.YOTO_CLIENT_SECRET
)
```

### Logging Secrets Safely

```python
# logging_config.py
import logging
import re

class SecretFilter(logging.Filter):
    """Filter to redact secrets from logs"""
    
    def __init__(self):
        super().__init__()
        self.patterns = [
            re.compile(r'YOTO_CLIENT_ID=\w+'),
            re.compile(r'YOTO_CLIENT_SECRET=\w+'),
            re.compile(r'password=\w+'),
            re.compile(r'token=\w+'),
        ]
    
    def filter(self, record):
        """Redact secrets in log messages"""
        for pattern in self.patterns:
            record.msg = pattern.sub('[REDACTED]', str(record.msg))
        return True

# Apply filter to logger
logger = logging.getLogger(__name__)
logger.addFilter(SecretFilter())

# Safe logging
logger.info(f"Authenticating with Yoto API (Client ID: {Config.YOTO_CLIENT_ID[:8]}...)")
```

## Secret Rotation

### Rotating Yoto Credentials

```bash
# 1. Generate new credentials from Yoto Developer Portal
# 2. Update GitHub Secrets
gh secret set YOTO_CLIENT_ID --body "new_client_id"
gh secret set YOTO_CLIENT_SECRET --body "new_client_secret"

# 3. Update Railway variables
railway variables set YOTO_CLIENT_ID="new_client_id" -e production
railway variables set YOTO_CLIENT_SECRET="new_client_secret" -e production

# 4. Redeploy services
railway redeploy -e production

# 5. Verify old credentials are revoked
```

### Rotating Railway Token

```bash
# 1. Generate new token from Railway dashboard
# 2. Update GitHub Secret
gh secret set RAILWAY_TOKEN --body "new_railway_token"

# 3. Test deployment with new token
# Trigger deployment via git push or manual trigger
```

## Security Best Practices

### ✅ DO:

1. **Store all secrets in GitHub Secrets** for CI/CD access
2. **Sync secrets to Railway** via automated workflows
3. **Use different credentials** for dev/staging/prod when possible
4. **Rotate secrets regularly** (quarterly for production)
5. **Limit access** to GitHub repository secrets (admin only)
6. **Use environment-specific values** (different DEBUG, LOG_LEVEL per env)
7. **Validate secrets exist** before application starts
8. **Log secret usage without values** (e.g., "Using Yoto Client ID: abc...123")
9. **Use .env.example** as template (commit) and .env for actual values (ignore)
10. **Implement secret rotation procedures** and document them

### ❌ DON'T:

1. **Never commit secrets** to git (use .gitignore)
2. **Never log secret values** in full
3. **Never share secrets** via chat, email, or documents
4. **Never hardcode secrets** in application code
5. **Never use production secrets** in development/testing
6. **Never store secrets** in code comments or documentation
7. **Never use weak secrets** (e.g., "password123")
8. **Never skip secret validation** on application startup
9. **Never expose secrets** in error messages or stack traces
10. **Never use same secret** across multiple environments

## Troubleshooting

### Secret Not Found Error

```python
# Error: ValueError: YOTO_CLIENT_ID environment variable is required

# Solution 1: Check GitHub Secret exists
gh secret list

# Solution 2: Verify Railway variable is set
railway variables -e production | grep YOTO_CLIENT_ID

# Solution 3: Check GitHub Actions workflow has secret access
# Ensure secret is referenced correctly: ${{ secrets.YOTO_CLIENT_ID }}
```

### Secret Value Empty in Railway

```bash
# Problem: Variable set but shows as empty

# Solution: Variables might need quotes for special characters
railway variables set YOTO_CLIENT_SECRET='secret_with_$pecial_chars' -e production

# Or use Railway dashboard to set complex values
```

### GitHub Actions Cannot Access Secret

```yaml
# Problem: Secret not accessible in workflow

# Solution: Check repository settings
# 1. Secrets must be added at repository level
# 2. Workflow must have proper permissions
# 3. Secret name is case-sensitive

jobs:
  deploy:
    runs-on: ubuntu-latest
    permissions:
      contents: read
    env:
      YOTO_CLIENT_ID: ${{ secrets.YOTO_CLIENT_ID }}  # Correct
      # NOT: ${{ secrets.yoto_client_id }}  # Wrong (case)
```

### Railway Deployment Fails After Secret Update

```bash
# Problem: New secrets not loaded in running service

# Solution: Redeploy to load new variables
railway redeploy -e production

# Or: Restart service
railway restart -s web -e production
```

## Automation Scripts

### Sync GitHub Secrets to Railway

```bash
#!/bin/bash
# sync-secrets.sh - Sync secrets from GitHub to Railway

set -e

ENVIRONMENT=${1:-production}

echo "Syncing secrets to Railway environment: $ENVIRONMENT"

# Get secrets from GitHub (requires gh CLI authenticated)
YOTO_CLIENT_ID=$(gh secret get YOTO_CLIENT_ID)
YOTO_CLIENT_SECRET=$(gh secret get YOTO_CLIENT_SECRET)

# Set in Railway
railway variables set YOTO_CLIENT_ID="$YOTO_CLIENT_ID" -e "$ENVIRONMENT"
railway variables set YOTO_CLIENT_SECRET="$YOTO_CLIENT_SECRET" -e "$ENVIRONMENT"

echo "✓ Secrets synced successfully"

# Redeploy to apply changes
echo "Redeploying service..."
railway redeploy -e "$ENVIRONMENT"

echo "✓ Deployment triggered"
```

### Validate Secrets Exist

```python
#!/usr/bin/env python3
# validate-secrets.py - Validate all required secrets are set

import os
import sys

REQUIRED_SECRETS = [
    "YOTO_CLIENT_ID",
    "YOTO_CLIENT_SECRET",
    "DATABASE_URL",
]

OPTIONAL_SECRETS = [
    "REDIS_URL",
    "SENTRY_DSN",
]

def validate_secrets():
    """Validate required secrets are set"""
    missing = []
    
    for secret in REQUIRED_SECRETS:
        if not os.getenv(secret):
            missing.append(secret)
    
    if missing:
        print(f"❌ Missing required secrets: {', '.join(missing)}")
        return False
    
    print("✓ All required secrets are set")
    
    # Check optional secrets
    for secret in OPTIONAL_SECRETS:
        if not os.getenv(secret):
            print(f"⚠️  Optional secret not set: {secret}")
    
    return True

if __name__ == "__main__":
    if not validate_secrets():
        sys.exit(1)
```

## Checklist: Setting Up Secrets

### Initial Setup

- [ ] Create Yoto Developer account and obtain Client ID/Secret
- [ ] Add YOTO_CLIENT_ID to GitHub Secrets
- [ ] Add YOTO_CLIENT_SECRET to GitHub Secrets
- [ ] Generate Railway API token
- [ ] Add RAILWAY_TOKEN to GitHub Secrets
- [ ] Create .env.example file with all required variables (commit)
- [ ] Create .env file with actual values (do NOT commit)
- [ ] Add .env to .gitignore
- [ ] Set up Railway variables for production environment
- [ ] Set up Railway variables for staging environment
- [ ] Verify secrets in GitHub Actions workflow
- [ ] Test deployment with secrets
- [ ] Document secret rotation procedure

### For Each New Environment

- [ ] Create environment in Railway
- [ ] Set YOTO_CLIENT_ID variable
- [ ] Set YOTO_CLIENT_SECRET variable
- [ ] Set environment-specific variables (DEBUG, LOG_LEVEL)
- [ ] Configure database connection (DATABASE_URL)
- [ ] Test deployment
- [ ] Verify secrets are loaded correctly

---

**Security Note**: This document describes secret management patterns. Never include actual secret values in documentation or commit them to version control. Always use placeholder values in examples.
