---
name: requesting-code-review
description: Use when completing tasks, implementing major features, or before merging to verify work meets requirements
---

# Requesting Code Review

Dispatch superpowers:code-reviewer subagent to catch issues before they cascade.

**Core principle:** Review early, review often.

## When to Request Review

**Mandatory:**
- After each task in subagent-driven development
- After completing major feature
- Before merge to main

**Optional but valuable:**
- When stuck (fresh perspective)
- Before refactoring (baseline check)
- After fixing complex bug

## How to Request

**1. Get git SHAs:**
```bash
BASE_SHA=$(git rev-parse HEAD~1)  # or origin/main
HEAD_SHA=$(git rev-parse HEAD)
```

**2. For C# .NET Projects (ECTSystem):**
```bash
# Verify build and tests before requesting review
dotnet build ECTSystem.sln --no-restore
dotnet test AF.ECT.Tests --filter "YourNewTests" --no-build

# Get SHAs
BASE_SHA=$(git log --oneline | grep "Previous task" | head -1 | awk '{print $1}')
HEAD_SHA=$(git rev-parse HEAD)

# Get diff summary
git diff $BASE_SHA $HEAD_SHA --stat
```

**3. Dispatch code-reviewer subagent:**

Use Task tool with superpowers:code-reviewer type, fill template at `code-reviewer.md`

**Placeholders:**
- `{WHAT_WAS_IMPLEMENTED}` - What you just built
- `{PLAN_OR_REQUIREMENTS}` - What it should do
- `{BASE_SHA}` - Starting commit
- `{HEAD_SHA}` - Ending commit
- `{DESCRIPTION}` - Brief summary

**4. Act on feedback:**
- Fix Critical issues immediately
- Fix Important issues before proceeding
- Note Minor issues for later
- Push back if reviewer is wrong (with reasoning)

## Example

```
[Just completed Task 2: Add verification function]

You: Let me request code review before proceeding.

BASE_SHA=$(git log --oneline | grep "Task 1" | head -1 | awk '{print $1}')
HEAD_SHA=$(git rev-parse HEAD)

[Dispatch superpowers:code-reviewer subagent]
  WHAT_WAS_IMPLEMENTED: Verification and repair functions for conversation index
  PLAN_OR_REQUIREMENTS: Task 2 from docs/plans/deployment-plan.md
  BASE_SHA: a7981ec
  HEAD_SHA: 3df7661
  DESCRIPTION: Added verifyIndex() and repairIndex() with 4 issue types

[Subagent returns]:
  Strengths: Clean architecture, real tests
  Issues:
    Important: Missing progress indicators
    Minor: Magic number (100) for reporting interval
  Assessment: Ready to proceed

You: [Fix progress indicators]
[Continue to Task 3]
```

**C# .NET Example (ECTSystem):**

```
[Just completed Task 2: Add GetWorkflowCount gRPC method]

You: Let me request code review before proceeding.

dotnet build ECTSystem.sln --no-restore
  ✓ Build succeeded, 0 errors, 0 warnings
  
dotnet test AF.ECT.Tests --filter "GetWorkflowCount" --no-build
  ✓ 3 tests passed

BASE_SHA=$(git log --oneline | grep "feat: Add GetUser" | head -1 | awk '{print $1}')
HEAD_SHA=$(git rev-parse HEAD)
git diff $BASE_SHA $HEAD_SHA --stat
  AF.ECT.Shared/Protos/workflow.proto
  AF.ECT.Server/Services/WorkflowServiceImpl.cs
  AF.ECT.Tests/Unit/WorkflowServiceTests.cs

[Dispatch superpowers:code-reviewer subagent]
  WHAT_WAS_IMPLEMENTED: GetWorkflowCount gRPC method with filtering by status
  PLAN_OR_REQUIREMENTS: Task 2 from docs/plans/2025-01-10-workflow-enhancements.md
  BASE_SHA: d4a9c2f
  HEAD_SHA: e8f1a3b
  DESCRIPTION: Added gRPC method with InMemory + SQL Server tests, audit logging

[Subagent returns]:
  Strengths: Good test coverage (both InMemory and integration), proper exception handling, audit logging included
  Issues:
    Important: Missing validation for status enum value
    Minor: Magic number 1000 for max results
  Assessment: Fix validation, then ready to proceed

You: [Add enum validation, extract MAX_RESULTS constant]
dotnet test AF.ECT.Tests --filter "GetWorkflowCount" --no-build
  ✓ 5 tests passed (added validation tests)

[Dispatch code-reviewer subagent again]
Code reviewer: ✅ Approved - ready for next task
[Continue to Task 3]
```

## Integration with Workflows

**Subagent-Driven Development:**
- Review after EACH task
- Catch issues before they compound
- Fix before moving to next task

**Executing Plans:**
- Review after each batch (3 tasks)
- Get feedback, apply, continue

**Ad-Hoc Development:**
- Review before merge
- Review when stuck

## Red Flags

**Never:**
- Skip review because "it's simple"
- Ignore Critical issues
- Proceed with unfixed Important issues
- Argue with valid technical feedback

**If reviewer wrong:**
- Push back with technical reasoning
- Show code/tests that prove it works
- Request clarification

See template at: requesting-code-review/code-reviewer.md
