# Rubric YAML Schema Reference

Complete reference for evaluation rubric configuration files.

## Full Schema

```yaml
# Rubric: [Human-readable name]
# Version: [semver]
# Purpose: [Brief description]

name: rubric-name                    # Required: kebab-case identifier
version: 1.0.0                       # Required: semantic version
description: |                       # Required: multi-line description
  What this rubric evaluates and when to use it.
  Include context about target content/behavior.

# Scope Configuration
scope:
  type: content                      # Required: content | behavior | both
  file_patterns:                     # Optional: glob patterns to filter
    - "*.md"
    - "**/*.tsx"
  exclude_patterns:                  # Optional: patterns to skip
    - "**/node_modules/**"
    - "**/*.test.ts"

# Criteria Definitions
criteria:
  criterion-name:                    # kebab-case criterion identifier
    weight: 25                       # Required: 0-100, all must sum to 100
    threshold: 80                    # Required: 0-100, minimum to pass
    description: |                   # Required: what this measures
      Clear description of what this criterion evaluates.

    checks:                          # Required: list of checks
      # Pattern-based checks
      - type: absence               # Content should NOT contain
        pattern: "regex-pattern"    # Regex to match
        message: "Feedback text"    # Shown when check fails

      - type: presence              # Content MUST contain
        pattern: "regex-pattern"
        message: "Feedback text"

      - type: pattern               # Content should match format
        pattern: "regex-pattern"
        message: "Feedback text"

      # LLM-evaluated checks
      - type: custom
        prompt: |                   # Question for LLM evaluation
          Does this content meet [specific criterion]?
        message: "Feedback text"    # Shown when check fails

    examples:                        # Optional but recommended
      pass: |                        # Example of passing content
        "Good example that meets criterion"
      fail: |                        # Example of failing content
        "Bad example that violates criterion"

  # Additional criteria...
  another-criterion:
    weight: 25
    threshold: 75
    description: "..."
    checks: [...]

# Passing Requirements
passing:
  min_score: 75                      # Required: overall minimum (0-100)
  required_criteria:                 # Optional: must-pass criteria
    - criterion-name                 # These must pass regardless of score
```

## Field Reference

### Root Level

| Field | Required | Type | Description |
|-------|----------|------|-------------|
| `name` | Yes | string | Unique identifier (kebab-case, 3-50 chars) |
| `version` | Yes | string | Semantic version (X.Y.Z) |
| `description` | Yes | string | Multi-line purpose description |
| `scope` | Yes | object | What this rubric evaluates |
| `criteria` | Yes | object | Map of criterion definitions |
| `passing` | Yes | object | Pass/fail requirements |

### Scope Object

| Field | Required | Type | Description |
|-------|----------|------|-------------|
| `type` | Yes | enum | `content`, `behavior`, or `both` |
| `file_patterns` | No | array | Glob patterns to include |
| `exclude_patterns` | No | array | Glob patterns to exclude |

### Criterion Object

| Field | Required | Type | Description |
|-------|----------|------|-------------|
| `weight` | Yes | number | 0-100, contribution to total score |
| `threshold` | Yes | number | 0-100, minimum score to pass |
| `description` | Yes | string | What this criterion measures |
| `checks` | Yes | array | List of check definitions |
| `examples` | No | object | Pass/fail example content |

### Check Object

| Field | Required | Type | Description |
|-------|----------|------|-------------|
| `type` | Yes | enum | `absence`, `presence`, `pattern`, `custom` |
| `pattern` | For pattern types | string | Regex pattern to match |
| `prompt` | For custom type | string | LLM evaluation question |
| `message` | Yes | string | Feedback when check fails |

### Passing Object

| Field | Required | Type | Description |
|-------|----------|------|-------------|
| `min_score` | Yes | number | 0-100, overall minimum score |
| `required_criteria` | No | array | Criterion names that must pass |

## Check Type Details

### `absence`

Fails if pattern IS found in content.

```yaml
- type: absence
  pattern: "\\b(very|really|basically)\\b"
  message: "Remove filler words"
```

Use for: Forbidden words, anti-patterns, deprecated syntax

### `presence`

Fails if pattern is NOT found in content.

```yaml
- type: presence
  pattern: "^#\\s+.+"
  message: "Document must have a title heading"
```

Use for: Required elements, mandatory sections, expected patterns

### `pattern`

Validates content matches expected format.

```yaml
- type: pattern
  pattern: "^[A-Z][a-z]"
  message: "Sentences should start with capital letter"
```

Use for: Format validation, structure checks, naming conventions

### `custom`

LLM evaluates content against prompt.

```yaml
- type: custom
  prompt: "Is the error message actionable? Does it tell the user what went wrong and how to fix it?"
  message: "Error messages should include context and recovery steps"
```

Use for: Nuanced evaluation, context-dependent quality, subjective criteria

## Regex Pattern Tips

- Escape special characters: `\.`, `\(`, `\[`
- Word boundaries: `\b` prevents partial matches
- Case insensitive: Use `(?i)` prefix
- Multiline: Use `(?m)` for `^`/`$` matching line starts/ends
- Common patterns:
  - Words: `\b(word1|word2)\b`
  - URLs: `https?://[^\s]+`
  - Email: `[\w.-]+@[\w.-]+`
  - Version: `v?\d+\.\d+\.\d+`

## Validation Rules

1. All `weight` values must sum to exactly 100
2. `threshold` must be between 0 and 100
3. `min_score` must be between 0 and 100
4. `required_criteria` entries must match criterion names
5. `name` must be valid kebab-case (lowercase, hyphens, 3-50 chars)
6. Pattern checks require `pattern` field
7. Custom checks require `prompt` field
