#!/usr/bin/env bash

# security-check - Security audit, vulnerability scanning, and secrets detection
# Part of Claude Skills system

set -euo pipefail

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
MAGENTA='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Icons
CHECK_MARK="✓"
CROSS_MARK="✗"
WARNING="⚠"
INFO="ℹ"

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Usage information
usage() {
    cat << EOF
${CYAN}security-check${NC} - Security audit, vulnerability scanning, and secrets detection

${YELLOW}USAGE:${NC}
    security-check <command> [options]

${YELLOW}COMMANDS:${NC}
    ${GREEN}audit${NC}                    Run npm/yarn/pnpm audit with detailed report
    ${GREEN}secrets [path]${NC}           Scan for exposed secrets/API keys (default: .)
    ${GREEN}licenses${NC}                 Check dependency licenses for compliance
    ${GREEN}env${NC}                      Audit environment variable security
    ${GREEN}headers${NC}                  Check HTTP security headers (requires server)
    ${GREEN}cors${NC}                     Validate CORS configuration
    ${GREEN}sql [path]${NC}               Detect potential SQL injection risks
    ${GREEN}xss [path]${NC}               Check for XSS vulnerabilities
    ${GREEN}report${NC}                   Generate comprehensive security report

${YELLOW}OPTIONS:${NC}
    -h, --help               Show this help message
    -v, --verbose            Verbose output
    --json                   Output in JSON format
    --fix                    Attempt to fix issues (where applicable)
    --severity <level>       Filter by severity: low, moderate, high, critical

${YELLOW}EXAMPLES:${NC}
    security-check audit
    security-check secrets src/
    security-check licenses
    security-check env
    security-check report
    security-check audit --severity high
    security-check secrets --json

${YELLOW}USE WHEN:${NC}
    - "security scan"
    - "check vulnerabilities"
    - "find exposed secrets"
    - "audit licenses"
    - "security report"

EOF
    exit 0
}

# Helper functions
log_info() {
    echo -e "${BLUE}${INFO}${NC} $1"
}

log_success() {
    echo -e "${GREEN}${CHECK_MARK}${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}${WARNING}${NC} $1"
}

log_error() {
    echo -e "${RED}${CROSS_MARK}${NC} $1"
}

log_section() {
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${MAGENTA}$1${NC}"
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
}

# Detect package manager
detect_package_manager() {
    if [ -f "pnpm-lock.yaml" ]; then
        echo "pnpm"
    elif [ -f "yarn.lock" ]; then
        echo "yarn"
    elif [ -f "package-lock.json" ]; then
        echo "npm"
    else
        echo "npm"
    fi
}

# Audit command - Run dependency security audit
cmd_audit() {
    log_section "Security Audit"
    
    local pm=$(detect_package_manager)
    local severity_filter="${SEVERITY:-}"
    local fix_mode="${FIX_MODE:-false}"
    
    log_info "Using package manager: $pm"
    
    if [ ! -f "package.json" ]; then
        log_error "No package.json found in current directory"
        exit 1
    fi
    
    case "$pm" in
        npm)
            if [ "$fix_mode" = "true" ]; then
                log_info "Running npm audit fix..."
                npm audit fix || true
            else
                log_info "Running npm audit..."
                if [ -n "$severity_filter" ]; then
                    npm audit --audit-level="$severity_filter" || true
                else
                    npm audit || true
                fi
            fi
            ;;
        yarn)
            log_info "Running yarn audit..."
            if [ -n "$severity_filter" ]; then
                yarn audit --level "$severity_filter" || true
            else
                yarn audit || true
            fi
            ;;
        pnpm)
            log_info "Running pnpm audit..."
            if [ -n "$severity_filter" ]; then
                pnpm audit --audit-level "$severity_filter" || true
            else
                pnpm audit || true
            fi
            ;;
    esac
    
    log_success "Audit complete"
}

# Secrets command - Scan for exposed secrets
cmd_secrets() {
    log_section "Secrets Scanning"
    
    local scan_path="${1:-.}"
    local issues_found=0
    
    log_info "Scanning for exposed secrets in: $scan_path"
    
    # Common secret patterns
    declare -A patterns=(
        ["AWS Access Key"]='AKIA[0-9A-Z]{16}'
        ["AWS Secret Key"]='aws[_-]?secret[_-]?access[_-]?key'
        ["API Key"]='api[_-]?key[[:space:]]*[=:][[:space:]]*["\x27][a-zA-Z0-9_\-]{20,}["\x27]'
        ["Bearer Token"]='bearer[[:space:]]+[a-zA-Z0-9_\-\.]{20,}'
        ["GitHub Token"]='gh[pousr]_[0-9a-zA-Z]{36,}'
        ["Private Key"]='-----BEGIN (RSA |EC |DSA )?PRIVATE KEY-----'
        ["Password in Code"]='password[[:space:]]*[=:][[:space:]]*["\x27][^"\x27]{4,}["\x27]'
        ["Slack Token"]='xox[baprs]-[0-9a-zA-Z]{10,48}'
        ["Stripe Key"]='sk_live_[0-9a-zA-Z]{24,}'
        ["Google API Key"]='AIza[0-9A-Za-z\-_]{35}'
        ["JWT Token"]='eyJ[a-zA-Z0-9_-]*\.eyJ[a-zA-Z0-9_-]*\.[a-zA-Z0-9_-]*'
    )
    
    # Files to exclude
    local exclude_patterns="node_modules|\.git|dist|build|coverage|\.next|\.env\.example"
    
    for secret_type in "${!patterns[@]}"; do
        local pattern="${patterns[$secret_type]}"
        
        # Search for pattern
        local results=$(grep -rniE "$pattern" "$scan_path" 2>/dev/null | grep -vE "$exclude_patterns" || true)
        
        if [ -n "$results" ]; then
            log_error "Found potential $secret_type:"
            echo "$results" | while read -r line; do
                echo "  ${RED}→${NC} $line"
            done
            ((issues_found++))
        fi
    done
    
    # Check for .env files not in .gitignore
    log_info "Checking .env file security..."
    if [ -f ".gitignore" ]; then
        for env_file in $(find "$scan_path" -name ".env*" -not -name "*.example" 2>/dev/null); do
            if ! grep -q "$(basename "$env_file")" .gitignore 2>/dev/null; then
                log_warning "Environment file not in .gitignore: $env_file"
                ((issues_found++))
            fi
        done
    fi
    
    if [ "$issues_found" -eq 0 ]; then
        log_success "No secrets found!"
    else
        log_warning "Found $issues_found potential security issues"
        echo ""
        log_info "Recommendations:"
        echo "  1. Remove secrets from code"
        echo "  2. Use environment variables instead"
        echo "  3. Add .env files to .gitignore"
        echo "  4. Rotate any exposed credentials"
        echo "  5. Use secret management services (AWS Secrets Manager, etc.)"
    fi
}

# Licenses command - Check dependency licenses
cmd_licenses() {
    log_section "License Compliance Check"
    
    if [ ! -f "package.json" ]; then
        log_error "No package.json found"
        exit 1
    fi
    
    # Potentially problematic licenses
    local problematic_licenses=("GPL-3.0" "AGPL" "LGPL")
    
    # Check if license-checker is available
    if ! command -v npx &> /dev/null; then
        log_error "npx not found. Please install Node.js"
        exit 1
    fi
    
    log_info "Analyzing dependency licenses..."
    
    # Use license-checker via npx
    if npx license-checker --summary 2>/dev/null; then
        log_success "License summary generated"
        
        echo ""
        log_info "Checking for problematic licenses..."
        
        local issues_found=0
        for license in "${problematic_licenses[@]}"; do
            if npx license-checker --onlyAllow "MIT;Apache-2.0;BSD;ISC" 2>&1 | grep -i "$license" &>/dev/null; then
                log_warning "Found potentially problematic license: $license"
                ((issues_found++))
            fi
        done
        
        if [ "$issues_found" -eq 0 ]; then
            log_success "No problematic licenses found"
        fi
    else
        log_info "Installing license-checker temporarily..."
        npx license-checker --summary
    fi
}

# Env command - Audit environment variables
cmd_env() {
    log_section "Environment Security Audit"
    
    local issues_found=0
    
    # Check for .env files
    log_info "Checking .env files..."
    local env_files=$(find . -maxdepth 2 -name ".env*" -type f 2>/dev/null | grep -v node_modules || true)
    
    if [ -z "$env_files" ]; then
        log_warning "No .env files found"
    else
        echo "$env_files" | while read -r file; do
            log_info "Analyzing: $file"
            
            # Check file permissions
            local perms=$(stat -c "%a" "$file" 2>/dev/null || stat -f "%A" "$file" 2>/dev/null)
            if [ "$perms" != "600" ] && [ "$perms" != "400" ]; then
                log_warning "  Insecure permissions ($perms) - should be 600 or 400"
                ((issues_found++))
            else
                log_success "  Permissions OK ($perms)"
            fi
            
            # Check for empty values
            local empty_vars=$(grep -E "^[A-Z_]+=\s*$" "$file" 2>/dev/null || true)
            if [ -n "$empty_vars" ]; then
                log_warning "  Empty environment variables found:"
                echo "$empty_vars" | while read -r var; do
                    echo "    ${YELLOW}→${NC} $var"
                done
                ((issues_found++))
            fi
            
            # Check for weak secrets (short passwords, etc.)
            local weak_secrets=$(grep -iE "(password|secret|key)=['\"]?.{1,8}['\"]?\s*$" "$file" 2>/dev/null || true)
            if [ -n "$weak_secrets" ]; then
                log_warning "  Potentially weak secrets (< 8 chars):"
                echo "$weak_secrets" | while read -r secret; do
                    echo "    ${YELLOW}→${NC} ${secret%%=*}=****"
                done
                ((issues_found++))
            fi
        done
    fi
    
    # Check .gitignore
    log_info "Checking .gitignore..."
    if [ -f ".gitignore" ]; then
        if grep -q "^\.env" .gitignore; then
            log_success ".env files are in .gitignore"
        else
            log_error ".env files NOT in .gitignore!"
            ((issues_found++))
        fi
    else
        log_warning "No .gitignore found"
        ((issues_found++))
    fi
    
    if [ "$issues_found" -eq 0 ]; then
        log_success "Environment security looks good!"
    else
        log_warning "Found $issues_found security concerns"
    fi
}

# Headers command - Check HTTP security headers
cmd_headers() {
    log_section "HTTP Security Headers Check"
    
    local url="${1:-http://localhost:3000}"
    
    log_info "Checking security headers for: $url"
    
    if ! command -v curl &> /dev/null; then
        log_error "curl not found. Please install curl"
        exit 1
    fi
    
    # Required security headers
    declare -A required_headers=(
        ["X-Content-Type-Options"]="nosniff"
        ["X-Frame-Options"]="DENY or SAMEORIGIN"
        ["X-XSS-Protection"]="1; mode=block"
        ["Strict-Transport-Security"]="max-age=31536000"
        ["Content-Security-Policy"]="restrictive policy"
        ["Referrer-Policy"]="no-referrer or strict-origin-when-cross-origin"
    )
    
    # Fetch headers
    local headers=$(curl -sI "$url" 2>/dev/null || echo "")
    
    if [ -z "$headers" ]; then
        log_error "Could not connect to $url"
        echo "Make sure the server is running"
        exit 1
    fi
    
    local issues_found=0
    
    for header in "${!required_headers[@]}"; do
        if echo "$headers" | grep -qi "^$header:"; then
            local value=$(echo "$headers" | grep -i "^$header:" | cut -d':' -f2- | xargs)
            log_success "$header: $value"
        else
            log_error "$header: Missing (recommended: ${required_headers[$header]})"
            ((issues_found++))
        fi
    done
    
    # Check for information disclosure
    if echo "$headers" | grep -qi "^Server:"; then
        local server=$(echo "$headers" | grep -i "^Server:" | cut -d':' -f2- | xargs)
        log_warning "Server header reveals information: $server"
        log_info "Consider hiding or genericizing the Server header"
    fi
    
    if echo "$headers" | grep -qi "^X-Powered-By:"; then
        local powered=$(echo "$headers" | grep -i "^X-Powered-By:" | cut -d':' -f2- | xargs)
        log_warning "X-Powered-By reveals technology: $powered"
        log_info "Consider removing the X-Powered-By header"
    fi
    
    if [ "$issues_found" -gt 0 ]; then
        echo ""
        log_warning "Found $issues_found missing security headers"
        log_info "See: https://owasp.org/www-project-secure-headers/"
    else
        log_success "All recommended security headers present!"
    fi
}

# CORS command - Validate CORS configuration
cmd_cors() {
    log_section "CORS Configuration Check"
    
    local url="${1:-http://localhost:3000}"
    
    log_info "Checking CORS configuration for: $url"
    
    if ! command -v curl &> /dev/null; then
        log_error "curl not found. Please install curl"
        exit 1
    fi
    
    # Test CORS with OPTIONS request
    local cors_headers=$(curl -sI -X OPTIONS \
        -H "Origin: https://example.com" \
        -H "Access-Control-Request-Method: POST" \
        "$url" 2>/dev/null || echo "")
    
    if [ -z "$cors_headers" ]; then
        log_error "Could not connect to $url"
        exit 1
    fi
    
    log_info "CORS headers:"
    
    # Check Access-Control-Allow-Origin
    if echo "$cors_headers" | grep -qi "Access-Control-Allow-Origin:"; then
        local origin=$(echo "$cors_headers" | grep -i "Access-Control-Allow-Origin:" | cut -d':' -f2- | xargs)
        if [ "$origin" = "*" ]; then
            log_warning "Access-Control-Allow-Origin: * (too permissive!)"
            echo "  ${YELLOW}→${NC} Consider restricting to specific origins"
        else
            log_success "Access-Control-Allow-Origin: $origin"
        fi
    else
        log_info "No CORS headers (CORS may not be configured)"
    fi
    
    # Check other CORS headers
    for header in "Access-Control-Allow-Methods" "Access-Control-Allow-Headers" "Access-Control-Max-Age"; do
        if echo "$cors_headers" | grep -qi "^$header:"; then
            local value=$(echo "$cors_headers" | grep -i "^$header:" | cut -d':' -f2- | xargs)
            log_info "$header: $value"
        fi
    done
    
    # Check for credentials
    if echo "$cors_headers" | grep -qi "Access-Control-Allow-Credentials: true"; then
        if echo "$cors_headers" | grep -qi "Access-Control-Allow-Origin: \*"; then
            log_error "Dangerous: Allow-Credentials + Allow-Origin:* combination!"
            echo "  ${RED}→${NC} This allows any origin to make authenticated requests"
        else
            log_info "Access-Control-Allow-Credentials: true (with specific origin)"
        fi
    fi
}

# SQL command - Detect SQL injection risks
cmd_sql() {
    log_section "SQL Injection Risk Detection"
    
    local scan_path="${1:-.}"
    local issues_found=0
    
    log_info "Scanning for SQL injection risks in: $scan_path"
    
    # Patterns that might indicate SQL injection vulnerabilities
    local patterns=(
        'query.*\+.*[\$\`]'                    # String concatenation in queries
        'execute\([^?]*\$'                      # Direct variable in execute
        'SELECT.*\$\{.*\}'                      # Template literals in SQL
        'WHERE.*\+.*\$'                         # String concat in WHERE
        'sql.*`.*\$\{.*\}`'                     # Tagged template without proper escaping
    )
    
    local exclude="node_modules|\.git|dist|build|coverage"
    
    for pattern in "${patterns[@]}"; do
        local results=$(grep -rniE "$pattern" "$scan_path" --include="*.ts" --include="*.js" --include="*.tsx" --include="*.jsx" 2>/dev/null | grep -vE "$exclude" || true)
        
        if [ -n "$results" ]; then
            log_warning "Potential SQL injection risk found:"
            echo "$results" | while read -r line; do
                echo "  ${YELLOW}→${NC} $line"
            done
            ((issues_found++))
        fi
    done
    
    # Check for parameterized queries (good practice)
    log_info "Checking for parameterized queries..."
    local good_patterns=$(grep -rn "\?" "$scan_path" --include="*.ts" --include="*.js" 2>/dev/null | grep -i -E "query|execute" | grep -vE "$exclude" | wc -l || echo 0)
    
    if [ "$good_patterns" -gt 0 ]; then
        log_success "Found $good_patterns instances of parameterized queries (good!)"
    fi
    
    if [ "$issues_found" -eq 0 ]; then
        log_success "No obvious SQL injection risks found"
    else
        log_warning "Found $issues_found potential SQL injection risks"
        echo ""
        log_info "Recommendations:"
        echo "  1. Use parameterized queries with placeholders (?)"
        echo "  2. Use ORMs like Prisma or TypeORM"
        echo "  3. Never concatenate user input into SQL strings"
        echo "  4. Use prepared statements"
        echo "  5. Validate and sanitize all user inputs"
    fi
}

# XSS command - Check for XSS vulnerabilities
cmd_xss() {
    log_section "XSS Vulnerability Detection"
    
    local scan_path="${1:-.}"
    local issues_found=0
    
    log_info "Scanning for XSS vulnerabilities in: $scan_path"
    
    # Patterns that might indicate XSS vulnerabilities
    local patterns=(
        'innerHTML.*=.*[\$\`]'                  # Setting innerHTML with variables
        'dangerouslySetInnerHTML'               # React dangerous HTML
        'document\.write\('                     # document.write
        'eval\('                                # eval usage
        'v-html'                                # Vue v-html directive
        '\[innerHTML\]'                         # Angular innerHTML binding
    )
    
    local exclude="node_modules|\.git|dist|build|coverage"
    
    for pattern in "${patterns[@]}"; do
        local results=$(grep -rniE "$pattern" "$scan_path" --include="*.ts" --include="*.js" --include="*.tsx" --include="*.jsx" --include="*.vue" 2>/dev/null | grep -vE "$exclude" || true)
        
        if [ -n "$results" ]; then
            log_warning "Potential XSS risk found:"
            echo "$results" | while read -r line; do
                echo "  ${YELLOW}→${NC} $line"
            done
            ((issues_found++))
        fi
    done
    
    # Check for CSP configuration
    log_info "Checking for Content Security Policy..."
    if grep -rq "Content-Security-Policy" "$scan_path" 2>/dev/null; then
        log_success "Found CSP configuration"
    else
        log_warning "No CSP configuration found"
        echo "  ${YELLOW}→${NC} Consider adding Content-Security-Policy headers"
        ((issues_found++))
    fi
    
    if [ "$issues_found" -eq 0 ]; then
        log_success "No obvious XSS vulnerabilities found"
    else
        log_warning "Found $issues_found potential XSS vulnerabilities"
        echo ""
        log_info "Recommendations:"
        echo "  1. Avoid innerHTML and dangerouslySetInnerHTML"
        echo "  2. Use textContent instead of innerHTML"
        echo "  3. Sanitize user input with DOMPurify"
        echo "  4. Implement Content Security Policy (CSP)"
        echo "  5. Use framework built-in escaping (React escapes by default)"
        echo "  6. Never use eval() with user input"
    fi
}

# Report command - Generate comprehensive security report
cmd_report() {
    log_section "Comprehensive Security Report"
    
    local report_file="security-report-$(date +%Y%m%d-%H%M%S).txt"
    
    log_info "Generating comprehensive security report..."
    echo "Security Report - $(date)" > "$report_file"
    echo "===========================================" >> "$report_file"
    echo "" >> "$report_file"
    
    # Run all checks and append to report
    {
        echo "1. DEPENDENCY AUDIT"
        echo "-------------------"
        cmd_audit 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
        echo "2. SECRETS SCAN"
        echo "---------------"
        cmd_secrets . 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
        echo "3. ENVIRONMENT SECURITY"
        echo "----------------------"
        cmd_env 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
        echo "4. SQL INJECTION RISKS"
        echo "---------------------"
        cmd_sql . 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
        echo "5. XSS VULNERABILITIES"
        echo "---------------------"
        cmd_xss . 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
        echo "6. LICENSE COMPLIANCE"
        echo "--------------------"
        cmd_licenses 2>&1 | sed 's/\x1b\[[0-9;]*m//g'
        echo ""
        
    } >> "$report_file"
    
    log_success "Report generated: $report_file"
    
    # Display summary
    echo ""
    log_info "Summary:"
    echo "  • Dependency audit: Complete"
    echo "  • Secrets scan: Complete"
    echo "  • Environment check: Complete"
    echo "  • SQL injection scan: Complete"
    echo "  • XSS vulnerability scan: Complete"
    echo "  • License compliance: Complete"
    
    # Open report if possible
    if command -v cat &> /dev/null; then
        echo ""
        read -p "View report now? (y/n) " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            cat "$report_file"
        fi
    fi
}

# Main command dispatcher
main() {
    # Parse options
    VERBOSE=false
    JSON_OUTPUT=false
    FIX_MODE=false
    SEVERITY=""
    
    while [[ $# -gt 0 ]]; do
        case "$1" in
            -h|--help)
                usage
                ;;
            -v|--verbose)
                VERBOSE=true
                shift
                ;;
            --json)
                JSON_OUTPUT=true
                shift
                ;;
            --fix)
                FIX_MODE=true
                shift
                ;;
            --severity)
                SEVERITY="$2"
                shift 2
                ;;
            audit|secrets|licenses|env|headers|cors|sql|xss|report)
                COMMAND="$1"
                shift
                break
                ;;
            *)
                log_error "Unknown option: $1"
                usage
                ;;
        esac
    done
    
    # If no command provided, show usage
    if [ -z "${COMMAND:-}" ]; then
        usage
    fi
    
    # Execute command
    case "$COMMAND" in
        audit)
            cmd_audit "$@"
            ;;
        secrets)
            cmd_secrets "$@"
            ;;
        licenses)
            cmd_licenses "$@"
            ;;
        env)
            cmd_env "$@"
            ;;
        headers)
            cmd_headers "$@"
            ;;
        cors)
            cmd_cors "$@"
            ;;
        sql)
            cmd_sql "$@"
            ;;
        xss)
            cmd_xss "$@"
            ;;
        report)
            cmd_report "$@"
            ;;
        *)
            log_error "Unknown command: $COMMAND"
            usage
            ;;
    esac
}

# Run main function
main "$@"
