"""共通ユーティリティモジュール for shiiman-google."""

import functools
import json
import os
import sys
import time
from typing import Any, Callable, Dict, List, Optional

from google.auth.exceptions import RefreshError
from google.auth.transport.requests import Request
from google.oauth2.credentials import Credentials


# 設定ディレクトリ
CONFIG_DIR = os.path.expanduser("~/.config/shiiman-google")
TOKENS_DIR = os.path.join(CONFIG_DIR, "tokens")
CLIENTS_DIR = os.path.join(CONFIG_DIR, "clients")
ACTIVE_PROFILE_FILE = os.path.join(CONFIG_DIR, "active-profile")


def expand_path(path: str) -> str:
    """パスを展開する。"""
    return os.path.expanduser(path)


def get_active_profile() -> str:
    """アクティブなプロファイル名を取得する。"""
    if os.path.exists(ACTIVE_PROFILE_FILE):
        with open(ACTIVE_PROFILE_FILE, "r", encoding="utf-8") as f:
            return f.read().strip() or "default"
    return "default"


def print_profile_header(show_all: bool = False) -> None:
    """現在のプロファイル情報をヘッダーとして表示する。

    Args:
        show_all: True の場合、利用可能な全プロファイルも表示
    """
    active = get_active_profile()
    print(f"[プロファイル: {active}]")

    if show_all:
        profiles = list_profiles()
        if len(profiles) > 1:
            others = [p for p in profiles if p != active]
            print(f"  切り替え可能: {', '.join(others)}")
    print()


def get_token_path(profile: Optional[str] = None) -> str:
    """トークンファイルのパスを取得する。"""
    if profile is None:
        profile = get_active_profile()
    return os.path.join(TOKENS_DIR, f"{profile}.json")


def list_profiles() -> List[str]:
    """登録済みの全プロファイル名を取得する。"""
    if not os.path.exists(TOKENS_DIR):
        return []
    profiles = []
    for filename in os.listdir(TOKENS_DIR):
        if filename.endswith(".json"):
            profiles.append(filename[:-5])  # .json を除去
    return sorted(profiles)


def load_credentials(
    token_path: str,
    scopes: List[str],
    auto_refresh: bool = True,
) -> Credentials:
    """認証情報を読み込み、必要に応じてリフレッシュする。

    Args:
        token_path: トークンファイルのパス
        scopes: 必要なスコープ
        auto_refresh: 期限切れ時に自動リフレッシュするか

    Returns:
        認証情報

    Raises:
        FileNotFoundError: トークンファイルが存在しない場合
        RefreshError: トークンのリフレッシュに失敗した場合
    """
    path = expand_path(token_path)

    if not os.path.exists(path):
        raise FileNotFoundError(
            f"トークンファイルが見つかりません: {path}\n"
            "「Google 認証して」または「ログインして」と言って認証を行ってください。"
        )

    creds = Credentials.from_authorized_user_file(path, scopes=scopes)

    # 期限切れの場合はリフレッシュを試みる
    if auto_refresh and creds and creds.expired and creds.refresh_token:
        try:
            creds.refresh(Request())
            # リフレッシュ後のトークンを保存
            save_credentials(path, creds)
        except RefreshError as e:
            raise RefreshError(
                f"トークンのリフレッシュに失敗しました。再認証が必要です。\n"
                f"「Google 認証して」と言って再認証を行ってください。\n"
                f"詳細: {e}"
            ) from e

    return creds


def save_credentials(token_path: str, creds: Credentials) -> None:
    """認証情報をファイルに保存する。"""
    path = expand_path(token_path)
    os.makedirs(os.path.dirname(path), exist_ok=True)

    data = {
        "token": creds.token,
        "refresh_token": creds.refresh_token,
        "token_uri": creds.token_uri,
        "client_id": creds.client_id,
        "client_secret": creds.client_secret,
        "scopes": creds.scopes,
    }

    with open(path, "w", encoding="utf-8") as f:
        json.dump(data, f, indent=2)

    # セキュリティ: ファイルパーミッションを制限
    os.chmod(path, 0o600)


def print_error(message: str) -> None:
    """エラーメッセージを標準エラー出力に表示する。"""
    print(f"エラー: {message}", file=sys.stderr)


def print_table(items: List[Dict[str, Any]], headers: List[str]) -> None:
    """データをテーブル形式で出力する。

    Args:
        items: 出力するデータのリスト
        headers: ヘッダー（キー名）のリスト
    """
    if not items:
        print("データがありません。")
        return

    # ヘッダー出力
    print("\t".join(headers))

    # データ出力
    for item in items:
        row = []
        for header in headers:
            value = item.get(header, "")
            # 改行やタブを置換
            if isinstance(value, str):
                value = value.replace("\n", " ").replace("\t", " ")
            row.append(str(value))
        print("\t".join(row))


def print_json(items: Any) -> None:
    """データを JSON 形式で出力する。"""
    print(json.dumps(items, ensure_ascii=False, indent=2))


def format_output(
    items: Any,
    headers: Optional[List[str]] = None,
    output_format: str = "table",
) -> None:
    """指定されたフォーマットでデータを出力する。

    Args:
        items: 出力するデータ
        headers: テーブル形式の場合のヘッダー
        output_format: 出力形式 ("table" or "json")
    """
    if output_format == "json":
        print_json(items)
    else:
        if headers and isinstance(items, list):
            print_table(items, headers)
        else:
            print_json(items)


def retry_with_backoff(
    max_retries: int = 3,
    base_delay: float = 1.0,
    max_delay: float = 60.0,
    retry_on_status: tuple = (429, 500, 502, 503, 504),
) -> Callable:
    """API呼び出しを指数バックオフでリトライするデコレータ。

    Args:
        max_retries: 最大リトライ回数（デフォルト: 3）
        base_delay: 初回リトライまでの待機秒数（デフォルト: 1秒）
        max_delay: 最大待機秒数（デフォルト: 60秒）
        retry_on_status: リトライ対象のHTTPステータスコード

    Returns:
        デコレータ関数

    使用例:
        @retry_with_backoff(max_retries=3)
        def call_api():
            ...
    """
    from googleapiclient.errors import HttpError

    def decorator(func: Callable) -> Callable:
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            last_exception = None
            for attempt in range(max_retries + 1):
                try:
                    return func(*args, **kwargs)
                except HttpError as e:
                    last_exception = e
                    status = e.resp.status

                    # リトライ対象外のステータスはすぐに例外を再送出
                    if status not in retry_on_status:
                        raise

                    # 最後の試行の場合は例外を再送出
                    if attempt >= max_retries:
                        raise

                    # 指数バックオフで待機（1s → 2s → 4s ...）
                    delay = min(base_delay * (2 ** attempt), max_delay)
                    print(
                        f"API呼び出しがステータス {status} で失敗しました。"
                        f"{delay:.1f}秒後にリトライします... (試行 {attempt + 1}/{max_retries})",
                        file=sys.stderr,
                    )
                    time.sleep(delay)

                except (ConnectionError, TimeoutError) as e:
                    last_exception = e
                    # 最後の試行の場合は例外を再送出
                    if attempt >= max_retries:
                        raise

                    # ネットワークエラーもリトライ
                    delay = min(base_delay * (2 ** attempt), max_delay)
                    print(
                        f"ネットワークエラーが発生しました。"
                        f"{delay:.1f}秒後にリトライします... (試行 {attempt + 1}/{max_retries})",
                        file=sys.stderr,
                    )
                    time.sleep(delay)

            # ここには通常到達しないが、念のため
            if last_exception:
                raise last_exception

        return wrapper
    return decorator


def handle_api_error(func):
    """API 呼び出しのエラーハンドリングデコレータ。

    retry_with_backoff と組み合わせて使用する場合は、
    handle_api_error を外側に配置してください。

    使用例:
        @handle_api_error
        @retry_with_backoff()
        def call_api():
            ...
    """
    from googleapiclient.errors import HttpError

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except FileNotFoundError as e:
            print_error(str(e))
            sys.exit(1)
        except RefreshError as e:
            print_error(str(e))
            sys.exit(1)
        except HttpError as e:
            if e.resp.status == 401:
                print_error(
                    "認証エラー: トークンが無効です。\n"
                    "「Google 認証して」と言って再認証を行ってください。"
                )
            elif e.resp.status == 403:
                print_error(
                    f"権限エラー: この操作を実行する権限がありません。\n詳細: {e}"
                )
            elif e.resp.status == 404:
                print_error(f"リソースが見つかりません。\n詳細: {e}")
            elif e.resp.status == 429:
                print_error(
                    "API レート制限に達しました。リトライ後も失敗しました。"
                    "しばらく待ってから再試行してください。"
                )
            else:
                print_error(f"API エラー: {e}")
            sys.exit(1)
        except (ConnectionError, TimeoutError) as e:
            print_error(
                f"ネットワークエラー: 接続に失敗しました。\n"
                f"インターネット接続を確認してください。\n詳細: {e}"
            )
            sys.exit(1)
        except Exception as e:
            print_error(f"予期しないエラーが発生しました: {e}")
            sys.exit(1)

    return wrapper
