#!/usr/bin/env python3
"""
Scan AGENTS.md and extract available skills information.
Outputs a Markdown table to assets/skill_list.md
"""

import re
import sys
from pathlib import Path

def parse_skill_block(skill_xml: str) -> dict:
    """Parse a single <skill> block and extract fields."""
    def extract_tag(tag: str, text: str) -> str:
        match = re.search(rf'<{tag}>(.*?)</{tag}>', text, re.DOTALL)
        return match.group(1).strip() if match else ""
    
    return {
        "name": extract_tag("name", skill_xml),
        "description": extract_tag("description", skill_xml),
        "commands": extract_tag("commands", skill_xml),
        "location": extract_tag("location", skill_xml),
    }

def scan_agents_md(agents_md_path: Path) -> list:
    """Scan AGENTS.md and extract all skills."""
    content = agents_md_path.read_text(encoding="utf-8")
    
    # Extract <available_skills> block
    match = re.search(r'<available_skills>(.*?)</available_skills>', content, re.DOTALL)
    if not match:
        print("Error: <available_skills> block not found in AGENTS.md")
        sys.exit(1)
    
    available_skills_block = match.group(1)
    
    # Extract each <skill> block
    skill_blocks = re.findall(r'<skill>(.*?)</skill>', available_skills_block, re.DOTALL)
    
    skills = []
    for block in skill_blocks:
        skill = parse_skill_block(block)
        if skill["name"]:  # Only add if name exists
            skills.append(skill)
    
    return skills

def generate_markdown(skills: list) -> str:
    """Generate Markdown content from skills list."""
    lines = [
        "# Available Skills",
        "",
        "This document lists all available skills configured in `AGENTS.md`.",
        "",
        "## Skills Summary",
        "",
        "| Name | Command | Description |",
        "| :--- | :--- | :--- |",
    ]
    
    for skill in skills:
        name = skill["name"]
        cmd = skill["commands"]
        desc = skill["description"][:80] + "..." if len(skill["description"]) > 80 else skill["description"]
        lines.append(f"| **{name}** | `{cmd}` | {desc} |")
    
    lines.extend([
        "",
        "## Detailed Information",
        "",
    ])
    
    for skill in skills:
        lines.extend([
            f"### {skill['name']}",
            "",
            f"- **Command**: `{skill['commands']}`",
            f"- **Location**: {skill['location']}",
            f"- **Description**: {skill['description']}",
            "",
        ])
    
    lines.extend([
        "---",
        "",
        "*Generated by skill-list*",
    ])
    
    return "\n".join(lines)

def main():
    # Determine paths
    script_dir = Path(__file__).parent
    skill_dir = script_dir.parent
    project_root = skill_dir.parent.parent.parent  # .claude/skills/skill-list -> project root
    
    agents_md_path = project_root / "AGENTS.md"
    output_path = skill_dir / "assets" / "skill_list.md"
    
    if not agents_md_path.exists():
        print(f"Error: AGENTS.md not found at {agents_md_path}")
        sys.exit(1)
    
    # Scan and parse
    skills = scan_agents_md(agents_md_path)
    
    if not skills:
        print("Warning: No skills found in AGENTS.md")
        sys.exit(0)
    
    # Generate markdown
    markdown_content = generate_markdown(skills)
    
    # Check if content has changed
    output_path.parent.mkdir(parents=True, exist_ok=True)
    content_changed = True
    
    if output_path.exists():
        existing_content = output_path.read_text(encoding="utf-8")
        if existing_content == markdown_content:
            content_changed = False
    
    # Save to file only if content changed
    if content_changed:
        output_path.write_text(markdown_content, encoding="utf-8")
        print(f"✅ Successfully scanned {len(skills)} skills")
        print(f"📄 Output saved to: {output_path}")
    else:
        print(f"✅ Successfully scanned {len(skills)} skills")
        print(f"ℹ️  No changes detected, skipping file update")
    
    # Also print for AI to display to user
    print("\n" + "=" * 50)
    print("SKILL_LIST_OUTPUT_START")
    print(markdown_content)
    print("SKILL_LIST_OUTPUT_END")
    print("=" * 50)

if __name__ == "__main__":
    main()
