---
name: skill-writer
description: Create well-structured Agent Skills for Claude Code. Use when the user asks to create a skill, write a skill, make a skill, build a new skill, or wants to teach Claude how to do something specific. Generates SKILL.md files following official best practices with proper YAML frontmatter, clear instructions, and examples.
allowed-tools:
  - Read
  - Write
  - Glob
  - Grep
  - Bash
  - AskUserQuestion
---

# Skill Writer

## Overview

This Skill helps you create new Agent Skills for Claude Code following official best practices. It generates properly formatted SKILL.md files with YAML frontmatter, clear instructions, and examples that make it easy for Claude to understand when and how to use the skill.

## Prerequisites

- Understanding of what task the new Skill should accomplish
- Access to the project's `.claude/skills/` directory for project Skills, or `~/.claude/skills/` for personal Skills

## Instructions

### Step 1: Analyze the Request and Derive Requirements

**First, derive as much as possible from the user's initial prompt**:

1. **Skill purpose**: What specific task should this Skill accomplish? (usually clear from the request)
2. **When to use**: When should Claude automatically apply this Skill? (infer from the task description)
3. **Scope**: Project-level (`.claude/skills/`) or personal (`~/.claude/skills/`)? (default to personal unless project-specific)
4. **Tools needed**: What tools will the Skill need access to? (infer from the task - e.g., git operations need Bash and Read)
5. **Supporting files**: Will this need additional reference files or scripts? (infer from complexity)

**Only use AskUserQuestion if something is genuinely unclear**:
- Use it to clarify ambiguous requirements when the prompt doesn't provide enough detail
- Use it to choose between equally valid approaches when the preference matters
- Don't ask about things that can be reasonably inferred from context

**Example when to ask**:
- User says "create a skill for database work" but doesn't specify which database → Ask which database system
- User says "review code" but team has specific standards → Ask if there are existing standards to reference

**Example when NOT to ask**:
- User says "create a skill for git commits" → Tools needed are clearly Bash, Read, Grep (no need to ask)
- User says "personal skill" → Scope is clear (no need to ask)

### Step 2: Create the Skill Name

Generate a skill name that:
- Uses only lowercase letters, numbers, and hyphens (no spaces, underscores, or special characters)
- Is descriptive but concise (max 64 characters)
- Matches what the user would naturally say
- Will be used as the directory name
- Should match the directory name exactly

**Examples**: `commit-helper`, `pr-reviewer`, `database-query-v2`, `api-tester`

### Step 3: Write a Clear Description

The description is critical - Claude uses it to decide when to apply the Skill.

**Required elements**:
1. **What it does**: Specific capabilities and actions
2. **When to use**: Trigger keywords users would naturally say
3. **Key features**: What makes this Skill valuable

**Good description pattern**:
```
[Action] [target] [method]. Use when [trigger phrases]. [Key capabilities].
```

**Example**:
```yaml
description: Generate commit messages from git diffs following conventional commit format. Use when the user asks to create a commit, write a commit message, or mentions committing changes. Analyzes staged changes and suggests clear, structured commit messages.
```

### Step 4: Determine Required Tools

List tools the Skill needs access to. Use YAML list format for readability.

**Common tool combinations**:

**File operations**:
```yaml
allowed-tools:
  - Read
  - Write
  - Glob
  - Grep
```

**Git operations**:
```yaml
allowed-tools:
  - Read
  - Bash
  - Grep
```

**Browser automation**:
```yaml
allowed-tools:
  - Read
  - Write
  - mcp__claude-in-chrome__tabs_context_mcp
  - mcp__claude-in-chrome__read_page
  - mcp__claude-in-chrome__get_page_text
  - mcp__claude-in-chrome__computer
```

**Web research**:
```yaml
allowed-tools:
  - WebSearch
  - WebFetch
  - Read
  - Write
```

If no tool restrictions are needed, omit the `allowed-tools` field.

### Step 5: Structure the Instructions

Create clear, actionable step-by-step instructions using this template:

```markdown
## Instructions

### Step 1: [First Action]

[Clear guidance on what to do first]

### Step 2: [Next Action]

[What to do next, with specific details]

### Step 3: [Continue...]

[Keep steps focused and actionable]
```

**Best practices for instructions**:
- Use imperative voice ("Check the file", not "The file should be checked")
- Provide specific examples within steps
- Include error handling or edge cases
- Keep steps focused (one main action per step)
- Use code blocks for technical syntax
- Add warnings for critical requirements

### Step 6: Add Examples

Include at least one complete example showing:
1. User request
2. Skill's process/steps
3. Expected output or result

**Example structure**:
```markdown
## Examples

### Example: [Scenario Name]

**User request**: "[What the user asks for]"

**Process**:
1. [First step taken]
2. [Second step taken]
3. [Result delivered]

**Output**: [What the user receives]
```

### Step 7: Add Supporting Sections

Include these optional but valuable sections:

**Prerequisites** (if needed):
```markdown
## Prerequisites

- [Required tools, packages, or setup]
- [Environment requirements]
```

**Key Principles** (for complex Skills):
```markdown
## Key Principles

- [Core concept 1]
- [Core concept 2]
```

**Common Pitfalls** (for error-prone tasks):
```markdown
## Common Pitfalls to Avoid

1. **[Mistake name]** - [Why it's wrong and how to avoid it]
2. **[Another mistake]** - [Guidance]
```

### Step 8: Create Directory and File

1. Create the Skill directory:
   ```bash
   mkdir -p [path]/[skill-name]
   ```

2. Write the SKILL.md file with all components:
   - YAML frontmatter (name, description, allowed-tools)
   - Overview section
   - Prerequisites (if applicable)
   - Step-by-step instructions
   - Examples
   - Supporting sections

**CRITICAL YAML formatting requirements**:
- The frontmatter MUST start with `---` on line 1 (no blank lines before it)
- The frontmatter MUST end with `---` before the Markdown content
- Use spaces for indentation, NOT tabs
- Invalid YAML will prevent the Skill from loading

**Performance guideline**:
- Keep SKILL.md under 500 lines for optimal performance
- If content exceeds this, split detailed reference material into separate files (see Step 7)

3. Verify the file was created successfully

### Step 9: Validate and Test Recommendations

Inform the user:

1. **Skill created**: `[path]/[skill-name]/SKILL.md`
2. **Verify it loads**: Ask Claude "What Skills are available?"
3. **Test it**: Try a request that matches the Skill's description
4. **Iterate**: Adjust the description if Claude doesn't automatically use the Skill when expected

## Examples

### Example: Creating a Simple Skill

**User request**: "Create a skill that helps me write better README files"

**Process**:

1. **Gather requirements**:
   - Purpose: Generate comprehensive README.md files
   - Trigger: "create readme", "write readme", "generate readme"
   - Scope: Personal skill (useful across all projects)
   - Tools: Read, Write, Glob (to check existing files)

2. **Generate name**: `readme-generator`

3. **Write description**:
   ```yaml
   description: Generate comprehensive README.md files with standard sections. Use when the user asks to create a readme, write a readme, or generate documentation. Includes installation, usage, contributing guidelines, and license sections.
   ```

4. **Structure instructions**:
   - Step 1: Analyze project structure
   - Step 2: Identify key components
   - Step 3: Generate README sections
   - Step 4: Format with proper markdown

5. **Create file**: `.claude/skills/readme-generator/SKILL.md`

6. **Test**: Ask "Create a README for this project"

### Example: Creating a Skill with Multiple Files

**User request**: "Make a skill for code review that checks our team's standards"

**Process**:

1. **Requirements**:
   - Purpose: Review code against team standards
   - Supporting file needed: `standards.md` with team conventions
   - Scope: Project skill (team-specific)

2. **Structure**:
   ```
   .claude/skills/code-reviewer/
   ├── SKILL.md
   └── standards.md
   ```

3. **SKILL.md references standards**:
   ```markdown
   ## Instructions

   ### Step 1: Load Team Standards

   Read [standards.md](standards.md) for the complete list of team conventions.
   ```

4. **Create both files** and inform user how to maintain `standards.md`

## Key Principles

**Clear descriptions win**: The description determines when Claude uses the Skill. Include specific trigger words users would naturally say.

**Progressive disclosure**: Put essential info in SKILL.md, detailed reference in separate files. Claude loads supporting files only when needed.

**Actionable steps**: Instructions should be clear enough that Claude can follow them autonomously without additional user input.

**Examples matter**: Concrete examples help Claude understand expected behavior and output format.

**Keep it focused**: Each Skill should do one thing well. Multiple focused Skills are better than one complex Skill.

**Smart requirement gathering**: Derive requirements from the user's initial prompt first. Only use AskUserQuestion when something is genuinely ambiguous or when choosing between equally valid approaches that require user preference.

## Skill File Template

Use this template as a starting point:

```markdown
---
name: your-skill-name
description: [What it does]. Use when [trigger phrases]. [Key capabilities].
allowed-tools:
  - ToolName1
  - ToolName2
---

# [Skill Display Name]

## Overview

[1-2 sentence summary of what this Skill does and why it's valuable]

## Prerequisites

- [Required tools, packages, or setup]

## Instructions

### Step 1: [Action]

[Clear, specific guidance]

### Step 2: [Next Action]

[More guidance]

## Examples

### Example: [Scenario]

**User request**: "[What user asks]"

**Process**: [How the Skill handles it]

## Key Principles

- [Important concept 1]
- [Important concept 2]
```

## Common Pitfalls to Avoid

1. **Vague descriptions** - "Helps with files" doesn't tell Claude when to use it. Be specific: "Organize project files by type and date. Use when the user asks to organize files, clean up directories, or sort files."

2. **Missing trigger keywords** - Include words users would naturally say in the description.

3. **Too many responsibilities** - One Skill = one clear purpose. Split complex workflows into multiple Skills.

4. **Unclear instructions** - Use specific commands, file paths, and examples. Avoid ambiguous language.

5. **No examples** - Always include at least one complete example showing the Skill in action.

6. **Wrong tool restrictions** - If `allowed-tools` is too restrictive, the Skill won't function. Test thoroughly.

7. **Generic names** - `helper`, `utility`, or `tool` don't communicate purpose. Use descriptive names like `commit-message-generator` or `test-runner`.

8. **Asking unnecessary questions** - Don't use AskUserQuestion to gather information that's already clear from the user's prompt. Only ask when something is genuinely ambiguous. The user expects you to infer reasonable defaults from context.

9. **Invalid YAML formatting** - Forgetting to start with `---` on line 1, using tabs instead of spaces, or having blank lines before the frontmatter will prevent the Skill from loading.

## Reference: YAML Frontmatter Fields

All available fields for SKILL.md frontmatter:

| Field | Required | Description |
|-------|----------|-------------|
| `name` | Yes | Skill name (lowercase letters, numbers, hyphens only, max 64 chars) |
| `description` | Yes | What the Skill does and when to use it (max 1024 chars) |
| `allowed-tools` | No | Tools Claude can use without asking permission |
| `model` | No | Specific model to use (e.g., `claude-sonnet-4-20250514`) |
| `context` | No | Set to `fork` to run in isolated sub-agent context |
| `agent` | No | Agent type when `context: fork` (e.g., `Explore`, `Plan`, `general-purpose`, or custom agent from `.claude/agents/`). Defaults to `general-purpose` if not specified. Only applicable with `context: fork`. |
| `hooks` | No | Lifecycle hooks (PreToolUse, PostToolUse, Stop) |
| `user-invocable` | No | Show in slash menu (default: true). Doesn't affect Skill tool or auto-discovery. |
| `disable-model-invocation` | No | Set to `true` to block Claude from invoking the Skill programmatically via the Skill tool, while still allowing manual `/skill-name` invocation. |

Most Skills only need `name`, `description`, and optionally `allowed-tools`.

**Skill visibility control**:
- `user-invocable: false` - Hides from slash menu, but Claude can still invoke it automatically or via Skill tool
- `disable-model-invocation: true` - Blocks Skill tool invocation, but users can still type `/skill-name`
