---
name: smith-gh-pr
description: GitHub PR workflows including creation, review cycles, merge strategies, and stacked PRs. Use when creating PRs, reviewing code, merging branches, or managing stacked PR workflows. Covers rebase decision trees and AI-generated descriptions.
---

# GitHub PR Workflows

<metadata>

- **Load if**: Creating PRs, reviewing code, merging, stacked PRs
- **Prerequisites**: @smith-principles/SKILL.md, @smith-standards/SKILL.md, `@smith-git/SKILL.md`

</metadata>

## CRITICAL (Primacy Zone)

<required>

- MUST run quality checks before creating PR
- MUST ensure branch is up-to-date before requesting review or merging
- MUST link to related issues
- MUST have all CI checks passing before merge

</required>

## Avoid GitHub MCP

<forbidden>

- GitHub MCP tools (`mcp__github__*`) - hard to control pagination (token waste), less complete than CLI, requires personal token

</forbidden>

**Use instead**: `gh pr-review` extension, `gh api`, or GraphQL queries

## PR Title Format

Follow conventional commits format. See `@smith-style/SKILL.md` for details.

## PR Creation Workflow

<required>

**Pre-PR checklist:**
1. Run linter and formatter
2. Run tests
3. Rebase onto parent branch (not always main - check stacked PRs)
4. Push to remote

</required>

**AI-generated descriptions**: Analyze full diff, read ALL commits, identify tickets, generate structured summary (What/Why/Testing/Dependencies).

## Working on Existing PRs

<required>

- ALWAYS get actual branch name: `gh pr view {PR} --json headRefName`
- ALWAYS check for review comments before making changes
- ALWAYS update PR title/body after pushing changes

</required>

## Code Review Cycle

1. Fetch review comments using `gh pr-review` (see "Fetching Review Comments" below)
2. **Get ALL comments including Nitpicks** - don't skip minor issues
3. Categorize: Actionable > Nitpick > Clarification > Discussion
4. **Proactively audit similar issues** in other files not explicitly mentioned
5. Implement fixes with confidence scoring (high: implement, low: ask)
   - **High confidence**: Small surface area change, aligns with existing patterns, covered by tests
   - **Low confidence**: Ambiguous behavior, architectural impact, requires design discussion
6. Reply to comments with commit SHA
7. **Resolve threads after addressing** - don't leave resolved issues open
8. Re-check for new comments after CI passes

<required>

**Code review response rules:**
- **File-inline comments** (on specific lines): MUST reply in-thread using `gh pr-review comments reply --thread-id {PRRT_xxx}`, NOT as PR-level comment. This keeps discussion traceable to the code location.
- **PR-level comments** (general discussion, `<details>` blocks): Reply with `gh pr comment` or GitHub's "Quote reply"
- Reply with commit SHA, then resolve thread with `gh pr-review threads resolve`
- Proactive audit: search codebase for similar issues before committing
- **CodeRabbit `<details>` comments** (Nitpicks, Duplicated, Outside diff range): These appear in PR thread, not inline on files. Use GitHub's "Quote reply" to include Markdown blockquote of the essential part (e.g., `> The redundant text...`), making response traceable
- Research questionable suggestions before implementing (see `@smith-research/SKILL.md`)

</required>

<forbidden>

- NEVER reply to file-inline comments with `gh pr comment` (loses thread context)
- NEVER mention `@copilot` in replies (triggers unwanted sub-PRs)

</forbidden>

## Fetching Review Comments

<required>

**Use `gh pr-review`** over `gh api` - structured output with thread IDs.

All commands require `--pr {number} -R {owner}/{repo}` for numeric PR selectors.

</required>

**Install**: `gh extension install agynio/gh-pr-review` (consider pinning to vetted SHA)

**On `gh pr-review` errors:**
1. Check if extension installed: `gh extension list | grep pr-review`
2. Verify command syntax (common: missing `--pr`, wrong `-R` format)
3. Verify repo name: `gh repo view --json nameWithOwner`
4. If not installed: `gh extension install agynio/gh-pr-review`

**List unresolved threads**: `gh pr-review threads list --pr {number} -R {owner}/{repo} --unresolved`

### gh-pr-review Commands

**View reviews** (use `--unresolved`, `--reviewer`, `--states` to filter):
`gh pr-review review view --pr {number} -R {owner}/{repo}`

**Reply to thread**:
`gh pr-review comments reply --pr {number} -R {owner}/{repo} --thread-id {PRRT_xxx} --body "..."`

**Resolve/unresolve thread**:
`gh pr-review threads resolve --pr {number} -R {owner}/{repo} --thread-id {PRRT_xxx}`

Output includes `thread_id` (PRRT_xxx format) needed for reply/resolve operations.

### REST API (Single Comments)

URL patterns map to API endpoints:
- `#issuecomment-{id}` → `gh api repos/{owner}/{repo}/issues/comments/{id}`
- `#discussion_r{id}` → `gh api repos/{owner}/{repo}/pulls/comments/{id}`
- `#pullrequestreview-{id}` → `gh api repos/{owner}/{repo}/pulls/{pr}/reviews/{id}`

## Rebase Decision Tree

**Behind base, no conflicts, explicit "update"**: AUTO-REBASE
**Behind base, no conflicts, not explicit**: ASK user
**Behind base, conflicts detected**: INFORM + ASK
**Parent PR merged**: INFORM + OFFER cascade
**About to request review, outdated**: BLOCK + INFORM

**Staleness thresholds**: <5 commits (fresh), 5-10 (notify), >10 (recommend), >20 (urgent)

**Note**: The above decision tree provides guidance during active development. The MUST requirement for up-to-date branches is enforced when requesting review or merging.

## Merge Strategies

**Merge commit**: Feature branches with meaningful history
**Squash**: Small fixes, docs, single logical change
**Rebase**: Linear history required, clean commits

## Stacked PRs

For stacked PRs, merge parent WITHOUT `--delete-branch`, then:
```shell
gh pr edit {CHILD} --base main
git rebase --onto origin/main feat/parent_branch
git push --force-with-lease
git push origin --delete feat/parent_branch
```

## Claude Code Plugin Integration

<context>

**When plugins are available, prefer plugin commands:**

- **`/code-review`**: Launches 4 parallel agents with confidence scoring (threshold 80)
- **`/commit-push-pr`**: Commits, pushes, and creates PR in one step
- **pr-review-toolkit agents**: Specialized review via Task tool
  - `code-reviewer` - CLAUDE.md compliance, style, bugs
  - `silent-failure-hunter` - Error handling issues
  - `pr-test-analyzer` - Test coverage gaps

**Plugin commands complement** (not replace) manual `gh` workflows.

</context>

<related>

- `@smith-gh-cli/SKILL.md` - GitHub CLI commands, pagination limits
- `@smith-stacks/SKILL.md` - Stacked PR workflows
- `@smith-git/SKILL.md` - Git operations, rebase
- `@smith-style/SKILL.md` - Conventional commits, branch naming
- `@smith-tests/SKILL.md` - Testing standards (pre-PR checklist)
- `@smith-research/SKILL.md` - Research best practices before implementing review feedback
- `@smith-validation/SKILL.md` - Debugging, root cause analysis for review issues

</related>

## ACTION (Recency Zone)

**Create PR:**
```shell
gh pr create --title "feat: add feature" --body "..." --assignee @me
```

**Merge PR:**
```shell
gh pr merge {PR} --squash
```

Options: `--squash`, `--merge`, or `--rebase`

**Post-merge cleanup:**
```shell
git checkout main && git fetch --prune origin && git pull
git branch -d feat/my_feature
```

**Check freshness:**
```shell
git fetch origin
BEHIND=$(git rev-list HEAD..origin/main --count)
```
