# Social Media Examples

## Example 1: LinkedIn Post (Thought Leadership)

```markdown
---
type: social_post
platform: linkedin
status: draft
scheduled_time: 2026-01-08T10:00:00Z
content_type: thought_leadership
---

The future of work isn't about AI replacing humans.

It's about humans + AI working together.

After 6 months of building an AI assistant for my business, here's what I learned:

1️⃣ AI handles the repetitive tasks I used to dread
   • Email triage
   • Invoice tracking
   • Social media scheduling

2️⃣ I focus on what humans do best
   • Strategic decisions
   • Client relationships
   • Creative problem-solving

3️⃣ The result?
   • 40% more time for deep work
   • Zero dropped balls
   • Better work-life balance

The key insight: Don't ask "What can AI do?"

Ask: "What should I NOT be doing?"

Then let AI handle those things.

What repetitive tasks are eating up your time?

#FutureOfWork #AI #Productivity #Automation
```

## Example 2: Twitter/X Thread

```markdown
---
type: social_post
platform: twitter
status: draft
scheduled_time: 2026-01-08T14:00:00Z
content_type: thought_leadership
format: thread
---

🧵 I built an AI employee that works 24/7.

Here's what it does (and doesn't do):

---

1/ It monitors my email inbox 📧

When something important comes in, it:
- Summarizes the message
- Drafts a response
- Waits for my approval

I just review and hit send.

---

2/ It tracks my finances 💰

Every transaction is:
- Automatically categorized
- Checked against budget
- Flagged if unusual

I get a weekly CEO briefing every Monday.

---

3/ It manages social media 📱

Posts are:
- Drafted based on my content calendar
- Scheduled for optimal times
- Never posted without my review

The key: Human-in-the-loop for everything important.

---

4/ What it DOESN'T do ❌

- Make decisions for me
- Send messages without approval
- Access anything financial without confirmation

Autonomy with guardrails.

---

5/ The tech stack:

- Claude Code (AI brain)
- Obsidian (knowledge base)
- Python watchers (sensors)
- MCP servers (hands)

All running locally. My data stays mine.

---

Want to build your own?

I'm documenting everything.

Follow for updates 👇

#AI #Automation #BuildInPublic
```

## Example 3: Facebook Business Update

```markdown
---
type: social_post
platform: facebook
status: draft
scheduled_time: 2026-01-08T13:00:00Z
content_type: business_update
---

🎉 Exciting news from our team!

We just hit a major milestone: 100 happy clients served in 2025!

When we started this journey, we had a simple goal: make [service] accessible to small businesses.

Fast forward to today, and we've:
✅ Helped 100 businesses grow
✅ Processed $1M+ in transactions
✅ Maintained a 4.9/5 client satisfaction rating

None of this would be possible without YOU - our amazing community of clients, partners, and supporters.

To celebrate, we're offering 20% off all new projects this month.

Use code: THANKYOU100

Here's to the next 100! 🚀

What would you like to see from us in 2026? Drop a comment below! 👇
```

## Example 4: Instagram Post (Visual-First)

```markdown
---
type: social_post
platform: instagram
status: draft
scheduled_time: 2026-01-08T11:30:00Z
content_type: engagement
---

Your morning routine sets the tone for your entire day. ☀️

Swipe to see the 5 habits that transformed my productivity:

1️⃣ No phone for first 30 minutes
2️⃣ 10-minute journaling session
3️⃣ Quick workout (even just stretching!)
4️⃣ Review today's top 3 priorities
5️⃣ One thing I'm grateful for

Which one are you trying first? Comment below! 👇

---

**Image**: Carousel with 5 slides, each showing one habit with simple graphics

**Hashtags**: #MorningRoutine #Productivity #Entrepreneur #StartupLife #WorkLifeBalance #Motivation #DailyHabits #SuccessMindset #BusinessOwner #PersonalGrowth
```

## Example 5: Content Calendar Generation

```python
# generate_calendar.py
from datetime import datetime, timedelta
from pathlib import Path

def generate_content_calendar(vault_path: str, weeks: int = 4):
    """Generate a social media content calendar."""

    calendar = []
    start_date = datetime.now()

    # Content themes by day
    themes = {
        0: ('LinkedIn', 'thought_leadership', 'Industry insights'),
        1: ('Twitter', 'engagement', 'Community discussion'),
        2: ('LinkedIn', 'business_update', 'Company news'),
        3: ('Twitter', 'thought_leadership', 'Quick tips'),
        4: ('Instagram', 'engagement', 'Weekend inspiration'),
    }

    for week in range(weeks):
        for day_offset in range(5):  # Monday to Friday
            day = start_date + timedelta(weeks=week, days=day_offset)
            day_of_week = day.weekday()

            if day_of_week in themes:
                platform, content_type, topic = themes[day_of_week]
                calendar.append({
                    'date': day.strftime('%Y-%m-%d'),
                    'platform': platform,
                    'content_type': content_type,
                    'topic': topic,
                    'status': 'planned'
                })

    # Generate markdown calendar
    content = f"""# Social Media Calendar
Generated: {datetime.now().isoformat()}

## Upcoming Posts

| Date | Platform | Type | Topic | Status |
|------|----------|------|-------|--------|
"""

    for item in calendar:
        content += f"| {item['date']} | {item['platform']} | {item['content_type']} | {item['topic']} | {item['status']} |\n"

    # Save calendar
    vault = Path(vault_path)
    calendar_file = vault / 'Social_Media_Calendar.md'
    calendar_file.write_text(content)

    print(f"Generated calendar with {len(calendar)} posts")
    print(f"Saved to: {calendar_file}")

    return calendar
```

## Example 6: Post Approval Request

```markdown
---
type: approval_request
action: social_post
platform: linkedin
created: 2026-01-07T14:00:00Z
expires: 2026-01-09T14:00:00Z
status: pending
priority: medium
requestor: social-media-manager
---

## Post Preview

**Platform**: LinkedIn
**Scheduled**: January 8, 2026 at 10:00 AM
**Type**: Thought Leadership

---

[Post content as shown in Example 1]

---

## Checklist
- [x] Grammar and spelling checked
- [x] Hashtags relevant and appropriate
- [x] No sensitive information
- [x] Aligns with brand voice
- [x] Optimal posting time

## Instructions
- **To Approve**: Move to `/Approved/`
- **To Edit**: Modify content, then approve
- **To Reject**: Move to `/Rejected/`
```
