#!/usr/bin/env node

/**
 * SonarCloud Security Triage Script
 *
 * Applies triage decisions from CSV to SonarCloud by updating issue/hotspot statuses.
 *
 * Usage: node apply-triage.mjs <csv-file> [--dry-run]
 *
 * Environment: SONARCLOUD_TOKEN (required)
 *
 * Security:
 *   - API token sourced from environment variable (never hardcoded)
 *   - Token validated at startup before any API calls
 *   - Token never logged or exposed in error messages
 *
 * Idempotency & Error Handling:
 *   - Gracefully handles partial failures (logs error, continues processing)
 *   - Tracks all failures with row number, project, and error details
 *   - SonarCloud APIs are idempotent (safe to re-run on same data)
 *   - Dry-run mode available for safe testing before applying changes
 *   - Comprehensive error reporting at end of batch operation
 */

import { readFileSync } from 'fs';

const SONARCLOUD_BASE_URL = 'https://sonarcloud.io/api';
const SONARCLOUD_TOKEN = process.env.SONARCLOUD_TOKEN;

// Parse command line arguments
const args = process.argv.slice(2);
if (args.length === 0) {
  console.error('❌ Error: CSV file path is required');
  console.error('Usage: node apply-triage.mjs <csv-file> [--dry-run]');
  process.exit(1);
}

const csvFile = args[0];
const dryRun = args.includes('--dry-run');

// Validate token
if (!SONARCLOUD_TOKEN) {
  console.error('❌ Error: SONARCLOUD_TOKEN environment variable is not set');
  console.error('Set it: export SONARCLOUD_TOKEN=your_token_here');
  process.exit(1);
}

/**
 * Sleep for specified milliseconds
 */
function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * Parse CSV file
 */
function parseCSV(filePath) {
  const content = readFileSync(filePath, 'utf-8');
  const lines = content.split('\n').filter(line => line.trim());

  if (lines.length === 0) {
    throw new Error('CSV file is empty');
  }

  // Parse header
  const header = lines[0].split(',').map(h => h.trim().replace(/^"|"$/g, ''));

  // Find required column indices
  const urlIndex = header.indexOf('URL');
  const typeIndex = header.indexOf('Type');
  const actionIndex = header.indexOf('Action');
  const resolutionIndex = header.indexOf('Resolution');
  const commentIndex = header.indexOf('Comment');
  const projectIndex = header.indexOf('Project');

  if (urlIndex === -1 || typeIndex === -1 || actionIndex === -1) {
    throw new Error('CSV must have URL, Type, and Action columns');
  }

  // Parse rows
  const rows = [];
  for (let i = 1; i < lines.length; i++) {
    const line = lines[i];
    if (!line.trim()) continue;

    // Simple CSV parsing (handles quoted fields with commas)
    const values = [];
    let current = '';
    let inQuotes = false;

    for (let j = 0; j < line.length; j++) {
      const char = line[j];

      if (char === '"') {
        inQuotes = !inQuotes;
      } else if (char === ',' && !inQuotes) {
        values.push(current.trim());
        current = '';
      } else {
        current += char;
      }
    }
    values.push(current.trim()); // Last value

    const row = {
      project: projectIndex !== -1 ? values[projectIndex] : '',
      type: values[typeIndex] || '',
      url: values[urlIndex] || '',
      action: values[actionIndex] || '',
      resolution: resolutionIndex !== -1 ? values[resolutionIndex] : '',
      comment: commentIndex !== -1 ? values[commentIndex]?.replace(/^"|"$/g, '') : '',
      rowNumber: i + 1
    };

    rows.push(row);
  }

  return rows;
}

/**
 * Extract issue or hotspot key from URL
 */
function extractKeyFromURL(url, type) {
  if (type === 'SECURITY_HOTSPOT') {
    // Extract from: https://sonarcloud.io/project/security_hotspots?id=...&hotspots=AZPV1fTp...
    const match = url.match(/hotspots=([^&]+)/);
    return match ? match[1] : null;
  } else if (type === 'VULNERABILITY') {
    // Extract from: https://sonarcloud.io/project/issues?open=AZnP1S0b...&id=...
    const match = url.match(/open=([^&]+)/);
    return match ? match[1] : null;
  }
  return null;
}

/**
 * Make authenticated request to SonarCloud API with retry logic
 */
async function callSonarCloudAPI(endpoint, params) {
  const url = new URL(`${SONARCLOUD_BASE_URL}${endpoint}`);

  // Convert params to form data
  const formData = new URLSearchParams();
  Object.entries(params).forEach(([key, value]) => {
    if (value !== undefined && value !== null && value !== '') {
      formData.append(key, value);
    }
  });

  const maxRetries = 3;
  let attempt = 0;

  while (attempt < maxRetries) {
    try {
      const response = await fetch(url.toString(), {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${SONARCLOUD_TOKEN}`,
          'Content-Type': 'application/x-www-form-urlencoded'
        },
        body: formData.toString()
      });

      // Handle rate limiting
      if (response.status === 429) {
        console.warn('   ⚠️  Rate limit hit, waiting 60 seconds...');
        await sleep(60000);
        attempt++;
        continue;
      }

      // Handle errors
      if (!response.ok) {
        const text = await response.text();
        return {
          success: false,
          status: response.status,
          error: text || response.statusText
        };
      }

      return { success: true, status: response.status };

    } catch (error) {
      attempt++;
      if (attempt >= maxRetries) {
        return {
          success: false,
          error: error.message
        };
      }
      console.warn(`   ⚠️  Request failed (attempt ${attempt}/${maxRetries}), retrying...`);
      await sleep(2000 * attempt);
    }
  }
}

/**
 * Update a security hotspot
 */
async function updateHotspot(hotspotKey, resolution, comment, dryRun) {
  if (dryRun) {
    return { success: true, dryRun: true };
  }

  return await callSonarCloudAPI('/hotspots/change_status', {
    hotspot: hotspotKey,
    status: 'REVIEWED',
    resolution: resolution,
    comment: comment
  });
}

/**
 * Update a vulnerability issue
 */
async function updateIssue(issueKey, transition, comment, dryRun) {
  if (dryRun) {
    return { success: true, dryRun: true };
  }

  return await callSonarCloudAPI('/issues/do_transition', {
    issue: issueKey,
    transition: transition,
    comment: comment
  });
}

/**
 * Main execution
 */
async function main() {
  console.log('🔧 SonarCloud Security Triage\n');
  console.log(`Input file: ${csvFile}`);
  console.log(`Dry run: ${dryRun ? 'YES (no changes will be made)' : 'NO'}\n`);

  try {
    // Parse CSV
    console.log('📄 Parsing CSV...');
    const rows = parseCSV(csvFile);
    console.log(`   Total rows: ${rows.length}`);

    // Filter rows with triage decisions (non-empty Action)
    const triageRows = rows.filter(row => row.action && row.action.trim() !== '');
    console.log(`   Rows with triage decisions: ${triageRows.length}\n`);

    if (triageRows.length === 0) {
      console.warn('⚠️  No rows to process (Action column is empty for all rows)');
      console.log('\nTip: Add triage decisions to the Action column');
      process.exit(0);
    }

    // Process each row
    console.log('Processing...\n');
    const results = {
      success: 0,
      failed: 0,
      skipped: 0,
      errors: []
    };

    for (let i = 0; i < triageRows.length; i++) {
      const row = triageRows[i];
      const progress = `[${i + 1}/${triageRows.length}]`;

      // Extract key from URL
      const key = extractKeyFromURL(row.url, row.type);
      if (!key) {
        console.log(`${progress} ⚠️  Row ${row.rowNumber}: Could not extract key from URL (skipped)`);
        results.skipped++;
        results.errors.push({
          row: row.rowNumber,
          project: row.project,
          error: 'Could not extract key from URL'
        });
        continue;
      }

      // Validate and process based on type
      let result;

      if (row.type === 'SECURITY_HOTSPOT') {
        // Validate resolution
        if (!['SAFE', 'FIXED'].includes(row.resolution)) {
          console.log(`${progress} ❌ Row ${row.rowNumber}: Invalid resolution "${row.resolution}" (must be SAFE or FIXED)`);
          results.failed++;
          results.errors.push({
            row: row.rowNumber,
            project: row.project,
            error: `Invalid resolution: ${row.resolution}`
          });
          continue;
        }

        if (dryRun) {
          console.log(`${progress} [DRY RUN] Would update hotspot ${key}:`);
          console.log(`   Project: ${row.project}`);
          console.log(`   Action: Change status to REVIEWED (${row.resolution})`);
          console.log(`   Comment: "${row.comment}"\n`);
          results.success++;
        } else {
          result = await updateHotspot(key, row.resolution, row.comment, dryRun);

          if (result.success) {
            console.log(`${progress} ✅ Hotspot ${key} → REVIEWED (${row.resolution})`);
            results.success++;
          } else {
            console.log(`${progress} ❌ Hotspot ${key} → ${result.status || 'Error'}: ${result.error}`);
            results.failed++;
            results.errors.push({
              row: row.rowNumber,
              project: row.project,
              key: key,
              error: result.error
            });
          }
        }

      } else if (row.type === 'VULNERABILITY') {
        // Validate transition
        const validTransitions = ['confirm', 'falsepositive', 'wontfix', 'resolve'];
        const transition = row.action.toLowerCase();

        if (!validTransitions.includes(transition)) {
          console.log(`${progress} ❌ Row ${row.rowNumber}: Invalid transition "${transition}"`);
          results.failed++;
          results.errors.push({
            row: row.rowNumber,
            project: row.project,
            error: `Invalid transition: ${transition}`
          });
          continue;
        }

        if (dryRun) {
          console.log(`${progress} [DRY RUN] Would update issue ${key}:`);
          console.log(`   Project: ${row.project}`);
          console.log(`   Action: Transition to ${transition}`);
          console.log(`   Comment: "${row.comment}"\n`);
          results.success++;
        } else {
          result = await updateIssue(key, transition, row.comment, dryRun);

          if (result.success) {
            console.log(`${progress} ✅ Issue ${key} → ${transition}`);
            results.success++;
          } else {
            console.log(`${progress} ❌ Issue ${key} → ${result.status || 'Error'}: ${result.error}`);
            results.failed++;
            results.errors.push({
              row: row.rowNumber,
              project: row.project,
              key: key,
              error: result.error
            });
          }
        }

      } else {
        console.log(`${progress} ⚠️  Row ${row.rowNumber}: Unknown type "${row.type}" (skipped)`);
        results.skipped++;
        continue;
      }

      // Small delay to be nice to the API
      if (!dryRun && i < triageRows.length - 1) {
        await sleep(500);
      }
    }

    // Summary
    console.log('\n📊 Summary:');
    console.log(`   ✅ Successfully updated: ${results.success}`);
    console.log(`   ❌ Failed: ${results.failed}`);
    console.log(`   ⏭️  Skipped: ${results.skipped}`);

    if (results.errors.length > 0) {
      console.log('\n❌ Failed updates:');
      results.errors.forEach(err => {
        console.log(`   - Row ${err.row} (${err.project}): ${err.error}`);
      });
    }

    if (dryRun) {
      console.log('\n💡 This was a dry run. Run without --dry-run to apply changes.');
    }

    console.log('\n✅ All done!');

  } catch (error) {
    console.error(`\n❌ Error: ${error.message}`);
    console.error(error.stack);
    process.exit(1);
  }
}

// Run the script
main();
