#!/bin/bash

# space-monitor - Disk space monitoring and management for WSL/Linux
# Part of Claude Code Agent Skills

set -e

# Colors
RED='\033[0;31m'
ORANGE='\033[0;33m'
YELLOW='\033[1;33m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Thresholds (in GB)
CRITICAL_GB=2
WARNING_GB=5
CAUTION_GB=10

# Get available space in GB for a path
get_space_gb() {
    local path=$1
    df -BG "$path" 2>/dev/null | tail -1 | awk '{print $4}' | sed 's/G//'
}

# Get used space in GB
get_used_gb() {
    local path=$1
    df -BG "$path" 2>/dev/null | tail -1 | awk '{print $3}' | sed 's/G//'
}

# Get total space in GB
get_total_gb() {
    local path=$1
    df -BG "$path" 2>/dev/null | tail -1 | awk '{print $2}' | sed 's/G//'
}

# Get percentage used
get_percent() {
    local path=$1
    df -h "$path" 2>/dev/null | tail -1 | awk '{print $5}' | sed 's/%//'
}

# Color code based on available space
get_color() {
    local avail_gb=$1
    if [ "$avail_gb" -lt "$CRITICAL_GB" ]; then
        echo "$RED"
    elif [ "$avail_gb" -lt "$WARNING_GB" ]; then
        echo "$ORANGE"
    elif [ "$avail_gb" -lt "$CAUTION_GB" ]; then
        echo "$YELLOW"
    else
        echo "$GREEN"
    fi
}

# Get status emoji
get_status() {
    local avail_gb=$1
    if [ "$avail_gb" -lt "$CRITICAL_GB" ]; then
        echo "🔴 CRITICAL"
    elif [ "$avail_gb" -lt "$WARNING_GB" ]; then
        echo "🟠 WARNING"
    elif [ "$avail_gb" -lt "$CAUTION_GB" ]; then
        echo "🟡 CAUTION"
    else
        echo "🟢 HEALTHY"
    fi
}

# Check command
cmd_check() {
    echo -e "${BLUE}=== Disk Space Status ===${NC}"
    echo ""

    # Check Windows C: drive if in WSL
    if [ -d "/mnt/c" ]; then
        local c_avail=$(get_space_gb "/mnt/c")
        local c_used=$(get_used_gb "/mnt/c")
        local c_total=$(get_total_gb "/mnt/c")
        local c_percent=$(get_percent "/mnt/c")
        local c_color=$(get_color "$c_avail")
        local c_status=$(get_status "$c_avail")

        echo -e "${c_color}Windows C: Drive${NC}"
        echo -e "  Status: $c_status"
        echo -e "  Available: ${c_color}${c_avail}GB${NC} / ${c_total}GB"
        echo -e "  Used: ${c_used}GB (${c_percent}%)"
        echo ""
    fi

    # Check WSL/Linux root
    local root_avail=$(get_space_gb "/")
    local root_used=$(get_used_gb "/")
    local root_total=$(get_total_gb "/")
    local root_percent=$(get_percent "/")
    local root_color=$(get_color "$root_avail")
    local root_status=$(get_status "$root_avail")

    echo -e "${root_color}WSL/Linux Root${NC}"
    echo -e "  Status: $root_status"
    echo -e "  Available: ${root_color}${root_avail}GB${NC} / ${root_total}GB"
    echo -e "  Used: ${root_used}GB (${root_percent}%)"
    echo ""

    # Quick recommendations
    if [ "$c_avail" -lt "$WARNING_GB" ] 2>/dev/null || [ "$root_avail" -lt "$WARNING_GB" ]; then
        echo -e "${YELLOW}💡 Quick Actions:${NC}"
        echo "  • Run: space-monitor cleanup"
        echo "  • Run: space-monitor analyze"
    fi
}

# Watch command - monitor space during operation
cmd_watch() {
    local command="$*"

    if [ -z "$command" ]; then
        echo "Usage: space-monitor watch <command>"
        exit 1
    fi

    echo -e "${BLUE}=== Monitoring Space During Operation ===${NC}"
    echo "Command: $command"
    echo ""

    # Get initial space
    local c_before=$(get_space_gb "/mnt/c" 2>/dev/null || echo "0")
    local root_before=$(get_space_gb "/")

    echo "Starting space:"
    [ "$c_before" != "0" ] && echo "  C: ${c_before}GB available"
    echo "  Root: ${root_before}GB available"
    echo ""

    # Run command
    eval "$command"
    local exit_code=$?

    echo ""

    # Get final space
    local c_after=$(get_space_gb "/mnt/c" 2>/dev/null || echo "0")
    local root_after=$(get_space_gb "/")

    # Calculate changes
    local c_change=$(awk "BEGIN {print $c_before - $c_after}")
    local root_change=$(awk "BEGIN {print $root_before - $root_after}")

    echo -e "${BLUE}=== Space Usage Summary ===${NC}"
    if [ "$c_after" != "0" ]; then
        echo "C: Drive:"
        echo "  Before: ${c_before}GB"
        echo "  After: ${c_after}GB"
        printf "  Change: %.2fGB\n" "$c_change"
    fi
    echo "Root:"
    echo "  Before: ${root_before}GB"
    echo "  After: ${root_after}GB"
    printf "  Change: %.2fGB\n" "$root_change"

    # Warning if critical
    if command -v bc &>/dev/null && [ "$c_after" != "0" ]; then
        local is_critical=$(echo "$c_after < $CRITICAL_GB" | bc -l 2>/dev/null || echo "0")
        if [ "$is_critical" -eq 1 ]; then
            echo -e "\n${RED}⚠️  WARNING: C: drive critically low!${NC}"
        fi
    fi

    exit $exit_code
}

# Analyze command
cmd_analyze() {
    echo -e "${BLUE}=== Disk Space Analysis ===${NC}"
    echo ""

    # Docker usage
    echo -e "${YELLOW}Docker Resources:${NC}"
    if command -v docker &> /dev/null; then
        docker system df 2>/dev/null || echo "  Docker not running"
    else
        echo "  Docker not installed"
    fi
    echo ""

    # Largest directories in home
    echo -e "${YELLOW}Largest Directories in ~/${NC}"
    du -sh ~/* 2>/dev/null | sort -hr | head -10
    echo ""

    # Node modules
    echo -e "${YELLOW}Node Modules:${NC}"
    local node_count=$(find ~ -name "node_modules" -type d 2>/dev/null | wc -l)
    if [ "$node_count" -gt 0 ]; then
        echo "  Found $node_count node_modules directories"
        find ~ -name "node_modules" -type d -exec du -sh {} \; 2>/dev/null | sort -hr | head -5
    else
        echo "  No node_modules found"
    fi
    echo ""

    # Caches
    echo -e "${YELLOW}Caches:${NC}"
    if [ -d ~/.cache ]; then
        du -sh ~/.cache 2>/dev/null
    fi
    if [ -d ~/.npm ]; then
        du -sh ~/.npm 2>/dev/null
    fi
    echo ""
}

# Cleanup command
cmd_cleanup() {
    local auto_mode=false
    local docker_only=false

    while [ "$#" -gt 0 ]; do
        case $1 in
            --auto) auto_mode=true; shift ;;
            --docker) docker_only=true; shift ;;
            *) shift ;;
        esac
    done

    echo -e "${BLUE}=== Cleanup Recommendations ===${NC}"
    echo ""

    if [ "$docker_only" = false ]; then
        echo -e "${YELLOW}Safe to Clean:${NC}"

        # Check caches
        if [ -d ~/.cache ]; then
            local cache_size=$(du -sh ~/.cache 2>/dev/null | cut -f1)
            echo "  • Clear cache: rm -rf ~/.cache/* (${cache_size})"
        fi

        if [ -d ~/.npm ]; then
            local npm_size=$(du -sh ~/.npm 2>/dev/null | cut -f1)
            echo "  • NPM cache: npm cache clean --force (${npm_size})"
        fi

        # Temp files
        echo "  • Temp files: sudo rm -rf /tmp/*"

        # Old logs
        echo "  • System logs: sudo journalctl --vacuum-time=7d"

        echo ""
    fi

    # Docker cleanup
    if command -v docker &> /dev/null; then
        echo -e "${YELLOW}Docker Cleanup:${NC}"

        # Get Docker stats
        if docker info &>/dev/null; then
            local unused=$(docker system df 2>/dev/null | grep "Build Cache" | awk '{print $4}' || echo "Unknown")
            echo "  • Unused images: docker image prune -a (Reclaim: $unused)"
            echo "  • Stopped containers: docker container prune"
            echo "  • Unused volumes: docker volume prune"
            echo "  • Build cache: docker builder prune"
            echo "  • Everything: docker system prune -a --volumes"
        else
            echo "  Docker not running"
        fi
    fi

    echo ""

    # Auto cleanup if requested
    if [ "$auto_mode" = true ]; then
        echo -e "${GREEN}Running auto-cleanup...${NC}"

        # Safe cleanups
        [ -d ~/.cache ] && rm -rf ~/.cache/* && echo "✓ Cleared ~/.cache"
        command -v npm &>/dev/null && npm cache clean --force &>/dev/null && echo "✓ Cleared npm cache"
        sudo rm -rf /tmp/* 2>/dev/null && echo "✓ Cleared /tmp"

        echo ""
        echo "Auto-cleanup complete! Run 'space-monitor check' to see results."
    fi
}

# Trends command (placeholder)
cmd_trends() {
    echo -e "${BLUE}=== Space Usage Trends ===${NC}"
    echo ""
    echo "Feature coming soon!"
    echo ""
    echo "Will track:"
    echo "  • Historical space usage"
    echo "  • Growth rate"
    echo "  • Time until full prediction"
}

# Help
show_help() {
    cat << EOF
space-monitor - Monitor and manage disk space

Usage:
    space-monitor <command> [options]

Commands:
    check, status              Show current space status
    watch <command>            Monitor space during command execution
    analyze                    Deep dive into space usage
    cleanup [--auto] [--docker] Get cleanup recommendations
    trends [--predict]         Show usage trends (coming soon)
    help                       Show this help

Examples:
    space-monitor check
    space-monitor watch "docker compose build"
    space-monitor analyze
    space-monitor cleanup --auto
    space-monitor cleanup --docker

Thresholds:
    🟢 Healthy:   > ${CAUTION_GB}GB free
    🟡 Caution:   ${WARNING_GB}-${CAUTION_GB}GB free
    🟠 Warning:   ${CRITICAL_GB}-${WARNING_GB}GB free
    🔴 Critical:  < ${CRITICAL_GB}GB free

EOF
}

# Main
main() {
    local command=${1:-help}
    shift || true

    case $command in
        check|status)
            cmd_check
            ;;
        watch)
            cmd_watch "$@"
            ;;
        analyze)
            cmd_analyze
            ;;
        cleanup)
            cmd_cleanup "$@"
            ;;
        trends)
            cmd_trends "$@"
            ;;
        help|--help|-h)
            show_help
            ;;
        *)
            echo "Unknown command: $command"
            echo "Run 'space-monitor help' for usage"
            exit 1
            ;;
    esac
}

main "$@"
