# Atomic Task Extraction Strategies

Strategies for breaking down complex tasks into commits <=250 lines.

## Principles

1. **Logical Completeness**: Each commit should be a complete unit of work
2. **Dependency Order**: Implement dependencies first
3. **Testability**: Each commit should be independently testable
4. **Rollback Safety**: Any commit should be safely revertable

## Extraction Strategies

### By Layer (Architecture)

```
Layer 1: Data models and schemas (~150 lines)
├── models/new_model.py
└── schemas/new_schema.py

Layer 2: Database layer (~100 lines)
├── database/migrations/xxx_migration.py
└── database/repository.py

Layer 3: Business logic (~200 lines)
├── services/new_service.py
└── services/handlers.py

Layer 4: API layer (~200 lines)
├── api/new_router.py
└── api/endpoints.py

Layer 5: UI layer (~250 lines)
├── templates/new_page.html
└── static/new_styles.css
```

### By Feature (Functionality)

```
Commit 1: Core functionality (~200 lines)
├── CoreService class
└── Main business logic

Commit 2: Error handling (~100 lines)
├── Custom exceptions
└── Error handlers

Commit 3: Validation (~150 lines)
├── Input validators
└── Pydantic schemas

Commit 4: Tests (~200 lines)
├── test_core.py
└── test_validators.py
```

### By Dependency

```
Commit 1: Infrastructure (no deps) (~150 lines)
├── config.py
├── base classes
└── constants.py

Commit 2: Core functionality (depends on infra) (~200 lines)
├── services/core.py
└── uses config.py

Commit 3: Integration (depends on core) (~200 lines)
├── api/router.py
└── uses core.py

Commit 4: UI (depends on API) (~250 lines)
├── templates/
└── uses API endpoints
```

## Size Estimation Guide

| File Type | Lines per File | Files per Commit |
|-----------|----------------|------------------|
| Config | 30-50 | 3-5 |
| Models | 50-100 | 2-3 |
| Services | 100-150 | 1-2 |
| API Routers | 80-120 | 1-2 |
| Templates | 50-100 | 2-4 |
| Tests | 100-150 | 1-2 |

## Commit Naming

Use descriptive names that indicate scope:

```
feat(thermocalc): add Pydantic schemas for API
feat(thermocalc): implement HTTP client with retry logic
feat(thermocalc): add /health and /calculate endpoints
feat(thermocalc): create main UI template with HTMX
test(thermocalc): add client and router tests
```

## Example: ThermoCalc Integration Breakdown

```
Total: ~800 lines

Session 1: Config and Schemas (200 lines)
  ├─ config.py - ThermocalcConfig (40)
  ├─ schemas.py - All response models (100)
  ├─ exceptions.py - Exception hierarchy (40)
  └─ dependencies.py - FastAPI DI (20)

Session 2: HTTP Client (150 lines)
  ├─ client.py - ThermocalcClient class (120)
  └─ tests/test_client.py (30)

Session 3: Router Integration (200 lines)
  ├─ router.py - API endpoints (80)
  ├─ events.py - Domain events (50)
  ├─ event_handlers.py - Event handlers (50)
  └─ tests/test_router.py (20)

Session 4: UI Templates (250 lines)
  ├─ index.html - Main page (100)
  ├─ partials/result_table.html (50)
  ├─ partials/error_message.html (30)
  ├─ partials/loading.html (20)
  └─ static/thermocalc.css (50)
```
