# Advanced SQLModel Patterns and Best Practices

## Overview

This document covers advanced patterns and best practices for working with SQLModel, including complex relationships, validation patterns, and performance optimizations.

## Advanced Relationship Patterns

### Self-Referencing Relationships
```python
from sqlmodel import SQLModel, Field, Relationship
from typing import Optional, List

class Category(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    name: str
    parent_id: Optional[int] = Field(default=None, foreign_key="category.id")

    # Self-referencing relationships
    parent: Optional["Category"] = Relationship(
        sa_relationship_kwargs={"remote_side": "Category.id"}
    )
    children: List["Category"] = Relationship(
        back_populates="parent",
        sa_relationship_kwargs={"remote_side": "Category.id"}
    )
```

### Polymorphic Associations
```python
from sqlmodel import SQLModel, Field, Relationship
from typing import Optional

class Tag(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    name: str

class TagAssociation(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    tag_id: int = Field(foreign_key="tag.id")
    entity_type: str  # 'post' or 'product'
    entity_id: int

    tag: Tag = Relationship()
```

### Complex Many-to-Many with Additional Fields
```python
from sqlmodel import SQLModel, Field, Relationship
from datetime import datetime
from typing import List

class Student(SQLModel, table=True):
    id: int = Field(default=None, primary_key=True)
    name: str

class Course(SQLModel, table=True):
    id: int = Field(default=None, primary_key=True)
    name: str

class Enrollment(SQLModel, table=True):
    student_id: int = Field(foreign_key="student.id", primary_key=True)
    course_id: int = Field(foreign_key="course.id", primary_key=True)
    enrolled_at: datetime = Field(default_factory=datetime.utcnow)
    grade: Optional[str] = None

    student: Student = Relationship()
    course: Course = Relationship()

class StudentWithEnrollments(Student):
    enrollments: List[Enrollment] = Relationship(back_populates="student")
```

## Advanced Validation Patterns

### Custom Validators
```python
from sqlmodel import SQLModel, Field
from pydantic import validator
from typing import Optional

class Product(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    name: str
    price: float
    category: str

    @validator("price")
    def price_must_be_positive(cls, v):
        if v <= 0:
            raise ValueError("Price must be positive")
        return v

    @validator("name")
    def name_must_not_be_empty(cls, v):
        if not v.strip():
            raise ValueError("Name cannot be empty or whitespace")
        return v.title()  # Capitalize the name
```

### Complex Field Validation
```python
from sqlmodel import SQLModel, Field
from pydantic import root_validator
from typing import Optional

class Discount(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    percentage: float = Field(ge=0, le=100)  # Percentage between 0-100
    start_date: datetime
    end_date: datetime
    min_amount: Optional[float] = Field(default=None, ge=0)

    @root_validator
    def validate_date_range(cls, values):
        start_date = values.get("start_date")
        end_date = values.get("end_date")

        if start_date and end_date and start_date > end_date:
            raise ValueError("Start date must be before end date")

        return values
```

## Performance Optimization Patterns

### Index Optimization
```python
from sqlmodel import SQLModel, Field
from sqlalchemy import Index
from typing import Optional

class User(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    email: str = Field(unique=True, index=True)  # Both unique constraint and index
    username: str = Field(unique=True, index=True)
    created_at: datetime = Field(default_factory=datetime.utcnow, index=True)
    status: str = Field(index=True)

    __table_args__ = (
        # Composite index for common query patterns
        Index("idx_user_status_created", "status", "created_at"),
        Index("idx_user_email_partial", "email", postgresql_where=Field("status") == "active"),
    )
```

### Eager Loading Patterns
```python
from sqlmodel import Session, select
from sqlalchemy.orm import selectinload

def get_user_with_posts(user_id: int, session: Session):
    """Get user with their posts loaded eagerly to avoid N+1 queries."""
    statement = (
        select(User)
        .options(selectinload(User.posts))  # Eager load posts
        .where(User.id == user_id)
    )
    return session.exec(statement).first()

def get_users_with_posts_multiple(users_ids: list, session: Session):
    """Get multiple users with their posts."""
    statement = (
        select(User)
        .options(selectinload(User.posts))
        .where(User.id.in_(users_ids))
    )
    return session.exec(statement).all()
```

## Async Patterns (Future Support)

While SQLModel currently doesn't have built-in async support, here are patterns for when it's added:

```python
# Future async pattern (not currently available)
from sqlmodel import SQLModel, Field
from sqlmodel.ext.asyncio.session import AsyncSession
from sqlalchemy.ext.asyncio import create_async_engine

class Hero(SQLModel, table=True):
    id: Optional[int] = Field(default=None, primary_key=True)
    name: str

# Async session usage (hypothetical)
async def get_hero_async(hero_id: int, session: AsyncSession):
    statement = select(Hero).where(Hero.id == hero_id)
    result = await session.exec(statement)
    return result.first()
```

## Migration Strategies

### Handling Schema Changes
```python
# For Alembic migrations with SQLModel
from alembic import op
import sqlalchemy as sa

def upgrade():
    # Add a column
    op.add_column('user', sa.Column('phone', sa.String(20)))

    # Add an index
    op.create_index(op.f('ix_user_phone'), 'user', ['phone'])

    # Create a new table
    op.create_table('profile',
        sa.Column('id', sa.Integer(), nullable=False),
        sa.Column('bio', sa.Text()),
        sa.Column('user_id', sa.Integer(), nullable=True),
        sa.ForeignKeyConstraint(['user_id'], ['user.id'], ),
        sa.PrimaryKeyConstraint('id')
    )
```

## Testing Patterns

### Test Isolation with Factories
```python
import factory
from sqlmodel import Session
from .models import User

class UserFactory(factory.Factory):
    class Meta:
        model = User

    name = factory.Faker("name")
    email = factory.Sequence(lambda n: f"user{n}@example.com")
    is_active = True

def test_user_creation():
    user = UserFactory()
    assert user.name
    assert "@" in user.email
```

### Testing with Temporary Database
```python
import tempfile
import pytest
from sqlmodel import create_engine, SQLModel
from sqlalchemy.pool import StaticPool

@pytest.fixture
def test_session():
    """Create a test database session."""
    # Use in-memory SQLite for testing
    engine = create_engine(
        "sqlite:///:memory:",
        connect_args={"check_same_thread": False},
        poolclass=StaticPool
    )
    SQLModel.metadata.create_all(engine)

    with Session(engine) as session:
        yield session

    engine.dispose()
```

## Error Handling Patterns

### Transaction Management
```python
from sqlmodel import Session
from sqlalchemy.exc import IntegrityError

def create_user_safe(user_data: UserCreate, session: Session):
    """Create a user with proper error handling."""
    try:
        user = User(**user_data.dict())
        session.add(user)
        session.commit()
        session.refresh(user)
        return user
    except IntegrityError as e:
        session.rollback()
        # Log the error
        print(f"Integrity error creating user: {e}")
        # Return appropriate error response
        raise ValueError("User already exists or violates constraints")
    except Exception as e:
        session.rollback()
        # Log the error
        print(f"Unexpected error creating user: {e}")
        raise
```

### Validation Error Handling
```python
from pydantic.error_wrappers import ValidationError

def validate_and_create_user(raw_data: dict, session: Session):
    """Validate input data and create user."""
    try:
        # This will raise ValidationError if data is invalid
        user_create = UserCreate(**raw_data)
        return create_user_safe(user_create, session)
    except ValidationError as e:
        # Handle Pydantic validation errors
        error_messages = []
        for error in e.errors():
            field = error['loc'][0]
            message = error['msg']
            error_messages.append(f"{field}: {message}")

        raise ValueError(f"Validation errors: {'; '.join(error_messages)}")
```

## Integration with FastAPI

### Dependency with Error Handling
```python
from fastapi import Depends, HTTPException
from sqlmodel import Session

def get_session() -> Session:
    """Dependency that provides a database session with error handling."""
    with Session(engine) as session:
        try:
            yield session
        except Exception as e:
            session.rollback()
            raise HTTPException(status_code=500, detail=f"Database error: {str(e)}")

def get_active_user(
    user_id: int,
    session: Session = Depends(get_session)
) -> User:
    """Get an active user or raise 404."""
    user = session.get(User, user_id)
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    if not user.is_active:
        raise HTTPException(status_code=400, detail="User is inactive")
    return user
```

These patterns represent advanced SQLModel usage that goes beyond basic CRUD operations, providing solutions for complex data modeling, performance optimization, and robust error handling.