#!/usr/bin/env python3
"""
SQLModel Helper Scripts

This module contains utility functions for common SQLModel operations:
- Model generation
- Database setup
- CRUD operation templates
"""

import argparse
from pathlib import Path
from datetime import datetime


def create_sqlmodel(model_name: str, fields: list, relationships: list = None):
    """
    Generate a SQLModel class definition with specified fields and relationships.

    Args:
        model_name: Name of the model class
        fields: List of tuples (field_name, field_type, field_options)
        relationships: List of relationship definitions
    """
    # Import statements
    imports = ["from sqlmodel import SQLModel, Field"]
    if relationships:
        imports.append("from sqlmodel import Relationship")
        imports.append("from typing import List, Optional")
    else:
        imports.append("from typing import Optional")

    # Build the class definition
    class_def = f"class {model_name}(SQLModel, table=True):\n"

    # Add fields
    if not fields:
        fields = [("id", "Optional[int]", "Field(default=None, primary_key=True)")]

    for field_name, field_type, field_options in fields:
        if field_options:
            class_def += f"    {field_name}: {field_type} = {field_options}\n"
        else:
            class_def += f"    {field_name}: {field_type}\n"

    # Add relationships if any
    if relationships:
        class_def += "\n"
        for rel_name, rel_def in relationships:
            class_def += f"    {rel_name}: {rel_def}\n"

    # Combine imports and class definition
    full_code = "\n".join(imports) + "\n" + class_def

    return full_code


def create_basic_crud(model_name: str, model_file: str = "models.py"):
    """
    Generate basic CRUD operations for a given model.

    Args:
        model_name: Name of the model class
        model_file: Name of the file where the model is defined
    """
    crud_code = f'''from sqlmodel import Session, select
from typing import Optional
from .{model_file.replace(".py", "")} import {model_name}


def create_{model_name.lower()}({model_name.lower()}_data: {model_name}, session: Session) -> {model_name}:
    """Create a new {model_name} record."""
    session.add({model_name.lower()}_data)
    session.commit()
    session.refresh({model_name.lower()})
    return {model_name.lower()}


def get_{model_name.lower()}(session: Session, {model_name.lower()}_id: int) -> Optional[{model_name}]:
    """Retrieve a {model_name} by ID."""
    statement = select({model_name}).where({model_name}.id == {model_name.lower()}_id)
    return session.exec(statement).first()


def get_{model_name.lower()}s(session: Session, offset: int = 0, limit: int = 100) -> list[{model_name}]:
    """Retrieve multiple {model_name} records."""
    statement = select({model_name}).offset(offset).limit(limit)
    return session.exec(statement).all()


def update_{model_name.lower()}(session: Session, {model_name.lower()}_id: int, {model_name.lower()}_data: {model_name}) -> Optional[{model_name}]:
    """Update a {model_name} record."""
    db_{model_name.lower()} = session.get({model_name}, {model_name.lower()}_id)
    if db_{model_name.lower()}:
        # Update fields based on {model_name.lower()}_data
        for field, value in {model_name.lower()}_data.__dict__.items():
            if hasattr(db_{model_name.lower()}, field) and value is not None:
                setattr(db_{model_name.lower()}, field, value)

        session.add(db_{model_name.lower()})
        session.commit()
        session.refresh(db_{model_name.lower()})
        return db_{model_name.lower()}
    return None


def delete_{model_name.lower()}(session: Session, {model_name.lower()}_id: int) -> bool:
    """Delete a {model_name} record."""
    {model_name.lower()} = session.get({model_name}, {model_name.lower()}_id)
    if {model_name.lower()}:
        session.delete({model_name.lower()})
        session.commit()
        return True
    return False
'''
    return crud_code


def create_database_setup(model_imports: list):
    """
    Generate database setup code with engine and session creation.

    Args:
        model_imports: List of model names to import for metadata creation
    """
    imports = ", ".join(model_imports)
    setup_code = f'''from sqlmodel import create_engine, SQLModel
from sqlalchemy import engine


# Database URL - adjust as needed
DATABASE_URL = "sqlite:///./sqlmodel_database.db"
engine = create_engine(DATABASE_URL, echo=True)  # Set echo=False in production


def create_db_and_tables():
    """Create database tables for all SQLModel classes."""
    SQLModel.metadata.create_all(engine)
    print("Database tables created successfully!")


if __name__ == "__main__":
    create_db_and_tables()
'''
    return setup_code


def main():
    parser = argparse.ArgumentParser(
        description="SQLModel Helper Scripts: Generate models, CRUD operations, and database setup code"
    )
    parser.add_argument(
        "action",
        choices=["model", "crud", "database"],
        help="Action to perform: 'model' to generate a model, 'crud' for CRUD operations, 'database' for setup code"
    )
    parser.add_argument("--model-name", help="Name of the model class")
    parser.add_argument("--fields", nargs="+",
                       help="Fields in format: name:type:options (e.g., name:str:'Field(index=True)')")
    parser.add_argument("--model-file", default="models.py",
                       help="Name of the file where the model is defined (for CRUD generation)")
    parser.add_argument("--model-imports", nargs="+",
                       help="List of model names to import (for database setup)")
    parser.add_argument("--output", "-o", help="Output file path (default: print to stdout)")

    args = parser.parse_args()

    result = ""

    if args.action == "model":
        if not args.model_name:
            print("Error: --model-name is required for 'model' action")
            return

        # Parse fields
        fields = []
        if args.fields:
            for field_spec in args.fields:
                parts = field_spec.split(":", 2)  # Split into max 3 parts
                if len(parts) >= 2:
                    name = parts[0]
                    field_type = parts[1]
                    options = parts[2] if len(parts) > 2 else ""
                    fields.append((name, field_type, options))

        result = create_sqlmodel(args.model_name, fields)

    elif args.action == "crud":
        if not args.model_name:
            print("Error: --model-name is required for 'crud' action")
            return

        result = create_basic_crud(args.model_name, args.model_file)

    elif args.action == "database":
        if not args.model_imports:
            print("Error: --model-imports is required for 'database' action")
            return

        result = create_database_setup(args.model_imports)

    if args.output:
        with open(args.output, 'w') as f:
            f.write(result)
        print(f"Generated code saved to {args.output}")
    else:
        print(result)


if __name__ == "__main__":
    import sys
    # Only run examples if no command line arguments provided
    if len(sys.argv) == 1:
        # Example usage
        example_fields = [
            ("id", "Optional[int]", "Field(default=None, primary_key=True)"),
            ("name", "str", "Field(index=True, min_length=1, max_length=100)"),
            ("email", "str", "Field(unique=True, index=True)"),
            ("is_active", "bool", "Field(default=True)")
        ]

        example_code = create_sqlmodel("User", example_fields)
        print("# Example SQLModel generated:")
        print(example_code)

        print("\n# Example CRUD operations:")
        example_crud = create_basic_crud("User")
        print(example_crud)

        print("\n# Example database setup:")
        example_db_setup = create_database_setup(["User"])
        print(example_db_setup)
    else:
        # Parse and execute command line arguments
        main()
