#!/bin/bash
# supabase-helper - Common Supabase operations
# Part of Claude Skills

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Find project with supabase
find_supabase_project() {
    local current_dir="$PWD"
    while [[ "$current_dir" != "/" ]]; do
        if [[ -d "$current_dir/supabase" ]]; then
            echo "$current_dir"
            return 0
        fi
        current_dir=$(dirname "$current_dir")
    done
    
    # Check common locations
    for dir in "$HOME/projects/archon/Archon" "$HOME/projects/te-reo-academy"; do
        if [[ -d "$dir/supabase" ]]; then
            echo "$dir"
            return 0
        fi
    done
    
    echo ""
    return 1
}

PROJECT_ROOT=$(find_supabase_project)

show_usage() {
    cat << EOF
${BLUE}Supabase Helper Tool${NC}

Usage: supabase-helper <command> [args]

Commands:
  reset               Reset database to clean state
  seed [file]         Run seed data (default: all seeds)
  types               Generate TypeScript types from database
  functions           List edge functions
  deploy-fn <name>    Deploy specific edge function
  status              Show Supabase service status
  studio              Open Supabase Studio in browser
  migrations          List all migrations
  migrate-up          Apply pending migrations
  migrate-down        Rollback last migration

Examples:
  supabase-helper reset
  supabase-helper seed
  supabase-helper types
  supabase-helper deploy-fn hello-world
  supabase-helper status

EOF
}

check_supabase() {
    if [[ -z "$PROJECT_ROOT" ]]; then
        echo -e "${RED}Error: No Supabase project found${NC}"
        echo "Please run from a project with a supabase/ directory"
        exit 1
    fi
    
    if ! command -v supabase &> /dev/null; then
        echo -e "${RED}Error: Supabase CLI not installed${NC}"
        echo "Install: brew install supabase/tap/supabase"
        exit 1
    fi
}

run_reset() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${YELLOW}⚠️  This will delete all data and reset the database!${NC}"
    read -p "Are you sure? (yes/no): " confirm
    
    if [[ "$confirm" != "yes" ]]; then
        echo "Cancelled."
        exit 0
    fi
    
    echo -e "${BLUE}Resetting database...${NC}"
    supabase db reset
    echo -e "${GREEN}✓ Database reset complete${NC}"
}

run_seed() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    local seed_file="$1"
    
    if [[ -n "$seed_file" ]]; then
        echo -e "${BLUE}Running seed file: $seed_file${NC}"
        psql "$DATABASE_URL" -f "supabase/seed/$seed_file"
    else
        echo -e "${BLUE}Running database reset (includes seeds)...${NC}"
        supabase db reset
    fi
    
    echo -e "${GREEN}✓ Seeding complete${NC}"
}

run_types() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${BLUE}Generating TypeScript types...${NC}"
    
    # Determine output location
    local output_dir="src/types"
    if [[ -d "archon-ui-main/src/types" ]]; then
        output_dir="archon-ui-main/src/types"
    elif [[ -d "src/types" ]]; then
        output_dir="src/types"
    else
        mkdir -p src/types
        output_dir="src/types"
    fi
    
    supabase gen types typescript --local > "$output_dir/supabase.ts"
    echo -e "${GREEN}✓ Types generated at $output_dir/supabase.ts${NC}"
}

run_functions() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${BLUE}=== Edge Functions ===${NC}\n"
    
    if [[ -d "supabase/functions" ]]; then
        ls -1 supabase/functions/ | grep -v "^_" || echo "No functions found"
    else
        echo "No functions directory found"
    fi
}

run_deploy_fn() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    local fn_name="$1"
    
    if [[ -z "$fn_name" ]]; then
        echo -e "${RED}Error: Function name required${NC}"
        echo "Usage: supabase-helper deploy-fn <function-name>"
        exit 1
    fi
    
    echo -e "${BLUE}Deploying function: $fn_name${NC}"
    supabase functions deploy "$fn_name"
    echo -e "${GREEN}✓ Function deployed${NC}"
}

run_status() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${BLUE}=== Supabase Status ===${NC}\n"
    supabase status
}

run_studio() {
    check_supabase
    
    local studio_url="http://localhost:54323"
    
    echo -e "${BLUE}Opening Supabase Studio...${NC}"
    echo -e "${GREEN}URL: $studio_url${NC}"
    
    if command -v xdg-open &> /dev/null; then
        xdg-open "$studio_url" 2>/dev/null &
    elif command -v open &> /dev/null; then
        open "$studio_url" 2>/dev/null &
    else
        echo -e "${YELLOW}Please open manually: $studio_url${NC}"
    fi
}

run_migrations() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${BLUE}=== Migrations ===${NC}\n"
    
    if [[ -d "supabase/migrations" ]]; then
        ls -1 supabase/migrations/
    else
        echo "No migrations found"
    fi
}

run_migrate_up() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${BLUE}Applying migrations...${NC}"
    supabase db push
    echo -e "${GREEN}✓ Migrations applied${NC}"
}

run_migrate_down() {
    check_supabase
    cd "$PROJECT_ROOT"
    
    echo -e "${YELLOW}Rolling back last migration...${NC}"
    echo -e "${RED}Note: This requires manual SQL or db reset${NC}"
    echo "Use: supabase db reset (full reset)"
}

# Main command router
COMMAND="${1:-}"

case "$COMMAND" in
    reset)
        run_reset
        ;;
    seed)
        run_seed "$2"
        ;;
    types)
        run_types
        ;;
    functions)
        run_functions
        ;;
    deploy-fn)
        run_deploy_fn "$2"
        ;;
    status)
        run_status
        ;;
    studio)
        run_studio
        ;;
    migrations)
        run_migrations
        ;;
    migrate-up)
        run_migrate_up
        ;;
    migrate-down)
        run_migrate_down
        ;;
    ""|-h|--help|help)
        show_usage
        ;;
    *)
        echo -e "${RED}Error: Unknown command '$COMMAND'${NC}\n"
        show_usage
        exit 1
        ;;
esac
