# コンテンツ最適化ガイド

本文のSEO最適化と読みやすさを向上させるための実践ガイドです。

## 見出し構造の最適化

### H1-H6の階層ルール

```markdown
# 記事タイトル (H1) - ページに1つのみ

## メインセクション (H2)
### サブセクション (H3)
#### 詳細項目 (H4)
##### 補足説明 (H5)
###### 注釈レベル (H6)

## 次のメインセクション (H2)
```

**ベストプラクティス:**
- H1は1ページに1つだけ
- 階層を飛ばさない（H2→H4は NG）
- H2-H3を中心に構成（H5-H6は稀に使用）
- 各見出しにキーワードまたは関連語を含める

### テックブログ向け見出し構造例

```markdown
# React Hooks完全ガイド｜useState/useEffectの使い方と実践例7選 (H1)

## React Hooksとは？ (H2)
### React Hooksが登場した背景 (H3)
### React Hooksのメリット (H3)

## useStateの使い方 (H2)
### useStateの基本構文 (H3)
#### 実装例: カウンター機能 (H4)
#### 実装例: フォーム入力 (H4)
### useStateの注意点 (H3)

## useEffectの使い方 (H2)
### useEffectの基本構文 (H3)
#### 実装例: データフェッチ (H4)
#### 実装例: イベントリスナー (H4)

## カスタムフックの作成 (H2)
### カスタムフックとは (H3)
### 実践例: useLocalStorage (H3)

## まとめ (H2)

## よくある質問 (H2)
### Q1: useStateとuseReducerの違いは？ (H3)
### Q2: useEffectのクリーンアップとは？ (H3)
```

## キーワード最適化

### キーワード密度

**推奨密度: 1-3%**

```
記事全体の文字数: 3000文字
主要キーワード「React Hooks」: 30-90回程度

計算式:
キーワード密度 = (キーワード出現回数 ÷ 総単語数) × 100
```

### キーワード配置戦略

**重要な配置場所:**

1. **タイトル** (H1)
2. **導入部分** (最初の100単語以内)
3. **主要見出し** (H2-H3)
4. **結論部分**
5. **メタディスクリプション**
6. **URL**
7. **画像のalt属性**

**例:**
```markdown
# React Hooks完全ガイド｜useState/useEffectの使い方

React Hooksは、React 16.8で導入された新機能です。
本記事では、React Hooksの基本から実践的な使い方まで、
useState、useEffectを中心に徹底解説します。

## React Hooksとは？

React Hooksを使うことで、関数コンポーネントでも
状態管理やライフサイクル機能を扱えるようになりました...
```

### 共起語(LSIキーワード)の活用

主要キーワード「React Hooks」の共起語:
- useState、useEffect、useContext
- 関数コンポーネント、クラスコンポーネント
- 状態管理、副作用
- カスタムフック
- React 16.8、React 18

**これらを自然に文章に含める**

## 段落構成と読みやすさ

### 1段落の長さ

```markdown
✅ 適切な長さ (3-5文、80-120文字程度)
React Hooksは、関数コンポーネントで状態管理を行える機能です。
従来のクラスコンポーネントと比べて、コードがシンプルになります。
本記事では、基本的な使い方を実例で解説します。

❌ 長すぎる (10文以上、200文字超)
React Hooksは、関数コンポーネントで状態管理を行える機能です。従来のクラスコンポーネントでは...という問題がありました。しかし、Hooksを使うことで...が可能になりました。具体的には...があり、さらに...という利点もあります。また...
```

### スキャン可能な構造

**箇条書きの活用:**
```markdown
React Hooksの主なメリット:
- コード量が削減される
- ロジックの再利用が容易
- テストがシンプルになる
- 学習コストが低い
```

**番号付きリストの活用:**
```markdown
useStateの使い方（3ステップ）:
1. ReactからuseStateをインポート
2. コンポーネント内でuseStateを呼び出す
3. 状態値と更新関数を使用する
```

### コードブロックの最適化

**シンタックスハイライト付き:**
````markdown
```javascript
import { useState } from 'react';

function Counter() {
  const [count, setCount] = useState(0);

  return (
    <div>
      <p>カウント: {count}</p>
      <button onClick={() => setCount(count + 1)}>
        +1
      </button>
    </div>
  );
}
```
````

**コメント付きコード:**
````markdown
```javascript
// 状態の初期化
const [count, setCount] = useState(0);

// イベントハンドラ
const handleClick = () => {
  setCount(count + 1); // カウントを1増やす
};
```
````

## 導入部分の最適化

### PASONA の法則

**Problem (問題)**: 読者の課題を提示
```markdown
React Hooksの使い方で悩んでいませんか？
useStateとuseEffectの違いが分からない...
```

**Agitation (煽動)**: 問題の深刻さを強調
```markdown
正しく理解していないと、無駄なレンダリングやメモリリークが
発生し、アプリのパフォーマンスが低下します。
```

**Solution (解決策)**: 記事が提供する解決策
```markdown
本記事では、React Hooksの基本から実践まで、
実例7つで分かりやすく解説します。
```

**Offer (提案)**: 具体的な内容
```markdown
useState、useEffect、カスタムフックの使い方を学び、
効率的なReact開発ができるようになります。
```

**Narrowing Down (絞込)**: 対象者の明確化
```markdown
初心者から中級者まで、すぐに実践できる内容です。
```

**Action (行動)**: 読み進めるよう促す
```markdown
それでは、React Hooksの世界へ飛び込みましょう！
```

## 内部リンクの最適化

### 自然な文脈でのリンク

```markdown
✅ 自然なアンカーテキスト
React Hooksをさらに深く学びたい方は、
[カスタムフックの実践的な使い方](./custom-hooks-guide)も
ぜひご覧ください。

❌ 不自然なアンカーテキスト
詳細は[こちら](./custom-hooks-guide)をクリック。
```

### 関連記事セクション

```markdown
## 関連記事

本記事と合わせて読みたい記事:

- [React 18の新機能完全ガイド](./react-18-guide)
  - Concurrent Rendering、Suspenseの使い方を解説
- [React パフォーマンス最適化](./react-performance)
  - useMemo、useCallbackによる最適化手法
- [TypeScript + React開発](./typescript-react)
  - 型安全なReact開発のベストプラクティス
```

## 画像・図解の最適化

### alt属性の最適化

```html
✅ 説明的なalt属性
<img src="react-hooks-usestate-example.png"
     alt="React HooksのuseStateを使ったカウンター実装例のコードスニペット">

❌ 簡略すぎるalt属性
<img src="image1.png" alt="画像">
```

### 図解・スクリーンショットの活用

```markdown
## useStateのライフサイクル

![useStateのライフサイクル図](./images/usestate-lifecycle.png)
*useStateの状態更新フロー: コンポーネント → useState → レンダリング*

上図のように、useStateは...
```

## 表の活用

**重要:** noteでは表（マークダウンテーブル）が使用できません。note用の記事では箇条書きで代替してください。

**一般的なブログプラットフォーム（はてな、WordPress等）:**
```markdown
## Hooksの比較

| Hook名 | 用途 | 主なユースケース |
|--------|------|------------------|
| useState | 状態管理 | フォーム入力、トグル状態 |
| useEffect | 副作用処理 | データフェッチ、イベント登録 |
| useContext | コンテキスト利用 | グローバル状態の参照 |
| useReducer | 複雑な状態管理 | 複数の状態を持つフォーム |
```

**note用の代替形式:**
```markdown
## Hooksの比較

**useState（状態管理）**
- 主なユースケース: フォーム入力、トグル状態

**useEffect（副作用処理）**
- 主なユースケース: データフェッチ、イベント登録

**useContext（コンテキスト利用）**
- 主なユースケース: グローバル状態の参照

**useReducer（複雑な状態管理）**
- 主なユースケース: 複数の状態を持つフォーム
```

## 結論部分の最適化

### まとめセクション

```markdown
## まとめ

本記事では、React Hooksの基本から実践まで解説しました。

**重要ポイント:**
- useState: 状態管理の基本
- useEffect: 副作用処理とクリーンアップ
- カスタムフック: ロジックの再利用

**次のステップ:**
1. 実際にコードを書いて動作確認
2. カスタムフックを作成してみる
3. パフォーマンス最適化を学ぶ

React Hooksを使いこなして、効率的な開発を実現しましょう！
```

## チェックリスト

### 見出し構造
- [ ] H1が1つだけ
- [ ] 階層構造が論理的
- [ ] 主要見出しにキーワード含有

### キーワード
- [ ] 密度1-3%
- [ ] 導入部分に含まれる
- [ ] 自然な使用
- [ ] 共起語を含む

### 読みやすさ
- [ ] 1段落3-5文
- [ ] 箇条書き活用
- [ ] コードブロックにハイライト
- [ ] 適切な改行と空白

### リンク・画像
- [ ] 内部リンク3-5個
- [ ] アンカーテキストが説明的
- [ ] 画像にalt属性
- [ ] 図解・表を活用
