# Transaction Token Reference Pattern

## Overview
Transaction YAML files use a dot notation system to reference tokens and validators defined in `public/yaml/validator-registry-v1.yaml`. This creates a clean separation between logical references and actual blockchain implementation details.

## Token Reference Format
```
<system>.<token-name>
```
Maps to registry path: `systems > <system> > tokens > <token-name>`

### Examples:
- `global-state.access-token-user` → `systems > global-state > tokens > access-token-user`
- `index-validators.index-scripts-ref-token` → `systems > index-validators > tokens > index-scripts-ref-token`
- `course.course-state-token` → `systems > course > tokens > course-state-token`
- `project.contributor-state-token` → `systems > project > tokens > contributor-state-token`

## Validator Reference Format
```
<system>.<validator-name>
```
Maps to registry path: `systems > <system> > validators > <validator-name>`

### Examples:
- `index-validators.index-scripts` → `systems > index-validators > validators > index-scripts`
- `global-state.global-state` → `systems > global-state > validators > global-state`
- `project.treasury-scripts` → `systems > project > validators > treasury-scripts`

## Value Field Format
All value fields must use bullet-point list format:
```yaml
value:
  - "5000000 lovelace"
  - "1 global-state.access-token-user"
```

## Required Comment Header
All transaction files should include this comment at the top:
```yaml
# Token references use dot notation to map to validator-registry-v1.yaml
# Format: <system>.<token-name> maps to registry path: systems > <system> > tokens > <token-name>
```

## Minting Policy Rules
**Critical**: In minting operations, use actual policy ID placeholders (e.g., `<course_nft_policyid>`) that match the asset-id in the registry for tokens being minted. For all other references, use dot notation.

### Example:
```yaml
mints:
  - id: course-nft
    policy: "<course_nft_policyid>"  # Use policy ID placeholder
    tokens:
      - '1 course.course-nft'        # Use dot notation for token reference
```

## UTXO Conservation Rule
If there's an input token, it must appear in outputs unless it's being burned. Always add wallet outputs for input tokens that aren't consumed.

### Example:
```yaml
inputs:
  - id: instance_admin
    value: 
      - "1 instance.instance-admin-token"

outputs:
  - id: instance_admin  # Must include unless token is burned
    value:
      - "1 instance.instance-admin-token"
```

## Files Successfully Updated
1. **access-token-mint.yaml** ✅ - Template/reference file
2. **admin-init-course.yaml** ✅ - Course initialization
3. **admin-init-project-step-1.yaml** ✅ - Project init step 1
4. **admin-init-project-step-2.yaml** ✅ - Project init step 2
5. **admin-add-course-creators.yaml** ✅ - Add course creators
6. **admin-add-project-creators.yaml** ✅ - Add project creators

## Pattern Verification Process
For each file:
1. Add header comment
2. Update all token references to dot notation
3. Update validator addresses to dot notation
4. Fix minting policies to use `<policy_id>` placeholders
5. Update observer references to include `-cbor-obs`
6. Verify UTXO conservation (inputs → outputs)
7. Use consistent placeholder naming (e.g., `<alias_to_add>`, `<current_creator_1>`)
8. Cross-reference all names against registry

## Benefits
- **Maintainability**: Asset IDs and validator addresses centralized in registry
- **Readability**: Semantic names instead of cryptic IDs  
- **Type safety**: Dot notation clearly indicates hierarchy
- **Environment flexibility**: Different environments can use different implementations
- **UTXO Accuracy**: Proper token flow tracking

## Implementation Notes
1. The registry contains actual asset IDs with placeholders (e.g., `<access_token_policyid>.u<alias>`)
2. Transaction files use dot notation for references, policy ID placeholders only in minting
3. All token/validator references must exist in the registry
4. The registry serves as the source of truth for all validators and tokens
5. Observer names typically end in `-cbor-obs`
6. Use consistent placeholder naming: `<alias_to_add>`, `<current_creator_1>`, `<creator_alias_a>`, etc.

## Transaction Type Patterns
- **State-change transactions** (add/remove creators): No minting, just datum updates + UTXO conservation
- **Initialization transactions**: Mint new tokens + reference existing instance tokens  
- **Simple access management**: Input/output token conservation with datum changes